(* :Title: Graph Plot *)

(* :Context: DiscreteMath`GraphPlot` *)

(* :Author: Yifan Hu *)

(* :Summary: A collection of graph theory related function,
    including functions for aesthetic plotting of graphs *)

(* :Copyright: Copyright 2004-2005, Wolfram Research, Inc. *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 5.1 *)

(* :History:
    Version 1.0, Oct. 2004, Yifan Hu.
*)

(* :Keywords:
Graph theory.
*)

(* :Sources:

*)

(* :Warnings: *)

(* :Limitations:  *)

(* :Discussion:

*)

BeginPackage["DiscreteMath`GraphPlot`",{"DiscreteMath`Combinatorica`","Graphics`Arrow`"}];

GraphPlot::usage = "GraphPlot[g, options] calculates a visually appealing 2D layout of the vertices of a graph g and plots the graph using this layout.";

GraphPlot3D::usage = "GraphPlot3D[g, options] calculates a visually appealing 3D layout of the vertices of a graph g and plots the graph using this layout.";

GraphCoordinates::usage = "GraphCoordinates[g, options] calculates a visually appealing 2D layout of the vertices of a graph g and returns the coordinates of the vertices.";

GraphCoordinates3D::usage = "GraphCoordinates3D[g, options] calculates a visually appealing 3D layout of the vertices of a graph g and returns the coordinates of the vertices.";

TreePlot::usage = "TreePlot[g, options] generates a plot of graph g as a tree. TreePlot[g, root, options] generates a plot of the graph g as a tree, with vertex r as the root of the tree.";

GraphDistance::usage="GraphDistance[g, i, j] calculates the distance between vertices i and j in a graph g. If there is no path from i to j, the distance is Infinity";

MaximalBipartiteMatching::usage="MaximalBipartiteMatching[g] returns the maximal matching of the bipartite graph as represented by a matrix g. For a m \[Times] n matrix, the output is a list of index pairs, of form {{i_1, j_1}, ..., {i_k, j_k},...} where 1 <= i_k <= m and 1 <= j_k <= n";

MaximalIndependentVertexSet::usage="MaximalIndependentVertexSet[g] gives a maximal independent vertex set of an undirected graph g.";

MaximalIndependentEdgeSet::usage="MaximalIndepndentEdgeSet[g] returns a maximal independent edge set, also known as maximal matching, of an undirected graph g.";

MinCut::usage="MinCut[g,k] gives the minimal partitioning of graph into k parts. The graph is represented by an undirected graph g. The partition groups the vertices into k groups, such that the number of edges between these vertices are minimized approximately.";

PseudoDiameter::usage="PseudoDiameter[g] gives the pseudo diameter of the undirected graph g, and the two vertices that achieve this diameter. If the graph is disconnected, then the diameter for each of the component, together with the indices of the two vertices that achieves this diameter, are returned.";

StrongComponents::usage="StrongComponents[g] returns a list of all strongly connected components in the directed graph as represented by a matrix g.";

VertexList::usage="VertexList[g] returns a list of all vertices."

GraphToSparseArray::usage="GraphToSparseArray[g] returns the sparse array representing the graph g, with g in the Combinatorica format";

(*================ Error messages =========== *)

StrongComponents::sqma = GraphDistance::sqma = "`1` is not a square matrix";

MaximalIndependentVertexSet::vtxwgt = "`1` is not a list of length equal to the number of vertices.";

MaximalIndependentVertexSet::rug = MinCut::rug = PseudoDiameter::rug = MaximalIndependentEdgeSet::rug = "Argument `1` at position 1 does not represent an undirected graph.";

MaximalIndependentEdgeSet::wgt = "The value of option Weighted->`1` must be either True or False.";

MaximalIndependentEdgeSet::symat = "`1` is not a symmetric matrix.";

MinCut::kgtwo = "The value of the second argument, `1`, must be an integer >= 1";
GraphDistance::iind = TreePlot::iind = "Argument `1` at position `3` is not a vertex index between 1 and `2`.";
GraphDistance::rind = TreePlot::rind = "Argument `1` at position `2` is not a valid vertex.";

Options[MaximalIndependentEdgeSet] = {DiscreteMath`GraphPlot`Weighted->False};
Options[PseudoDiameter] = {DiscreteMath`GraphPlot`Aggressive->False};

Begin["`Private`"];

(*  option processing *)
StringName[sym_Symbol] := SymbolName[sym];
StringName[name_] := name;
SetAttributes[processOptionNames, Listable];
processOptionNames[(r : (Rule | RuleDelayed))[name_Symbol, val_]] := 
    r[SymbolName[name], val];
processOptionNames[opt___] := opt;
filterOptions[hiddenOpts_List, command_Symbol, options___?OptionQ] := 
  (filterOptions[
   First /@ processOptionNames[Flatten[{Options[command], hiddenOpts}]], 
   options]);
filterOptions[command_Symbol, options___?OptionQ] := 
  (filterOptions[{}, command, options]);
filterOptions[optnames_List, options___?OptionQ] := 
  (Select[Flatten[{options}], 
   MemberQ[optnames, StringName[First[#]]] &]);

DeleteOptions[optnames_List, opts___] := 
 Select[
   Flatten[{opts}], !MemberQ[optnames, StringName[First[#]]]&]

orderOptions[opts_]:= Module[{x}, x=Ordering[Map[StringName[#[[1]]]&, opts]]; opts[[x]]];

gopt = Flatten[{DeleteOptions[{"PlotRange", "AspectRatio"}, Options[Graphics]], Options[SparseArray`GraphPlacement]/.{"VertexStyleFunction"->DiscreteMath`GraphPlot`VertexStyleFunction, "EdgeStyleFunction"->DiscreteMath`GraphPlot`EdgeStyleFunction,"VertexCoordinates"->DiscreteMath`GraphPlot`VertexCoordinates}, 
PlotRange->All, AspectRatio->Automatic}];

Options[GraphPlot] = orderOptions[gopt];

gopt3 = Flatten[{DeleteOptions[{"PlotRange", "AspectRatio"}, Options[Graphics3D]], Options[SparseArray`GraphPlacement3D]/.{"VertexStyleFunction"->DiscreteMath`GraphPlot`VertexStyleFunction, "EdgeStyleFunction"->DiscreteMath`GraphPlot`EdgeStyleFunction,"VertexCoordinates"->DiscreteMath`GraphPlot`VertexCoordinates}, 
PlotRange->All, AspectRatio->Automatic}];

Options[GraphPlot3D] = orderOptions[gopt3];

topt = Flatten[{DeleteOptions[{"Method","RandomSeed"}, Options[GraphPlot]], {DiscreteMath`GraphPlot`RootPosition->Top, DiscreteMath`GraphPlot`TreeSizeFunction->(1&)}}];

Options[TreePlot] = orderOptions[topt];


indhash = .;

GraphToSparseArray[G_DiscreteMath`Combinatorica`Graph]:= G2S[G];

RuleListQ[x_] := (ListQ[x] && Length[x]>0 &&
   Apply[And, 
    Map[(Head[#] === Rule)&, 
     x, {1}]]);

G2S[A_?MatrixQ] := A;

(* this generate the list of vertices from a rule list *)
VertexList[G_DiscreteMath`Combinatorica`Graph]:= Module[
   {l, n},
   l = DiscreteMath`Combinatorica`ToAdjacencyLists[G, EdgeWeight];
   n = Length[l];
   Table[i, {i, n}]
]
VertexList[G_?MatrixQ]:= Module[
   {n},
   n = Dimensions[G][[1]];
   Table[i, {i, n}]
]
VertexList[G_] := Module[{n, index, hash, res},
    (*rule list case*)
    Clear[hash];
    index[x_] := If[hash[x] =!= 1, hash[x] = 1; Sow[x], {}];
    res = Reap[Map[index, Flatten[Map[List @@ # &, G], 1]]];
    res[[2,1]]
]

setHash[G_]:= Module[{f, index, ind, nz = 1},
   (* set up hashing based on a rule list. 
      indhash is cached so other function can also use it. *)
   Clear[indhash];
   index[x_] := If[NumberQ[indhash[x]], indhash[x], indhash[x] = nz++];
   f[Rule[x_, y_]] := {index[x], index[y]};
   ind = Map[f[#] &, G]
];

G2S[G_] := 
  Module[{ind, n, a, b}, 
   ind = setHash[G];
   n = Max[ind];
   SparseArray[{ind -> Table[1, {Length[ind]}]}, {n, n}]
];

RulesToSparseArrayWithLabels[G_] := 
  Module[{ind, n, hash, f, a, b, nz = 1,labels}, 
   Clear[hash];
   labels = {};
   index[x_] := If[NumberQ[hash[x]], hash[x], 
           labels = {labels, x}; hash[x] = nz++];
   f[Rule[x_, y_]] := {index[x], index[y]};
   ind = Map[f[#] &, G];
   n = Max[ind];
   {SparseArray[{ind -> Table[_, {Length[ind]}]}, {n, n}],Flatten[labels]}
];


G2S[G_DiscreteMath`Combinatorica`Graph]:= Module[
   {l, n, ind,row},
   row[i_Integer,neighb_List]:= Map[({i,#[[1]]}->#[[2]])&, neighb];
   l = DiscreteMath`Combinatorica`ToAdjacencyLists[G,EdgeWeight];
   n = Length[l];
   ind = Thread[row[Table[i,{i,n}], l]];
	SparseArray[Flatten[ind],{n,n}]
];

GraphPlot[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[G2S[G], 2, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[G2S[G], 2, "Coordinates", opts]},
   res/;(res =!= $Failed)
];

GraphPlot[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 2, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 2, "Coordinates", opts]},
   res/;(res =!= $Failed)
];

GraphPlot[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[A, 2, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[A, 2, "Coordinates", opts]},
   res/;(res =!= $Failed)
];

GraphPlotInternalRuleList23D[G_, dim_Integer, task_String, opts___?OptionQ]:= Module[
   {},
   If [dim == 2,
     GraphPlotInternalRuleList23D[GraphPlot, G, dim, task, opts],
     GraphPlotInternalRuleList23D[GraphPlot3D, G, dim, task, opts]
   ]
]

GraphPlotInternalRuleList23D[caller_, G_, dim_Integer, task_String, opts___?OptionQ]:= Module[
   {A, labels, vtxfun, vlabelThreshold, n},
   {A, labels} = RulesToSparseArrayWithLabels[G];
   (* if VertexStyleFunction->Automatic/All, plot 
      the vertex labels *)
   n = Dimensions[A][[1]];
   vtxfun =  "VertexStyleFunction"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[GraphPlot]];

   vlabelThreshold = "VertexLabelThreshold"/.
               processOptionNames[
                   Developer`SystemOptions["GraphPlotOptions"][[2]]];

   If [StringName[vtxfun] === "All",
     vtxfun = (Text[labels[[#]], #]&)
   ];

   If [StringName[vtxfun] === "Automatic" && n <= vlabelThreshold,
       vtxfun = (Text[labels[[#]], #]&)
   ];

   GraphPlot23DInternal[caller, A, dim, task, Sequence@@Flatten[{"VertexStyleFunction"->vtxfun, opts}]]
];

FixPlotRange[ g_]:=
  Block[ {pltr, asp,$DisplayFunction=Identity},
        pltr =FullOptions[g,PlotRange];
        asp =FullOptions[g,AspectRatio];
        pltr=
          Which[
            asp>100.,
            ReplacePart[pltr,FixSize[pltr[[1]]],1];
            Show[g,PlotRange->pltr],

            asp<0.01,
            ReplacePart[pltr,FixSize[pltr[[2]]],2];
            Show[g,PlotRange->pltr],

            True, Show[g]]
    ]


FixSize[{n0_,n1_}]:=
  Module[{d},
    d=If[Abs[n0]>10.^-2,Abs[n1],1.];
    m=(n0+n1)/2.;
    {m-d,m+d}
    ] 

(* we set the aspect ratio for tree by default to Sqrt[asp] *)
setDefaultAspectRatio[asp_, coord_]:= Module[
  {xmin,xmax,ymin,ymax,res},
  (* make sure this is 2D *)
  If [StringName[asp] === "Automatic" && Length[coord[[1]]] == 2,
     {{xmin,xmax},{ymin,ymax}} = Map[{Min[#],Max[#]}&,Transpose[coord]];
     If [xmax-xmin <= 0 || ymax-ymin <= 0,
        res = asp,
        res = Sqrt[(ymax-ymin)/(xmax-xmin)]
     ],
     res = asp
  ];
  res
];

GraphPlot23DInternal[A_?MatrixQ, dim_Integer, task_String, opts___?OptionQ] := Module[
   {},
   If [dim == 2,
      GraphPlot23DInternal[GraphPlot, A, dim, task, opts],
      GraphPlot23DInternal[GraphPlot3D, A, dim, task, opts]
   ]
]

GraphPlot23DInternal[caller_, A_?MatrixQ, dim_Integer, task_String, opts___?OptionQ] := Module[
   {res, range, gg, asp, spgplot, gphs, mthd, out},
   
   If [dim == 2,
       spgplot = SparseArray`Private`GraphPlot;
       gphs = System`Graphics,
       spgplot = SparseArray`Private`GraphPlot3D;
       gphs = System`Graphics3D
   ];
   res = spgplot[caller, A,
               filterOptions[{Plot->something, Rotation->something}, spgplot, opts]];
   range = "PlotRange"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
   asp = "AspectRatio"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
   If [!ListQ[res], out = $Failed];
	If [ListQ[res] && task === "Plot",
       mthd = "Method"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[caller]];
       If [ListQ[mthd], mthd = mthd[[1]]];
       (* we set the aspect ratio for tree by default to Sqrt[asp] *)
	    If [StringName[mthd] === "LayeredDrawing" && dim == 2, 
            asp = setDefaultAspectRatio[asp, res[[1]]]
       ];
       gg = gphs[Apply[GraphicsComplex, res],
          PlotRange->range, AspectRatio->asp,
          Sequence@@filterOptions[gphs, DeleteOptions[{"Method","PlotRange","AspectRatio"},opts]]];


   	 If [StringName[asp] === "Automatic",
      	  gg = FixPlotRange[gg]
	    ];	

       Show[gg];
       out = gg
   ];
   If [ListQ[res] && task === "Coordinates",
       out = res[[1]];
   ];
   out
];


GraphPlot3D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[G2S[G], 3, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates3D[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[G2S[G], 3, "Coordinates", opts]},
   res/;(res =!= $Failed)
];

GraphPlot3D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 3, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates3D[G_?RuleListQ, opts___?OptionQ] := With[
   {res = GraphPlotInternalRuleList23D[G, 3, "Coordinates", opts]},
   res/;(res =!= $Failed)
];

GraphPlot3D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[A, 3, "Plot", opts]},
   res/;(res =!= $Failed)
];

GraphCoordinates3D[A_?MatrixQ, opts___?OptionQ] := With[
   {res = GraphPlot23DInternal[A, 3, "Coordinates", opts]},
   res/;(res =!= $Failed)
];


StructurallySymmetricMatrixQ[A_?MatrixQ]:= Module[
  {B = SparseArray[A], ind, val},
  If [SquareMatrixQ[B],
     ind = B@NonzeroPositions[];
     val = Table[1, {Length[ind]}];
     B = SparseArray[Thread[Rule[ind,val]],Dimensions[B]];
     Return [(B == Transpose[B])]
  ];
  False
];

SymmetricMatrixQ[A_?MatrixQ]:= Module[
  {B=A, res = False},
  If [SquareMatrixQ[B],
     res = (B == Transpose[B]);
     If [res =!= True && res =!= False, res = False]
  ];
  res
];

SquareMatrixQ[A_?MatrixQ]:= Module[
  {},
  {m,n}=Dimensions[A];
  If [m =!= n, False, True]
];

GraphDistance[G_?RuleListQ, i_, j_]:= With[
      {res = Catch[GraphDistanceInternal[G2S[G], checkVertex[GraphDistance, G, i, 2], checkVertex[GraphDistance, G, j, 3]]]},
      res /; (res =!= $Failed)
];
GraphDistance[G_DiscreteMath`Combinatorica`Graph, i_Integer, j_Integer]:= With[
      {res = Catch[GraphDistanceInternal[G2S[G], checkVertex[GraphDistance, G, i, 2], checkVertex[GraphDistance, G, j, 3]]]},
      res /; (res =!= $Failed)
];
GraphDistance[A_?MatrixQ, i_Integer, j_Integer]:= With[
      {res = Catch[GraphDistanceInternal[A, checkVertex[GraphDistance, A, i, 2], checkVertex[GraphDistance, A, j, 3]]]},
      res /; (res =!= $Failed)
];

GraphDistanceInternal[A_?MatrixQ, i_Integer, j_Integer]:= Module[
   {res, n = Dimensions[A][[1]]},
   If [!SquareMatrixQ[A],
       Message[GraphDistance::sqma, A];
       Return[$Failed]
   ];
   res = SparseArray`GraphDistance[A, i, j];
   If [!IntegerQ[res], Return[$Failed]];
   If [res <= 0, Infinity, res]
];

MaximalBipartiteMatching[G_?RuleListQ]:= With[
      {res = MaximalBipartiteMatchingInternal[G2S[G]]},
      res /; (res =!= $Failed)
];

MaximalBipartiteMatching[G_DiscreteMath`Combinatorica`Graph]:= With[
      {res = MaximalBipartiteMatchingInternal[G2S[G]]},
      res /; (res =!= $Failed)
];

MaximalBipartiteMatching[A_?MatrixQ]:= With[
      {res = MaximalBipartiteMatchingInternal[A]},
      res /; (res =!= $Failed)
];

MaximalBipartiteMatchingInternal[A_?MatrixQ]:= Module[
   {res, n = Dimensions[A][[1]]},
   res = SparseArray`MaximalBipartiteMatching[A];
   If [!List[res], Return[$Failed]];
   res
];

getUndirectedGraph[caller_, G_?RuleListQ]:= Module[{A},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
	A = G2S[G];
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, G];
       Throw[$Failed]
   ];
   A
];
getUndirectedGraph[caller_, G_DiscreteMath`Combinatorica`Graph]:= Module[{A},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
	A = G2S[G];
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, G];
       Throw[$Failed]
   ];
   A
];
getUndirectedGraph[caller_, A_?MatrixQ]:= Module[{},
   (* check that the matrix from data is symmetric
      and is thus undirected *)
   If [!StructurallySymmetricMatrixQ[A],
       Message[caller::rug, A];
       Throw[$Failed]
   ];
   A
];

MaximalIndependentVertexSet[G_?RuleListQ]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], Automatic]], vtx = VertexList[G]},
   vtx[[res]] /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_DiscreteMath`Combinatorica`Graph]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], Automatic]]},
   res /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_?RuleListQ, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], vtxwgt]], vtx = VertexList[G]},
   vtx[[res]] /; (res =!= $Failed)
];
MaximalIndependentVertexSet[G_DiscreteMath`Combinatorica`Graph, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, G], vtxwgt]]},
   res /; (res =!= $Failed)
];

MaximalIndependentVertexSet[A_?MatrixQ]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, A], Automatic]]},
   res /; (res =!= $Failed)
];
MaximalIndependentVertexSet[A_?MatrixQ, vtxwgt_]:= With[
   {res = Catch[MaximalIndependentVertexSetInternal[getUndirectedGraph[MaximalIndependentVertexSet, A], vtxwgt]]},
   res /; (res =!= $Failed)
];
MaximalIndependentVertexSetInternal[A_?MatrixQ, vtxwgt_]:= Module[
   {n=Dimensions[A][[1]], dims = Dimensions[vtxwgt]},

   (* this should already be checked *)
   If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   If [vtxwgt =!= Automatic && (Length[dims] =!= 1 ||
         dims[[1]] =!= n),
      Message[MaximalIndependentVertexSet::vtxwgt, vtxwgt];
      Return[$Failed]
   ];
   SparseArray`MaximalIndependentVertexSet[A, vtxwgt]
];

MaximalIndependentEdgeSet[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[getUndirectedGraph[MaximalIndependentEdgeSet, G], opts]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
MaximalIndependentEdgeSet[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[getUndirectedGraph[MaximalIndependentEdgeSet, G], opts]]},
   res /; (res =!= $Failed)
];
MaximalIndependentEdgeSet[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[MaximalIndependentEdgeSetInternal[getUndirectedGraph[MaximalIndependentEdgeSet, A], opts]]},
   res /; (res =!= $Failed)
];

MaximalIndependentEdgeSetInternal[A_?MatrixQ, opts___?OptionQ]:= Module[
   {weighted},

   (* this should be already checked with getUndirectedGraph*)
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];

   weighted = 
        "Weighted" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[MaximalIndependentEdgeSet]];

   If [StringName[weighted] =!= "True" && StringName[weighted] =!= "False",
      Message[MaximalIndependentEdgeSet::wgt, weighted];
      Return[$Failed]
   ];
   If [StringName[weighted] === "True",
  	   If [!SymmetricMatrixQ[A],
         Message[MaximalIndependentEdgeSet::symat, A];
         Return[$Failed]
      ]
   ];

   SparseArray`MaximalMatching[A, "Weighted"->weighted]
];


MinCut[G_?RuleListQ, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, G], k]], vtx = VertexList[G]},
   Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
MinCut[G_DiscreteMath`Combinatorica`Graph, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, G], k]]},
   res /; (res =!= $Failed)
];
MinCut[A_?MatrixQ, k_Integer]:= With[
   {res = Catch[MinCutInternal[getUndirectedGraph[MinCut, A], k]]},
   res /; (res =!= $Failed)
];

MinCutInternal[A_?MatrixQ, k_Integer]:= Module[
   {},
   (* this should already be checked by getUndirectedGraph*)
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   If [k < 1,
      Message[MinCut::kgtwo, k];
      Return[$Failed]
   ];
   If [k == 1, Return[{Table[i,{i,Dimensions[A][[1]]}]}]];
   SparseArray`MinCut[A, k]
];

PseudoDiameterToVertex[res_, vtxlist_]:= Module[
  (* convert results from a rule list, {{diam,{i,j}},...}
     to {{diam,{v_i, v_j}}, ...} *)
  {},
  Map[{#[[1]], vtxlist[[#[[2]]]]} &, res]
]

PseudoDiameter[G_?RuleListQ, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, G], opts]], vtx = VertexList[G]},
   PseudoDiameterToVertex[res, vtx] /; (res =!= $Failed)
];

PseudoDiameter[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, G], opts]]},
   res /; (res =!= $Failed)
];

PseudoDiameter[A_?MatrixQ, opts___?OptionQ]:= With[
   {res = Catch[PseudoDiameterInternal[getUndirectedGraph[PseudoDiameter, A], opts]]},
   res /; (res =!= $Failed)
];

PseudoDiameterInternal[A_?MatrixQ, opts___?OptionQ]:= Module[
   {agg, res},

   (* this should already be checked by getUndirectedGraph *)
	If [!StructurallySymmetricMatrixQ[A],
      Return[$Failed]
   ];
   agg = 
        "Aggressive" /. processOptionNames[Flatten[{opts}]] /. 
          processOptionNames[Options[PseudoDiameter]];

   If [StringName[agg] =!= "True" && StringName[agg] =!= "False",
      Message[PseudoDiameter::agg, agg];
      Return[$Failed]
   ];
 
   res = SparseArray`PseudoDiameter[A, Aggressive->agg];

   (* converting from {{d, i, j},...} to {{d,{i,j}},...} *)
	If [!ListQ[res], Return[$Failed]];
   Map[{#[[1]], {#[[2]], #[[3]]}} &, res]
];
 


StrongComponents[G_?RuleListQ]:= With[
      {res = StrongComponentsInternal[G2S[G]], vtx = VertexList[G]},
      Map[vtx[[#]]&, res] /; (res =!= $Failed)
];
StrongComponents[G_DiscreteMath`Combinatorica`Graph]:= With[
      {res = StrongComponentsInternal[G2S[G]]},
      res /; (res =!= $Failed)
];
StrongComponents[A_?MatrixQ]:= With[
      {res = StrongComponentsInternal[A]},
      res /; (res =!= $Failed)
];

StrongComponentsInternal[A_?MatrixQ]:= Module[
   {res},
   If [!SquareMatrixQ[A],
       Message[StrongComponents::sqma, A];
       Return[$Failed]
   ];
   SparseArray`StronglyConnectedComponents[A]
];

TreePlot[G_DiscreteMath`Combinatorica`Graph, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[TreePlot, G2S[G], 2, "Plot", TreeOptions[opts]]]},
   res/;(res =!= $Failed)
];

TreePlot[G_DiscreteMath`Combinatorica`Graph, root_Integer, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[TreePlot, G2S[G], 2, "Plot", TreeOptions[checkVertex[TreePlot, G, root, 2], opts]]]},
   res/;(res =!= $Failed)
];

TreePlot[G_?RuleListQ, opts___?OptionQ] := With[
   {res = Catch[GraphPlotInternalRuleList23D[TreePlot, G, 2, "Plot", TreeOptions[opts]]]},
   res/;(res =!= $Failed)
];

TreePlot[G_?RuleListQ, root_, opts___?OptionQ] := With[
   {res = Catch[GraphPlotInternalRuleList23D[TreePlot, G, 2, "Plot", TreeOptions[checkVertex[TreePlot, G, root, 2], opts]]]},
   res/;(res =!= $Failed)
];

TreePlot[A_?MatrixQ, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[TreePlot, A, 2, "Plot", TreeOptions[opts]]]},
   res/;(res =!= $Failed)
];

TreePlot[A_?MatrixQ, root_Integer, opts___?OptionQ] := With[
   {res = Catch[GraphPlot23DInternal[TreePlot, A, 2, "Plot", TreeOptions[checkVertex[TreePlot, A, root, 2], opts]]]},
   res/;(res =!= $Failed)
];

checkVertex[caller_, A_?MatrixQ, vertex_Integer, argnum_Integer]:= Module[{n},
   n = Dimensions[A][[1]];
   (* vertex mush be positive integer <= n *)
   If [vertex > n || vertex <= 0, Message[caller::iind, vertex, n, argnum]; Throw[$Failed]];
   vertex
]

checkVertex[caller_, G_?RuleListQ, vertex_, argnum_Integer]:= Module[{ind, n, vertex1},
   ind = setHash[G];
   n = Max[ind];
	vertex1 = indhash[vertex];
   (* vertex must be a member of the rule list entries *)
   If [!IntegerQ[vertex1] || vertex1 <= 0 || vertex1 > n, Message[caller::rind, vertex, argnum]; 
        Throw[$Failed]];
   vertex1
]

checkVertex[caller_, A_DiscreteMath`Combinatorica`Graph, vertex_Integer, argnum_Integer]:= Module[{l, n},

   l = DiscreteMath`Combinatorica`ToAdjacencyLists[A,EdgeWeight];
   n = Length[l];

   (* vertex mush be positive integer <= n *)
   If [vertex > n || vertex <= 0, Message[caller::iind, vertex, n, argnum]; Throw[$Failed]];
   vertex
]




TreeOptions[root_Integer, opts___?OptionQ]:= Module[
   {},
   TreeOptionsInternal[root, opts]
];
TreeOptions[opts___?OptionQ]:= Module[
   {},
   TreeOptionsInternal[-1, opts]
];


TreeOptionsInternal[root0_Integer, opts___?OptionQ]:= Module[
   {root = root0, opts2, res, rp, tsfun, rotation = 0},
	rp = "RootPosition"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[TreePlot]];
   tsfun = "TreeSizeFunction"/.processOptionNames[Flatten[{opts}]]/.processOptionNames[Options[TreePlot]];
	opts2 = DeleteOptions[{"Method"},opts];
	Switch[StringName[rp],
      "Left", rotation = 270,
      "Right", rotation = 90,
      "Top", rotation = 0,
      "Bottom", rotation = 180,
      "Center", rotation = 0,
      _, Message[TreePlot::rp, root]; Throw[$Failed]
   ];

   If [root < 0, root = Automatic];

   If [StringName[rp] === "Center",       
      res = Sequence@@Flatten[{Method->{"RadialDrawing", "Root"->root, "TreeSizeFunction"->tsfun}, "Rotation"->rotation, opts2}],
      res = Sequence@@Flatten[{Method->{"LayeredDrawing", "Root"->root, "TreeSizeFunction"->tsfun}, "Rotation"->rotation, opts2}]
   ];
   res
];



(*==============================================
 utilities routine that layout balls of different
 radiuses so that they do not clash with each other and
 occupies the space evenly. Used for layout 
 disconnected graphs
*)
ComponentsLayout[radiuses_List] := ComponentsLayout[2, radiuses, 0.];
ComponentsLayout[radiuses_List, gap_] := ComponentsLayout[2, radiuses, gap];
ComponentsLayout[dim_Integer, radiuses_List] := 
 ComponentsLayout[dim, radiuses, 0.];
ComponentsLayout[dim_Integer, radiuses_List, gap_] := Module[
   {diams , scale, max, band, bin, order, revord, width, areas, arealist, nz, 
    lens, unit, centers, area, i, j, min, totalSquares, upperDiams, quad, res,
     vol = 4},
   (* lay out a list of circles with radius gives in the list radius, 
   return the center of the circle,
   WHen plotting reasonable the circles at their centers, 
   the layout looks. only dim = 2 work now.
   *)
   diams = 2*radiuses;
   diams += gap/2;
   order = Reverse[Ordering[diams]];
   diams = diams[[order]];
   scale = Max[diams];
   diams /= scale;(* scale*)
   max = Max[diams]; min = Min[diams];
   band = Ceiling[Log[vol, max]];
   bin = Map[Ceiling[Log[vol, #]] &, diams];
   areas = Map[(vol^#) &, bin]/vol^Max[bin];
   totalSquares = Ceiling[Total[areas]];
   width = Ceiling[Sqrt[totalSquares]];
   area  = 0; 
   lens = {};
   nz = 0;
   Do[
    If [(area = area + areas[[i]]) <= 1,
      nz++,
      area =  areas[[i]];
     lens = {lens, nz}; nz = 1],
    {i, Length[areas]}
    ];
   lens = {lens, nz};
   lens = Flatten[lens];
   arealist = areas; 
   arealist = 
    Map[(res = Take[arealist, #]; arealist = Drop[arealist, #]; res) &, 
     lens];
   unit = (vol^band) ;
   centers = 
    Partition[
     unit Flatten[
       Table[{j, i}, {i, width - 1, 0, -1}, {j, 0, width - 1, 1}]], {2}];
   SetAttributes[{squareAdd, OctreeFullQ, OctreeSetFull, OctreeSetSta}, 
    HoldAll];
   OctreeFullQ[octree_] := (octree =!= None && octree[[4]]);
   OctreeEmptyQ[octree_] := (octree === None);
   OctreeSetFull[octree_] := (octree[[4]] = True);
   OctreeInit[width_, x_] := {width, x, Table[None, {vol}], False, 1};
   OctreeSetSta[octree_, i_] := (octree[[5]] = i);
   stuffSquare[areas_, x_] := Module[
     {width, octree, coord},
     (* stuff a list of areas in a square at origin x.
       Return a list of ceters each area will reside
      *)
     coord = Table[1, {Length[areas]}];
     width = 1;
     octree = OctreeInit[width, x];
     Do[
      (*Print["===== adding ", areas[[i]], " octree = ", octree];*)
      
      coord[[i]] = squareAdd[octree, areas[[i]]],
      {i, Length[areas]}];
     
     coord
     ];
   quad[i_] := (Switch[i, 1, {-1, 1}, 2, {1, 1}, 3, {-1, -1}, 4, {1, -1}, _, 
      Throw[$Failed]]);
   squareAdd[octree_, area_] := Module[
     {areaWidth, width, x, i, leaf, coord = "Full", full, leafs, sta},
     areaWidth = Sqrt[area];
     If [OctreeFullQ[octree], Return["Full"]];
     {width, x, leafs, full, sta} = octree;

     If [areaWidth >= width, OctreeSetFull[octree]; Return[x]];
     Do[
      leaf = leafs[[i]];
      If [OctreeFullQ[leaf], Continue[]];
      If [OctreeEmptyQ[leaf], 
       leafs[[i]] = OctreeInit[width/2, x + width/4*quad[i]]];
      leaf = leafs[[i]];
      coord = squareAdd[leaf, area];
      leafs[[i]] = leaf;
      octree[[3]] = leafs;
      If [coord =!= "Full", Break[], OctreeSetSta[octree, i + 1]];
      
      
      
      , {i, sta, 4}];
     coord
     ];
   upperDiams = unit*Map[Sqrt[#] &, areas];
   coord = unit * Map[stuffSquare[#, {0, 0}] &, arealist];
   Do[
    coord[[i]] = Map[(# + centers[[i]]) &, coord[[i]]];
    , {i, Length[coord]}];
   coord = Partition[Flatten[coord], {2}];
   
   (* this put an object at the top left corner of the square instead of the default
	   position which is the center
   Do[
    coord[[i]] = coord[[i]] + {-0.5, .5}(upperDiams[[i]] - diams[[i]]);
    , {i, Length[coord]}];
   *)

   revord = Table[1, {Length[order]}];
   revord[[order]] = Range[Length[order]];
   (* in case dim > 2, pad zeros to coordinates *)
   
   res = Map[PadRight[#, dim] &, (coord*scale)[[revord]]];
   res = Developer`ToPackedArray[N[res]];
   res
   
   ];

(* ============ end ComponentsLayout =========*)

GraphicsComplex[coord_, rest_] := Module[
   {res, dim},
   dim = Dimensions[coord][[2]];
   Unprotect[Point, Line, Arrow, Text, Circle, Disk];
   Point[i_Integer]:= Point[coord[[i]]];
   Line[{i_Integer, j_Integer}] := Line[{coord[[i]], coord[[j]]}];
   Arrow[{i_Integer, j_Integer}, opt___?OptionQ]:= If [dim == 2,
      Arrow[coord[[i]], coord[[j]], opt],
      Arrow3D[coord[[i]], coord[[j]], opt]
   ];
   Arrow[i_Integer, j_Integer, opt___?OptionQ]:= If [dim == 2,
      Arrow[coord[[i]], coord[[j]], opt],
      Arrow3D[coord[[i]], coord[[j]], opt]
   ];
   Arrow[{x_List,y_List}, opt___?OptionQ]:= Arrow[x,y,opt];
   Circle[i_Integer, diam_]:= Circle[coord[[i]],diam];
   Disk[i_Integer, diam_]:= Disk[coord[[i]],diam];
	Text[s_, i_Integer] := Text[s, coord[[i]]];
(*
   If [dim == 2,
     res = Show[Graphics[rest]],
     res = Show[Graphics3D[rest]]
   ];
*)
   res = rest;
	Clear[Point, Line, Arrow,Text,Circle,Disk];
   Protect[Point, Line, Arrow,Text,Circle,Disk];
   res
];

(*============ 
The Arrow3D function is copied from
http://library.wolfram.com/infocenter/TechNotes/4117
*)

 
Options[Arrow3D] = {HeadColor -> GrayLevel[.3],HeadSize -> .05};
 
 
ArrowHead[pt1_,pt2_,opts___?OptionQ] :=
  Module[{a,b,c,aa,bb,cc,nrm1,nrm2,ttt,circ,pts,arrowpolys,hdcolor,hdsz,hdrad},
   
    hdcolor=HeadColor/.{opts}/.Options[Arrow3D];
    hdsz = HeadSize/.{opts}/.Options[Arrow3D];
     
    {aa,bb,cc} = pt2-pt1;
     
    (*Constructing an orthonormal basis for the arrowhead*)
     
    Which[aa == 0 && bb == 0, nrm1 = {0, 1, 0};
      nrm2 = {1, 0, 0}, aa == 0 && cc == 0,
     nrm1 = {1, 0, 0}; nrm2 = {0, 0, 1},
     bb == 0 && cc == 0, nrm1 = {0, 0, 1};
      nrm2 = {0, 1, 0}, aa == 0, nrm1 = {1, 0, 0};
      nrm2 = Cross[nrm1, {0, bb, cc}/Norm[{bb, cc}]],
     bb == 0, nrm1 = {0, 1, 0};
      nrm2 = Cross[nrm1, {aa, 0, cc}/Norm[{aa, cc}]],
     cc == 0, nrm1 = {0, 0, 1};
      nrm2 = Cross[nrm1, {aa, bb, 0}/Norm[{aa, bb}]],
     True, nrm1 = {-(((bb + cc)*Abs[aa])/
          (aa*Sqrt[2*aa^2 + (bb + cc)^2])),
        Abs[aa]/Sqrt[2*aa^2 + (bb + cc)^2],
        Abs[aa]/Sqrt[2*aa^2 + (bb + cc)^2]};
      nrm2 = {((bb - cc)*Abs[aa])/
   Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
     aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)],
  -(((aa^2 + cc*(bb + cc))*Abs[aa])/
    (aa*Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
       aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)])),
  ((aa^2 + bb*(bb + cc))*Abs[aa])/
   (aa*Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
      aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)])}];Which[aa == 0 && bb == 0, nrm1 = {0, 1, 0};
      nrm2 = {1, 0, 0}, aa == 0 && cc == 0,
     nrm1 = {1, 0, 0}; nrm2 = {0, 0, 1},
     bb == 0 && cc == 0, nrm1 = {0, 0, 1};
      nrm2 = {0, 1, 0}, aa == 0, nrm1 = {1, 0, 0};
      nrm2 = Cross[nrm1, {0, bb, cc}/Norm[{bb, cc}]],
     bb == 0, nrm1 = {0, 1, 0};
      nrm2 = Cross[nrm1, {aa, 0, cc}/Norm[{aa, cc}]],
     cc == 0, nrm1 = {0, 0, 1};
      nrm2 = Cross[nrm1, {aa, bb, 0}/Norm[{aa, bb}]],
     True, nrm1 = {-(((bb + cc)*Abs[aa])/
          (aa*Sqrt[2*aa^2 + (bb + cc)^2])),
        Abs[aa]/Sqrt[2*aa^2 + (bb + cc)^2],
        Abs[aa]/Sqrt[2*aa^2 + (bb + cc)^2]};
      nrm2 = {((bb - cc)*Abs[aa])/
   Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
     aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)],
  -(((aa^2 + cc*(bb + cc))*Abs[aa])/
    (aa*Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
       aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)])),
  ((aa^2 + bb*(bb + cc))*Abs[aa])/
   (aa*Sqrt[2*aa^4 + (bb + cc)^2*(bb^2 + cc^2) +
      aa^2*(3*bb^2 + 2*bb*cc + 3*cc^2)])}];
     
    hdrad = If[Head[hdsz]===UniformSize,hdsz[[1]]/5,
      Sqrt[((pt2 -(hdsz pt1 + (1 - hdsz)pt2)).(pt2 -(hdsz pt1 + (1 -
                            hdsz)pt2)))]/5];
     
    circ1[ttt_] :=
      hdrad nrm1 Cos[ttt]+
          hdrad nrm2 Sin[ttt]+(hdsz pt1 + (1 - hdsz)pt2);
            
    circ2[ttt_] := (hdvec = pt2+hdsz[[1]] (pt1-pt2)/Norm[(pt1-pt2)];
      hdrad nrm1 Cos[ttt]+
          hdrad nrm2 Sin[ttt]+hdvec);
            
    pts =If[Head[hdsz]===UniformSize,
        Partition[ N[Table[circ2[t],{t,0,2 Pi,Pi/5}]],2,1],
        Partition[ N[Table[circ1[t],{t,0,2 Pi,Pi/5}]],2,1]];
     
    arrowpolys =
      Map[{EdgeForm[],SurfaceColor[hdcolor],
            Polygon[Flatten[{#,{pt2}},1]]}&,pts]
    ]
     
Arrow3D[pt1_,pt2_,opts___?OptionQ]:= If[Chop[pt2-pt1]=={0,0,0}, Point[pt1],
{Line[{pt1,pt2}],ArrowHead[pt1,pt2,opts]}]
 

End[];(*end private*)
EndPackage[];
