(* :Mathematica Version: 3.0 *)

(* :Name: Graphics`Graphics3D` *)

(* :Title: Graphics3D *)

(* :Author: Wolfram Research, Inc. *)
(* :Copyright: Copyright 1990-2005, Wolfram Research, Inc.*)

(* :History:
        Created March 1991 by John M. Novak. A collection of functions
        originally intended for Graphics.m by Michael Chan and Kevin McIsaac,
        with modifications by Bruce Sawhill and ECM.  Modifications to Project
        and Shadow by John M. Novak.
        V1.0.5 April 1994 by John M. Novak. Minor fixes to ScatterPlot3D.
        Modifications to StackGraphics for correct display of polygons
          by Tom Wickham-Jones. Change to ShadowPlot3D and ListShadowPlot3D
          to use a ColorFunction option in place of the current Color
          option (which is retained for compatibility); this was done at
          the suggestion of John Fultz.
	GeneralizedBarChart3D, Histogram3D, ECM, October 1997.
*)
(* :Summary:
This package provides special functions for plotting in three
dimensions.  Special formats include bar charts, scatter plots,
surface plots, shadow plots, projections, and histograms.
*)

(* :Context: Graphics`Graphics3D` *)

(* :Package Version: 1.1 *)

(* :Keywords:
	Graphics, 3D, Surface, Project, Shadow, histogram
*)

(* :Warning: Adds to the definition of the function Graphics3D. *)

BeginPackage["Graphics`Graphics3D`", "Graphics`Common`GraphicsCommon`",
		"Utilities`FilterOptions`"]

ScatterPlot3D::usage = 
"ScatterPlot3D[{{x1, y1, z1}, ...}, (options)] plots points in three 
dimensions as a scatter plot.";

ListSurfacePlot3D::usage =
"ListSurfacePlot3D[{{{x11, y11, z11}, ...},{{x12, y12, z12}, ...}, ...}, (options)]
plots a matrix of points in three dimensions as a surface." ;

ShadowPlot3D::usage =
"ShadowPlot3D[f, {x, xmin, xmax}, {y, ymin, ymax}] plots the function f over
the the x and y ranges with polygons shaded according to the height of the
surface, with a projection of the surface onto the x-y plane.";

(* The Color option determines whether the plot is in color or gray scale.
SurfaceMesh determines whether a mesh is drawn on the surface.
ShadowMesh determines whether a mesh is drawn on the projection.
SurfaceMeshStyle determines the style of the mesh on the surface.
ShadowMeshStyle determines the style of the mesh on the projection.
ShadowPosition determines the location of the projection. *)

Color::usage = 
"Color is an option for ShadowPlot3D and ListShadowPlot3D, which determines
whether the plot should be drawn in color.  If True, the ColorFunction option
is employed (set to Hue by default), otherwise a greyscale is used.";

SurfaceMesh::usage =
"SurfaceMesh is an option for ShadowPlot3D and ListShadowPlot3D, which
determines whether a mesh is drawn on the surface.";

ShadowMesh::usage =
"ShadowMesh is an option for ShadowPlot3D and ListShadowPlot3D, which determines
whether a mesh is drawn on the projection.";

SurfaceMeshStyle::usage =
"SurfaceMeshStyle is an option for ShadowPlot3D and ListShadowPlot3D, which
defines the style of the mesh on the surface.";

ShadowMeshStyle::usage =
"ShadowMeshStyle is an option for ShadowPlot3D and ListShadowPlot3D, which
defines the style of the mesh on the projection.";

ShadowPosition::usage =
"ShadowPosition is an option for ShadowPlot3D and ListShadowPlot3D, which
determines whether the projection is above or below the surface (in the
positive or negative z direction).";

ListShadowPlot3D::usage =
"ListShadowPlot3D[array, (opts)] generates a surface representing an array of
height values with polygons shaded according to the height of the surface and a
projection of the surface onto the x-y plane.";

Project::usage = 
"Project[graphic, point] projects an image of the graphic onto a plane
perpendicular to the line from the center of the graphic to point.
Project[graphic, {e1, e2}, point] projects an image of the graphic onto a plane
with basis vectors {e1, e2} at point, along the line from the origin to point.
Project[graphic, {e1,e2},point,center] project as before, except along the
line from center to point.  The projection is as seen from Infinity.";

Shadow::usage = 
"Shadow[graphic, (opts)] projects images of the graphic onto the
x-y, x-z, and y-z planes.  Options XShadow, YShadow, ZShadow,
XShadowPosition, YShadowPosition, and ZShadowPosition determine
which projections are shown and where they are located.  The
magnitude of the positions is scaled so that 1 is the width of the
plot on the given axis; it is measured from the center of the
plot.";

XShadow::usage =
"XShadow is an option for Shadow that determines whether to draw a
projection of the graphic in the x direction.";

YShadow::usage =
"YShadow is an option for Shadow that determines whether to draw a
projection of the graphic in the y direction.";

ZShadow::usage =
"ZShadow is an option for Shadow that determines whether to draw a
projection of the graphic in the z direction.";

XShadowPosition::usage = 
"XShadowPosition is an option for Shadow that determines whether the
projection of the graphic is in the positive or negative x direction.";

YShadowPosition::usage = 
"YShadowPosition is an option for Shadow that determines whether the
projection of the graphic is in the positive or negative y direction.";

ZShadowPosition::usage = 
"ZShadowPosition is an option for Shadow that determines whether the
projection of the graphic is in the positive or negative z direction.";

Histogram3D::usage =
"Histogram3D[{{x1, y1}, {x2, y2}, ...}] generates a three-dimensional bar graph
representing a histogram of the bivariate data {{x1, y1}, {x2, y2}, ...}.  The
cross-sectional area of each solid bar is proportional to the area of the 
rectangular region defining the respective category, and the volume of the bar
is proportional to the frequency with which the data fall in that category.
Histogram range and categories may be specified using the options HistogramRange
and HistogramCategories.
Histogram[{{f11, f12, ... f1l}, ... {fk1, fk2, ..., fkl}},
FrequencyData -> True] generates a histogram of the bivariate data 
{{f11, f12, ... f1l}, ... {fk1, fk2, ..., fkl}}, where fij is the frequency
with which the original data falls into the category {i, j}." 

BarChart3D::usage =
"BarChart3D[list, opts] creates a three-dimensional bar graph of the 
rectangular matrix list.  BarChart3D[{{{z, style},...}...},opts] creates a bar
graph with a specific style for each bar.  BarChart3D[{{{x, y, z}, style}...}] 
creates a bar graph of bars scattered at specific x and y coordinates with
height z and a specific style.";

GeneralizedBarChart3D::usage =
"GeneralizedBarChart3D[{{{xpos1, ypos1}, height1, {xwidth1, ywidth1}},
{{xpos2, ypos2}, height2, {xwidth2, ywidth2}}, ...}] generates a 
three-dimensional bar graph with the solid bars at the given positions, heights,
and widths."

(* The XSpacing and YSpacing options control the space between bars in the
X and Y directions respectively.  SolidBarEdges and SolidBarEdgeStyle are
options determining the style of the edges of the cuboids making up the bars
of the bar chart.  SolidBarStyle is a style for the faces of the cuboids.
The odd naming convention is to avoid shadowing similar options in Graphics.m.
BarChart3D also accepts all options valid for Graphics3D. *)

XSpacing::usage =
"XSpacing is an option for BarChart3D, which determines the amount of space
between bars in the x direction.  XSpacing may be set to any real number
between 0 and 1.";

YSpacing::usage =
"YSpacing is an option for BarChart3D, which determines the amount of space
between bars in the Y direction.  YSpacing may be set to any real number
between 0 and 1.";

SolidBarEdges::usage =
"SolidBarEdges is an option for BarChart3D, which determines whether the edges
of the bars are drawn.";

SolidBarEdgeStyle::usage =
"SolidBarEdgeStyle is an option for BarChart3D, which determines the style of
the edges of the bars.";

SolidBarStyle::usage = 
"SolidBarStyle is an option for BarChart3D, which determines the style of the
faces of the bars.";

StackGraphics::usage =
"StackGraphics[{g1, g2, ...}] generates a Graphics3D object corresponding to a
\"stack\" of two-dimensional graphics objects.";

TransformGraphics3D::usage =
"TransformGraphics3D[graphics3d, f] applies the function f to all lists of
coordinates in graphics3d.";

SkewGraphics3D::usage = 
"SkewGraphics3D[graphics, m] applies the matrix m to all coordinates in graphics.";

Graphics3D::usage =
"Graphics3D[primitives, options] represents a three-dimensional graphic
image.  Graphics3D[graphics] projects a two-dimensional graphic image into
a three-dimensional graphic image."; 

(* Attach usage messages to symbols from GraphicsCommon.m. *)
If[Head[FrequencyData::usage] === MessageName,
   FrequencyData::usage =
"FrequencyData is an option of histogram functions that specifies whether
the data argument represents the original data (FrequencyData -> False) or
the frequencies with which the original data fall in the respective
categories (FrequencyData -> True).  The default is FrequencyData -> False.";
   HistogramCategories::usage =
"HistogramCategories is an option of histogram functions that specifies
the categories in the histogram.  When HistogramCategories->Automatic
(default), the categories are chosen to be intervals of equal width and
the number of categories is chosen based on the data.  The number of
equally sized categories may be specified using HistogramCategories->m for
univariate data, or HistogramCategories -> {m, n} for bivariate data.
Unequally sized categories may be specified using HistogramCategories->
{c1, c2, c3, ... } for univariate data, where the cutoffs represent the
categories {c1 <= x < c2, c2 <= x < c3, ...}.  For bivariate data,
unequally sized categories in both dimensions may be specified using
HistogramCategories->{{c1, c2, c3, ...}, {d1, d2, d3, ...}}."; 
   HistogramRange::usage =
"HistogramRange is an option of histogram functions that specifies the lower
and upper limits of the points to be included in the histogram.  Possible
values are Automatic (default), HistogramRange -> {min, max}, or for bivariate
data, HistogramRange -> {{xmin, xmax}, {ymin, ymax}}.";
   HistogramScale::usage =
"HistogramScale is an option of histogram functions that specifies the
way in which the bar heights are to be scaled.  HistogramScale ->
Automatic (default) implies that, for equally sized intervals, there is no
scaling (yielding raw frequencies), and for unequally sized intervals, there is 
scaling according to interval size (yielding frequency densities).
HistogramScale -> True implies that the heights are scaled by the interval sizes
regardless of whether the interval sizes are equal.  HistogramScale -> k 
implies that the heights are scaled so that the bar areas (for univariate
histograms) or bar volumes (for bivariate histograms) sum to the positive number
k.  In particular, HistogramScale -> 1 gives a probabity density plot.";
   ApproximateIntervals::usage =
"ApproximateIntervals is an option of histogram functions that specifies
whether the HistogramCategories or HistogramRange settings should be adjusted
so that the interval boundaries are described by simple numbers.  The default
is ApproximateIntervals -> Automatic, which means that intervals are
adjusted when HistogramCategories is set to Automatic or a positive integer,
but not adjusted when HistogramCategories is set to a specific list of
cutoffs.  Other possible settings for ApproximateIntervals are True or False.";
   IntervalCenters::usage =
"IntervalCenters is a possible value for the Ticks option of
histogram functions.  For example, Histogram[data, Ticks -> IntervalCenters]
Automatic}], or Histogram[data, Ticks -> IntervalCenters], places the ticks of
the category axis at the interval centers, and sets the ticks of the frequency
axis automatically.";
   IntervalBoundaries::usage =
"IntervalBoundaries is a possible value for the Ticks option of histogram
functions.  For example, Histogram[data, Ticks -> {IntervalBoundaries,
Automatic}], or Histogram[data, Ticks -> IntervalBoundaries] places the ticks
of the category axis at the interval boundaries, and sets the ticks of the
frequency axis automatically.";
];
(* NOTE that it is possible that one would want to specify categories that
	tesselate some rectangular range in the x-y plane in a more general way
	than the cross of x-cutoffs and y-cutoffs.  However, that is
	not supported yet, and specifying x-cutoffs and y-cutoffs is the
	most general tesselation allowed by HistogramCategories.  *)


Begin["`Private`"]

(* Define a better NumberQ *)

numberQ[x_] := NumberQ[N[x]]

(* vector cross product *)

cross[v1_?VectorQ, v2_?VectorQ] :=
	Module[{m=Minors[{v1,v2},2][[1]]},
		{m[[3]], -m[[2]], m[[1]]}
	]	/; Length[v1]==Length[v2]==3

(* Unit vector in the vec direction *)

normalize[vec:{_,_,_}] :=
	vec/Sqrt[Apply[Plus,vec^2]]


(* BarChart3D *)

BarChart3D::badxspacing = "XSpacing must be between 0 and 1.";

BarChart3D::badyspacing = "YSpacing must be between 0 and 1.";

Options[BarChart3D] =  {XSpacing -> 0, YSpacing -> 0,SolidBarEdges->True,
	SolidBarEdgeStyle->GrayLevel[0],SolidBarStyle->GrayLevel[.5]} ~Join~
	Options[Graphics3D];

SetOptions[BarChart3D, PlotRange -> All, BoxRatios -> {1,1,1},
						Axes -> Automatic,
						Ticks -> Automatic]

BarChart3D[list:{{_?numberQ..}..}, opts___] :=
	BarChart3D[Flatten[Table[{{x,y,list[[x,y]]},
				  SolidBarStyle/.{opts}/.Options[BarChart3D]},
				 {x,Length[list]},
				 {y,Length[Transpose[list]]}
				],1],opts]

BarChart3D[list:{{{_?numberQ,_}..}..}, opts___] :=
	BarChart3D[Flatten[Table[{{x,y,list[[x,y,1]]},
				  list[[x,y,2]]},
				 {x,Length[list]},
				 {y,Length[Transpose[list]]}
				],1],opts]

(* NOTE that BarSpacing is an option of Graphics`Graphics`BarChart and
	XSpacing and YSpacing are options of Graphics`Graphics3D`BarChart3D. *)
BarChart3D[list:{{{_?numberQ,_?numberQ,_?numberQ},_}...},opts___] :=
  Module[{x,y,xs,ys,xspacing,yspacing,boxopts,g3dopts,list1},
	xspacing = XSpacing /. {opts} /. Options[BarChart3D];
	If[xspacing>1 || xspacing<0,
			(Message[BarChart3D::badxspacing];xspacing=0)];
	yspacing = YSpacing /. {opts} /. Options[BarChart3D];
	If[yspacing>1 || yspacing<0,
			(Message[BarChart3D::badyspacing];yspacing=0)];
	If[TrueQ[SolidBarEdges/.{opts}/.Options[BarChart3D]],
	   edges = EdgeForm[SolidBarEdgeStyle/.{opts}/.Options[BarChart3D]],
	   edges = EdgeForm[]];
  	g3dopts = ({FilterOptions[Graphics3D, ##]}&) @@ 
				Flatten[{opts, Options[BarChart3D]}];
  	xs = (1-xspacing)/2;
  	ys = (1-yspacing)/2;
  	list1 = Transpose[Map[#[[1]]&,list]];
        Show[
   		Graphics3D[Map[Flatten[{#[[2]],edges,
   				Cuboid[{#[[1,1]]-xs, #[[1,2]]-ys, 0},
				   	{#[[1,1]]+xs, #[[1,2]]+ys, #[[1,3]]}]
				      }]&,
			    list]],
		   Flatten[{g3dopts}]
	]
  ]                                        


(* Histogram3D *)

(* Histogram3D does not have the BarChart3D options XSpacing or YSpacing. 
	The option HistogramBins functions like the option
        PlotPoints;  HistogramRange functions like PlotRange.
*)

Histogram3D::ticks =
"`` is not a valid tick specification.  Taking Ticks->Automatic."

Histogram3D::rcount = "Frequency count of data in categories failed."

Histogram3D::rdhc =
"Warning: `` is not a valid setting for HistogramCategories when
FrequencyData -> False.  When the data is raw (not frequencies),
HistogramCategories should specify (i) Automatic, (ii) a positive integer
denoting the total number of bivariate categories that the data is to be
divided into, or (iii) a vector {xhc, yhc}.  Either xhc or yhc may be
(i) Automatic, (ii) a positive integer denoting the number of categories that
the corresponding component of the data is to be divided into, or (iii) a
vector of monotonically increasing numbers denoting cutoffs.
Taking HistogramCategories -> Automatic."

Histogram3D::fdhc =
"Warning: `` is not a valid setting for HistogramCategories when
FrequencyData -> True.  When the data represents frequencies,
HistogramCategories should specify Automatic (implying {Automatic, Automatic}),
or {xcutoffs, Automatic}, or {Automatic, ycutoffs}, or {xcutoffs, ycutoffs},
where xcutoffs and ycutoffs are vectors of monotonically increasing numbers.
Taking HistogramCategories -> Automatic."

Histogram3D::badrg =
"Warning: `` is not a valid value for a component of the setting for
the HistogramRange option.  Taking the component to be Automatic."

Histogram3D::rd2d =
"When FrequencyData -> False, the data must be in the form of
two-dimensional data {{x1, y1}, {x2, y2}, ...}."

Histogram3D::fdfail =
"When FrequencyData -> True and HistogramCategories -> {xcutoffs, ycutoffs}, 
the length of the xcutoffs vector should be exactly one more than the number of
rows in the frequency data matrix, and the length of the ycutoffs vector
should be exactly one more than the number of columns in the frequency data
matrix."

Histogram3D::lt1 =
"Warning: `1` point with `2` component strictly less than `3`
is not included in histogram."

Histogram3D::lt =
"Warning: `1` points with `2` components strictly less than `3`
are not included in histogram."

Histogram3D::gtet1 =
"Warning: `1` point with `2` component greater than or equal to `3`
is not included in histogram."

Histogram3D::gtet =
"Warning: `1` points with `2` components greater than or equal to `3`
are not included in histogram."

Histogram3D::range =
"Warning: `` is not a valid setting for HistogramRange.
Taking HistogramRange -> Automatic."

Histogram3D::noapprox =
"ApproximateIntervals -> `` is a not a valid setting when 
HistogramCategories->{c1, c2, ..., cm}.  Taking ApproximateIntervals -> False.";


Options[Histogram3D] =
	{
	ApproximateIntervals -> Automatic,
	FrequencyData -> False,
	HistogramCategories -> Automatic,
        HistogramRange -> Automatic,
	HistogramScale -> Automatic,
	SolidBarEdges -> True,		 (* opt of GeneralizedBarChart3D *)
	SolidBarEdgeStyle -> GrayLevel[0], (* opt of GeneralizedBarChart3D *)	
	SolidBarStyle -> SurfaceColor[GrayLevel[1]],	 (* opt of GeneralizedBarChart3D *)
	Ticks -> Automatic 		 (* opt of Graphics3D *)
	};


(* Note: Histogram3D calls RangeCounts to compute frequencies
        and GeneralizedBarChart3D for plotting. *)
(* Note:
rc = RangeCounts[data2D, {0, .2, .4, .6, .8, 1.0}, {0, .5, 1.0}];
Dimensions[rc] -> {7, 4}
Dimensions[ Map[#[[{2, 3}]]&, rc[[{2, 3, 4, 5, 6}]] ] ] ->  {5, 2}
*)

Histogram3D[mat_?MatrixQ, opts___?OptionQ] :=
 Module[{approximate, fdata, hcat, range, scale, sbedges, sbedgestyle, sbstyle,
	 ticks, dim,
	 rangeX, rangeY, (* x and y components of HistogramRange *)	 
	 countdata, numberOfBinsX, numberOfBinsY, countdataX, countdataY,
	 hcatX, hcatY, (* x and y components of HistogramCategories *)
	dataminX, datamaxX, (* min and max as determined by x-component of
				data and option HistogramRange *)
	dataminY, datamaxY, (* min and max as determined by y-component of
                                data and option HistogramRange *)
	cutoffsX, cutoffsY, 
	binminX, binmaxX, (* min and max of x coordinate bin boundaries *)
	binminY, binmaxY, (* min and max of y coordinate bin boundaries *)
	listX, listY, totalcount, cutoffs,
	leftTailCountX, rightTailCountX, leftTailCountY, rightTailCountY,
	binwidthsX, binwidthsY, binareas, binareasFlatList,
	bincentersX, bincentersY, axisticksX, axisticksY,
	bincenters, binwidths,
	phwdata, (* position-height-width data for GeneralizedBarChart3D *)
	groptslist,
	volume	(* volume of histogram; used for scaling z axis so that
			histogram has unit volume *)
	},
  (

     (* Error check for HistogramRange setting. *)
     If[range === Automatic, range = {Automatic, Automatic}];
     If[!( ListQ[range] && Length[range] == 2 ),
	   Message[Histogram3D::badrg, range];
           range = {Automatic, Automatic}   ];
     {rangeX, rangeY} = range;		

     If[TrueQ[fdata],

        (* PROCESS MATRIX assuming that it represents FREQUENCIES
	   on a 2D grid.  That is,
	   Histogram3D[{{7, 4}, {1, 4}, {3, 2}, {1, 3}, {2, 3}},
		FrequencyData -> True,
		HistogramCategories -> {{cx0, cx1, cx2, cx3, cx4, cx5},
			{cy0, cy1, cy2}}]
	   implies that there are 3 data in the range 
		cx2 <= x < cx3 && cy0 <= y <= cy1 .
	*)
        (* Define countdata, numberOfBins, xmin, xmax, cutoffs. *)
         

        countdata = mat;
	{numberOfBinsX, numberOfBinsY} = Dimensions[countdata]; 
	countdataX = Map[Apply[Plus, #]&, countdata];
	countdataY = Map[Apply[Plus, #]&, Transpose[countdata]];

	(* Error check for HistogramCategories setting. *)
	If[hcat === Automatic, hcat = {Automatic, Automatic}];
	If[!( ListQ[hcat] && Length[hcat] == 2 &&
	     (hcat[[1]]===Automatic || monotoneIncreasingVectorQ[hcat[[1]]]) &&
	     (hcat[[2]]===Automatic || monotoneIncreasingVectorQ[hcat[[2]]]) ),
           Message[Histogram3D::fdhc, hcat];
           hcat = {Automatic, Automatic}   ];
	{hcatX, hcatY} = hcat;

	{dataminX, datamaxX} = findRange[rangeX,
		If[hcatX === Automatic, Range[0, numberOfBinsX], hcatX]
		];
	{dataminY, datamaxY} = findRange[rangeY,
		If[hcatY === Automatic, Range[0, numberOfBinsY], hcatY]
                ];

	If[hcatX === Automatic,
                cutoffsX = dataminX + (datamaxX-dataminX)/numberOfBinsX*
			Range[0, numberOfBinsX],
                cutoffsX = findCutoffs1[hcatX, dataminX, datamaxX, countdataX];
                numberOfBinsX = Length[cutoffsX]-1
           ];
        {binminX, binmaxX} = {First[cutoffsX], Last[cutoffsX]};
	If[hcatY === Automatic,
                cutoffsY = dataminY + (datamaxY-dataminY)/numberOfBinsY*
			Range[0, numberOfBinsY],
                cutoffsY = findCutoffs1[hcatY, dataminY, datamaxY, countdataY];
                numberOfBinsY = Length[cutoffsY]-1
           ];
        {binminY, binmaxY} = {First[cutoffsY], Last[cutoffsY]},

	
	(* ===================================================== *)
        (* PROCESS LIST assuming that it represents RAW DATA. *)
        (* Define countdata, numberOfBins, binmin, binmax, cutoffs. *)
        (* ===================================================== *)

        (* === Define min and max of rangeX, and count x data in range. ==== *)
	listX = Map[First, mat];
        {dataminX, datamaxX} = findRange[rangeX, listX];
        (* === Define min and max of rangeY, and count y data in range. ==== *)
	listY = Map[Last, mat];
        {dataminY, datamaxY} = findRange[rangeY, listY];

	(* ================== Count data in range. ===================== *)
	totalcount = Count[mat, {x_, y_} /; dataminX <= x <= datamaxX &&
						dataminY <= y <= datamaxY];

	(* =========== Define category cutoffs for raw data. =========== *)
	(* Error check for HistogramCategories setting. *)
        (* Note that RangeCounts considers intervals of the form
                {{binminX <= x < etc, binminY <= y < etc}, ..., 
		 {etc <= x < binmaxX, etc <= y < binmaxY}}. *)
	(* Define {cutoffsX, cutoffsY. *)
	If[hcat === Automatic || PositiveIntegerQ[hcat],
	   (* look at the distribution as a whole *)
	   cutoffs = findCutoffs3[hcat,
		 {dataminX, datamaxX}, {dataminY, datamaxY}, 
		totalcount, approximate];
	   If[Head[cutoffs] === findCutoffs3, Return[$Failed]];
	   {cutoffsX, cutoffsY} = cutoffs,	
	   (* look at the x and y components separately *)
	   If[ListQ[hcat] && Length[hcat] == 2 &&
	      (hcat[[1]]===Automatic || monotoneIncreasingVectorQ[hcat[[1]]] ||
		PositiveIntegerQ[hcat[[1]]]) &&
	      (hcat[[2]]===Automatic || monotoneIncreasingVectorQ[hcat[[2]]] ||
		PositiveIntegerQ[hcat[[2]]]) ,
	      (* = HistogramCategories points to a valid list of two items. = *)
	      {hcatX, hcatY} = hcat;
              cutoffsX = findCutoffs2[hcatX, dataminX, datamaxX,
		 If[IntegerQ[hcatY], totalcount/hcatY, (totalcount)^(2/3)],
                 approximate];
              cutoffsY = findCutoffs2[hcatY, dataminY, datamaxY,
		 If[IntegerQ[hcatX], totalcount/hcatX, (totalcount)^(2/3)],
                 approximate],
	      (* ==== HistogramCategories points to an invalid setting. ==== *)	
              Message[Histogram3D::rdhc, hcat];
	      cutoffs = findCutoffs3[hcat,
		 {dataminX, datamaxX}, {dataminY, datamaxY}, 
		totalcount, approximate];
              If[Head[cutoffs] === findCutoffs3, Return[$Failed]];
              {cutoffsX, cutoffsY} = cutoffs	
	   ]
	];
        numberOfBinsX = Length[cutoffsX]-1;
        {binminX, binmaxX} = {First[cutoffsX], Last[cutoffsX]};
        numberOfBinsY = Length[cutoffsY]-1;
        {binminY, binmaxY} = {First[cutoffsY], Last[cutoffsY]};


	(* ========== Warning messages for points not plotted. ======== *)
	(* If histogram range is to be determined automatically, *)
        (* presumably because the user wants all points to be plotted, *)
        (* warn user if some points will be excluded from histogram. *)
        If[ (rangeX === Automatic ||
             (VectorQ[rangeX] && Length[rangeX] == 2 &&
                 rangeX[[1]] === Automatic) ) &&
                        dataminX < binminX,
             leftTailCountX = Count[mat, z_ /; z[[1]] < binminX];
             If[leftTailCountX == 1,
                Message[Histogram3D::lt1, leftTailCountX, "x", binminX] ];
             If[leftTailCountX > 1,
                Message[Histogram3D::lt, leftTailCountX, "x", binminX] ]
	];
        If[ (rangeX === Automatic ||
             (VectorQ[rangeX] && Length[rangeX] == 2 &&
                 rangeX[[2]] === Automatic) ) &&
                        datamaxX >= binmaxX,
             rightTailCountX = Count[mat, z_ /; z[[1]] >= binmaxX];
             If[rightTailCountX == 1,
                Message[Histogram3D::gtet1, rightTailCountX, "x", binmaxX] ];
             If[rightTailCountX > 1,
                Message[Histogram3D::gtet, rightTailCountX, "x", binmaxX] ]
	];
        If[ (rangeY === Automatic ||
             (VectorQ[rangeY] && Length[rangeY] == 2 &&
                 rangeY[[1]] === Automatic) ) &&
                        dataminY < binminY,
             leftTailCountY = Count[mat, z_ /; z[[2]] < binminY];
             If[leftTailCountY == 1,
                Message[Histogram3D::lt1, leftTailCountY, "y", binminY] ];
             If[leftTailCountY > 1,
                Message[Histogram3D::lt, leftTailCountY, "y", binminY] ];
	];
        If[ (rangeY === Automatic ||
             (VectorQ[rangeY] && Length[rangeY] == 2 &&
                 rangeY[[2]] === Automatic) ) &&
                        datamaxY >= binmaxY,
             rightTailCountY = Count[mat, z_ /; z[[2]] >= binmaxY];
             If[rightTailCountY == 1,
                Message[Histogram3D::gtet1, rightTailCountY, "y", binmaxY] ];
             If[rightTailCountY > 1,
                Message[Histogram3D::gtet, rightTailCountY, "y", binmaxY] ]
	];

	(* =========== Compute category counts for raw data. =========== *)
        countdata = RangeCounts[mat, cutoffsX, cutoffsY];
        If[Head[countdata] === RangeCounts,
                Message[Histogram3D::rcount];
                Return[$Failed] ];
        (* Dimensions of data should be {numberOfBinsX+2, numberOfBinsY+2}.
                Eliminate elements
             of data corresponding to the ranges x < binminX and x >= binmaxX
		and y < binminY and y >= binmaxY. *)
	(* eliminate first and last row *)
        countdata = Drop[Drop[countdata, 1], -1]; 
	(* eliminate first and last col *)
	countdata = Transpose[Drop[Drop[Transpose[countdata], 1], -1]]

     ]; (* end If TrueQ[fdata] *)
	
     (* ============================================================= *)
     (* ============================================================= *)
     (* Use countdata, cutoffsX, cutoffsY, numberOfBinsX, numberOfBinsY *)
     (* 	 binminX, binmaxX, binminY, and binmaxY to   *)
     (*              generate histogram.                              *)
     (* ============================================================= *)
     (* ============================================================= *)

     (* ================= Scale category counts. ================ *)
     binwidthsX = Drop[cutoffsX, 1] - Drop[cutoffsX, -1];
     binwidthsY = Drop[cutoffsY, 1] - Drop[cutoffsY, -1];
     binareas = Outer[Times, binwidthsX, binwidthsY];
     binareasFlatList = Flatten[binareas];
     If[TrueQ[scale] || !Apply[Equal, binareasFlatList],
           (* Make the volume of the solid bar proportional to the frequency
		associated with the bar. *)
	   countdata = countdata/binareas
     ];
     bincentersX = Drop[FoldList[Plus, binminX, binwidthsX], -1] +
                1/2 binwidthsX;
     bincentersY = Drop[FoldList[Plus, binminY, binwidthsY], -1] +
                1/2 binwidthsY;

     (* =============================================================== *)
     (*  Define category axis ticks from                                *)
     (*      bincenters, countdata, and ticks.                          *)
     (* =============================================================== *)
     (* Note that it is not possible to figure out what Automatic setting of
	   category axis ticks would be for, say, ScatterPlot3D of
	   data3D = Flatten[MapThread[Append, {bincenters, countdata}, 2], 1],
	   where bincenters = Outer[List, bincentersX, bincentersY],
	   because FullOptions does not work for extracting the Ticks setting
           of a Graphics3D object.  So "autoticks" cannot be defined as in
	   the case of Graphics`Graphics`Histogram. ECM '97. *)	

     (* Process the Ticks setting. *)
     If[MatchQ[ticks, Automatic | IntervalCenters | IntervalBoundaries],
                ticks = {ticks, ticks, Automatic}];
     If[ticks === None, ticks = {None, None, None}];
     (* Check the Ticks setting, and reset to Automatic if the setting is
                illegal. *)
     If[!(ListQ[ticks] && Length[ticks] == 3 && ticksCheckQ[ticks]),
           Message[Histogram3D::ticks, ticks];
           ticks = {Automatic, Automatic, Automatic}];
     axisticksX = Switch[ticks[[1]],
                _?ListQ, (* ticksCheckQ has already checked for
                                monotoneIncreasingVectorQ *)
                   Map[neatTick, ticks[[1]] ],
                IntervalBoundaries,
                   (
                     Map[neatTick, cutoffsX]
                   ),
                IntervalCenters,
                   (
                     Map[neatTick, bincentersX]
                   ),
                None, (* no category axis ticks *)
                        None,
                _, (* place x axis ticks automatically *)
                        Automatic
        ];
     axisticksY = Switch[ticks[[2]],
                _?ListQ, (* ticksCheckQ has already checked for
                                monotoneIncreasingVectorQ *)
                        Map[neatTick, ticks[[2]] ],
                IntervalBoundaries,
                   (
                     Map[neatTick, cutoffsY]
                   ),
                IntervalCenters,
                   (
                     Map[neatTick, bincentersY]
                   ),
                None, (* no category axis ticks *)
                        None,
                _, (* place y axis ticks automatically *)
                        Automatic
        ];
     ticks = {axisticksX, axisticksY, ticks[[3]]};

     (* ==================================================================== *)
     (* ======= Define phwdata (position, height, width). ================== *)
     (* ======= { {{xpos1, ypos1}, height1, {xwidth1, ywidth1}}, ...} ====== *)
     (* ==================================================================== *)
     (* Note that XSpacing and YSpacing are each assumed to be 0 here.  *)
     (* If you want to add that option to Histogram3D, the option should be *)
     (* processed here.  (Some would say that histograms of discrete data *)
     (* ought to have columns separated from each other, i.e., with  *)
     (* XSpacing or YSpacing greater than zero, depending on whether the *)
     (* x or y variable is discrete.) *)

     bincenters = Outer[List, bincentersX, bincentersY]; (* position *)
     binwidths = Outer[List, binwidthsX, binwidthsY]; (* widths *)
     phwdata = Flatten[
	MapThread[Join,
		 {bincenters, Map[List, countdata, {2}], binwidths},
		 2], 1];
     (* Now phwdata has the form {{xpos1, ypos1, height1, xwidth1, ywidth1},
	{xpos2, ypos2, height2, xwidth2, ywidth2}...}. *)
     phwdata = Map[{#[[{1, 2}]], #[[3]], #[[{4, 5}]]}&, phwdata];


     (* ============= Extract any other options relevent to Graphics3D. ==== *)
     groptslist = {FilterOptions[Graphics3D, opts]};

     (* ====== Scale solid bar heights according to HistogramScale -> k ==== *)
     (* NOTE that phwdata has the form...
	{  {{xpos1, ypos1}, height1, {xwidth1, ywidth1}},
	   {{xpos2, ypos2}, height2, {xwidth2, ywidth2}}, ...} *)
     If[NumberQ[scale] && FreeQ[scale, Complex] && scale > 0,
        volume = Apply[Plus, Map[(#[[2]]#[[3, 1]]#[[3, 2]])&, phwdata]];
	phwdata = Map[{#[[1]], #[[2]] scale/volume, #[[3]]}&, phwdata]
     ];

     (* =================== GeneralizedBarChart3D =================== *)
     GeneralizedBarChart3D[phwdata,
	PlotRange -> All,	(* option of Graphics3D *)
	SolidBarEdges->sbedges,	(* option of GeneralizedBarChart3D *)
   	SolidBarEdgeStyle->sbedgestyle, (* option of GeneralizedBarChart3D *)
   	SolidBarStyle->sbstyle, (* option of GeneralizedBarChart3D *)
	Ticks -> ticks,		(* option of Graphics3D *)
	(* groptslist includes any other options relevent to Graphics3D *)
	Apply[Sequence, groptslist]
     ]
  ) /;  (
	{approximate, fdata, hcat, range, scale, sbedges, sbedgestyle, sbstyle, 
	 ticks} =
	 {ApproximateIntervals, FrequencyData, HistogramCategories,
	  HistogramRange, HistogramScale,
		SolidBarEdges, SolidBarEdgeStyle, SolidBarStyle,
	  	Ticks} /. {opts} /. Options[Histogram3D];
	dim = Dimensions[mat];
	If[TrueQ[fdata] && ListQ[hcat] && Length[hcat] == 2,
	   If[VectorQ[hcat[[1]]],
	      If[dim[[1]]+1 == Length[hcat[[1]]],
		 True,
		 Message[Histogram3D::fdfail];  False],
	      True] &&
	   If[VectorQ[hcat[[2]]],
	      If[dim[[2]]+1 == Length[hcat[[2]]],
	         True,
		 Message[Histogram3D::fdfail];  False],
	      True],
           True] &&
	If[!TrueQ[fdata],
	   If[Length[mat[[1]]] != 2,
	      Message[Histogram3D::rd2d];  False,
	      True],
	   True]	
        )
	
 ] (* end Histogram3D *)
 

(* Interpret the HistogramCategories option when the
        data is frequency data. *)
findCutoffs1[hcat_, datamin_, datamax_, data_] :=
        Module[{countdata = data, cutoffs = hcat, n},
                (* If range specifies something more restrictive than
                        the given categories, then trim cutoffs. *)
                If[datamin >= First[cutoffs],
                   While[!(cutoffs[[1]] <= datamin < cutoffs[[2]]),
                        countdata = Drop[countdata, 1];
                        cutoffs = Drop[cutoffs, 1]]
                ];
                If[datamax < Last[cutoffs],
                   While[!(n = Length[cutoffs];
                       cutoffs[[n-1]] <= datamax < cutoffs[[n]]),
                        countdata = Drop[countdata, -1];
                        cutoffs = Drop[cutoffs, -1]]
                ];
                cutoffs
        ] (* end findCutoffs1 *)


(* Interpret the HistogramCategories option when the
        data is raw data, FOR A SINGLE COMPONENT x or y. *)
(* Example:
       findCutoffs2[hcatX, dataminX, datamaxX, totalcount, approximate]
*)
findCutoffs2[hcat_, datamin_, datamax_, totalcount_, approximate_] :=
   Module[{numberOfBins, cutoffs, binmin, binmax, bindelta},
           If[monotoneIncreasingVectorQ[hcat],
              (* Intervals are NOT approximated when they are
                        specifically requested using HistogramCategories. *)
              If[!(approximate===Automatic || approximate===False),
                         Message[Histogram3D::noapprox, approximate]];
              cutoffs = hcat;
              (* If range specifies something more restrictive than
                                the given categories, then trim cutoffs. *)
              If[datamin >= First[cutoffs],
                 While[!(cutoffs[[1]] <= datamin < cutoffs[[2]]),
                                cutoffs = Drop[cutoffs, 1]]
              ];
              If[datamax < Last[cutoffs],    
                 While[!(n = Length[cutoffs];
                       cutoffs[[n-1]] <= datamax < cutoffs[[n]]),
                        cutoffs = Drop[cutoffs, -1]]
              ],
	      (* ==================================================== *)
              (* hcat === Automatic || PositiveIntegerQ[hcat] *)
              If[PositiveIntegerQ[hcat],
                 numberOfBins = hcat,
                 (* hcat === Automatic *)
                 numberOfBins = Sqrt[totalcount]
	      ];
              If[approximate === Automatic || TrueQ[approximate],
		 (* make the intervals approximate and make them neat *)
                 {binmin, binmax, bindelta} =
                        approximateIntervals[datamin, datamax, numberOfBins];
                 numberOfBins = Round[(binmax-binmin)/bindelta],
		 (* make the cutoffs exact, ignore neatness *)
		 numberOfBins = Round[numberOfBins];
                 {binmin, binmax, bindelta} =
                         {datamin, datamax, (datamax-datamin)/numberOfBins}
              ];
              cutoffs = binmin + bindelta Range[0, numberOfBins]
           ]; (* end If monotoneIncreasingVectorQ[hcat] *)
           cutoffs
   ] (* end findCutoffs2 *)



(* Interpret the HistogramCategories option when the
        data is raw data, FOR BOTH COMPONENTS x and y. *)
(* Example:
       findCutoffs3[hcat, {dataminX, datamaxX}, {dataminY, datamaxY},
		 totalcount, approximate]
		returns
	{cutoffsX, cutoffsY}
*)
findCutoffs3[hcat_, {dataminX_, datamaxX_}, {dataminY_, datamaxY_},
	 totalcount_, approximate_] :=
   Module[{numberOfBinsOnAnAxis, binminX, binmaxX, bindeltaX,
		binminY, binmaxY, bindeltaY, numberOfBinsX, numberOfBinsY},
	   If[PositiveIntegerQ[hcat],
	      (* hcat gives the total number of bins desired by the user *)
	      numberOfBinsOnAnAxis = Sqrt[hcat],
	      (* hcat === Automatic *)
	      (* NOTE formerly used
                    numberOfBins = Ceiling[(totalcount)^(1/3)]; *)	
	      numberOfBinsOnAnAxis = (totalcount)^(1/3)
	   ];
       If[numberOfBinsOnAnAxis < 2, numberOfBinsOnAnAxis = 2];
	   If[approximate === Automatic || TrueQ[approximate],
	      (* make the cutoffs approximate and make them neat *)
	      {binminX, binmaxX, bindeltaX} =
                   approximateIntervals[dataminX, datamaxX,
			 numberOfBinsOnAnAxis];
              numberOfBinsX = Round[(binmaxX-binminX)/bindeltaX];
	      {binminY, binmaxY, bindeltaY} =
                   approximateIntervals[dataminY, datamaxY,
			 numberOfBinsOnAnAxis];
              numberOfBinsY = Round[(binmaxY-binminY)/bindeltaY],
	      (* make the cutoffs exact, ignore neatness *)	
	      numberOfBinsX = numberOfBinsY = Round[numberOfBinsOnAnAxis];	
              {binminX, binmaxX, bindeltaX} =
                   {dataminX, datamaxX,
			 (datamaxX-dataminX)/numberOfBinsX};
              {binminY, binmaxY, bindeltaY} =
                   {dataminY, datamaxY,
			 (datamaxY-dataminY)/numberOfBinsY}
           ];
	   (* returning {cutoffsX, cutoffsY} *)
	   {binminX + bindeltaX Range[0, numberOfBinsX],
	    binminY + bindeltaY Range[0, numberOfBinsY]}
   ] (* end findCutoffs3 *)



neatTick[t_] := If[TrueQ[Round[t]==t], Round[t],
                   If[Head[t] === Rational, N[t], t]]


(* interpret the HistogramRange option *)
findRange[range_, list_] :=
  Module[{min, max},
   (
        {min, max} = {Min[list], Max[list]};
        max += 10 $MachineEpsilon; (* this is done so that the maximum data 
                                point is included in an interval that is closed 
                                on the left and open on the right *)
        Switch[range,
                Automatic | {Automatic, Automatic}, {min, max},
                {l_?NumberQ, u_?NumberQ} /; FreeQ[{l, u}, Complex] && l < u,
                                range,
                {l_?NumberQ, Automatic} /; FreeQ[l, Complex] && l < max,
                                {range[[1]], max},
                {Automatic, u_?NumberQ} /; FreeQ[u, Complex] && min < u,
                                {min, range[[2]]},
                _, (Message[Histogram::range, range];
                    {min, max})
        ]
   )
  ]


ticksCheckQ[{x_, y_, z_}] :=
        (x === None || x === Automatic || monotoneIncreasingVectorQ[x] ||
         x === IntervalBoundaries || x === IntervalCenters) &&
        (y === None || y === Automatic || monotoneIncreasingVectorQ[y] ||
         y === IntervalBoundaries || y === IntervalCenters) &&
        (z === None || z === Automatic || monotoneIncreasingVectorQ[z])

monotoneIncreasingVectorQ[x_] :=
   Module[{positions},
        positions = If[VectorQ[x], x, Map[If[ListQ[#], First[#], #]&, x] ];
        VectorQ[positions, NumberQ] && FreeQ[positions, Complex] &&
        Apply[Less, positions]
   ]

(* RangeCounts expects cutoffs {c0, c1, c2, ..., cm, cn}, specifying
        intervals {c0 <= x < c1, c1 <= x < c2, ..., cm <= x < cn}.
   Thus the {first, last, delta} returned by approximateIntervals specifies
        {first <= x < first + delta, ..., last - delta <= x < last}.
*)
(* approximateIntervals[min, max, numOfInt] defines a set of
        approximately numOfInt intervals, covering the range {min, max},
        and having boundaries expressible in terms of simple numbers. *)
approximateIntervals[min_, max_, numOfInt_] :=
   Module[{nmin = N[min], nmax = N[max], spacing, t,
                 nicebins, first, last, delta},
        (* start with handling the bad case of min and max being so
           close together that having multiple bins doesn't make sense;
           user can override with specific bins if this exceptional case
           is actually desired. *)
        If[Abs[(max - min)/(spacing = If[# == 0., 1, #]&[Max[Abs[{min, max}]]])] < 10^-5,
            spacing = 0.2 spacing;
            Return[{min - 1.5 spacing, min + 1.5 spacing, spacing}]
        ];
        (* ======= The following code is similar to LinearScale. ===== *)
        (* It uses TickSpacing[, Nearest], rather than the default
                 TickSpacing[, GreaterEqual]. *)
        spacing = TickSpacing[nmax-nmin, numOfInt,
                 {1, 2, 2.5, 5, 10}, Nearest];
        t = Range[Ceiling[nmin/spacing - 0.05] spacing, max, spacing] ;
        nicebins = Map[{#, If[Round[#]==#, Round[#], #]}&, t];
        (* =========================================================== *)
        {first, last} = {First[nicebins][[1]], Last[nicebins][[1]]};
        delta = nicebins[[2, 1]]-first;
        (* If x < first, then x will not be counted in an interval
                {first <= x < first + delta}.
           If x >= last, then x will not be counted in an interval.
                {last - delta <= x < last.
           Keep adding intervals until all points min <= x <= max are
                counted. *)
        While[min < first || max >= last,
        (* Make sure that min and max are included in default categories. *)
         If[min < first,
           nicebins = Join[
                Map[{#, If[Round[#]==#, Round[#], #]}&, {first-delta}],
                nicebins]
         ];
         If[max >= last,
           nicebins = Join[
                nicebins,
                Map[{#, If[Round[#]==#, Round[#], #]}&, {last+delta}]]
         ];
         {first, last} = {First[nicebins][[1]], Last[nicebins][[1]]}
        ];
        {first, last, delta}
   ]

PositiveIntegerQ[n_] := IntegerQ[n] && n > 0



(* GeneralizedBarChart3D *)

Options[GeneralizedBarChart3D] =
  {
   SolidBarEdges -> True,
   SolidBarEdgeStyle -> GrayLevel[0],
   SolidBarStyle -> SurfaceColor[GrayLevel[1]]
  };



(* NOTE that BarSpacing is NOT an option of
	Graphics`Graphics`GeneralizedBarChart,
	so XSpacing and YSpacing are NOT options of
	Graphics`Graphics3D`GeneralizedBarChart3D. *)
(* NOTE that the data in "list" are of the form...
	{  {{xpos1, ypos1}, height1, {xwidth1, ywidth1}},
	   {{xpos2, ypos2}, height2, {xwidth2, ywidth2}}, ...}
*)
GeneralizedBarChart3D[
	list:{ {{_?numberQ,_?numberQ}, _?numberQ, {_?numberQ,_?numberQ}}... },
	opts___] :=
  Module[{edges, g3dopts, barstyle},
   (
    If[TrueQ[SolidBarEdges/.Flatten[{opts, Options[GeneralizedBarChart3D]}]],
	    edges = EdgeForm[SolidBarEdgeStyle/.Flatten[{opts, Options[GeneralizedBarChart3D]}]],
	    edges = EdgeForm[]
    ];
    barstyle = SolidBarStyle/.Flatten[{opts, Options[GeneralizedBarChart3D]}];
    g3dopts = ({FilterOptions[Graphics3D, ##]}&) @@ 
		Flatten[{opts, Options[GeneralizedBarChart3D]}];
    Show[
   	Graphics3D[{If[ListQ[barstyle], Sequence @@ barstyle, barstyle], Map[Flatten[{edges,
   		Cuboid[{#[[1,1]]-#[[3, 1]]/2, #[[1,2]]-#[[3, 2]]/2, 0},
		       {#[[1,1]]+#[[3, 1]]/2, #[[1,2]]+#[[3, 2]]/2, #[[2]]}]
		      }]&,
		   list]}],
	Flatten[Join[
	   {g3dopts},
	   {Axes -> Automatic, BoxRatios -> {1,1,1}, PlotRange -> All} 
		]  ]
    ]
   ) 
  ] (* end GeneralizedBarChart3D *)                                        


(* ScatterPlot3D *)

Options[ScatterPlot3D] =
    Join[{PlotJoined->False,
                  PlotStyle -> GrayLevel[0]},
                 Options[Graphics3D]
    ];

SetOptions[ScatterPlot3D, Axes -> True]

ScatterPlot3D[l3:{{_, _, _}..}, opts___?OptionQ] :=
Module[{sty, join},
        {sty, join} = {PlotStyle, PlotJoined}/.Flatten[{opts}]/.
                   Options[ScatterPlot3D];
        If[join,
                Show[Graphics3D[ Flatten[{ sty, Line[l3] }] ],
                        FilterOptions[ Graphics3D, ##] & @@
                            Flatten[{opts, Options[ScatterPlot3D]}]
                ],
                Show[Graphics3D[ Flatten[{ sty, Map[Point,l3] }] ],
                        FilterOptions[ Graphics3D, ##] & @@
                            Flatten[{opts, Options[ScatterPlot3D]}]
                ]
        ]
]

(* Make Polygons from ParametricPlot3D.m by Roman Maeder.
		Used in ListSurfacePlot. *)

MakePolygons[vl_List] :=
    Module[{l = vl,
    	   l1 = Map[RotateLeft, vl],
    	   mesh},
	mesh = {l, l1, RotateLeft[l1], RotateLeft[l]};
	mesh = Map[Drop[#, -1]&, mesh, {1}];
	mesh = Map[Drop[#, -1]&, mesh, {2}];
	Polygon /@ Transpose[ Map[Flatten[#, 1]&, mesh] ]
    ]

(* ListSurfacePlot3D *)

Options[ListSurfacePlot3D] = Options[Graphics3D];

ListSurfacePlot3D[t3_List, opts___] :=
	Show[Graphics3D[MakePolygons[t3]],
		 FilterOptions[ListSurfacePlot3D,opts]] 

(* Modified MakePolygon, used in ShadowPlot3D *)

MakePolygonCoords[vl_List] :=
    Module[{l = vl,
    	   l1 = Map[RotateLeft, vl],
    	   mesh},
	mesh = {l, l1, RotateLeft[l1], RotateLeft[l]};
	mesh = Map[Drop[#, -1]&, mesh, {1}];
	mesh = Map[Drop[#, -1]&, mesh, {2}];
	Transpose[ Map[Flatten[#, 1]&, mesh] ]
    ]

(* ShadowPlot3D *)

Options[ShadowPlot3D] = 
	Join[{PlotPoints->15, Color->True, ColorFunction -> Hue,
		SurfaceMesh->True,
		SurfaceMeshStyle -> RGBColor[0,0,0],
		ShadowMesh -> True, 
		ShadowMeshStyle -> RGBColor[0,0,0],
		ShadowPosition -> -1}, Options[Graphics3D]];

SetOptions[ShadowPlot3D, BoxRatios -> {1,1,1}]

ShadowPlot3D[func_, {u_, umin_, umax_},  {v_, vmin_, vmax_}, opts___?OptionQ] :=
	Module[{plotpoints = PlotPoints /. {opts} /. Options[ShadowPlot3D]},
		SP0[MakePolygonCoords[
			Table[N[{u, v, func}],
				{u,umin,umax,(umax-umin)/plotpoints},
				{v,vmin,vmax,(vmax-vmin)/plotpoints}]],
			Flatten[{{opts},Options[ShadowPlot3D]}]]]
		

(* ListShadowPlot3D *)

Options[ListShadowPlot3D] = 
	Join[{Color -> True, ColorFunction -> Hue, SurfaceMesh->True, 
	SurfaceMeshStyle -> RGBColor[0,0,0],
	ShadowMesh -> True,
	ShadowMeshStyle -> RGBColor[0,0,0],
	ShadowPosition -> -1}, Options[Graphics3D]];

SetOptions[ListShadowPlot3D, BoxRatios -> {1,1,1}];

ListShadowPlot3D[list_?MatrixQ, opts___?OptionQ] :=
	SP0[MakePolygonCoords[MapIndexed[Append[Reverse[#2], #1]&, N[list],{2}]],
		Flatten[{{opts},Options[ListShadowPlot3D]}]]

SP0[list_, opts___] :=
	Module[{gopts, z, zmin, zmax, zrange, zshadow, shades,
			color, colorfunc, surfacemesh, surfacemeshstyle, 
			shadowmesh, shadowmeshstyle, pos, g},
		gopts = FilterOptions[Graphics3D, ##] & @@  opts;
		{color, colorfunc, surfacemesh, surfacemeshstyle, shadowmesh,
			 shadowmeshstyle, pos} = 
			{Color, ColorFunction, SurfaceMesh, SurfaceMeshStyle, 
				ShadowMesh, ShadowMeshStyle, 
				ShadowPosition} /. opts;
		z = Map[#[[3]]&,list,{-2}];
		{zmin, zmax} = {Min[z], Max[z]};
		zrange = zmax - zmin;
		zshadow = If[!TrueQ[pos == -1],
					zmax + zrange/2,
					zmin - zrange/2];
	        colorfunc = If[TrueQ[color], colorfunc, GrayLevel];
                shades = Map[colorfunc,
                                (Apply[Plus,z,{-2}]/4 - zmin)/zrange
                ];
                g = Transpose[{shades,Polygon /@ list}];
                Show[
                        Graphics3D[
                                {If[TrueQ[surfacemesh],
                                        EdgeForm[surfacemeshstyle],EdgeForm[]],
                                 g,
                                 If[TrueQ[shadowmesh],
                                        EdgeForm[shadowmeshstyle],EdgeForm[]],
                                tg3D[g, {#[[1]],#[[2]],zshadow}& ]}
                        ],
                        Flatten[{gopts, BoxRatios->{1,1,1}}]
                    ]
         ]


(* TransformGraphcs3D *)
TransformGraphics3D[obj_, f_] := tg3D[obj, f]

cuboidtopolygons[Cuboid[{xmin_, ymin_, zmin_}]] :=
        cuboidtopolygons[Cuboid[{xmin, ymin, zmin}, {xmin, ymin, zmin} + 1]]

cuboidtopolygons[Cuboid[{xmin_, ymin_, zmin_}, {xmax_, ymax_, zmax_}]] :=
  {Polygon[{{xmax, ymax, zmax}, {xmin, ymax, zmax}, {xmin, ymin, zmax},
     {xmax, ymin, zmax}}], Polygon[{{xmax, ymax, zmax}, {xmax, ymin, zmax},
     {xmax, ymin, zmin}, {xmax, ymax, zmin}}],
   Polygon[{{xmax, ymax, zmax}, {xmax, ymax, zmin}, {xmin, ymax, zmin},
     {xmin, ymax, zmax}}], Polygon[{{xmin, ymax, zmax}, {xmin, ymax, zmin},
     {xmin, ymin, zmin}, {xmin, ymin, zmax}}],
   Polygon[{{xmin, ymin, zmin}, {xmin, ymax, zmin}, {xmax, ymax, zmin},
     {xmax, ymin, zmin}}], Polygon[{{xmin, ymin, zmax}, {xmin, ymin, zmin},
     {xmax, ymin, zmin}, {xmax, ymin, zmax}}]}

tg3D[Graphics3D[g_, o___], f_, p_:0] := Graphics3D[tg3D[g, f, p], o]

tg3D[d_List, f_, p_:0] := Map[ tg3D[#, f, p]& , d ]

tg3D[GraphicsComplex[pts_, prims_, o___], f_, p_:0] :=
    GraphicsComplex[If[VectorQ[pts], pts, Map[f, pts]],
         tg3D[prims, f, Length[pts]]]

tg3D[Point[d], f_, p_:0]/;(p === 0 || !gcq[d, p]) := Point[f[d]]

tg3D[Line[d_List], f_, p_:0]/;(p === 0 || !gcq[d, p]) := Line[Map[f, d, {-2}]]

tg3D[Polygon[d_List], f_, p_:0]/;(p === 0 || !gcq[d, p]) := Polygon[Map[f, d, {-2}]]

tg3D[Cylinder[d:{{_,_,_},{_,_,_}}, r_?numberQ, t___], f_, p_:0] :=
    Circle[f[d], r, t]

tg3D[Sphere[d_List, r_?numberQ, t___], f_, p_:0] :=
        Sphere[f[d], r, t] 

tg3D[Text[expr_, d_List, opts___], f_, p_:0] := Text[expr, f[d], opts]

tg3D[c:Cuboid[{_,_,_}], f_, p_:0] :=
    tg3D[cuboidtopolygons[c], f, p]
    
tg3D[c:Cuboid[{_,_,_}, {_,_,_}], f_, p_:0] :=
    tg3D[cuboidtopolygons[c], f, p]

tg3D[expr_, f_, p_:0] := expr

(* transform an entire primitive, not just coordinates. This may be being too clever
    than necessary... In this case, the transform function needs to have specific
    knowledge of multipoints, multilines, etc. and of what to do inside a
    GraphicsComplex; if the f is being applied inside a GC, it will be given
    two arguments (obj and pointlist), and there will be two variables (newpoints
    and newcount) that the transform function can depend upon. The transform
    function must do the work of incrementing the counter, and adding rules
    of the form pthead[count] -> coord to newpoints. (Clearly less than ideal,
    this needs to be thought about and improved.) Also note global (in this context)
    symbol pthead used as a wrapper for new coordinates. *)
tg3Ds[Graphics3D[g_, o___], f_, p_:0] := Graphics3D[tg3Ds[g, f, p], o]

tg3Ds[d_List, f_, p_:0] := Map[ tg3Ds[#, f, p]& , d ]

tg3Ds[GraphicsComplex[pts_, prims_, o___], f_, p_:0] :=
 Module[{tg, allpoints = pts, tmppoints},
    Block[{newpoints = {}, newcount = 0},
       tg = tg3Ds[prims, f, If[VectorQ[pts] && p =!= 0, p[[pts]], pts]];
       tmppoints = newpoints;
       If[newpoints != {},
           tg = tg/.newpoints;
           allpts = allpts ~Join~ (Last /@ newpoints)
       ]
    ];
    If[VectorQ[pts] && p =!= 0,
        tmppoints = Map[(newcount++; pthead[newcount] -> Last[#])&, tmppoints];
        newpoints = newpoints ~Join~ tmppoints;
        allpoints = pts ~Join~ (First /@ tmppoints),
      (* else *)
        allpoints = pts ~Join~ (Last /@ tmppoints)
    ];
    GraphicsComplex[allpoints, tg]
]

tg3Ds[Point[d], f_, p_:0]/;(p === 0 || !gcq[d, p]) := f[Point[d]]

tg3Ds[Line[d_List], f_, p_:0]/;(p === 0 || !gcq[d, p]) := f[Line[d]]

tg3Ds[Polygon[d_List], f_, p_:0]/;(p === 0 || !gcq[d, p]) := f[Polygon[d]]

tg3Ds[c:Cylinder[d:{{_,_,_},{_,_,_}}, r_?numberQ, t___], f_, p_:0] := f[c]

tg3Ds[s:Sphere[d_List, r_?numberQ, t___], f_, p_:0] := f[s]

tg3Ds[t:Text[expr_, d_List, opts___], f_, p_:0] := f[t]

tg3Ds[c:Cuboid[{_,_,_}], f_, p_:0] := f[c]
    
tg3Ds[c:Cuboid[{_,_,_}, {_,_,_}], f_, p_:0] := f[c]

tg3Ds[expr_, f_, 0] := expr

tg3Ds[expr_, f_, pts_List] := f[expr, pts]

(* gcq tests for compliance with a GraphicsComplex indexed primitive *)
gcq[_Integer] := True
gcq[pts:{__Integer}, count_]/;(Max[pts] <= count && Min[pts] >= 1) := True
gcq[pts:{{_Integer..}..}, count_]/;(Max[pts] <= count && Min[pts] >= 1) := True
gcq[any_] := False

(* SkewGraphics3D *)

SkewGraphics3D[g_Graphics3D, m_?MatrixQ] :=
	TransformGraphics3D[g, (m . #)&]

(* Project *)

Project[g_Graphics3D, point:{_,_,_}] :=
	Module[{p1, p2, t1, t2, t3, b1, b2,c},
		p1 = PlotRange[g];
		c = Map[((#[[1]] + #[[2]])/2)&,p1];
		p2 = point-c;
		t1 = If[TrueQ[(t2 = cross[{0,0,1},p2]) == {0,0,0}],
				cross[{0,1,0},p2],t2];
		b1 = normalize[t1];
		t3 = cross[p2,b1];
		b2 = normalize[t3];
		Project[g,{b1,b2},point,c]]

Project[g_Graphics3D, basis:{{_,_,_},{_,_,_}}, location:{_,_,_},
          Optional[center:{_,_,_},{0,0,0}]] :=
	TransformGraphics3D[g,
		(Apply[Plus, (basis.(# - center)) basis] + location)&]

Project[g_,  point:{_,_,_}] := Project[Graphics3D[g],point]

Project[g_, basis:{{_,_,_},{_,_,_}}, location:{_,_,_},
                 Optional[center:{_,_,_},{0,0,0}]] :=
    Project[Graphics3D[g], basis, location, center]


(* Shadow *)

Options[Shadow] = 
	Join[{XShadow -> True, YShadow -> True, ZShadow -> True,
	XShadowPosition -> -1, YShadowPosition -> 1,
	ZShadowPosition -> -1}, Options[Graphics3D]];

Shadow[g_Graphics3D, opts___] :=
	Module[{xmin, xmax, ymin, ymax, zmin, zmax, xshadow, 
			yshadow, zshadow, xshadowposition, 
			yshadowposition, zshadowposition,
			image,br},
		{xshadow, yshadow, zshadow, xshadowposition, 
			yshadowposition, zshadowposition} = 
			{XShadow, YShadow, ZShadow, XShadowPosition,
			 YShadowPosition, ZShadowPosition} /. {opts} /.
			 	Options[Shadow];
		gopts = FilterOptions[Graphics3D, opts,
				Sequence @@ Options[Shadow]];
		{xmin, xmax, ymin, ymax, zmin, zmax} = 
			Flatten[PlotRange[g]];
		br = FullOptions[g,BoxRatios];
		image = {g};
		If[xshadow, 
			AppendTo[image,
				Project[g,
					{(xmax+xmin)/2 + xshadowposition (xmax - xmin),
						(ymax+ymin)/2,
						(zmax+zmin)/2}]];
			If[Abs[xshadowposition] > 1/2,
				br = br {(Abs[xshadowposition] + 1/2),1,1}]];
		If[yshadow, 
			AppendTo[image,
				Project[g,
					{(xmax+xmin)/2,
					(ymax+ymin)/2 + yshadowposition (ymax - ymin),
					(zmax+zmin)/2}]];
			If[Abs[yshadowposition] > 1/2,
				br = br {1, (Abs[yshadowposition] + 1/2),1}]];
		If[zshadow, 
			AppendTo[image,
				Project[g,
					{(xmax+xmin)/2,
					(ymax+ymin)/2,
					(zmax+zmin)/2 + zshadowposition (zmax - zmin)}]];
			If[Abs[zshadowposition] > 1/2,
				br = br {1,1, (Abs[zshadowposition] + 1/2)}]];
		Show[image,Flatten[{gopts,BoxRatios->br}]]]

Shadow[g_,opts___] := Shadow[Graphics3D[g],opts] (* handle other graphics *)

(* Graphics3D *)
Unprotect[Graphics3D];

Graphics3D[Graphics[primitives_,options___]] :=
	Graphics3D[ZTG[primitives, 0], BoxRatios->{1,1,1},
        Axes->{True, False, True},
        PlotRange->{Automatic, {-1,1}, Automatic},
        ViewPoint->{0, -1, -3}
	]

Protect[Graphics3D];


(* StackGraphics *)
(* allow graphics to be specified in a sequence instead of a list *)
StackGraphics[grs__Graphics, opts___?OptionQ] :=
    StackGraphics[{grs}, opts]

StackGraphics[list:{__Graphics}, opts___?OptionQ] :=
	Module[{i, pr, as},
        {pr, as} = {PlotRange, BoxRatios}/.
             Flatten[{opts, BoxRatios -> {1, 1, 1}, Options[Graphics3D]}];
      (* hopefully following not too clever for its own good...
          if pr is not a list, immediately do If clause; otherwise, it
          must be a list, we can drop the middle element and see if the
          result is numeric; if not, drop into if clause with 2-element pr *)
        If[!ListQ[pr] || !VectorQ[pr = Drop[pr, {2}], NumericQ],
            pr = PlotRange[Show[list,
                      DisplayFunction -> Identity, PlotRange -> pr]]
        ];
        If[as === Automatic,
            as = AspectRatio/.AbsoluteOptions[Show[list,
                     DisplayFunction -> Identity, PlotRange -> pr,
                     AspectRatio -> Automatic], AspectRatio],
          (* else it must be a triplet... *)
            as = Divide @@ Reverse @ Drop[as, {2}]
        ];
        Graphics3D[ Table[{ZTG[First[ list[[i]] ], i/Length[list], pr, as]}, 
		                 {i, Length[list]}],
             opts,
             BoxRatios->{1,1,1},
		     Axes->{True, False, True}
        ]
	]

(* ZTG transforms elements from 2D to 3D *)

ZTG[d_List, all___] := Map[ ZTG[#, all]& , d ]

ZTG[Point[d_List?(Depth[#] === 3 &), o___], z_, ___] := Point[Map[Insert[#, z, 2]&, d], o]

ZTG[Point[{x_, y_}, o___], z_, ___] := Point[{x, z, y}, o]

ZTG[(h:(Line | Polygon))[d_List?(Depth[#] === 4 &), o___], z_, ___] :=
    h[Map[Insert[#, z, 2]&, d, {2}], o]

ZTG[(h:(Line | Polygon))[d:{{_,_}...}, o___], z_, ___] :=
    h[ Map[Insert[#, z, 2]&, d], o]

(* ****** following only handles a simple GraphicsComplex; nested or ones with embedded
   exacts will fail, this needs to be fixed. *)
ZTG[GraphicsComplex[p_, r___], z_, ___] :=
    GraphicsComplex[Map[Insert[#, z, 2]&, p], r]

ZTG[(h:(Disk | Circle))[cent:{_,_}], all___] :=
    ZTG[h[cent, {1, 1}, {0, 2 Pi}], all]

ZTG[(h:(Disk | Circle))[cent:{_,_}, rad_?NumericQ], all___] :=
    ZTG[h[cent, {rad, rad}, {0, 2 Pi}], all]

ZTG[(h:(Disk | Circle))[cent:{_,_}, rad:{_,_}], all___] :=
    ZTG[h[cent, rad, {0, 2 Pi}], all]

ZTG[(h:(Disk | Circle))[cent:{_,_}, rad_?NumericQ, ang:{_,_}], all___] :=
    ZTG[h[cent, {rad, rad}, ang], all]

$ZTGFraction = 0.05; (* percent of longest plotrange to use in computation
                        of approximate increment for arcs *)

ZTG[(h:(Disk | Circle))[cent:{_,_}, {rlen_, rheight_}, {angmin_, angmax_}],
       z_, pr_, ___] :=
    Module[{pts, t, div},
      (* compute number of divisions to use in arc, based on a cirle of the
          larger radius divided into sections based on a global fraction of
          the longer plotrange *)
        div = Max[Map[(Subtract @@ Reverse[#])&, pr]] * $ZTGFraction;
        If[(div = 6.3 Max[rlen, rheight]/div) < 1, div = 1, div = Ceiling[div]];
      (* compute points, using above-computed divisions *)
        pts = Table[{rlen Sin[t], rheight Cos[t]},
                    {t, angmin, angmax, (angmax - angmin)/div}];
        pts = Map[# + cent &, pts];
        If[!(angmin === 0 && angmax == 2 Pi),
             pts = Append[pts, cent]
        ];
      (* don't bother with recursive ZTG call,
         since poly is guaranteed convex *)
        pts = Map[Insert[#, z, 2]&, pts];
        If[h === Disk,
            Polygon[pts],
            Line[pts]
        ]
    ]

ZTG[Rectangle[{lx_, ly_}, {ux_, uy_}], all___] :=
    ZTG[Polygon[{{lx, ly}, {lx, uy}, {ux, uy}, {ux, ly}}], all]

ZTG[Raster[mat_?MatrixQ, opts___?OptionQ], all___] :=
    ZTG[Raster[mat, {{0,0}, Reverse[Dimensions[mat]]}, opts], all]

ZTG[Raster[mat_?MatrixQ, coords:{{_,_},{_,_}}, opts___?OptionQ], all___] :=
   ZTG[Raster[mat, coords, {0, 1}, opts], all]

ZTG[Raster[mat_?MatrixQ,
           coords:{{_,_},{_,_}},
           {smin_, smax_}, opts___?OptionQ
      ], all___] :=
    Module[{cf, cfs, newmat, sdiff = smax - smin},
        {cf, cfs} = {ColorFunction, ColorFunctionScaling}/.
            Flatten[{opts, Options[Raster]}];
        If[cf === Automatic, cf = GrayLevel];
        If[TrueQ[cfs],
            newmat = Map[
                (cf[If[#< smin, 0, If[# > smax, 1, (# - smin)/sdiff]]])&,
                mat,
                {2}
            ],
            newmat = Map[cf, mat, {2}]
        ];
        ZTG[RasterArray[newmat, coords], all]
    ]

ZTG[RasterArray[colormat_?MatrixQ], all___] :=
    ZTG[RasterArray[colormat, {{0,0}, Reverse[Dimensions[colormat]]}], all]

ZTG[RasterArray[colormat_?MatrixQ, {{minx_, miny_}, {maxx_, maxy_}}], all___] :=
  Module[{xincr = (maxx - minx)/Length[First[colormat]],
          yincr = (maxy - miny)/Length[colormat]},
    ZTG[{EdgeForm[], Transpose[{colormat,
            Table[Polygon[{{i, j}, {i + xincr, j}, {i + xincr, j + yincr},
                           {i, j + yincr}}],
                  {j, miny, maxy - yincr, yincr},
                  {i, minx, maxx - xincr, xincr}
             ]}, {3, 1, 2}
        ]},
        all
    ]
  ]

(* For the following entries, the weird pattern is to prevent
   stub autoloading *)
(* but, 'subsumed' splines in existing lines and polys not handled yet... *)
ZTG[(_Symbol?(Context[#] === "Graphics`Spline`" &&
              SymbolName[#] === "Spline" &))[args___], all___] :=
   ZTG[Line[{
       Graphics`Spline`Private`splinesubsume[Symbol["Graphics`Spline`Spline"][args]]
   }], all]

(* Arrow support will need to be rewritten when the built-in Arrow primitive
   is added. *)
ZTG[(_Symbol?(Context[#] === "Graphics`Arrow`" &&
              SymbolName[#] === "Arrow" &))[args___], z_, pr_, as_] :=
   ZTG[evalarrow[as, pr, args], z, pr, as]

ZTG[Text[d_String, {x_, y_}, dd___], z_, ___] := Text[d, {x,z,y}, dd]

ZTG[expr_, ___] := expr

(* arrow support code -- taken from the V2.2 Windows top-level version
   of Arrow.m *)
norm[pt_] := Sqrt[pt . pt]

tmat[ar_, sf_, {xn_, yn_}, {xm_, ym_}] :=
	With[{tn = ArcTan[xm - xn, (ym - yn)/ar]},
         {{Cos[tn], Sin[tn], 0},
          {-Sin[tn], Cos[tn], 0},
          {0, 0, 1}} .
         {{sf, 0, 0},
          {0, sf ar, 0},
          {xm, ym, 1}}
    ]

transformpt[mat_, pt_] :=
	Take[Append[pt, 1] . mat, 2]

transformobj[mat_, l_List] := Map[transformobj[mat, #]&, l]

transformobj[mat_, Polygon[pts_]] :=
	Polygon[Map[transformpt[mat,#]&, pts]]

transformobj[mat_, Line[pts_]] :=
    Line[Map[transformpt[mat,#]&, pts]]

transformobj[mat_, Point[pt_]] :=
    Point[transformpt[mat,pt]]

transformobj[mat_, any_] := any

evalarrow[ar_, pr_, pt1_, pt1_, opts___] :=
     Module[{delx = -Subtract @@ pr[[1]], dely = -Subtract @@ pr[[2]],
              arfix},
        {scale, zshape} = {Symbol["Graphics`Arrow`HeadScaling"],
                           Symbol["Graphics`Arrow`ZeroShape"]}/.Flatten[{opts}]/.
                 Options[Symbol["Graphics`Arrow`Arrow"]];
        arfix = delx/(dely ar);
        Which[ scale === Automatic, scale = delx,
               scale === Absolute, scale = delx/100,
               True, scale = 1];
        If[zshape === Automatic, zshape = Point[{0,0}] ];
        transformobj[{{scale, 0, 0},
                      {0, scale arfix, 0},
                      {pt1[[1]], pt1[[2]], 1}},
            zshape
        ]
    ]

evalarrow[ar_, pr_, pt1_, pt2_, opts___] :=
	Module[{delx = -Subtract @@ pr[[1]], dely = -Subtract @@ pr[[2]],
           scale, hl, hw, hc, shape},
        {scale, hl, hw, hc, shape} = {
                Symbol["Graphics`Arrow`HeadScaling"],
                Symbol["Graphics`Arrow`HeadLength"],
                Symbol["Graphics`Arrow`HeadWidth"],
                Symbol["Graphics`Arrow`HeadCenter"],
                Symbol["Graphics`Arrow`HeadShape"]}/.Flatten[{opts}]/.
              Options[Symbol["Graphics`Arrow`Arrow"]];
        If[!NumberQ[N[hl]], hl = scale/.{Automatic -> .05,
                   Absolute -> 15, _ -> .2}
        ];
        Which[ scale === Automatic, scale = delx,
               scale === Absolute, scale = delx/120,
               scale === Relative,
                       scale = norm[{1, (delx ar)/dely} (pt2 - pt1)],
               True, scale = 1];
        If[shape === Automatic,
             shape = buildshape[hl, hw, hc]
        ];
		{Line[{pt1, pt2}],
		transformobj[tmat[dely/(delx ar) , scale, pt1, pt2],
			shape
        ]}
    ]

buildshape[len_, wid_, _?(#==0&)] :=
   Line[{{-len, -len wid/2}, {0,0}, {-len, len wid/2}}]

buildshape[len_, wid_, cen_] :=
    Block[{points = {{0,0}, {-len, -len wid/2}, {-len cen, 0},
             {-len, len wid/2}, {0,0}} },
         {Polygon[points], Line[points]}
    ]

(* ============================ RangeCounts ================================ *)

(* RangeCounts[{{x1, y1}, {x2, y2}, ...}, {cx1, cx2, ...}, {cy1, cy2, ...}] *)
RangeCounts[list_List, clists:{_?NumberQ...}...] :=
        Module[ { trees, vals, f,
                  n = Length[{clists}], nbins = Map[Length, {clists}] + 1,
                  split, counts, i},
                trees = Map[MakeTree, {clists}] ;
                f = Map[Function[{x}, TreeFind[#, x]]&, trees];
                vals = Map[MapIndexed[Function[{y, z},
			 (
			 f[[z[[1]]]][y]
			 )  ],
                         #]&, list];
                split = Split[Sort[vals]];
                counts = Array[0&, nbins];  i = Array[0&, n];
                Scan[(
                      While[(
                             First[#] =!= i),
                             i = increment[i, nbins, 0]]; (* next bin *)
                      counts = ReplacePart[counts, Length[#], i+1];
                      i = increment[i, nbins, 0])&, split];
                counts
        ] /; Dimensions[list][[2]] == Length[{clists}]


(* ============================ increment =============================== *)
(* increment allows you to locate the next bin in a multidimensional
        array of bins, where "next" is the order in which Sort would
        sort the bins. *)
(* i is a list, denoting the current bin *)
(* max is a list, denoting the dimensions of the bin array *)
(* origin is an integer (typically 0 or 1),
         indicating the origin of the bin indexing scheme  *)
(*
 increment[{1, 1, 1}, {2, 2, 2}, 1]                     {1, 1, 2}
 increment[{1, 1, 2}, {2, 2, 2}, 1]                     {1, 2, 1}
        ...
 increment[{2, 2, 2}, {2, 2, 2}, 1]                     {1, 1, 1}
*)

increment[i_, max_, origin_] :=
 Block[{j = i},
        Scan[(If[j[[#]] < max[[#]],
                 j[[#]] = j[[#]]+1;  Return[],
                 j[[#]] = origin])&,
                 Reverse[Range[Length[i]]] ];
        j
 ]

(* ================================ MakeTree ============================ *)

MakeTree[{}] := {}

MakeTree[list_List] :=
   Block[{n, t},
                n = Length[list];
                t = Transpose[{Sort[list], Range[n]}] ;
                MakeTree0[ 1, n ]
   ]

MakeTree0[i_,j_] := Block[{midpoint,diff},
        diff = j-i;
        Which[
           diff==3, {t[[i+1]],{t[[i]],{},{}},{t[[i+2]],{},{t[[i+3]],{},{}}}},
           diff==2, {t[[i+1]],{t[[i]],{},{}},{t[[j]],{},{}}},
           diff==1, {t[[i]],{},{t[[j]],{},{}}},
           diff==0, {t[[i]],{},{}},
           True, (
                        midpoint = i + Quotient[diff,2];
                        {t[[midpoint]],
                                MakeTree0[i,midpoint-1],
                                MakeTree0[midpoint+1,j]}
                 )
           ]]


(* ================================ TreeFind ============================ *)

TreeFind[{}, e_] := 0

TreeFind[tree_List, e_] :=
         Block[{found=0, bar=e},
                TreeFind0[tree];
                found
         ]

TreeFind0[tree_] :=
    Block[{m, k},
        {m, k} = First[tree] ;
        Which[
              bar < m, TreeFind0[tree[[2]]],
              bar > m, found = k ;TreeFind0[tree[[3]]],
              True, found = k; Return[]
        ]
    ]

TreeFind0[{}] = 1


(* ================================ TickSpacing ========================= *)

(* NOTE: 10/97: Eliminated check that argument "n" was an integer. *)
TickSpacing[dx_, n_, prefs_List, method_:GreaterEqual] :=
        Module[ { dist=N[dx/n], scale, prefsdelta, min, pos } ,
                scale = 10.^Floor[Log[10., dist]] ;
                dist /= scale ;
                If[dist < 1, dist *= 10 ; scale /= 10] ;
                If[dist >= 10, dist /= 10 ; scale *= 10] ;
                scale * Switch[method,
                        GreaterEqual,
                        (* "nice" tick spacing is greater than or equal to
                                requested tick spacing *)
                        First[Select[prefs, (dist <= #)&]],
                        LessEqual,
                        (* "nice" tick spacing is less than or equal to
                                requested tick spacing *)
                        First[Select[Reverse[prefs], (dist >= #)&]],
                        Nearest,
                        (* "nice" tick spacing is the taken from the
                                element of "prefs" nearest to "dist" *)
                        prefsdelta = Map[Abs[#-dist]&, prefs];
                        min = Min[prefsdelta];
                        pos = Position[prefsdelta, min][[1, 1]];
                        prefs[[pos]]
                ]
        ]



End[] (* Private` *)

EndPackage[] (* Graphics`Graphics3D` *)

(* :Examples:
	<<Graphics/ParametricPlot3D.m
	g1 = CylindricalPlot3D[ r^2,{r,0,1},{phi,0,2 Pi}];
	Show[ TransformGraphics3D[ g1, Cos[#] & ] ]

	Show[ Graphics3D[ Plot[ Sin[t],{t,0,Pi}]]]

	g1 = CylindricalPlot3D[ r^2,{r,0,1},{phi,0,2 Pi}];

	Show[ SkewGraphics3D[ g1, {{1,2,0},{0,1,0},{0,0,1}}] ]

	g1 = Table[ Plot[x^n, {x,0,5}], {n,5}]; Show[ StackGraphics[ g1]] 
	
	g1 = Plot[ Sin[x],{x,0,Pi}];
	g2 = Plot[ Sin[x+0.5],{x,0,Pi}];
	g3 = Plot[ Sin[x+1],{x,0,Pi}];
	Show[ StackGraphics[{g1,g2,g3}] ]
	
	BarChart3D[ { { 1,2,3},{4,5,6}}]
	
	ScatterPlot3D[ Table[ { t,Sin[t],Cos[t]},{t,0,10,0.1}]]

	ScatterPlot3D[ Table[ { t,Sin[t],Cos[t]},{t,0,10,0.1}],PlotJoined->True]

	ListSurfacePlot3D[ Table[ {i,j, Sin[i j] },{i,1,10},{j,1,10}]]

	graphics = Plot3D[Sin[x y],{x,0,Pi},{y,0,Pi}];
	Show[ Project[ graphics, {1,1,0}] ]

	Shadow[ graphics, ZShadow -> False ]

*)


