
(*:Copyright: Copyright 1989-2005, Wolfram Research, Inc. *)

(*:Mathematica Version: 4.0 *)

(*:Package Version: 1.8.5 *)

(*:Name: Graphics`Polyhedra` *)

(*:Title: Graphics with Platonic Polyhedra  *)

(*:Context: Graphics`Polyhedra` *)

(*:Author: Roman Maeder *)

(*:Keywords:
	Polyhedron, vertices, faces, Stellate, Geodesate,
	Truncate, OpenTruncate,
	Tetrahedron, Cube, Octahedron,
	Dodecahedron, Icosahedron, Hexahedron,
	GreatDodecahedron, SmallStellatedDodecahedron,
	GreatStellatedDodecahedron, GreatIcosahedron
*)

(*:Requirements: None. *)

(*:Warnings:
	Faces and Vertices fail on
	GreatDodecahedron, SmallStellatedDodecahedron, and
	GreatStellatedDodecahedron.
*)

(*:Limitations:
	Truncate sometimes creates non-convex 3D polygons which are
	not displayed correctly.
*)


(*:Source:
	Roman E. Maeder: Programming in Mathematica, 2nd Ed.,
	Addison-Wesley, 1991.
*)

(*:History:
  Version 1.5: 
        Added the functions Geodesate, Truncate, and 
           OpenTruncate, Jeff Adams,  August 1992.
  Version 1.6: 
        Fixed Truncate so that it works with v1.1 Polyhedra.m
           polyhedra specifications.
        Set Options[Polyhedron] = Options[Graphics3D] and made PlotRange->All
	   the default setting.
        Allow the ratio value for Truncate and OpenTruncate to include 0.5.
  Version 1.7:
	Uses standard FilterOptions.
  Version 1.8:
	Moved definitions for Tetrahedron, Cube, Octahedron,
		Dodecahedron, Icosahedron, and Hexahedron
		to Geometry`Polytopes` and added Geometry`Polytopes`
		to BeginPackage.  Now both packages can be used together
		without symbol shadowing.
  Version 1.8.5 by John M. Novak, February 1998 -- Geometry`Polytopes`
        modified to return exact values for the vertices, so a numeric
        cache of the values is now being created. Polyhedron[] is redefined
        to use the numeric cache. Also fixed so that the standard polyhedra
        defined as stellations of other polyhedra scale and translate right.
*)


(*:Summary:
This package provides graphics primitives for rendering various
Platonic polyhedra and functions for accessing the coordinates
of the vertices and the vertex numbers of the faces.  The package
also includes functions for operating on polyhedra, such as
Stellate, Geodesate, Truncate, and OpenTruncate.
*)

(*:Discussion: *)

BeginPackage["Graphics`Polyhedra`", "Geometry`Polytopes`"]

Polyhedron::usage = "Polyhedron[name] gives a Graphics3D object
representing the specified solid centered at the origin and with
unit distance to the midpoints of the edges.  Polyhedron[name,
center, size] uses the given center and size. The possible names
are in the list Polyhedra."

Stellate::usage = "Stellate[expr, (ratio:2)] replaces each polygon
in expr by a pyramid with the polygon as its base.  Stellation
ratios less than 1 give concave figures."

Geodesate::usage =
"Geodesate[expr, n] replaces each polygon in expr by the projection onto the circumscribed sphere of the order n regular tessellation of that polygon.  Geodesate[expr, n, {x, y, z}, radius] does the projection onto the sphere of size radius centered at {x, y, z}."

Truncate::usage =
"Truncate[expr] truncates each edge of each polygon in expr. Truncate[expr, ratio] truncates to the specified ratio of the edge length."

OpenTruncate::usage =
"OpenTruncate[expr] truncates each edge of each polygon in expr without filling in with a polygon. OpenTruncate[expr, ratio] truncates to the specified ratio of the edge length."

Polyhedra::usage = "Polyhedra is a list of the known polyhedra."

Polyhedra = {Tetrahedron, Cube, Octahedron,
	Dodecahedron, Icosahedron, Hexahedron,
	GreatDodecahedron, SmallStellatedDodecahedron,
	GreatStellatedDodecahedron, GreatIcosahedron}

Unprotect @@ Polyhedra;

Map[(Evaluate[#]::"usage" =
        StringJoin[Evaluate[#]::"usage",
        " It may also be used with the Polyhedron function."])&,
      Take[Polyhedra, 6]]

Map[(Evaluate[#]::"usage" =
        StringJoin[ToString[#], 
        " is a kind of polyhedron, for use with the Polyhedron function."])&,
			Take[Polyhedra, -4]]

Begin["`Private`"]

Needs["Utilities`FilterOptions`"]

Options[ Polyhedron] = Options[ Graphics3D];
SetOptions[ Polyhedron, PlotRange->All];

(* definition of NVertices at end of package to take effect after
   all Vertices defined *)

Polyhedron[name_Symbol, others___ ] := 
    PolyGraphics3D[ NVertices[name],
	 Faces[name], others ] /; MemberQ[Polyhedra, name]

PolyGraphics3D[ vertices_, faces_, opts___?OptionQ] :=
	Graphics3D[ Polygon[vertices[[#]]]& /@ faces,
		  {FilterOptions[Polyhedron, opts, Options[Polyhedron]]} ]

PolyGraphics3D[ vertices_, faces_, pos_?(!OptionQ[#]&), opts___?OptionQ] :=
	PolyGraphics3D[ vertices, faces, pos, 1.0, opts]
	
PolyGraphics3D[ vertices_, faces_, pos_, scale_, opts___?OptionQ] :=
    Graphics3D[ Polygon /@ 
	   Map[scale # + pos &, (vertices[[#]]&) /@ faces, {2}],
		  {FilterOptions[Polyhedron, opts, Options[Polyhedron]]} ]

(* a redundant definition, but a quick fix for the polyhedra that are
   defined as stellations of other polyhedra *)
scaleandtranslate[polys_] := polys
scaleandtranslate[polys_, pos_] := scaleandtranslate[polys, pos, 1]
scaleandtranslate[polys_, pos_, scale_] :=
     polys/.Polygon[l_] :> Polygon[Map[scale # + pos &, l]]

norm[ v_ ] := Sqrt[Plus @@ (v^2)]
apex[ v_ ] := Plus @@ v / Length[v]

GreatDodecahedron/:
Polyhedron[ GreatDodecahedron, nonopts___, opts___?OptionQ ] :=
	scaleandtranslate[Stellate[ Polyhedron[Icosahedron, opts], 3(Sqrt[5]-2) ],
	  nonopts]

SmallStellatedDodecahedron/:
Polyhedron[ SmallStellatedDodecahedron, nonopts___, opts___?OptionQ ] :=
	scaleandtranslate[Stellate[ Polyhedron[Dodecahedron, opts], Sqrt[5] ],
	   nonopts]

GreatStellatedDodecahedron/:
Polyhedron[ GreatStellatedDodecahedron, nonopts___, opts___?OptionQ ] :=
	scaleandtranslate[Stellate[ Polyhedron[Icosahedron, opts], 3 ], nonopts]

AdjacentTo[face_, flist_] :=
	Select[flist, Length[Intersection[face, #]] == 2&]

Opposite[face_, flist_] :=
	Block[ {adjacent, next},
		adjacent = AdjacentTo[ face, flist ];
		next = AdjacentTo[#, flist]& /@ adjacent;
		next = Complement[#, {face}]& /@ next;
		Flatten[ Intersection @@ #& /@ next ]
	]

GreatIcosahedron/: Vertices[GreatIcosahedron] := Vertices[Icosahedron];

GreatIcosahedron/: Faces[GreatIcosahedron] :=
	GreatIcosahedron/: Faces[GreatIcosahedron] =
		Opposite[#, Faces[Icosahedron]]& /@ Faces[Icosahedron];


StellateFace[face_List, k_] :=
	Block[ { apex,  n = Length[face], i } ,
		apex = N [ k Apply[Plus, face] / n ] ;
		Table[ Polygon[ {apex, face[[i]], face[[ Mod[i, n] + 1 ]] }],
		    {i, n} ]
	]
	
Stellate[poly_, k_:2] := 
	Flatten[ poly /. Polygon[x_] :> StellateFace[x, k] ] /;
		NumberQ[N[k]]


Normalize[v_] := v/Sqrt[Plus @@ (v^2)];

(* triangular polygons*)
GeodesateFace[face:{_,_,_}, n_, center_, size_] := Block[{i, j, vtab},
    
        vtab = Table[size Normalize[{n-i-j,i,j}.face/n -center]+center, 
			{i, 0, n}, {j, 0, n-i}];
        Flatten[Table[Polygon[{vtab[[i,j]], vtab[[i+1,j]], vtab[[i,j+1]]}],
                {i, n}, {j, n+1-i}]] ~Join~
        Flatten[Table[Polygon[{vtab[[i,j]], vtab[[i-1,j]], vtab[[i,j-1]]}],
                {i, 2, n+1}, {j, 2, n+2-i}]]
        ]

(* rectangular polygons *)
GeodesateFace[face:{a_, b_, c_, d_}, n_, center_, size_] :=
    Block[{mid = (Plus @@ face)/4, ord = If[n > 1, n - 1, 1]},
        Map[GeodesateFace[#, ord, center, size]&,
            {{a, b, mid}, {b, c, mid}, {c, d, mid}, {d, a, mid}}
        ]
    ]

(* following is old version that I am leaving in place so that a user
    can hook it in easily from the top level. For a poly with coplanar
    points, this leads to overlapping polygons that can take forever to
    render in a complex shape. Also, all other shapes use a triangular
    tesselation. So, new code uses triangular tess. *)
RectGeodesateFace[face:{_,_,_,_}, n_, center_, size_] := Block[{i, j, k, vtab},
  
        vtab = Table[size Normalize[{n-i-j-k,i,j,k}.face/n -center]+center, 
			{i, 0, n}, {j, 0, n-i},{k,0,n-i-j}];
        Flatten[Table[Polygon[{vtab[[i,j,k]], vtab[[i+1,j,k]], vtab[[i,j+1,k]],
			 vtab[[i,j,k+1]]}],{i, n}, {j, n+1-i},{k, n+2-i-j}]] ~Join~
        Flatten[Table[Polygon[{vtab[[i,j,k]], vtab[[i-1,j,k]], vtab[[i,j-1,k]], 
			 vtab[[i,j,k-1]]}],{i, 2, n+1}, {j, 2, n+2-i},{k, 2, n+3-i-j}]]
        ]
		
(* 5-sided and higher polygons *)		
GeodesateFace[face_List, n_, center_, size_] := 
	Block[{tripoint = (Plus @@ face)/Length[face]},

	GeodesateFace[#,n,center,size]& /@ Map[ Join[#,{tripoint}]&,
		Join[ Partition[face,2,1], {{First[face],Last[face]}} ] ]
	]
	
$DefaultGeodesateN = 2;

Geodesate::tessv =
"Warning: `1` must be a positive integer.  The default Geodesate tessellation of `2` will be used."
	
Geodesate[poly_, n_, center_List:{0,0,0}, radius_:1] :=
	(Message[Geodesate::tessv, n, $DefaultGeodesateN];
	Geodesate[poly, $DefaultGeodesateN, center, radius]) /; !(IntegerQ[n] && Positive[n])
	
Geodesate[poly_, n_:$DefaultGeodesateN, center_List:{0,0,0}, radius_:1] := 
	Flatten[ poly /. {Polygon[{a_, mid___, b_}/;TrueQ[a==b]] :>
	        GeodesateFace[{a, mid}, n, center, radius],
	        Polygon[x_] :> GeodesateFace[x, n, center, radius]} ] /;
		NumberQ[N[radius]]


$DefaultTruncateRatio = 0.3;

Truncate::ratiov =
"Warning: `1` must be a number between 0 and 0.5.  The default Truncate ratio of `2` will be used."
OpenTruncate::ratiov = Truncate::ratiov

$TruncList = {}
$TruncHalfList = {}
	
truncList[face_, 0.5] :=  (0.5 (Plus @@ #))& /@ 
      Join[ Partition[face,2,1], {{Last[face],First[face]}} ]
			
truncList[face_, ratio_] := Flatten[{ratio(#[[2]]-#[[1]])+#[[1]], 
	(1-ratio)(#[[2]]-#[[1]])+#[[1]]}& /@ 
		Join[ Partition[face,2,1], {{Last[face],First[face]}} ] ,1];
		 
OpenTruncateFace[face_List,ratio_] := Polygon[ truncList[ face,ratio] ]

OpenTruncate[poly_, ratio_] :=
	(Message[OpenTruncate::ratiov, ratio, $DefaultTruncateRatio];
	OpenTruncate[poly, $DefaultTruncateRatio]) /; !(NumberQ[N[ratio]] && 0 <= N[ratio] <= 0.5)

OpenTruncate[poly_, ratio_?(#==0&)] := poly

OpenTruncate[poly_, ratio_:$DefaultTruncateRatio] := 
	Flatten[ poly /. Polygon[x_] :> OpenTruncateFace[x, N[ratio]] ] /; 
		NumberQ[N[ratio]]
		
TruncateFace[face_List, 0.5] :=  Block[{array, array2},
	array = truncList[face, 0.49]; 
	AppendTo[ $TruncList, Partition[RotateLeft[array],2]];
	array2 = Flatten[(0.5 {Plus @@ #,Plus @@ #})& /@ Partition[array,2],1];
	AppendTo[ $TruncHalfList, Partition[RotateLeft[array2],2]];
	Polygon[array]]
	
TruncateFace[face_List, ratio_] :=  Block[{array},
	array = truncList[face, ratio]; 
	AppendTo[ $TruncList, Partition[RotateLeft[array],2]];
	Polygon[array]]


combineSame[ x_List] := Block[{set},
	 set = If[ norm[x[[2]]-x[[1]]] > norm[x[[3]]-x[[2]]],
	      RotateLeft[x], x];
	 Polygon[(0.5  Plus @@ #)& /@ Partition[set,2]] ]

Truncate[poly_, ratio_] :=
	(Message[Truncate::ratiov, ratio, $DefaultTruncateRatio];
	Truncate[poly, $DefaultTruncateRatio]) /; !(NumberQ[N[ratio]] && 0 <= N[ratio] <= 0.5)

Truncate[poly_, ratio_?(# == 0 &)] := poly

Truncate[poly_, ratio_:$DefaultTruncateRatio] := Block[{pol, res, reslist = {}, nratio = N[ratio],
	search, begin, val},
	$TruncList = {};
	$TruncHalfList = {};
	pol = Flatten[ poly /. Polygon[x_] :> TruncateFace[x, nratio] ];
	
	res = Round[$TruncList*100000000]/100000000.;
	res = Sort[ Flatten[MapIndexed[{#1,#2}&,res,{3}],2], 
			OrderedQ[{#1[[1]], #2[[1]]}]& ];
	res = Join[{res[[{2,1,2}]]}, Partition[res,3,1], 
			{res[[Length[res]-{1,0,1}]]}];
	res = Select[res, (#[[2,1]] == #[[1,1]] || #[[2,1]] == #[[3,1]])&];
	
	If[Length[res] > 0,
		res =  Partition[Transpose[Transpose[res][[2]]][[2]] ,2];
		While[res =!= {},
			search = {First[res][[2]]} /. {a_,b_,c_} -> {a,b,If[c==2,1,2]};
			begin = search;
			res = Rest[res];
			While[(val = Select[res, MemberQ[#,search[[1]]]&, 1]) =!= {},
				search = Complement[val[[1]],search] /. 
					{a_,b_,c_} -> {a,b,If[c==2,1,2]};
				begin = Join[begin,search];
				res = DeleteCases[res,val[[1]]];
				];
			AppendTo[reslist,begin];
			];
		If[ nratio === 0.5,
		    reslist = Apply[($TruncHalfList[[##]])&, reslist,{2}];
			  pol = pol /. Polygon[x_] :> combineSame[x],
		    reslist = Apply[($TruncList[[##]])&, reslist,{2}] ];
		If[ Head[pol] === Graphics3D,
		     Join[Graphics3D[Join[First[pol], Polygon /@ reslist]], Rest[pol]],
			 Join[pol, Polygon /@ reslist] ]
	    ,
		pol]  
    ] /; NumberQ[N[ratio]]


(* Compatibility with V1.1 Polyhedra.m *)

GreatStellatedDodecahedron[opts___] :=
	Polyhedron[GreatStellatedDodecahedron, opts][[1]]

SmallStellatedDodecahedron[opts___] :=
	Polyhedron[SmallStellatedDodecahedron, opts][[1]]

GreatDodecahedron[opts___] := Polyhedron[GreatDodecahedron, opts][[1]]

Dodecahedron[opts___] := Polyhedron[Dodecahedron, opts][[1]]

GreatIcosahedron[opts___] :=
	Polyhedron[GreatIcosahedron, opts][[1]]

Icosahedron[opts___] := Polyhedron[Icosahedron, opts][[1]]

Octahedron[opts___] := Polyhedron[Octahedron, opts][[1]]

Cube[opts___] := Polyhedron[Cube, opts][[1]]

Hexahedron[opts___]:= Polyhedron[Hexahedron, opts][[1]] 

Tetrahedron[opts___] := Polyhedron[Tetrahedron, opts][[1]]

Scan[(NVertices[#] = N[Vertices[#]])&, Polyhedra];

End[]   (* Graphics`Polyhedra`Private` *)

EndPackage[]   (* Graphics`Polyhedra` *)

(*:Tests:
*)

(*:Examples:

Show[ Polyhedron[Dodecahedron] ]

Show[ Polyhedron[ GreatStellatedDodecahedron]]

Vertices[Dodecahedron] 

Faces[ Icosahedron ]

Show[ Stellate[ Polyhedron[ Octahedron ] ] ]

Show[ Geodesate[ Polyhedron[ Dodecahedron ], 3] ]

Show[ Truncate[ Polyhedron[ Dodecahedron ] ] ]

Show[ OpenTruncate[ Polyhedron[ Octahedron ] ] ]

*)

