(* :Title: FourierTrig *)

(* :Context: LinearAlgebra`FourierTrig` *)

(* :Author: Rob Knapp *)

(* :Summary:
    This package provides FourierCos and FourierSin, which compute 
    repectively the discrete cosine and sine transforms.  

    The normalization returned is not standard, but is used so that 
    both functions are their own inverse.  
*)

(* :Copyright: Copyright 1989-2005, Wolfram Research, Inc.*)

(* :Package Version: 5.0 *)

(* :Mathematica Version: 5.0 *)

(* :History:
    Created January 2000
*)

(* :Keywords: 
    Fourier        
*)

(* :Sources: 
    [VL] C. Van Loan, "Computational Frameworks for the Fast Fourier Transform"
*)

(* :Limitations:

    The transforms are correct only for real input.  No check is done for 
    complex input because this check is relatively expensive particularly
    for short sequence lengths.

    The implementation here does not use the fastest method for even lengths 
    described in VL.  The overhead of the extra expression manipulation 
    prevents this from being faster in Mathematica until sequence lengths
    bigger than 2^18 or so.

*)

(* :Example: 
    FourierSin[Table[Sin[Pi x^2],{x,.1,.9,.1}]] computes the discrete sine
    transform of the data generated from the function Sin[Pi x^2]
*)


BeginPackage["LinearAlgebra`FourierTrig`"]

Unprotect[FourierCos];
Unprotect[FourierSin];

FourierCos::usage = "FourierCos[list] computes the discrete cosine transform of a list of
real numbers."

FourierSin::usage = "FourierSin[list] computes the discrete sine transform of a list of
real numbers."

Begin["`Private`"]

Options[FourierCos] = {FourierParameters-> {0,1}}
Options[FourierSin] = {FourierParameters-> {0,1}}

FourierCos::fftl = FourierSin::fftl = Fourier::fftl;
FourierCos::fpopt = FourierSin::fpopt = Fourier::fpopt;
FourierCos::fpoptn = FourierSin::fpoptn = Fourier::fpoptn;
FourierCos::fpopt2 = FourierSin::fpopt2 = Fourier::fpopt2;

FourierCos[x_, opts___] := With[{res = dct[x, opts]}, res /; res =!= $Failed]

FourierSin[x_, opts___] := With[{res = dst[x, opts]}, res /; res =!= $Failed]

dct[x_, opts___] := Module[{
	dims = Dimensions[x], xx, fp, tmp, mlist},
    If[!ListQ[x], Fourier[x];Return[$Failed]];
    xx = Developer`ToPackedArray[x];
    fp = FourierParameters/. Flatten[{opts, Options[FourierCos]}];
    tmp = Part[
	    xx,
            Sequence @@ Map[Join[Range[#],Range[#-1,2,-1]]& ,dims]];
    mlist = Internal`DeactivateMessages[tmp = Fourier[tmp, FourierParameters->fp];$MessageList];
    Map[handlemessage[#, FourierCos, x, fp]&, mlist];
    If[Head[tmp] == Fourier, Return[$Failed]];
    Take[Re[tmp], Sequence @@ dims]
    ]

dst[x_, opts___] := Module[{
	dims = Dimensions[x], rank, xx, fp, tmp, zero, mlist},
    If[!ListQ[x], Fourier[x];Return[$Failed]];
    xx = Developer`ToPackedArray[x];
    If[Developer`PackedArrayQ[xx],
	zero = 0.;
	xx = N[xx],
    (* else *)
	zero = 0];
    rank = Length[dims];
    fp = FourierParameters/. Flatten[{opts, Options[FourierSin]}];
    tmp = PadLeft[xx, dims + 1, zero];
    tmp = Map[Join[#,RotateRight[Reverse[-#]]]&, tmp,{0, rank - 1}];
    mlist = Internal`DeactivateMessages[tmp = Fourier[tmp, FourierParameters->fp];$MessageList];
    Map[handlemessage[#, FourierSin, x, fp]&, mlist];
    If[Head[tmp] == Fourier, Return[$Failed]];
    Switch[Mod[rank, 4],
	0,  tmp = -Re[tmp],
	1,  tmp = Im[tmp],
	2, tmp = Re[tmp],
	3, tmp = -Im[tmp]];
    Take[tmp, Sequence @@ Map[{2,# + 1}&,dims]]
    ]

handlemessage[HoldForm[MessageName[Fourier, "fftl"]], newhead_, x_, fp_] := Message[MessageName[newhead, "fftl"], x];

handlemessage[HoldForm[MessageName[Fourier, string_String]], newhead_, x_, fp_] := Message[MessageName[newhead, string], fp];

End[]; (* `Private` context *)

Protect[FourierCos];
Protect[FourierSin];

EndPackage[];
