(* :Title: Orthogonalization *)

(* :Author: John M. Novak *)

(* :Summary:
This package provides the Gram-Schmidt routine for orthogonalizing a list
of vectors or a list of functions.
*)

(* :Context: LinearAlgebra`Orthogonalization` *)

(* :Package Version: 1.3 *)

(* :Copyright: Copyright 1991-2005, Wolfram Research, Inc. *)

(* :History: 
    Version 1.0 by John M. Novak (Wolfram Research), January 1991.
    Version 1.1 by John M. Novak, February 1991. Adds ability to handle
        function spaces and ability to generate normalized basis.
    Version 1.2 by John M. Novak, February 1999. In a small cleanup for
        handling linearly dependent case, uses a Developer` context function
    Version 1.2.1 by John M. Novak, May 1999. Modify previous version
        by emitting a message when linearly dependent case found,
        or zero-magnitude vector found under Projection or Normalize.
    Version 1.3 by John M. Novak, September 2000. Add Householder
        orthogonalization.
*)

(* :Keywords: orthogonalization, linear algebra, matrices, vector spaces *)

(* :Sources: Standard linear algebra and calculus texts *)

(* :Mathematica Version: 4.0 *)

(* :Example:
    Here is an example of using GramSchmidt with an arbitrary inner product
    in a function space:
    GramSchmidt[{1,x,x^2,x^3,x^4},
        InnerProduct->(Integrate[#1 #2,{x,-1,1}]&)]
    This generates the first five Legendre Polynomials.
*)

BeginPackage["LinearAlgebra`Orthogonalization`"]

(* Usage messages *)

Normalize::usage =
"Normalize[vec] normalizes a vector passed in; note that an inner
product can be specified with the option InnerProduct to allow, for
instance, function spaces to be used."

Projection::usage =
"Projection[v1,v2] projects a vector on another vector. Note that an
inner product can be specified with the option InnerProduct to allow,
for instance, function spaces."

GramSchmidt::usage =
"GramSchmidt[vectors] performs the GramSchmidt orthogonalization process
on a list of vectors. Note that an inner product can be specified,
allowing, for instance, a function space to used.  Also, the option
Normalized can be set to determine whether or not the basis is
orthonormal."

Householder::usage =
"Householder[vectors] performs Householder orthogonalization on a list
of numeric vectors, returning an orthonormal basis."

InnerProduct::usage =
"InnerProduct is an option to functions in the Orthogonalization
package specifying an inner product as a pure function.  The
default is Dot.  For example: InnerProduct->(Integrate[#1 #2,{x,-1,1}]&)."

Normalized::usage =
"Normalized is an option for GramSchmidt that determines
whether the basis generated is orthonormal or not.  If
True, then it is orthonormal, False is returned otherwise."

Begin["`Private`"]

(* zeroQ utility; Developer`ZeroQ is Listable *)
zeroQ[_List] := False
zeroQ[any_] := Developer`ZeroQ[any]

Options[Normalize] = {InnerProduct -> Dot};

Options[Projection] = {InnerProduct -> Dot};

Options[GramSchmidt] = {InnerProduct -> Dot, Normalized -> True};

Normalize::zeromag = Projection::zeromag =
"The magnitude of `2` was zero under the inner product `1`.";

GramSchmidt::zeromag =
"The magnitude of `2` was zero under the inner product `1`. This
suggests that the vectors are linearly dependent.";

(* Note the kludgy use of $inGS, a global that indicates whether
   Normalize and Projection are being called independently or from
   inside of GramSchmidt. The use is to suppress the zeromag messages
   during GramSchmidt, since that's fairly ugly in the case of a
   linearly dependent set. Instead, the $firstZeroVec global indicates
   whether a message should be emitted by GramSchmidt. *)

Normalize[vec_, opts___?OptionQ] :=
    Module[{ip, ipvec},
        {ip} = {InnerProduct}/.Flatten[{opts, Options[Normalize]}];
        normalize[vec, ip]
    ]

normalize[vec_, ip_] :=
    Module[{ipvec},
        If[zeroQ[ipvec = ip[vec, vec]],
            If[!TrueQ[$inGS], Message[Normalize::zeromag, ip, vec]];
            vec,
            vec/Sqrt[ipvec]
        ]
    ]

Projection[v1_, v2_, opts___?OptionQ] := 
    Module[{ip},
        {ip} = {InnerProduct}/.Flatten[{opts, Options[Projection]}];
        projection[v1, v2, ip]
    ]

projection[v1_, v2_, ip_] :=
    Module[{ipv2},
        If[zeroQ[ipv2 = ip[v2, v2]],
            If[!TrueQ[$inGS],
                Message[Projection::zeromag, ip, v2],
                If[TrueQ[$firstZeroVec],
                    Message[GramSchmidt::zeromag, ip, v2];
                    $firstZeroVec = False
                ]
            ];
            v2,
            (* was ip[v1, v2] v2/ipv2 *)
            ip[v2, v1] v2/ipv2
        ]
    ]

(* auxiliary internal function; projection of the vector on a list of vectors.
*)

multipleprojection[v1_, vecs_, ip_] := 
    Plus @@ Map[projection[v1, #, ip]&, vecs]

(*
The Gram-Schmidt method works by taking a list of vectors; starting with the
first vector, it finds a new basis vector in the orthogonal set by subtracting
from each vector the projection of that vector on the basis vectors determined
so far. Note that a different inner product can be specified, allowing the
use of this function in vector spaces.
*)

GramSchmidt[vecs_List, opts___?OptionQ] :=
    Block[{$inGS = True}, (* this is a kludge, but faster than a MessageBlock *)
      Module[{norm, ip, ans},
        {norm, ip} = {Normalized, InnerProduct}/.
            Flatten[{opts, Options[GramSchmidt]}];
        $firstZeroVec = True; (* also kludge for controlling messages *)
        ans = Fold[Join[#1,
                    {#2 - multipleprojection[#2, #1, ip]}]&,
                {}, vecs];
        If[$firstZeroVec && zeroQ[ip[#, #]],
            Message[GramSchmidt::zeromag, ip, #]
        ]&[Last[ans]];
        If[TrueQ[norm],
            Map[normalize[#, ip]&,ans],
            ans
        ]
      ]
    ]

(* Householder orthogonalization, implemented via QRDecomposition.
   Restrictied to numeric matrices. This is more stable than Gram-Schmidt,
   and better for numeric matrices, though it theoretically takes
   twice as many operations. It is less flexible, in that you can't
   specify the inner product and normalization. The bulk of the routine
   is to tack on the zero vectors if the vecs are linearly dependent. *)
Householder[vecs_?(MatrixQ[#, NumericQ]&)] := 
  Function[{mat},
     If[Length[mat] === Length[vecs],
     (* if no linearly dependent vecs, return result *)
       mat,
     (* otherwise create and tack on zero vecs, packing if appropriate *)
       Join[mat,
          If[Developer`PackedArrayQ[mat], Developer`ToPackedArray[#], #]&[
             Table[N[0, Precision[mat]], {Length[vecs] - Length[mat]},
                   {Length[First[vecs]]}
             ]
          ]
       ]
     ]
  ][First[QRDecomposition[Transpose[vecs]]]]

End[]

EndPackage[]
