(* :Title: ChemicalElements  *)

(* :Author: Stephen Wolfram  *)

(* :Summary:
This package provides functions giving basic properties and abundances 
of the chemical elements. Properties include atomic number, atomic weight,
melting and boiling points, heat of fusion, heat of vaporization,
density, thermal conductivity, electron configuration, ionization potential,  
and specific heat.  Abundances include solar system abundance, earth crust 
abundance and earth ocean abundance.
*)

(* :Context: Miscellaneous`ChemicalElements` *)

(* :Package Version: 1.4 *)

(* :Copyright: Copyright 1990-2005, Wolfram Research, Inc.*)
(* :History:
	original version by S. Wolfram. 
	expanded by ECM November 1990.
	support added for SolarSystemAbundance, EarthCrustAbundance,
		EarthOceanAbundance, IonizationPotential, and SpecificHeat,
		Barbara Ercolano, July 1997.
	updated Unnilquadium to Rutherfordium, Unnilpentium to Dubnium, and
		Unnilhexium to Seaborgium, and added Bohrium, Hassium, 
		Meitnerium, (names for elements 104-109 approved by the
		International Union of Pure and Applied Chemistry
		in August 1997), and Ununnilium, Unununium, and
		Ununbium (elements 110-112); updated atomic weights
		and some electron configurations, ECM, 1998.
*)
(* :Keywords: *)
(* :Source:
	The Elements, John Emsley, Oxford University Press, 1989; 
	CRC, 69th Edition;
	CRC, 75th Edition;
	CRC, 79th Edition
*)
(* :Warning: *)

(* :Mathematica Version: 2.0 *)

(* :Limitation: None. *)
(* :Discussion: *)

BeginPackage["Miscellaneous`ChemicalElements`", "Miscellaneous`Units`",
		"Miscellaneous`SIUnits`"]


Elements::usage =
"Elements is a list of the chemical elements described in the
ChemicalElements package."

ElementAbbreviations::usage =
"ElementAbbreviations is a list of the abbreviations of the chemical
elements described in the ChemicalElements package."

AtomicNumber::usage = 
"AtomicNumber[element] gives the atomic number of the specified
element."

AtomicWeight::usage = 
"AtomicWeight[element] gives the atomic weight of the specified
element."

MeltingPoint::usage =
"MeltingPoint[element] gives the melting point of the specified
element."

BoilingPoint::usage =
"BoilingPoint[element] gives the boiling point of the specified
element."

HeatOfFusion::usage =
"HeatOfFusion[element] gives the heat of fusion of the 
specified element."

HeatOfVaporization::usage =
"HeatOfVaporization[element] gives the heat of vaporization of the
specified element."

Density::usage =
"Density[element] gives the density of the specified element, usually
at 298 Kelvin and one atmosphere pressure.  A message is issued for
other temperatures and for special forms of the element."

ThermalConductivity::usage =
"ThermalConductivity[element] gives the thermal conductivity of
the specified element at 300 K."

StableIsotopes::usage =
"StableIsotopes[element] gives a list of the stable isotopes
of the specified element."

ElectronConfiguration::usage =
"ElectronConfiguration[element] depicts the electron configuration
of the specified element in a list of the form
{{1s},{2s,2p},{3s,3p,3d},{4s,4p,4d,4f},{5s,5p,5d,5f},{6s,6p,6d},{7s}}."
(* NOTE: The 79th Edition of the CRC indicates that electron configurations
	are given as
1 or K:		{s}
2 or L:		{s, p}
3 or M:		{s, p, d}
4 or N:		{s, p, d, f}
5 or O:		{s, p, d, f}
6 or P:		{s, p, d}
7 or Q:		{s}

yet the 6th oribital of Francium seems to be given as {2, 6, 0, 1}.
Whether the 1 should be associated with 6f or 7s is unclear.  This
should be clarified for all elements with atomic numbers 87 and above.
*)  

ElectronConfigurationFormat::usage =
"ElectronConfigurationFormat[element] puts the electron
configuration of the specified element in standard notation."

Abbreviation::usage = 
"Abbreviation[element] gives the standard abbreviation of the
specified element."

SolarSystemAbundance::usage = 
"SolarSystemAbundance[element] gives the relative abundance in
the solar system of the specified element."

EarthCrustAbundance::usage = 
"EarthCrustAbundance[element] gives the relative abundance in the
Earth's crust of the specified element."

EarthOceanAbundance::usage =  
"EarthOceanAbundance[element] gives the relative abundance in the
Earth's oceans of the specified element."

IonizationPotential::usage = 
"IonizationPotential[element] gives the ionization potential of the
specified element."

SpecificHeat::usage =  
"SpecificHeat[element] gives the specific heat of the specified
element."

Unknown::usage = 
"Unknown is a symbol returned for unknown element properties."

Elements = {Hydrogen, Helium, Lithium, Beryllium, Boron, Carbon, Nitrogen,
   Oxygen, Fluorine, Neon, Sodium, Magnesium, Aluminium, Silicon, Phosphorus,
   Sulfur, Chlorine, Argon, Potassium, Calcium, Scandium, Titanium, Vanadium,
   Chromium, Manganese, Iron, Cobalt, Nickel, Copper, Zinc, Gallium,
   Germanium, Arsenic, Selenium, Bromine, Krypton, Rubidium, Strontium,
   Yttrium, Zirconium, Niobium, Molybdenum, Technetium, Ruthenium, Rhodium,
   Palladium, Silver, Cadmium, Indium, Tin, Antimony, Tellurium, Iodine,
   Xenon, Caesium, Barium, Lanthanum, Cerium, Praseodymium, Neodymium,
   Promethium, Samarium, Europium, Gadolinium, Terbium, Dysprosium, Holmium,
   Erbium, Thulium, Ytterbium, Lutetium, Hafnium, Tantalum, Tungsten,
   Rhenium, Osmium, Iridium, Platinum, Gold, Mercury, Thallium, Lead,
   Bismuth, Polonium, Astatine, Radon, Francium, Radium, Actinium, Thorium,
   Protactinium, Uranium, Neptunium, Plutonium, Americium, Curium, Berkelium,
   Californium, Einsteinium, Fermium, Mendelevium, Nobelium, Lawrencium,
   Rutherfordium, Dubnium, Seaborgium, Bohrium, Hassium, Meitnerium,
   Ununnilium, Unununium, Ununbium}

Map[(#::"usage" =
	StringJoin[ToString[#]," is a chemical element."])&,
   Elements]

Map[(Evaluate[#[[1]]]::"usage" =
	StringJoin[ToString[#[[1]]],", commonly known as ",
		   ToString[#[[2]]],", is a chemical element."])&,
   {{Natrium,Sodium},{Aluminum,Aluminium},{Sulphur,Sulfur},{Kalium,Potassium},
    {Stibium,Antimony},{Cesium,Caesium},{Wolfram,Tungsten},
    {Kurchatovium,Rutherfordium},{Unnilquadium,Rutherfordium},
    {Unnilpentium,Dubnium},{Unnilhexium,Seaborgium}
}]


Begin["`Private`"]

AtomicWeight::unstable =
	"No stable isotope of `1` exists."
(*  Atomic mass number of isotope having the longest known half life returned. *)
MeltingPoint::form =
	"Melting point is for the `2` form of `1`."
BoilingPoint::form =
	"Boiling point is for the `2` form of `1`."
HeatOfFusion::form =
	"Heat of fusion is for the `2` form of `1`."
HeatOfVaporization::form =
	"Heat of vaporization is for the `2` form of `1`."
Density::form =
	"Density is for the `2` form of `1`."
Density::temp =
	"Density is for `1` at `2` Kelvin."
Density::tempform =
	"Density is for the `3` form of `1` at `2` Kelvin."
ThermalConductivity::form =
	"Thermal conductivity is for the `2` form of `1`."

AtomicWeightUnstable[name_Symbol, number_] :=
	(Message[AtomicWeight::unstable, name]; number)
MeltingPointException[name_Symbol, mp_, form_?StringQ] :=
	(Message[MeltingPoint::form, name, form]; mp Kelvin)
BoilingPointException[name_Symbol, bp_, form_?StringQ] :=
	(Message[BoilingPoint::form, name, form]; bp Kelvin)
HeatOfFusionException[name_Symbol, hof_, form_?StringQ] :=
	(Message[HeatOfFusion::form, name, form]; hof Kilo Joule / Mole)
HeatOfVaporizationException[name_Symbol, hov_, form_?StringQ] :=
	(Message[HeatOfVaporization::form, name, form]; hov Kilo Joule / Mole)
DensityException[name_Symbol, den_, temp_?NumberQ] :=
	(Message[Density::temp, name, temp]; den Kilogram/(Meter)^3)
DensityException[name_Symbol, den_, form_?StringQ] :=
	(Message[Density::form, name, form]; den Kilogram/(Meter)^3)
DensityException[name_Symbol, den_, temp_?NumberQ, form_?StringQ] :=
	 (Message[Density::tempform, name, temp, form]; den Kilogram/(Meter)^3)
ThermalConductivityException[name_Symbol, tc_, form_?StringQ] :=
	(Message[ThermalConductivity::form, name, form]; tc Watt/(Meter Kelvin)) 
AtomicWeightNA[name_Symbol] :=
        (Message[AtomicWeight::unknown, name]; Unknown)
MeltingPointNA[name_Symbol] :=
	(Message[MeltingPoint::unknown, name]; Unknown)
BoilingPointNA[name_Symbol] :=
	(Message[BoilingPoint::unknown, name]; Unknown)
HeatOfFusionNA[name_Symbol] :=
	(Message[HeatOfFusion::unknown, name]; Unknown)
HeatOfVaporizationNA[name_Symbol] :=
	(Message[HeatOfVaporization::unknown, name]; Unknown)
DensityNA[name_Symbol] :=
	(Message[Density::unknown, name]; Unknown)
ThermalConductivityNA[name_Symbol] :=
	(Message[ThermalConductivity::unknown, name]; Unknown)
ElectronConfigurationNA[name_Symbol] :=
	(Message[ElectronConfiguration::unknown, name]; Unknown) 
SolarSystemAbundanceNA[name_Symbol] := 
	(Message[SolarSystemAbundance::unknown, name]; Unknown)  
EarthCrustAbundanceNA[name_Symbol] := 
	(Message[EarthCrustAbundance::unknown, name]; Unknown)
EarthOceanAbundanceNA[name_Symbol] := 
	(Message[EarthOceanAbundance::unknown, name]; Unknown)
IonizationPotentialNA[name_Symbol] := 
	(Message[IonizationPotential::unknown, name]; Unknown)  
SpecificHeatNA[name_Symbol] := 
	(Message[SpecificHeat::unknown, name]; Unknown)  

AtomicWeight::unknown =
	"The atomic weight of `1` is not known."
MeltingPoint::unknown =
	"The melting point of `1` is not known."
BoilingPoint::unknown =
	"The boiling point of `1` is not known."
HeatOfFusion::unknown =
	"The heat of fusion of `1` is not known."
HeatOfVaporization::unknown =
	"The heat of vaporization of `1` is not known."
Density::unknown =
	"The density of `1` is not known."
ThermalConductivity::unknown =
	"The thermal conductivity of `1` is not known."
ElectronConfiguration::unknown =
	"The electron configuration of `1` is not known."
SolarSystemAbundance::unknown = 
	"The abundance of `1` in the solar system is not known."
EarthCrustAbundance::unknown = 
	"The abundance of `1` in the earth's crust is not known."
EarthOceanAbundance::unknown = 
	"The abundance of `1` in the earth's oceans is not known."
IonizationPotential::unknown = 
	"The ionization potential of `1` is not known."
SpecificHeat::unknown = 
	"The specific heat of `1` is not known."


ElectronConfigurationFormat[name_Symbol] :=
	Module[{l=ElectronConfiguration[name],i,j,orbital={"s","p","d","f"}},
	   If[!SameQ[l,Unknown],	
		Apply[SequenceForm,Drop[Flatten[
		    Table[
			Append[
			    Table[{ToString[i],orbital[[j]],
				   Superscript[l[[i,j]]]},
				   {j,1,Length[l[[i]]]}],
			    " "
			],
			{i,1,Length[l]}
		    ]
		],-1]],
		Unknown
	   ]
	] 


ElementAbbreviations =
	{"H","He","Li","Be","B","C","N","O","F","Ne","Na","Mg","Al","Si","P",
	 "S","Cl","Ar","K","Ca","Sc","Ti","V","Cr","Mn","Fe","Co","Ni","Cu",
	 "Zn","Ga","Ge","As","Se","Br","Kr","Rb","Sr","Y","Zr","Nb","Mo","Tc",
	 "Ru","Rh","Pd","Ag","Cd","In","Sn","Sb","Te","I","Xe","Cs","Ba","La",
	 "Ce","Pr","Nd","Pm","Sm","Eu","Gd","Tb","Dy","Ho","Er","Tm","Yb",
	 "Lu","Hf","Ta","W","Re","Os","Ir","Pt","Au","Hg","Tl","Pb","Bi","Po",
	 "At","Rn","Fr","Ra","Ac","Th","Pa","U","Np","Pu","Am","Cm","Bk","Cf",
	 "Es","Fm","Md","No","Lr", "Rf", "Db", "Sg", "Bh", "Hs", "Mt",
	"Uun","Uuu","Uub"}


Scan[SetAttributes[#,Listable]&,
    {AtomicNumber,AtomicWeight,MeltingPoint,BoilingPoint,HeatOfFusion,
     HeatOfVaporization,Density,ThermalConductivity,StableIsotopes,
     ElectronConfiguration,ElectronConfigurationFormat,Abbreviation, 
     EarthOceanAbundance, EarthCrustAbundance, SolarSystemAbundance, 
     IonizationPotential, SpecificHeat}]




AtomicNumber[Hydrogen] ^= 1
AtomicWeight[Hydrogen] ^= 1.00794
MeltingPoint[Hydrogen] ^= 14.01 Kelvin
BoilingPoint[Hydrogen] ^= 20.28 Kelvin
HeatOfFusion[Hydrogen] ^=  0.12 Kilo Joule / Mole
HeatOfVaporization[Hydrogen] ^= 0.46 Kilo Joule / Mole
Density[Hydrogen] ^:=  DensityException[Hydrogen, 76.0, 11]
ThermalConductivity[Hydrogen] ^:= ThermalConductivityException[Hydrogen,
					0.1815, "gaseous"]
StableIsotopes[Hydrogen] ^= {1,2} 
ElectronConfiguration[Hydrogen] ^= {{1}}
Abbreviation[Hydrogen] ^= "H"
SolarSystemAbundance[Hydrogen] ^= 0.91
EarthCrustAbundance[Hydrogen] ^= 
  0.0014
EarthOceanAbundance[Hydrogen] ^= 0.108 
IonizationPotential[Hydrogen] ^= 13.598 ElectronVolt
SpecificHeat[Hydrogen] ^= (14.304 Joule)/(Gram Kelvin)
 
AtomicNumber[Helium] ^= 2
AtomicWeight[Helium] ^= 4.002602
MeltingPoint[Helium] ^= 0.95 Kelvin			(* under pressure *)
BoilingPoint[Helium] ^= 4.216 Kelvin
HeatOfFusion[Helium] ^= 0.021 Kilo Joule / Mole
HeatOfVaporization[Helium] ^= 0.082 Kilo Joule / Mole
Density[Helium] ^:= DensityException[Helium, 124.8, 4, "liquid"] 
ThermalConductivity[Helium] ^:= ThermalConductivityException[Helium,
					0.152, "gaseous"]
StableIsotopes[Helium] ^= {3,4}
ElectronConfiguration[Helium] ^= {{2}}
Abbreviation[Helium] ^= "He"
SolarSystemAbundance[Helium] ^= 0.089
EarthCrustAbundance[Helium] ^= 8. 10^-9
EarthOceanAbundance[Helium] ^= 7.0 10^-12
IonizationPotential[Helium] ^= 24.587 ElectronVolt
SpecificHeat[Helium] ^= (5.193 Joule)/(Gram Kelvin)
 
AtomicNumber[Lithium] ^= 3
AtomicWeight[Lithium] ^= 6.941
MeltingPoint[Lithium] ^= 453.69 Kelvin
BoilingPoint[Lithium] ^= 1620. Kelvin
HeatOfFusion[Lithium] ^= 4.60 Kilo Joule / Mole
HeatOfVaporization[Lithium] ^= 147.7  Kilo Joule / Mole
Density[Lithium] ^=  534. Kilogram/(Meter)^3
ThermalConductivity[Lithium] ^= 84.7 Watt/(Meter Kelvin)
StableIsotopes[Lithium] ^=  {6,7}
ElectronConfiguration[Lithium] ^= {{2},{1}}
Abbreviation[Lithium] ^= "Li"
SolarSystemAbundance[Lithium] ^= 1.86 10^-9
EarthCrustAbundance[Lithium] ^= 0.00002
EarthOceanAbundance[Lithium] ^= 
  1.8 10^-7 
IonizationPotential[Lithium] ^= 5.392 ElectronVolt
SpecificHeat[Lithium] ^= 
  (3.582 Joule)/(Gram Kelvin)
 
AtomicNumber[Beryllium] ^= 4
AtomicWeight[Beryllium] ^= 9.012182
MeltingPoint[Beryllium] ^= 1551. Kelvin 		(*  +- 5. Kelvin *)
BoilingPoint[Beryllium] ^= 3243. Kelvin
HeatOfFusion[Beryllium] ^= 9.80 Kilo Joule / Mole
HeatOfVaporization[Beryllium] ^= 308.8 Kilo Joule / Mole
Density[Beryllium] ^= 1847.7 Kilogram/(Meter)^3 
ThermalConductivity[Beryllium] ^= 200. Watt/(Meter Kelvin) 
StableIsotopes[Beryllium] ^= {9} 
ElectronConfiguration[Beryllium] ^= {{2},{2}}
Abbreviation[Beryllium] ^= "Be"
SolarSystemAbundance[Beryllium] ^= 2.38 10^-11
EarthCrustAbundance[Beryllium] ^= 
  2.8 10^-6 
EarthOceanAbundance[Beryllium] ^= 
  5.6 10^-12 
IonizationPotential[Beryllium] ^= 9.323 ElectronVolt
SpecificHeat[Beryllium] ^= (1.825 Joule)/(Gram Kelvin)
 
AtomicNumber[Boron] ^= 5
AtomicWeight[Boron] ^= 10.81
MeltingPoint[Boron] ^= 2573. Kelvin
BoilingPoint[Boron] ^= 3931. Kelvin
HeatOfFusion[Boron] ^= 22.2 Kilo Joule / Mole
HeatOfVaporization[Boron] ^= 504.5 Kilo Joule / Mole
Density[Boron] ^=  2340. Kilogram/(Meter)^3
ThermalConductivity[Boron] ^= 27.0 Watt/(Meter Kelvin) 
StableIsotopes[Boron] ^= {10,11} 
ElectronConfiguration[Boron] ^= {{2},{2,1}}
Abbreviation[Boron] ^= "B"
SolarSystemAbundance[Boron] ^= 6.9 10^-10
EarthCrustAbundance[Boron] ^= 0.00001
EarthOceanAbundance[Boron] ^= 4.44 10^-6 
IonizationPotential[Boron] ^= 8.298 ElectronVolt
SpecificHeat[Boron] ^= (1.026 Joule)/(Gram Kelvin)
 
AtomicNumber[Carbon] ^= 6
AtomicWeight[Carbon] ^= 12.0107
MeltingPoint[Carbon] ^:= MeltingPointException[Carbon,
				3820., "diamond"]
BoilingPoint[Carbon] ^= 5100. Kelvin 	(* temperature at which C sublimes *)
HeatOfFusion[Carbon] ^= 105.0 Kilo Joule / Mole 
HeatOfVaporization[Carbon] ^= 710.9 Kilo Joule / Mole
Density[Carbon] ^:= DensityException[Carbon,3513.,"diamond"]
ThermalConductivity[Carbon] ^:= ThermalConductivityException[Carbon,
				1960., "graphite"]
StableIsotopes[Carbon] ^=  {12,13}
ElectronConfiguration[Carbon] ^= {{2},{2,2}}
Abbreviation[Carbon] ^= "C"
SolarSystemAbundance[Carbon] ^= 0.00033
EarthCrustAbundance[Carbon] ^= 0.0002 
EarthOceanAbundance[Carbon] ^= 0.000028 
IonizationPotential[Carbon] ^= 11.26 ElectronVolt
SpecificHeat[Carbon] ^= 
  (0.709 Joule)/(Gram Kelvin)
 
AtomicNumber[Nitrogen] ^= 7
AtomicWeight[Nitrogen] ^= 14.00674
MeltingPoint[Nitrogen] ^= 63.29 Kelvin
BoilingPoint[Nitrogen] ^= 77.4 Kelvin
HeatOfFusion[Nitrogen] ^= 0.720 Kilo Joule / Mole
HeatOfVaporization[Nitrogen] ^= 5.577  Kilo Joule / Mole
Density[Nitrogen] ^:= DensityException[Nitrogen,1026.,21]
ThermalConductivity[Nitrogen] ^:= ThermalConductivityException[Nitrogen,
				0.02598, "gaseous"]
StableIsotopes[Nitrogen] ^= {14,15}  
ElectronConfiguration[Nitrogen] ^= {{2},{2,3}}
Abbreviation[Nitrogen] ^= "N"
SolarSystemAbundance[Nitrogen] ^= 0.000102
EarthCrustAbundance[Nitrogen] ^= 0.000019 
EarthOceanAbundance[Nitrogen] ^= 5. 10^-7 
IonizationPotential[Nitrogen] ^= 14.534 ElectronVolt
SpecificHeat[Nitrogen] ^= (1.04 Joule)/(Gram Kelvin)
 
AtomicNumber[Oxygen] ^= 8
AtomicWeight[Oxygen] ^= 15.9994
MeltingPoint[Oxygen] ^= 54.8 Kelvin
BoilingPoint[Oxygen] ^= 90.19 Kelvin
HeatOfFusion[Oxygen] ^= 0.444 Kilo Joule / Mole
HeatOfVaporization[Oxygen] ^= 6.82 Kilo Joule / Mole
Density[Oxygen] ^:= DensityException[Oxygen,2000.,55]
ThermalConductivity[Oxygen] ^:= ThermalConductivityException[Oxygen,
				0.02674, "gaseous"]
StableIsotopes[Oxygen] ^= {16,17,18} 
ElectronConfiguration[Oxygen] ^= {{2},{2,4}}
Abbreviation[Oxygen] ^= "O"
SolarSystemAbundance[Oxygen] ^= 0.00078
EarthCrustAbundance[Oxygen] ^= 
  0.461
EarthOceanAbundance[Oxygen] ^= 0.857
IonizationPotential[Oxygen] ^= 13.618 ElectronVolt
SpecificHeat[Oxygen] ^= 
  (0.918 Joule)/(Gram Kelvin)
 
AtomicNumber[Fluorine] ^= 9
AtomicWeight[Fluorine] ^= 18.9984032
MeltingPoint[Fluorine] ^= 53.53 Kelvin
BoilingPoint[Fluorine] ^= 85.01 Kelvin
HeatOfFusion[Fluorine] ^= 1.02 Kilo Joule / Mole
HeatOfVaporization[Fluorine] ^= 3.26  Kilo Joule / Mole
Density[Fluorine] ^:= DensityException[Fluorine,1516.,85,"liquid"]
ThermalConductivity[Fluorine] ^:= ThermalConductivityException[Fluorine,
				0.0279, "gaseous"]
StableIsotopes[Fluorine] ^=  {19}
ElectronConfiguration[Fluorine] ^= {{2},{2,5}}
Abbreviation[Fluorine] ^= "F"
SolarSystemAbundance[Fluorine] ^= 2.7 10^-8
EarthCrustAbundance[Fluorine] ^= 0.000585 
EarthOceanAbundance[Fluorine] ^= 1.3 10^-6 
IonizationPotential[Fluorine] ^= 
  17.423 ElectronVolt
SpecificHeat[Fluorine] ^= (0.824 Joule)/(Gram Kelvin)
 
AtomicNumber[Neon] ^= 10
AtomicWeight[Neon] ^= 20.1797
MeltingPoint[Neon] ^= 24.48 Kelvin
BoilingPoint[Neon] ^= 27.10 Kelvin
HeatOfFusion[Neon] ^= 0.324 Kilo Joule / Mole
HeatOfVaporization[Neon] ^= 1.736  Kilo Joule / Mole
Density[Neon] ^:= DensityException[Neon,1444.,24]
ThermalConductivity[Neon] ^:= ThermalConductivityException[Neon,
				0.0493, "gaseous"]
StableIsotopes[Neon] ^=  {20,21,22}
ElectronConfiguration[Neon] ^= {{2},{2,6}}
Abbreviation[Neon] ^= "Ne"
SolarSystemAbundance[Neon] ^= 0.000112
EarthCrustAbundance[Neon] ^= 5. 10^-9
EarthOceanAbundance[Neon] ^= 
  1.20 10^-10
IonizationPotential[Neon] ^= 21.565 ElectronVolt
SpecificHeat[Neon] ^= (1.03 Joule)/(Gram Kelvin)
 
Natrium = Sodium
AtomicNumber[Sodium] ^= 11
AtomicWeight[Sodium] ^= 22.989770
MeltingPoint[Sodium] ^= 370.96 Kelvin
BoilingPoint[Sodium] ^= 1156.1 Kelvin
HeatOfFusion[Sodium] ^=  2.64 Kilo Joule / Mole
HeatOfVaporization[Sodium] ^= 99.2  Kilo Joule / Mole
Density[Sodium] ^= 971. Kilogram/(Meter)^3 
ThermalConductivity[Sodium] ^= 141. Watt/(Meter Kelvin) 
StableIsotopes[Sodium] ^=  {23}
ElectronConfiguration[Sodium] ^= {{2},{2,6},{1}}
Abbreviation[Sodium] ^= "Na"
SolarSystemAbundance[Sodium] ^= 1.87 10^-6
EarthCrustAbundance[Sodium] ^= 
  0.02360
EarthOceanAbundance[Sodium] ^= 0.0108
IonizationPotential[Sodium] ^= 5.139 ElectronVolt
SpecificHeat[Sodium] ^= (1.228 Joule)/(Gram Kelvin)
 
AtomicNumber[Magnesium] ^= 12
AtomicWeight[Magnesium] ^= 24.3050
MeltingPoint[Magnesium] ^= 922.0 Kelvin
BoilingPoint[Magnesium] ^= 1363. Kelvin
HeatOfFusion[Magnesium] ^= 9.04 Kilo Joule / Mole
HeatOfVaporization[Magnesium] ^= 127.6  Kilo Joule / Mole
Density[Magnesium] ^= 1738. Kilogram/(Meter)^3 
ThermalConductivity[Magnesium] ^= 156. Watt/(Meter Kelvin) 
StableIsotopes[Magnesium] ^=  {24,25,26}
ElectronConfiguration[Magnesium] ^= {{2},{2,6},{2}}
Abbreviation[Magnesium] ^= "Mg"
SolarSystemAbundance[Magnesium] ^= 0.000035
EarthCrustAbundance[Magnesium] ^= 
  0.0233
EarthOceanAbundance[Magnesium] ^= 0.00129
IonizationPotential[Magnesium] ^= 
  7.6460 ElectronVolt
SpecificHeat[Magnesium] ^= (1.026 Joule)/(Gram Kelvin)

Aluminum = Aluminium
AtomicNumber[Aluminium] ^= 13
AtomicWeight[Aluminium] ^= 26.981538
MeltingPoint[Aluminium] ^= 933.5 Kelvin
BoilingPoint[Aluminium] ^= 2740. Kelvin
HeatOfFusion[Aluminium] ^= 10.67 Kilo Joule / Mole
HeatOfVaporization[Aluminium] ^= 290.8  Kilo Joule / Mole
Density[Aluminium] ^=  2698. Kilogram/(Meter)^3
ThermalConductivity[Aluminium] ^= 237. Watt/(Meter Kelvin) 
StableIsotopes[Aluminium] ^= {27} 
ElectronConfiguration[Aluminium] ^= {{2},{2,6},{2,1}}
Abbreviation[Aluminium] ^= "Al"
SolarSystemAbundance[Aluminium] ^= 2.77 10^-6
EarthCrustAbundance[Aluminium] ^= 
  0.0823
EarthOceanAbundance[Aluminium] ^= 2. 10^-9 
IonizationPotential[Aluminium] ^= 
  5.986 ElectronVolt
SpecificHeat[Aluminium] ^= (0.897 Joule)/(Gram Kelvin)
 
AtomicNumber[Silicon] ^= 14
AtomicWeight[Silicon] ^= 28.0855
MeltingPoint[Silicon] ^= 1683. Kelvin
BoilingPoint[Silicon] ^= 2628. Kelvin
HeatOfFusion[Silicon] ^= 39.6  Kilo Joule / Mole
HeatOfVaporization[Silicon] ^= 383.3  Kilo Joule / Mole
Density[Silicon] ^=  2329. Kilogram/(Meter)^3
ThermalConductivity[Silicon] ^=  148 Watt/(Meter Kelvin)
StableIsotopes[Silicon] ^=  {28,29,30}
ElectronConfiguration[Silicon] ^= {{2},{2,6},{2,2}}
Abbreviation[Silicon] ^= "Si"
SolarSystemAbundance[Silicon] ^= 0.0000326
EarthCrustAbundance[Silicon] ^= 
  0.282
EarthOceanAbundance[Silicon] ^= 2.2 10^-6
IonizationPotential[Silicon] ^= 
  8.152 ElectronVolt
SpecificHeat[Silicon] ^= 
  (0.7050 Joule)/(Gram Kelvin)
 
AtomicNumber[Phosphorus] ^= 15
AtomicWeight[Phosphorus] ^= 30.973761
MeltingPoint[Phosphorus] ^:= MeltingPointException[Phosphorus, 317.3,"P4"]
BoilingPoint[Phosphorus] ^:= BoilingPointException[Phosphorus, 553., "P4"]
HeatOfFusion[Phosphorus] ^:= HeatOfFusionException[Phosphorus, 2.51, "P4"]
HeatOfVaporization[Phosphorus] ^:= HeatOfVaporizationException[Phosphorus,
					51.9, "P4"]
Density[Phosphorus] ^:= DensityException[Phosphorus,1820., "P4"]
ThermalConductivity[Phosphorus] ^=  0.235 Watt/(Meter Kelvin)
StableIsotopes[Phosphorus] ^=  {31}
ElectronConfiguration[Phosphorus] ^= {{2},{2,6},{2,3}}
Abbreviation[Phosphorus] ^= "P"
SolarSystemAbundance[Phosphorus] ^= 3.4 10^-7
EarthCrustAbundance[Phosphorus] ^= 
  0.00105
EarthOceanAbundance[Phosphorus] ^= 6.0 10^-8
IonizationPotential[Phosphorus] ^= 
  10.487 ElectronVolt
SpecificHeat[Phosphorus] ^= (0.769 Joule)/(Gram Kelvin)
 
Sulphur = Sulfur
AtomicNumber[Sulfur] ^= 16
AtomicWeight[Sulfur] ^= 32.066
MeltingPoint[Sulfur] ^:= MeltingPointException[Sulfur, 386.0, "alpha"]
BoilingPoint[Sulfur] ^= 717.824 Kelvin
HeatOfFusion[Sulfur] ^= 1.23 Kilo Joule / Mole
HeatOfVaporization[Sulfur] ^= 9.62  Kilo Joule / Mole
Density[Sulfur] ^:= DensityException[Sulfur,2070.,"alpha"]
ThermalConductivity[Sulfur] ^:= ThermalConductivityException[Sulfur,
				0.269, "alpha"]
StableIsotopes[Sulfur] ^=   {32,33,34,36}
ElectronConfiguration[Sulfur] ^= {{2},{2,6},{2,4}}
Abbreviation[Sulfur] ^= "S"
SolarSystemAbundance[Sulfur] ^= 0.0000326
EarthCrustAbundance[Sulfur] ^= 0.00035
EarthOceanAbundance[Sulfur] ^= 0.000905
IonizationPotential[Sulfur] ^= 10.36 ElectronVolt
SpecificHeat[Sulfur] ^= 
  (0.71 Joule)/(Gram Kelvin)
 
AtomicNumber[Chlorine] ^= 17
AtomicWeight[Chlorine] ^= 35.4527
MeltingPoint[Chlorine] ^= 172.2 Kelvin
BoilingPoint[Chlorine] ^= 239.6 Kelvin
HeatOfFusion[Chlorine] ^= 6.41 Kilo Joule / Mole
HeatOfVaporization[Chlorine] ^= 20.4033  Kilo Joule / Mole
Density[Chlorine] ^:= DensityException[Chlorine,2030.,113]
ThermalConductivity[Chlorine] ^:= ThermalConductivityException[Chlorine,
					0.0089, "gaseous"]
StableIsotopes[Chlorine] ^= {35,37}
ElectronConfiguration[Chlorine] ^= {{2},{2,6},{2,5}}
Abbreviation[Chlorine] ^= "Cl"
SolarSystemAbundance[Chlorine] ^= 1.7 10^-7
EarthCrustAbundance[Chlorine] ^= 0.000145
EarthOceanAbundance[Chlorine] ^= 
  0.0194
IonizationPotential[Chlorine] ^= 
  12.968 ElectronVolt
SpecificHeat[Chlorine] ^= (0.479 Joule)/(Gram Kelvin)
 
AtomicNumber[Argon] ^= 18
AtomicWeight[Argon] ^= 39.948
MeltingPoint[Argon] ^= 83.8 Kelvin
BoilingPoint[Argon] ^= 87.3 Kelvin
HeatOfFusion[Argon] ^= 1.21  Kilo Joule / Mole
HeatOfVaporization[Argon] ^= 6.53  Kilo Joule / Mole
Density[Argon] ^:= DensityException[Argon,1656.,40]
ThermalConductivity[Argon] ^:= ThermalConductivityException[Argon,
				0.01772, "gaseous"]
StableIsotopes[Argon] ^= {36,38,40} 
ElectronConfiguration[Argon] ^= {{2},{2,6},{2,6}}
Abbreviation[Argon] ^= "Ar"
SolarSystemAbundance[Argon] ^= 3.29 10^-6
EarthCrustAbundance[Argon] ^= 3.5 10^-6
EarthOceanAbundance[Argon] ^= 4.5 10^-7
IonizationPotential[Argon] ^= 15.76 ElectronVolt
SpecificHeat[Argon] ^= (0.52 Joule)/(Gram Kelvin)
 
Kalium = Potassium
AtomicNumber[Potassium] ^= 19
AtomicWeight[Potassium] ^= 39.0983
MeltingPoint[Potassium] ^= 336.8 Kelvin
BoilingPoint[Potassium] ^= 1047. Kelvin
HeatOfFusion[Potassium] ^=  2.40 Kilo Joule / Mole
HeatOfVaporization[Potassium] ^= 79.1  Kilo Joule / Mole
Density[Potassium] ^= 862. Kilogram/(Meter)^3 
ThermalConductivity[Potassium] ^=  102.4 Watt/(Meter Kelvin)
StableIsotopes[Potassium] ^=  {39,41}
ElectronConfiguration[Potassium] ^= {{2},{2,6},{2,6},{1}}
Abbreviation[Potassium] ^= "K"
SolarSystemAbundance[Potassium] ^= 1.23 10^-7
EarthCrustAbundance[Potassium] ^= 
  0.0209 
EarthOceanAbundance[Potassium] ^= 0.000399
IonizationPotential[Potassium] ^= 4.341 ElectronVolt
SpecificHeat[Potassium] ^= (0.757 Joule)/(Gram Kelvin)
 
AtomicNumber[Calcium] ^= 20
AtomicWeight[Calcium] ^= 40.078
MeltingPoint[Calcium] ^= 1112. Kelvin
BoilingPoint[Calcium] ^= 1757. Kelvin
HeatOfFusion[Calcium] ^= 9.33  Kilo Joule / Mole
HeatOfVaporization[Calcium] ^= 150.6  Kilo Joule / Mole
Density[Calcium] ^=  1550. Kilogram/(Meter)^3
ThermalConductivity[Calcium] ^=  200. Watt/(Meter Kelvin)
StableIsotopes[Calcium] ^=  {40,42,43,44,46,48}
ElectronConfiguration[Calcium] ^= {{2},{2,6},{2,6},{2}}
Abbreviation[Calcium] ^= "Ca"
SolarSystemAbundance[Calcium] ^= 2. 10^-6
EarthCrustAbundance[Calcium] ^= 
  0.0415 
EarthOceanAbundance[Calcium] ^= 0.000412
IonizationPotential[Calcium] ^= 6.113 ElectronVolt
SpecificHeat[Calcium] ^= (0.647 Joule)/(Gram Kelvin)
 
AtomicNumber[Scandium] ^= 21
AtomicWeight[Scandium] ^= 44.955910
MeltingPoint[Scandium] ^= 1814. Kelvin 
BoilingPoint[Scandium] ^= 3104. Kelvin
HeatOfFusion[Scandium] ^=  15.9 Kilo Joule / Mole
HeatOfVaporization[Scandium] ^= 376.1  Kilo Joule / Mole
Density[Scandium] ^=   2989. Kilogram/(Meter)^3
ThermalConductivity[Scandium] ^=  15.8 Watt/(Meter Kelvin)
StableIsotopes[Scandium] ^=  {45}
ElectronConfiguration[Scandium] ^= {{2},{2,6},{2,6,1},{2}}
Abbreviation[Scandium] ^= "Sc"
SolarSystemAbundance[Scandium] ^= 1.12 10^-9
EarthCrustAbundance[Scandium] ^= 0.000022
EarthOceanAbundance[Scandium] ^= 6.0 10^-13
IonizationPotential[Scandium] ^= 
  6.561 ElectronVolt
SpecificHeat[Scandium] ^= 
  (0.568 Joule)/(Gram Kelvin)
 
AtomicNumber[Titanium] ^= 22
AtomicWeight[Titanium] ^= 47.867
MeltingPoint[Titanium] ^= 1933. Kelvin
BoilingPoint[Titanium] ^= 3560. Kelvin
HeatOfFusion[Titanium] ^= 20.9  Kilo Joule / Mole
HeatOfVaporization[Titanium] ^= 425.5  Kilo Joule / Mole
Density[Titanium] ^=  4540. Kilogram/(Meter)^3
ThermalConductivity[Titanium] ^=  21.9 Watt/(Meter Kelvin)
StableIsotopes[Titanium] ^=  {46,47,48,49,50}
ElectronConfiguration[Titanium] ^= {{2},{2,6},{2,6,2},{2}}
Abbreviation[Titanium] ^= "Ti"
SolarSystemAbundance[Titanium] ^= 7.8 10^-8
EarthCrustAbundance[Titanium] ^= 
  0.00565
EarthOceanAbundance[Titanium] ^= 1. 10^-9
IonizationPotential[Titanium] ^= 6.828 ElectronVolt
SpecificHeat[Titanium] ^= (0.523 Joule)/(Gram Kelvin)
 
AtomicNumber[Vanadium] ^= 23
AtomicWeight[Vanadium] ^= 50.9415
MeltingPoint[Vanadium] ^= 2160. Kelvin
BoilingPoint[Vanadium] ^= 3650. Kelvin
HeatOfFusion[Vanadium] ^= 17.6 Kilo Joule / Mole
HeatOfVaporization[Vanadium] ^= 459.70  Kilo Joule / Mole
Density[Vanadium] ^=  6110. Kilogram/(Meter)^3
ThermalConductivity[Vanadium] ^=  30.7 Watt/(Meter Kelvin)
StableIsotopes[Vanadium] ^=  {51}
ElectronConfiguration[Vanadium] ^= {{2},{2,6},{2,6,3},{2}}
Abbreviation[Vanadium] ^= "V"
SolarSystemAbundance[Vanadium] ^= 9.6 10^-9
EarthCrustAbundance[Vanadium] ^= 0.00012
EarthOceanAbundance[Vanadium] ^= 2.5 10^-9
IonizationPotential[Vanadium] ^= 6.746 ElectronVolt
SpecificHeat[Vanadium] ^= 
  (0.489 Joule)/(Gram Kelvin)
 
AtomicNumber[Chromium] ^= 24
AtomicWeight[Chromium] ^= 51.9961
MeltingPoint[Chromium] ^= 2130. Kelvin		 	(* +- 20. Kelvin *)
BoilingPoint[Chromium] ^= 2945. Kelvin
HeatOfFusion[Chromium] ^= 15.3  Kilo Joule / Mole
HeatOfVaporization[Chromium] ^= 341.8  Kilo Joule / Mole
Density[Chromium] ^=  7190. Kilogram/(Meter)^3
ThermalConductivity[Chromium] ^=  93.7 Watt/(Meter Kelvin)
StableIsotopes[Chromium] ^=  {50,52,53,54}
ElectronConfiguration[Chromium] ^= {{2},{2,6},{2,6,5},{1}}
Abbreviation[Chromium] ^= "Cr"
SolarSystemAbundance[Chromium] ^= 4.4 10^-7
EarthCrustAbundance[Chromium] ^= 0.000102
EarthOceanAbundance[Chromium] ^= 3.0 10^-10 
IonizationPotential[Chromium] ^= 6.767 ElectronVolt
SpecificHeat[Chromium] ^= (0.449 Joule)/(Gram Kelvin)
 
AtomicNumber[Manganese] ^= 25
AtomicWeight[Manganese] ^= 54.938049
MeltingPoint[Manganese] ^= 1517. Kelvin
BoilingPoint[Manganese] ^= 2235. Kelvin
HeatOfFusion[Manganese] ^=  14.4 Kilo Joule / Mole
HeatOfVaporization[Manganese] ^= 220.5  Kilo Joule / Mole
Density[Manganese] ^=  7440. Kilogram/(Meter)^3
ThermalConductivity[Manganese] ^=  7.82 Watt/(Meter Kelvin)
StableIsotopes[Manganese] ^=  {55}
ElectronConfiguration[Manganese] ^= {{2},{2,6},{2,6,5},{2}}
Abbreviation[Manganese] ^= "Mn"
SolarSystemAbundance[Manganese] ^= 3.1 10^-7
EarthCrustAbundance[Manganese] ^= 
  0.00095
EarthOceanAbundance[Manganese] ^= 2.0 10^-10
IonizationPotential[Manganese] ^= 7.434 ElectronVolt
SpecificHeat[Manganese] ^= (0.479 Joule)/(Gram Kelvin)
 
AtomicNumber[Iron] ^= 26
AtomicWeight[Iron] ^= 55.845
MeltingPoint[Iron] ^= 1808. Kelvin
BoilingPoint[Iron] ^= 3023. Kelvin
HeatOfFusion[Iron] ^= 14.9  Kilo Joule / Mole
HeatOfVaporization[Iron] ^= 340.2  Kilo Joule / Mole
Density[Iron] ^=  7874. Kilogram/(Meter)^3
ThermalConductivity[Iron] ^=  80.2 Watt/(Meter Kelvin)
StableIsotopes[Iron] ^=  {54,56,57,58}
ElectronConfiguration[Iron] ^= {{2},{2,6},{2,6,6},{2}}
Abbreviation[Iron] ^= "Fe"
SolarSystemAbundance[Iron] ^= 0.0000294
EarthCrustAbundance[Iron] ^= 
  0.0563 
EarthOceanAbundance[Iron] ^= 2. 10^-9
IonizationPotential[Iron] ^= 7.902 ElectronVolt
SpecificHeat[Iron] ^= (0.449 Joule)/(Gram Kelvin)
 
AtomicNumber[Cobalt] ^= 27
AtomicWeight[Cobalt] ^= 58.933200
MeltingPoint[Cobalt] ^= 1768. Kelvin
BoilingPoint[Cobalt] ^= 3143. Kelvin
HeatOfFusion[Cobalt] ^= 15.2  Kilo Joule / Mole
HeatOfVaporization[Cobalt] ^= 382.4  Kilo Joule / Mole
Density[Cobalt] ^=  8900. Kilogram/(Meter)^3
ThermalConductivity[Cobalt] ^=  100. Watt/(Meter Kelvin)
StableIsotopes[Cobalt] ^=  {59}
ElectronConfiguration[Cobalt] ^= {{2},{2,6},{2,6,7},{2}}
Abbreviation[Cobalt] ^= "Co"
SolarSystemAbundance[Cobalt] ^= 7.3 10^-9
EarthCrustAbundance[Cobalt] ^= 0.000025
EarthOceanAbundance[Cobalt] ^= 2.0 10^-11 
IonizationPotential[Cobalt] ^= 7.881 ElectronVolt
SpecificHeat[Cobalt] ^= 
  (0.421 Joule)/(Gram Kelvin)
 
AtomicNumber[Nickel] ^= 28
AtomicWeight[Nickel] ^= 58.6934
MeltingPoint[Nickel] ^= 1726. Kelvin
BoilingPoint[Nickel] ^= 3005. Kelvin
HeatOfFusion[Nickel] ^= 17.6 Kilo Joule / Mole
HeatOfVaporization[Nickel] ^= 374.8  Kilo Joule / Mole
Density[Nickel] ^=  8902. Kilogram/(Meter)^3
ThermalConductivity[Nickel] ^=  90.7 Watt/(Meter Kelvin)
StableIsotopes[Nickel] ^=  {58,60,61,62,64}
ElectronConfiguration[Nickel] ^= {{2},{2,6},{2,6,8},{2}}
Abbreviation[Nickel] ^= "Ni"
SolarSystemAbundance[Nickel] ^= 1.61 10^-6
EarthCrustAbundance[Nickel] ^= 0.000084 
EarthOceanAbundance[Nickel] ^= 
  5.6 10^-10
IonizationPotential[Nickel] ^= 7.64 ElectronVolt
SpecificHeat[Nickel] ^= (0.444 Joule)/(Gram Kelvin)
 
AtomicNumber[Copper] ^= 29
AtomicWeight[Copper] ^= 63.546
MeltingPoint[Copper] ^= 1356.6 Kelvin
BoilingPoint[Copper] ^= 2840 Kelvin
HeatOfFusion[Copper] ^= 13.0  Kilo Joule / Mole
HeatOfVaporization[Copper] ^= 306.7  Kilo Joule / Mole
Density[Copper] ^=  8960. Kilogram/(Meter)^3
ThermalConductivity[Copper] ^=   401. Watt/(Meter Kelvin)
StableIsotopes[Copper] ^=  {63,65}
ElectronConfiguration[Copper] ^= {{2},{2,6},{2,6,10},{1}}
Abbreviation[Copper] ^= "Cu"
SolarSystemAbundance[Copper] ^= 1.7 10^-8
EarthCrustAbundance[Copper] ^= 0.00006
EarthOceanAbundance[Copper] ^= 2.5 10^-10
IonizationPotential[Copper] ^= 7.726 ElectronVolt
SpecificHeat[Copper] ^= (0.385 Joule)/(Gram Kelvin)
 
AtomicNumber[Zinc] ^= 30
AtomicWeight[Zinc] ^= 65.39
MeltingPoint[Zinc] ^= 692.73 Kelvin
BoilingPoint[Zinc] ^= 1180. Kelvin
HeatOfFusion[Zinc] ^= 6.67  Kilo Joule / Mole
HeatOfVaporization[Zinc] ^= 114.2  Kilo Joule / Mole
Density[Zinc] ^=  7133. Kilogram/(Meter)^3
ThermalConductivity[Zinc] ^=  116. Watt/(Meter Kelvin)
StableIsotopes[Zinc] ^=  {66,67,68,70}
ElectronConfiguration[Zinc] ^= {{2},{2,6},{2,6,10},{2}}
Abbreviation[Zinc] ^= "Zn"
SolarSystemAbundance[Zinc] ^= 4.11 10^-8
EarthCrustAbundance[Zinc] ^= 0.00007
EarthOceanAbundance[Zinc] ^= 4.9 10^-9
IonizationPotential[Zinc] ^= 9.394 ElectronVolt
SpecificHeat[Zinc] ^= (0.388 Joule)/(Gram Kelvin)
 
AtomicNumber[Gallium] ^= 31
AtomicWeight[Gallium] ^= 69.723
MeltingPoint[Gallium] ^= 302.93 Kelvin
BoilingPoint[Gallium] ^= 2676. Kelvin
HeatOfFusion[Gallium] ^= 5.59  Kilo Joule / Mole
HeatOfVaporization[Gallium] ^= 270.3  Kilo Joule / Mole
Density[Gallium] ^=  5907. Kilogram/(Meter)^3
ThermalConductivity[Gallium] ^=  40.6 Watt/(Meter Kelvin)
StableIsotopes[Gallium] ^=  {69,71}
ElectronConfiguration[Gallium] ^= {{2},{2,6},{2,6,10},{2,1}}
Abbreviation[Gallium] ^= "Ga"
SolarSystemAbundance[Gallium] ^= 1.23 10^-9
EarthCrustAbundance[Gallium] ^= 0.000019 
EarthOceanAbundance[Gallium] ^= 3.0 10^-11
IonizationPotential[Gallium] ^= 
  5.999 ElectronVolt
SpecificHeat[Gallium] ^= 
  (0.371 Joule)/(Gram Kelvin)
 
AtomicNumber[Germanium] ^= 32
AtomicWeight[Germanium] ^= 72.61
MeltingPoint[Germanium] ^= 1210.6 Kelvin
BoilingPoint[Germanium] ^= 3103. Kelvin
HeatOfFusion[Germanium] ^= 34.7 Kilo Joule / Mole
HeatOfVaporization[Germanium] ^= 327.6 Kilo Joule / Mole
Density[Germanium] ^=  5323. Kilogram/(Meter)^3
ThermalConductivity[Germanium] ^=  59.9 Watt/(Meter Kelvin)
StableIsotopes[Germanium] ^=  {70,72,73,74,76}
ElectronConfiguration[Germanium] ^= {{2},{2,6},{2,6,10},{2,2}}
Abbreviation[Germanium] ^= "Ge"
SolarSystemAbundance[Germanium] ^= 3.9 10^-9
EarthCrustAbundance[Germanium] ^= 
  1.5 10^-6 
EarthOceanAbundance[Germanium] ^= 5.0 10^-11
IonizationPotential[Germanium] ^= 7.9 ElectronVolt
SpecificHeat[Germanium] ^= (0.32 Joule)/(Gram Kelvin)
 
AtomicNumber[Arsenic] ^= 33
AtomicWeight[Arsenic] ^= 74.92160
MeltingPoint[Arsenic] ^= 83.78 Kelvin
BoilingPoint[Arsenic] ^= 889 Kelvin 	(* temperature at which As sublimes *)
HeatOfFusion[Arsenic] ^= 27.7 Kilo Joule / Mole
HeatOfVaporization[Arsenic] ^= 31.9 Kilo Joule / Mole
Density[Arsenic] ^:= DensityException[Arsenic,5780.,"alpha"]
ThermalConductivity[Arsenic] ^:= ThermalConductivityException[Arsenic,
				50.0, "alpha"]
StableIsotopes[Arsenic] ^=  {75}
ElectronConfiguration[Arsenic] ^= {{2},{2,6},{2,6,10},{2,3}}
Abbreviation[Arsenic] ^= "As"
SolarSystemAbundance[Arsenic] ^= 2.1 10^-10
EarthCrustAbundance[Arsenic] ^= 1.8 10^-6
EarthOceanAbundance[Arsenic] ^= 3.7 10^-9
IonizationPotential[Arsenic] ^= 
  9.815 ElectronVolt
SpecificHeat[Arsenic] ^= (0.329 Joule)/(Gram Kelvin)
 
AtomicNumber[Selenium] ^= 34
AtomicWeight[Selenium] ^= 78.96
MeltingPoint[Selenium] ^= 490. Kelvin
BoilingPoint[Selenium] ^= 958.1 Kelvin
HeatOfFusion[Selenium] ^=  5.1 Kilo Joule / Mole
HeatOfVaporization[Selenium] ^= 90. Kilo Joule / Mole 
Density[Selenium] ^=  4790. Kilogram/(Meter)^3
ThermalConductivity[Selenium] ^=  2.04 Watt/(Meter Kelvin)
StableIsotopes[Selenium] ^=  {74,76,77,78,80,82}
ElectronConfiguration[Selenium] ^= {{2},{2,6},{2,6,10},{2,4}}
Abbreviation[Selenium] ^= "Se"
SolarSystemAbundance[Selenium] ^= 2.03 10^-9
EarthCrustAbundance[Selenium] ^= 
  5. 10^-8
EarthOceanAbundance[Selenium] ^= 2.0 10^-9
IonizationPotential[Selenium] ^= 9.752 ElectronVolt
SpecificHeat[Selenium] ^= 
  (0.321 Joule)/(Gram Kelvin)
 
AtomicNumber[Bromine] ^= 35
AtomicWeight[Bromine] ^= 79.904
MeltingPoint[Bromine] ^= 265.9 Kelvin
BoilingPoint[Bromine] ^= 331.9 Kelvin
HeatOfFusion[Bromine] ^= 10.8  Kilo Joule / Mole
HeatOfVaporization[Bromine] ^= 30.5  Kilo Joule / Mole
Density[Bromine] ^:= DensityException[Bromine,4050.,123]
ThermalConductivity[Bromine] ^:= ThermalConductivityException[Bromine,
				0.122, "liquid"]
StableIsotopes[Bromine] ^=  {79,81}
ElectronConfiguration[Bromine] ^= {{2},{2,6},{2,6,10},{2,5}}
Abbreviation[Bromine] ^= "Br"
SolarSystemAbundance[Bromine] ^= 2.03 10^-9
EarthCrustAbundance[Bromine] ^= 
  2.4 10^-6
EarthOceanAbundance[Bromine] ^= 
  0.0000673
IonizationPotential[Bromine] ^= 11.814 ElectronVolt
SpecificHeat[Bromine] ^= (0.226 Joule)/(Gram Kelvin)
 
AtomicNumber[Krypton] ^= 36
AtomicWeight[Krypton] ^= 83.80
MeltingPoint[Krypton] ^= 116.6 Kelvin
BoilingPoint[Krypton] ^= 120.85 Kelvin
HeatOfFusion[Krypton] ^= 1.64  Kilo Joule / Mole
HeatOfVaporization[Krypton] ^= 9.05  Kilo Joule / Mole
Density[Krypton] ^:= DensityException[Krypton,2823.,117]
ThermalConductivity[Krypton] ^:= ThermalConductivityException[Krypton,
				0.00949, "gaseous"]
StableIsotopes[Krypton] ^=  {78,80,82,83,84,86}
ElectronConfiguration[Krypton] ^= {{2},{2,6},{2,6,10},{2,6}}
Abbreviation[Krypton] ^= "Kr"
SolarSystemAbundance[Krypton] ^= 1.5 10^-9
EarthCrustAbundance[Krypton] ^= 
  1.0 10^-10
EarthOceanAbundance[Krypton] ^= 2.1 10^-10
IonizationPotential[Krypton] ^= 14 ElectronVolt
SpecificHeat[Krypton] ^= 
  (0.248 Joule)/(Gram Kelvin)
 
AtomicNumber[Rubidium] ^= 37
AtomicWeight[Rubidium] ^= 85.4678
MeltingPoint[Rubidium] ^= 312.2 Kelvin
BoilingPoint[Rubidium] ^= 961. Kelvin
HeatOfFusion[Rubidium] ^= 2.20  Kilo Joule / Mole
HeatOfVaporization[Rubidium] ^= 75.7  Kilo Joule / Mole
Density[Rubidium] ^= 1532.   Kilogram/(Meter)^3
ThermalConductivity[Rubidium] ^=  58.2 Watt/(Meter Kelvin)
StableIsotopes[Rubidium] ^=  {85}
ElectronConfiguration[Rubidium] ^= {{2},{2,6},{2,6,10},{2,6},{1}}
Abbreviation[Rubidium] ^= "Rb"
SolarSystemAbundance[Rubidium] ^= 2.31 10^-10
EarthCrustAbundance[Rubidium] ^= 0.00009
EarthOceanAbundance[Rubidium] ^= 
  1.2 10^-7
IonizationPotential[Rubidium] ^= 
  4.177 ElectronVolt
SpecificHeat[Rubidium] ^= 
  (0.363 Joule)/(Gram Kelvin)
 
AtomicNumber[Strontium] ^= 38
AtomicWeight[Strontium] ^= 87.62
MeltingPoint[Strontium] ^= 1042. Kelvin
BoilingPoint[Strontium] ^= 1657 Kelvin
HeatOfFusion[Strontium] ^= 9.16  Kilo Joule / Mole
HeatOfVaporization[Strontium] ^= 154.4  Kilo Joule / Mole
Density[Strontium] ^=  2540. Kilogram/(Meter)^3
ThermalConductivity[Strontium] ^=  35.3 Watt/(Meter Kelvin)
StableIsotopes[Strontium] ^=  {84,86,87,88}
ElectronConfiguration[Strontium] ^= {{2},{2,6},{2,6,10},{2,6},{2}}
Abbreviation[Strontium] ^= "Sr"
SolarSystemAbundance[Strontium] ^= 7.7 10^-10
EarthCrustAbundance[Strontium] ^= 
  0.00037
EarthOceanAbundance[Strontium] ^= 7.9 10^-6
IonizationPotential[Strontium] ^= 5.695 ElectronVolt
SpecificHeat[Strontium] ^= 
  (0.301 Joule)/(Gram Kelvin)
 
AtomicNumber[Yttrium] ^= 39
AtomicWeight[Yttrium] ^= 88.90585
MeltingPoint[Yttrium] ^= 1795. Kelvin
BoilingPoint[Yttrium] ^= 3611. Kelvin
HeatOfFusion[Yttrium] ^= 17.2 Kilo Joule / Mole
HeatOfVaporization[Yttrium] ^= 367.4 Kilo Joule / Mole
Density[Yttrium] ^=  4469. Kilogram/(Meter)^3
ThermalConductivity[Yttrium] ^= 17.2 Watt/(Meter Kelvin) 
StableIsotopes[Yttrium] ^=  {89}
ElectronConfiguration[Yttrium] ^= {{2},{2,6},{2,6,10},{2,6,1},{2}}
Abbreviation[Yttrium] ^= "Y"
SolarSystemAbundance[Yttrium] ^= 1.51 10^-10
EarthCrustAbundance[Yttrium] ^= 0.000033
EarthOceanAbundance[Yttrium] ^= 
  1.2 10^-11
IonizationPotential[Yttrium] ^= 
  6.217 ElectronVolt
SpecificHeat[Yttrium] ^= (0.298 Joule)/(Gram Kelvin)
 
AtomicNumber[Zirconium] ^= 40
AtomicWeight[Zirconium] ^= 91.224
MeltingPoint[Zirconium] ^= 2125. Kelvin
BoilingPoint[Zirconium] ^= 4650. Kelvin
HeatOfFusion[Zirconium] ^= 23.0  Kilo Joule / Mole
HeatOfVaporization[Zirconium] ^= 566.7 Kilo Joule / Mole
Density[Zirconium] ^=  6506. Kilogram/(Meter)^3
ThermalConductivity[Zirconium] ^=  22.7 Watt/(Meter Kelvin)
StableIsotopes[Zirconium] ^=  {90,91,92,94}
ElectronConfiguration[Zirconium] ^= {{2},{2,6},{2,6,10},{2,6,2},{2}}
Abbreviation[Zirconium] ^= "Zr"
SolarSystemAbundance[Zirconium] ^= 3.72 10^-10
EarthCrustAbundance[Zirconium] ^= 
  0.000165
EarthOceanAbundance[Zirconium] ^= 3.0 10^-11
IonizationPotential[Zirconium] ^= 6.634 ElectronVolt
SpecificHeat[Zirconium] ^= (0.278 Joule)/(Gram Kelvin)
 
AtomicNumber[Niobium] ^= 41
AtomicWeight[Niobium] ^= 92.90638
MeltingPoint[Niobium] ^= 2741. Kelvin 
BoilingPoint[Niobium] ^= 5015. Kelvin 
HeatOfFusion[Niobium] ^= 27.2  Kilo Joule / Mole
HeatOfVaporization[Niobium] ^= 680.19  Kilo Joule / Mole
Density[Niobium] ^=  8570. Kilogram/(Meter)^3
ThermalConductivity[Niobium] ^=  53.7 Watt/(Meter Kelvin)
StableIsotopes[Niobium] ^=  {93}
ElectronConfiguration[Niobium] ^= {{2},{2,6},{2,6,10},{2,6,4},{1}}
Abbreviation[Niobium] ^= "Nb"
SolarSystemAbundance[Niobium] ^= 2.28 10^-11
EarthCrustAbundance[Niobium] ^= 0.00002
EarthOceanAbundance[Niobium] ^= 1.0 10^-11
IonizationPotential[Niobium] ^= 6.759 ElectronVolt
SpecificHeat[Niobium] ^= (0.265 Joule)/(Gram Kelvin)
 
AtomicNumber[Molybdenum] ^= 42
AtomicWeight[Molybdenum] ^= 95.94
MeltingPoint[Molybdenum] ^= 2890. Kelvin
BoilingPoint[Molybdenum] ^= 4885. Kelvin
HeatOfFusion[Molybdenum] ^= 27.6 Kilo Joule / Mole
HeatOfVaporization[Molybdenum] ^= 589.9  Kilo Joule / Mole
Density[Molybdenum] ^=  10220. Kilogram/(Meter)^3
ThermalConductivity[Molybdenum] ^=  138. Watt/(Meter Kelvin)
StableIsotopes[Molybdenum] ^=  {92,94,95,96,97,98,100}
ElectronConfiguration[Molybdenum] ^=
	{{2},{2,6},{2,6,10},{2,6,5},{1}}
Abbreviation[Molybdenum] ^= "Mo"
SolarSystemAbundance[Molybdenum] ^= 8.3 10^-11
EarthCrustAbundance[Molybdenum] ^= 
  1.2 10^-6
EarthOceanAbundance[Molybdenum] ^= 1. 10^-8
IonizationPotential[Molybdenum] ^= 
  7.092 ElectronVolt
SpecificHeat[Molybdenum] ^= 
  (0.251 Joule)/(Gram Kelvin)

AtomicNumber[Technetium] ^= 43
AtomicWeight[Technetium] ^:= AtomicWeightUnstable[Technetium, 98]
MeltingPoint[Technetium] ^= 2445. Kelvin
BoilingPoint[Technetium] ^= 5150. Kelvin
HeatOfFusion[Technetium] ^= 23.81 Kilo Joule / Mole
HeatOfVaporization[Technetium] ^= 585.22 Kilo Joule / Mole
Density[Technetium] ^=  11500. Kilogram/(Meter)^3
ThermalConductivity[Technetium] ^=  50.6 Watt/(Meter Kelvin)
StableIsotopes[Technetium] ^=  {}
ElectronConfiguration[Technetium] ^=
	{{2},{2,6},{2,6,10},{2,6,5},{2}}
Abbreviation[Technetium] ^= "Tc"
EarthCrustAbundance[Technetium] ^:= EarthCrustAbundanceNA[Technetium]
EarthOceanAbundance[Technetium] ^:= EarthOceanAbundanceNA[Technetium]
IonizationPotential[Technetium] ^= 7.28 ElectronVolt
SpecificHeat[Technetium] ^:= SpecificHeatNA[Technetium]
SolarSystemAbundance[Technetium] ^:= SolarSystemAbundanceNA[Technetium]
  
AtomicNumber[Ruthenium] ^= 44
AtomicWeight[Ruthenium] ^= 101.07
MeltingPoint[Ruthenium] ^= 2583. Kelvin 
BoilingPoint[Ruthenium] ^= 4173. Kelvin 
HeatOfFusion[Ruthenium] ^= 23.7  Kilo Joule / Mole
HeatOfVaporization[Ruthenium] ^= 567.  Kilo Joule / Mole
Density[Ruthenium] ^=  12370. Kilogram/(Meter)^3
ThermalConductivity[Ruthenium] ^=  117. Watt/(Meter Kelvin)
StableIsotopes[Ruthenium] ^=  {96,98,99,100,101,102,104}
ElectronConfiguration[Ruthenium] ^= {{2},{2,6},{2,6,10},{2,6,7},{1}}
Abbreviation[Ruthenium] ^= "Ru"
SolarSystemAbundance[Ruthenium] ^= 6.1 10^-11
EarthCrustAbundance[Ruthenium] ^= 
  1. 10^-9
EarthOceanAbundance[Ruthenium] ^= 7.0 10^-13
IonizationPotential[Ruthenium] ^= 
  7.361 ElectronVolt
SpecificHeat[Ruthenium] ^= 
  (0.238 Joule)/(Gram Kelvin)
 
AtomicNumber[Rhodium] ^= 45
AtomicWeight[Rhodium] ^= 102.90550
MeltingPoint[Rhodium] ^= 2239. Kelvin 
BoilingPoint[Rhodium] ^= 4000. Kelvin 
HeatOfFusion[Rhodium] ^= 21.55 Kilo Joule / Mole
HeatOfVaporization[Rhodium] ^= 494.34 Kilo Joule / Mole
Density[Rhodium] ^=  12410. Kilogram/(Meter)^3
ThermalConductivity[Rhodium] ^=  150. Watt/(Meter Kelvin)
StableIsotopes[Rhodium] ^=  {103}
ElectronConfiguration[Rhodium] ^= {{2},{2,6},{2,6,10},{2,6,8},{1}}
Abbreviation[Rhodium] ^= "Rh"
SolarSystemAbundance[Rhodium] ^= 1.12 10^-11
EarthCrustAbundance[Rhodium] ^= 
 1. 10^-9
EarthOceanAbundance[Rhodium] ^:= EarthOceanAbundanceNA[Rhodium] 
IonizationPotential[Rhodium] ^= 
  7.459 ElectronVolt
SpecificHeat[Rhodium] ^= 
  (0.243 Joule)/(Gram Kelvin)

AtomicNumber[Palladium] ^= 46
AtomicWeight[Palladium] ^= 106.42
MeltingPoint[Palladium] ^= 1825. Kelvin
BoilingPoint[Palladium] ^= 3413. Kelvin
HeatOfFusion[Palladium] ^= 17.2 Kilo Joule / Mole
HeatOfVaporization[Palladium] ^= 361.5 Kilo Joule / Mole
Density[Palladium] ^=  12020. Kilogram/(Meter)^3
ThermalConductivity[Palladium] ^=  71.8 Watt/(Meter Kelvin)
StableIsotopes[Palladium] ^=  {102,104,105,106,108,110}
ElectronConfiguration[Palladium] ^= {{2},{2,6},{2,6,10},{2,6,10}}
Abbreviation[Palladium] ^= "Pd"
SolarSystemAbundance[Palladium] ^= 4.5 10^-11
EarthCrustAbundance[Palladium] ^= 
  1.5 10^-8
EarthOceanAbundance[Palladium] ^:= EarthOceanAbundanceNA[Palladium] 
IonizationPotential[Palladium] ^= 
  8.337 ElectronVolt
SpecificHeat[Palladium] ^= 
  (0.244 Joule)/(Gram Kelvin)
 
AtomicNumber[Silver] ^= 47
AtomicWeight[Silver] ^= 107.8682
MeltingPoint[Silver] ^= 1235.08 Kelvin
BoilingPoint[Silver] ^= 2485. Kelvin
HeatOfFusion[Silver] ^= 11.3 Kilo Joule / Mole
HeatOfVaporization[Silver] ^= 257.7  Kilo Joule / Mole
Density[Silver] ^=  10500. Kilogram/(Meter)^3
ThermalConductivity[Silver] ^=  429. Watt/(Meter Kelvin)
StableIsotopes[Silver] ^=  {107,109}
ElectronConfiguration[Silver] ^= {{2},{2,6},{2,6,10},{2,6,10},{1}}
Abbreviation[Silver] ^= "Ag"
SolarSystemAbundance[Silver] ^= 1.58 10^-11
EarthCrustAbundance[Silver] ^= 7.5 10^-8
EarthOceanAbundance[Silver] ^= 4.0 10^-11
IonizationPotential[Silver] ^= 7.576 ElectronVolt
SpecificHeat[Silver] ^= 
  (0.235 Joule)/(Gram Kelvin)
 
AtomicNumber[Cadmium] ^= 48
AtomicWeight[Cadmium] ^= 112.411
MeltingPoint[Cadmium] ^= 594.1 Kelvin
BoilingPoint[Cadmium] ^= 1038. Kelvin
HeatOfFusion[Cadmium] ^= 6.11  Kilo Joule / Mole
HeatOfVaporization[Cadmium] ^= 100.0  Kilo Joule / Mole
Density[Cadmium] ^=  8650. Kilogram/(Meter)^3
ThermalConductivity[Cadmium] ^=  96.8 Watt/(Meter Kelvin)
StableIsotopes[Cadmium] ^=  {106,108,110,111,112,113,114,116}
ElectronConfiguration[Cadmium] ^= {{2},{2,6},{2,6,10},{2,6,10},{2}}
Abbreviation[Cadmium] ^= "Cd"
SolarSystemAbundance[Cadmium] ^= 5.3 10^-11
EarthCrustAbundance[Cadmium] ^= 1.5 10^-7
EarthOceanAbundance[Cadmium] ^= 1.1 10^-10
IonizationPotential[Cadmium] ^= 
  8.994 ElectronVolt
SpecificHeat[Cadmium] ^= (0.232 Joule)/(Gram Kelvin)
 
AtomicNumber[Indium] ^= 49
AtomicWeight[Indium] ^= 114.818
MeltingPoint[Indium] ^= 429.32 Kelvin
BoilingPoint[Indium] ^= 2353. Kelvin 
HeatOfFusion[Indium] ^= 3.27 Kilo Joule / Mole
HeatOfVaporization[Indium] ^= 231.8  Kilo Joule / Mole
Density[Indium] ^=  7310. Kilogram/(Meter)^3
ThermalConductivity[Indium] ^=  81.6 Watt/(Meter Kelvin)
StableIsotopes[Indium] ^=  {113}
ElectronConfiguration[Indium] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,1}}
Abbreviation[Indium] ^= "In"
SolarSystemAbundance[Indium] ^= 6. 10^-10
EarthCrustAbundance[Indium] ^= 2.5 10^-7
EarthOceanAbundance[Indium] ^= 2. 10^-8
IonizationPotential[Indium] ^= 5.786 ElectronVolt
SpecificHeat[Indium] ^= (0.233 Joule)/(Gram Kelvin)
 
AtomicNumber[Tin] ^= 50
AtomicWeight[Tin] ^= 118.710
MeltingPoint[Tin] ^= 505.118 Kelvin
BoilingPoint[Tin] ^= 2543. Kelvin
HeatOfFusion[Tin] ^= 7.20 Kilo Joule / Mole
HeatOfVaporization[Tin] ^= 296.2  Kilo Joule / Mole
Density[Tin] ^:= DensityException[Tin,7310.,"beta"]
ThermalConductivity[Tin] ^:= ThermalConductivityException[Tin,
				66.6, "alpha"]
StableIsotopes[Tin] ^=  {112,114,115,116,117,118,119,120,122,124}
ElectronConfiguration[Tin] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,2}}
Abbreviation[Tin] ^= "Sn"
SolarSystemAbundance[Tin] ^= 1.25 10^-10
EarthCrustAbundance[Tin] ^= 
  2.3 10^-6
EarthOceanAbundance[Tin] ^= 4.0 10^-12
IonizationPotential[Tin] ^= 7.344 ElectronVolt
SpecificHeat[Tin] ^= (0.228 Joule)/(Gram Kelvin)
 
Stibium = Antimony
AtomicNumber[Antimony] ^= 51
AtomicWeight[Antimony] ^= 121.760
MeltingPoint[Antimony] ^= 903.89 Kelvin
BoilingPoint[Antimony] ^= 1908. Kelvin
HeatOfFusion[Antimony] ^= 20.9  Kilo Joule / Mole
HeatOfVaporization[Antimony] ^= 165.8  Kilo Joule / Mole
Density[Antimony] ^=  6691. Kilogram/(Meter)^3
ThermalConductivity[Antimony] ^=  243. Watt/(Meter Kelvin)
StableIsotopes[Antimony] ^=  {121,123}
ElectronConfiguration[Antimony] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,3}}
Abbreviation[Antimony] ^= "Sb"
SolarSystemAbundance[Antimony] ^= 1.01 10^-11
EarthCrustAbundance[Antimony] ^= 2. 10^-7
EarthOceanAbundance[Antimony] ^= 
  2.4 10^-10
IonizationPotential[Antimony] ^= 8.64 ElectronVolt
SpecificHeat[Antimony] ^= 
  (0.207  Joule)/(Gram Kelvin)
 
AtomicNumber[Tellurium] ^= 52
AtomicWeight[Tellurium] ^= 127.60
MeltingPoint[Tellurium] ^= 722.7 Kelvin
BoilingPoint[Tellurium] ^= 1263. Kelvin
HeatOfFusion[Tellurium] ^=  13.5 Kilo Joule / Mole
HeatOfVaporization[Tellurium] ^= 104.6 Kilo Joule / Mole
Density[Tellurium] ^= 6240. Kilogram/(Meter)^3
ThermalConductivity[Tellurium] ^= 2.35 Watt/(Meter Kelvin)
StableIsotopes[Tellurium] ^= {120,122,124,125,126,128,130}
ElectronConfiguration[Tellurium] ^=
	{{2},{2,6},{2,6,10},{2,6,10},{2,4}}
Abbreviation[Tellurium] ^= "Te"
SolarSystemAbundance[Tellurium] ^= 1.57 10^-10
EarthCrustAbundance[Tellurium] ^= 
  1. 10^-9
EarthOceanAbundance[Tellurium] ^:= EarthOceanAbundanceNA[Tellurium] 
IonizationPotential[Tellurium] ^= 
  9.01 ElectronVolt
SpecificHeat[Tellurium] ^= (0.202 Joule)/(Gram Kelvin)
 
AtomicNumber[Iodine] ^= 53
AtomicWeight[Iodine] ^= 126.90447
MeltingPoint[Iodine] ^= 386.7 Kelvin
BoilingPoint[Iodine] ^= 457.50 Kelvin
HeatOfFusion[Iodine] ^=  15.27 Kilo Joule / Mole
HeatOfVaporization[Iodine] ^= 41.67 Kilo Joule / Mole
Density[Iodine] ^=  4930. Kilogram/(Meter)^3
ThermalConductivity[Iodine] ^=  0.449 Watt/(Meter Kelvin)
StableIsotopes[Iodine] ^=  {127}
ElectronConfiguration[Iodine] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,5}}
Abbreviation[Iodine] ^= "I"
SolarSystemAbundance[Iodine] ^= 2.9 10^-11
EarthCrustAbundance[Iodine] ^= 4.5 10^-7
EarthOceanAbundance[Iodine] ^= 6. 10^-8
IonizationPotential[Iodine] ^= 10.451 ElectronVolt
SpecificHeat[Iodine] ^= (0.145 Joule)/(Gram Kelvin)
 
AtomicNumber[Xenon] ^= 54
AtomicWeight[Xenon] ^= 131.29
MeltingPoint[Xenon] ^= 161.3 Kelvin
BoilingPoint[Xenon] ^= 166.1 Kelvin
HeatOfFusion[Xenon] ^=  3.10 Kilo Joule / Mole
HeatOfVaporization[Xenon] ^= 12.65  Kilo Joule / Mole
Density[Xenon] ^=  3540. Kilogram/(Meter)^3
ThermalConductivity[Xenon] ^:= ThermalConductivityException[Xenon,
				0.00569, "gaseous"]
StableIsotopes[Xenon] ^=  {129,130,131,132,134,136}
ElectronConfiguration[Xenon] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,6}}
Abbreviation[Xenon] ^= "Xe"
SolarSystemAbundance[Xenon] ^= 1.5 10^-10
EarthCrustAbundance[Xenon] ^= 
  3.0 10^-11
EarthOceanAbundance[Xenon] ^= 5.0 10^-11 
IonizationPotential[Xenon] ^= 12.13 ElectronVolt
SpecificHeat[Xenon] ^= (0.158 Joule)/(Gram Kelvin)
 
Cesium = Caesium
AtomicNumber[Caesium] ^= 55
AtomicWeight[Caesium] ^= 132.90545
MeltingPoint[Caesium] ^= 301.55 Kelvin
BoilingPoint[Caesium] ^= 951.6 Kelvin
HeatOfFusion[Caesium] ^=  2.09 Kilo Joule / Mole
HeatOfVaporization[Caesium] ^= 66.5  Kilo Joule / Mole
Density[Caesium] ^= 1873. Kilogram/(Meter)^3
ThermalConductivity[Caesium] ^= 35.9 Watt/(Meter Kelvin)
StableIsotopes[Caesium] ^= {133}
ElectronConfiguration[Caesium] ^=
	{{2},{2,6},{2,6,10},{2,6,10},{2,6},{1}}
Abbreviation[Caesium] ^= "Cs"
SolarSystemAbundance[Caesium] ^= 1.21 10^-11
EarthCrustAbundance[Caesium] ^= 3.0 10^-6
EarthOceanAbundance[Caesium] ^= 3.0 10^-10
IonizationPotential[Caesium] ^= 3.894 ElectronVolt
SpecificHeat[Caesium] ^= 
  (0.242 Joule)/(Gram Kelvin)
 
AtomicNumber[Barium] ^= 56
AtomicWeight[Barium] ^= 137.327
MeltingPoint[Barium] ^= 1002. Kelvin
BoilingPoint[Barium] ^= 1910. Kelvin
HeatOfFusion[Barium] ^=  7.66 Kilo Joule / Mole
HeatOfVaporization[Barium] ^= 150.9  Kilo Joule / Mole
Density[Barium] ^=  3594. Kilogram/(Meter)^3
ThermalConductivity[Barium] ^=  18.4 Watt/(Meter Kelvin)
StableIsotopes[Barium] ^=  {130,132,134,135,136,137,138}
ElectronConfiguration[Barium] ^= {{2},{2,6},{2,6,10},{2,6,10},{2,6},{2}}
Abbreviation[Barium] ^= "Ba"
SolarSystemAbundance[Barium] ^= 1.46 10^-10
EarthCrustAbundance[Barium] ^= 0.000425
EarthOceanAbundance[Barium] ^= 1.3 10^-8
IonizationPotential[Barium] ^= 5.212 ElectronVolt
SpecificHeat[Barium] ^= (0.204 Joule)/(Gram Kelvin)
 
AtomicNumber[Lanthanum] ^= 57
AtomicWeight[Lanthanum] ^= 138.9055
MeltingPoint[Lanthanum] ^= 1194. Kelvin
BoilingPoint[Lanthanum] ^= 3730. Kelvin 
HeatOfFusion[Lanthanum] ^=  10.04 Kilo Joule / Mole
HeatOfVaporization[Lanthanum] ^= 402.1	 Kilo Joule / Mole
Density[Lanthanum] ^=  6145. Kilogram/(Meter)^3
ThermalConductivity[Lanthanum] ^=  13.5 Watt/(Meter Kelvin)
StableIsotopes[Lanthanum] ^=  {138}
ElectronConfiguration[Lanthanum] ^=
	{{2},{2,6},{2,6,10},{2,6,10},{2,6,1},{2}}
Abbreviation[Lanthanum] ^= "La"
SolarSystemAbundance[Lanthanum] ^= 1.45 10^-11
EarthCrustAbundance[Lanthanum] ^= 0.000039
EarthOceanAbundance[Lanthanum] ^= 
  3.4 10^-12
IonizationPotential[Lanthanum] ^= 
  5.577 ElectronVolt
SpecificHeat[Lanthanum] ^= (0.195 Joule)/(Gram Kelvin)
 
AtomicNumber[Cerium] ^= 58
AtomicWeight[Cerium] ^= 140.116
MeltingPoint[Cerium] ^= 1072. Kelvin 
BoilingPoint[Cerium] ^= 3699. Kelvin 
HeatOfFusion[Cerium] ^= 8.87 Kilo Joule / Mole
HeatOfVaporization[Cerium] ^= 398. Kilo Joule / Mole
Density[Cerium] ^:= DensityException[Cerium,8240.,"alpha"]
ThermalConductivity[Cerium] ^=  11.4 Watt/(Meter Kelvin)
StableIsotopes[Cerium] ^=  {136,138,140,142}
ElectronConfiguration[Cerium] ^=
	{{2},{2,6},{2,6,10},{2,6,10,2},{2,6},{2}}
Abbreviation[Cerium] ^= "Ce"
SolarSystemAbundance[Cerium] ^= 3.7 10^-11
EarthCrustAbundance[Cerium] ^= 0.0000665
EarthOceanAbundance[Cerium] ^= 
  1.2 10^-12
IonizationPotential[Cerium] ^= 5.539 ElectronVolt
SpecificHeat[Cerium] ^= 
  (0.192 Joule)/(Gram Kelvin)
 
AtomicNumber[Praseodymium] ^= 59
AtomicWeight[Praseodymium] ^= 140.90765
MeltingPoint[Praseodymium] ^= 1204. Kelvin
BoilingPoint[Praseodymium] ^= 3785. Kelvin
HeatOfFusion[Praseodymium] ^=  11.3 Kilo Joule / Mole
HeatOfVaporization[Praseodymium] ^= 357. Kilo Joule / Mole
Density[Praseodymium] ^= 6773. Kilogram/(Meter)^3
ThermalConductivity[Praseodymium] ^=  12.5 Watt/(Meter Kelvin)
StableIsotopes[Praseodymium] ^=  {141}
ElectronConfiguration[Praseodymium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,3},{2,6},{2}}
Abbreviation[Praseodymium] ^= "Pr"
SolarSystemAbundance[Praseodymium] ^= 5.44 10^-12
EarthCrustAbundance[Praseodymium] ^= 
  9.2 10^-6
EarthOceanAbundance[Praseodymium] ^= 
  6.4 10^-13
IonizationPotential[Praseodymium] ^= 5.464 ElectronVolt
SpecificHeat[Praseodymium] ^= (0.193 Joule)/(Gram Kelvin)
 
AtomicNumber[Neodymium] ^= 60
AtomicWeight[Neodymium] ^= 144.24
MeltingPoint[Neodymium] ^= 1294. Kelvin 
BoilingPoint[Neodymium] ^= 3341. Kelvin
HeatOfFusion[Neodymium] ^= 7.113 Kilo Joule / Mole
HeatOfVaporization[Neodymium] ^= 328. Kilo Joule / Mole
Density[Neodymium] ^= 7007. Kilogram/(Meter)^3 
ThermalConductivity[Neodymium] ^=  16.5 Watt/(Meter Kelvin)
StableIsotopes[Neodymium] ^=  {142,143,145,146,148,150}
ElectronConfiguration[Neodymium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,4},{2,6},{2}}
Abbreviation[Neodymium] ^= "Nd"
SolarSystemAbundance[Neodymium] ^= 2.7 10^-11
EarthCrustAbundance[Neodymium] ^= 
  0.0000415
EarthOceanAbundance[Neodymium] ^= 
  2.8 10^-12
IonizationPotential[Neodymium] ^= 5.525 ElectronVolt
SpecificHeat[Neodymium] ^= (0.19 Joule)/(Gram Kelvin)
 
AtomicNumber[Promethium] ^= 61
AtomicWeight[Promethium] ^:= AtomicWeightUnstable[Promethium, 145] 
MeltingPoint[Promethium] ^= 1441. Kelvin
BoilingPoint[Promethium] ^= 3000. Kelvin 	(* "circa" or "estimated" *)
HeatOfFusion[Promethium] ^= 12.6 Kilo Joule / Mole
HeatOfVaporization[Promethium] ^:= HeatOfVaporizationNA[Promethium]
Density[Promethium] ^=  7220. Kilogram/(Meter)^3
ThermalConductivity[Promethium] ^=  17.9 Watt/(Meter Kelvin)   (* "estimated" *)
StableIsotopes[Promethium] ^=  {}
ElectronConfiguration[Promethium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,5},{2,6},{2}}
Abbreviation[Promethium] ^= "Pm"
SolarSystemAbundance[Promethium] ^:= SolarSystemAbundanceNA[Promethium] 
EarthCrustAbundance[Promethium] ^:= EarthCrustAbundanceNA[Promethium] 
EarthOceanAbundance[Promethium] ^:= EarthOceanAbundanceNA[Promethium] 
IonizationPotential[Promethium] ^= 
  5.55 ElectronVolt
SpecificHeat[Promethium] ^:= SpecificHeatNA[Promethium]
 
AtomicNumber[Samarium] ^= 62
AtomicWeight[Samarium] ^= 150.36
MeltingPoint[Samarium] ^= 1350. Kelvin
BoilingPoint[Samarium] ^= 2064. Kelvin
HeatOfFusion[Samarium] ^= 10.9 Kilo Joule / Mole 
HeatOfVaporization[Samarium] ^= 164.8  Kilo Joule / Mole
Density[Samarium] ^= 7520. Kilogram/(Meter)^3
ThermalConductivity[Samarium] ^= 13.3 Watt/(Meter Kelvin)
StableIsotopes[Samarium] ^= {144,150,152,154}
ElectronConfiguration[Samarium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,6},{2,6},{2}}
Abbreviation[Samarium] ^= "Sm"
SolarSystemAbundance[Samarium] ^= 8.42 10^-12
EarthCrustAbundance[Samarium] ^= 
  7.05 10^-6
EarthOceanAbundance[Samarium] ^= 
  4.5 10^-13
IonizationPotential[Samarium] ^= 5.644 ElectronVolt
SpecificHeat[Samarium] ^= (0.197 Joule)/(Gram Kelvin)
 
AtomicNumber[Europium] ^= 63
AtomicWeight[Europium] ^= 151.964
MeltingPoint[Europium] ^= 1095. Kelvin 
BoilingPoint[Europium] ^= 1870. Kelvin
HeatOfFusion[Europium] ^= 10.5 Kilo Joule / Mole
HeatOfVaporization[Europium] ^= 176. Kilo Joule / Mole 
Density[Europium] ^=  5243. Kilogram/(Meter)^3
ThermalConductivity[Europium] ^=  13.9 Watt/(Meter Kelvin)
StableIsotopes[Europium] ^=  {151,153}
ElectronConfiguration[Europium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,7},{2,6},{2}}
Abbreviation[Europium] ^= "Eu"
SolarSystemAbundance[Europium] ^= 3.17 10^-12
EarthCrustAbundance[Europium] ^= 2. 10^-6
EarthOceanAbundance[Europium] ^= 
   1.3 10^-13
IonizationPotential[Europium] ^= 
  5.67 ElectronVolt
SpecificHeat[Europium] ^= 
  (0.182 Joule)/(Gram Kelvin)
 
AtomicNumber[Gadolinium] ^= 64
AtomicWeight[Gadolinium] ^= 157.25
MeltingPoint[Gadolinium] ^= 1586. Kelvin
BoilingPoint[Gadolinium] ^= 3539. Kelvin
HeatOfFusion[Gadolinium] ^= 15.5 Kilo Joule / Mole
HeatOfVaporization[Gadolinium] ^= 301. Kilo Joule / Mole 
Density[Gadolinium] ^= 7900.4 Kilogram/(Meter)^3
ThermalConductivity[Gadolinium] ^= 10.6 Watt/(Meter Kelvin)
StableIsotopes[Gadolinium] ^= {154,155,156,157,158,160}
ElectronConfiguration[Gadolinium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,7},{2,6,1},{2}}
Abbreviation[Gadolinium] ^= "Gd"
SolarSystemAbundance[Gadolinium] ^= 1.07 10^-11
EarthCrustAbundance[Gadolinium] ^= 
  6.2 10^-6
EarthOceanAbundance[Gadolinium] ^= 
  7.0 10^-13
IonizationPotential[Gadolinium] ^= 6.15 ElectronVolt
SpecificHeat[Gadolinium] ^= 
  (0.236 Joule)/(Gram Kelvin)
 
AtomicNumber[Terbium] ^= 65
AtomicWeight[Terbium] ^= 158.92534
MeltingPoint[Terbium] ^= 1629. Kelvin
BoilingPoint[Terbium] ^= 3396. Kelvin
HeatOfFusion[Terbium] ^= 16.3 Kilo Joule / Mole
HeatOfVaporization[Terbium] ^= 391. Kilo Joule / Mole 
Density[Terbium] ^=  8229. Kilogram/(Meter)^3
ThermalConductivity[Terbium] ^=  11.1 Watt/(Meter Kelvin)
StableIsotopes[Terbium] ^=  {159}
ElectronConfiguration[Terbium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,9},{2,6},{2}}
Abbreviation[Terbium] ^= "Tb"
SolarSystemAbundance[Terbium] ^= 1.97 10^-12
EarthCrustAbundance[Terbium] ^= 
  1.2 10^-6
EarthOceanAbundance[Terbium] ^= 
  1.4 10^-13
IonizationPotential[Terbium] ^= 
  5.864 ElectronVolt
SpecificHeat[Terbium] ^= 
  (0.182 Joule)/(Gram Kelvin)
 
AtomicNumber[Dysprosium] ^= 66
AtomicWeight[Dysprosium] ^= 162.50
MeltingPoint[Dysprosium] ^= 1685. Kelvin
BoilingPoint[Dysprosium] ^= 2835. Kelvin
HeatOfFusion[Dysprosium] ^= 17.2  Kilo Joule / Mole
HeatOfVaporization[Dysprosium] ^= 293. Kilo Joule / Mole 
Density[Dysprosium] ^=  8550. Kilogram/(Meter)^3
ThermalConductivity[Dysprosium] ^=  10.7 Watt/(Meter Kelvin)
StableIsotopes[Dysprosium] ^=  {156,158,160,161,162,163,164}
ElectronConfiguration[Dysprosium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,10},{2,6},{2}}
Abbreviation[Dysprosium] ^= "Dy"
SolarSystemAbundance[Dysprosium] ^= 1.286 10^-11
EarthCrustAbundance[Dysprosium] ^= 
  5.2 10^-6
EarthOceanAbundance[Dysprosium] ^= 
  9.1 10^-13
IonizationPotential[Dysprosium] ^= 5.939 ElectronVolt
SpecificHeat[Dysprosium] ^= 
  (0.173 Joule)/(Gram Kelvin)
 
AtomicNumber[Holmium] ^= 67
AtomicWeight[Holmium] ^= 164.93032
MeltingPoint[Holmium] ^= 1747. Kelvin
BoilingPoint[Holmium] ^= 2968. Kelvin
HeatOfFusion[Holmium] ^= 17.2 Kilo Joule / Mole
HeatOfVaporization[Holmium] ^= 303. Kilo Joule / Mole 
Density[Holmium] ^=  8795. Kilogram/(Meter)^3
ThermalConductivity[Holmium] ^=  16.2 Watt/(Meter Kelvin)
StableIsotopes[Holmium] ^=  {165}
ElectronConfiguration[Holmium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,11},{2,6},{2}}
Abbreviation[Holmium] ^= "Ho"
SolarSystemAbundance[Holmium] ^= 2.9 10^-12
EarthCrustAbundance[Holmium] ^= 1.3 10^-6
EarthOceanAbundance[Holmium] ^= 2.2 10^-13
IonizationPotential[Holmium] ^= 6.022 ElectronVolt
SpecificHeat[Holmium] ^= (0.165 Joule)/(Gram Kelvin)
 
AtomicNumber[Erbium] ^= 68
AtomicWeight[Erbium] ^= 167.26
MeltingPoint[Erbium] ^= 1802. Kelvin
BoilingPoint[Erbium] ^= 3136. Kelvin
HeatOfFusion[Erbium] ^=  17.2 Kilo Joule / Mole
HeatOfVaporization[Erbium] ^= 280. Kilo Joule / Mole 
Density[Erbium] ^=  9066. Kilogram/(Meter)^3
ThermalConductivity[Erbium] ^= 14.3  Watt/(Meter Kelvin) 
StableIsotopes[Erbium] ^=  {162,164,166,167,168,170}
ElectronConfiguration[Erbium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,12},{2,6},{2}}
Abbreviation[Erbium] ^= "Er"
SolarSystemAbundance[Erbium] ^= 8.18 10^-12
EarthCrustAbundance[Erbium] ^= 3.5 10^-6
EarthOceanAbundance[Erbium] ^= 
  8.7 10^-13
IonizationPotential[Erbium] ^= 6.108 ElectronVolt
SpecificHeat[Erbium] ^= (0.168 Joule)/(Gram Kelvin)
 
AtomicNumber[Thulium] ^= 69
AtomicWeight[Thulium] ^= 168.93421
MeltingPoint[Thulium] ^= 1818. Kelvin
BoilingPoint[Thulium] ^= 2220. Kelvin
HeatOfFusion[Thulium] ^= 18.4 Kilo Joule / Mole
HeatOfVaporization[Thulium] ^= 247. Kilo Joule / Mole 
Density[Thulium] ^= 9321. Kilogram/(Meter)^3 
ThermalConductivity[Thulium] ^=  16.8 Watt/(Meter Kelvin)
StableIsotopes[Thulium] ^= {169} 
ElectronConfiguration[Thulium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,13},{2,6},{2}}
Abbreviation[Thulium] ^= "Tm"
SolarSystemAbundance[Thulium] ^= 1.23 10^-12
EarthCrustAbundance[Thulium] ^= 5.2 10^-7
EarthOceanAbundance[Thulium] ^= 1.7 10^-13
IonizationPotential[Thulium] ^= 6.184 ElectronVolt
SpecificHeat[Thulium] ^= (0.16 Joule)/(Gram Kelvin)
 
AtomicNumber[Ytterbium] ^= 70
AtomicWeight[Ytterbium] ^= 173.04
MeltingPoint[Ytterbium] ^= 1097. Kelvin
BoilingPoint[Ytterbium] ^= 1466. Kelvin
HeatOfFusion[Ytterbium] ^= 9.20 Kilo Joule / Mole
HeatOfVaporization[Ytterbium] ^= 159. Kilo Joule / Mole 
Density[Ytterbium] ^= 6965. Kilogram/(Meter)^3  
ThermalConductivity[Ytterbium] ^= 34.9 Watt/(Meter Kelvin) 
StableIsotopes[Ytterbium] ^= {168,170,171,172,173,174,176} 
ElectronConfiguration[Ytterbium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6},{2}}
Abbreviation[Ytterbium] ^= "Yb"
SolarSystemAbundance[Ytterbium] ^= 8.08 10^-12
EarthCrustAbundance[Ytterbium] ^= 3.2 10^-6
EarthOceanAbundance[Ytterbium] ^= 
  8.2 10^-13
IonizationPotential[Ytterbium] ^= 
  6.254 ElectronVolt
SpecificHeat[Ytterbium] ^= (0.155 Joule)/(Gram Kelvin)
 
AtomicNumber[Lutetium] ^= 71
AtomicWeight[Lutetium] ^= 174.967
MeltingPoint[Lutetium] ^= 1936. Kelvin
BoilingPoint[Lutetium] ^= 3668. Kelvin
HeatOfFusion[Lutetium] ^= 19.2 Kilo Joule / Mole
HeatOfVaporization[Lutetium] ^= 428. Kilo Joule / Mole 
Density[Lutetium] ^= 9840. Kilogram/(Meter)^3 
ThermalConductivity[Lutetium] ^= 16.4 Watt/(Meter Kelvin)  
StableIsotopes[Lutetium] ^= {175} 
ElectronConfiguration[Lutetium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,1},{2}}
Abbreviation[Lutetium] ^= "Lu"
SolarSystemAbundance[Lutetium] ^= 1.197 10^-12
EarthCrustAbundance[Lutetium] ^= 8. 10^-7
EarthOceanAbundance[Lutetium] ^= 
  1.5 10^-13
IonizationPotential[Lutetium] ^= 5.426 ElectronVolt
SpecificHeat[Lutetium] ^= (0.154 Joule)/(Gram Kelvin)
 
AtomicNumber[Hafnium] ^= 72
AtomicWeight[Hafnium] ^= 178.49
MeltingPoint[Hafnium] ^= 2503. Kelvin
BoilingPoint[Hafnium] ^= 5470. Kelvin
HeatOfFusion[Hafnium] ^=  25.5 Kilo Joule / Mole
HeatOfVaporization[Hafnium] ^= 570.7 Kilo Joule / Mole
Density[Hafnium] ^= 13310. Kilogram/(Meter)^3 
ThermalConductivity[Hafnium] ^= 23.0  Watt/(Meter Kelvin)
StableIsotopes[Hafnium] ^= {176,177,178,179,180} 
ElectronConfiguration[Hafnium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,2},{2}}
Abbreviation[Hafnium] ^= "Hf"
SolarSystemAbundance[Hafnium] ^= 5.02 10^-12
EarthCrustAbundance[Hafnium] ^= 3. 10^-6
EarthOceanAbundance[Hafnium] ^= 7.0 10^-12
IonizationPotential[Hafnium] ^= 6.825 ElectronVolt
SpecificHeat[Hafnium] ^= 
  (0.144 Joule)/(Gram Kelvin)
 
AtomicNumber[Tantalum] ^= 73
AtomicWeight[Tantalum] ^= 180.9479
MeltingPoint[Tantalum] ^= 3269. Kelvin
BoilingPoint[Tantalum] ^= 5698. Kelvin
HeatOfFusion[Tantalum] ^=  31.4 Kilo Joule / Mole
HeatOfVaporization[Tantalum] ^= 758.22  Kilo Joule / Mole
Density[Tantalum] ^=  16654. Kilogram/(Meter)^3
ThermalConductivity[Tantalum] ^=  57.5 Watt/(Meter Kelvin)
StableIsotopes[Tantalum] ^=  {181}
ElectronConfiguration[Tantalum] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,3},{2}}
Abbreviation[Tantalum] ^= "Ta"
SolarSystemAbundance[Tantalum] ^= 6.75 10^-13
EarthCrustAbundance[Tantalum] ^= 2.0 10^-6
EarthOceanAbundance[Tantalum] ^= 2.0 10^-12
IonizationPotential[Tantalum] ^= 
  7.89 ElectronVolt
SpecificHeat[Tantalum] ^= (0.14 Joule)/(Gram Kelvin)

Wolfram = Tungsten
AtomicNumber[Tungsten] ^= 74
AtomicWeight[Tungsten] ^= 183.84
MeltingPoint[Tungsten] ^= 3680. Kelvin
BoilingPoint[Tungsten] ^= 5930. Kelvin
HeatOfFusion[Tungsten] ^= 35.2 Kilo Joule / Mole 
HeatOfVaporization[Tungsten] ^= 824.2 Kilo Joule / Mole 
Density[Tungsten] ^= 19300. Kilogram/(Meter)^3 
ThermalConductivity[Tungsten] ^= 174. Watt/(Meter Kelvin) 
StableIsotopes[Tungsten] ^= {180,182,183,184,186} 
ElectronConfiguration[Tungsten] ^=
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,4},{2}}
Abbreviation[Tungsten] ^= "W"
SolarSystemAbundance[Tungsten] ^= 4.34 10^-12
EarthCrustAbundance[Tungsten] ^= 1.25 10^-6
EarthOceanAbundance[Tungsten] ^= 1.0 10^-10
IonizationPotential[Tungsten] ^= 7.98 ElectronVolt
SpecificHeat[Tungsten] ^= (0.132 Joule)/(Gram Kelvin)
 
AtomicNumber[Rhenium] ^= 75
AtomicWeight[Rhenium] ^= 186.207
MeltingPoint[Rhenium] ^= 3453. Kelvin
BoilingPoint[Rhenium] ^= 5900. Kelvin
HeatOfFusion[Rhenium] ^= 33.1  Kilo Joule / Mole
HeatOfVaporization[Rhenium] ^= 704.25  Kilo Joule / Mole
Density[Rhenium] ^= 21020. Kilogram/(Meter)^3 
ThermalConductivity[Rhenium] ^= 47.9 Watt/(Meter Kelvin) 
StableIsotopes[Rhenium] ^= {185} 
ElectronConfiguration[Rhenium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,5},{2}}
Abbreviation[Rhenium] ^= "Re"
SolarSystemAbundance[Rhenium] ^= 1.69 10^-12
EarthCrustAbundance[Rhenium] ^= 
  7.0 10^-10
EarthOceanAbundance[Rhenium] ^= 4.0 10^-12
IonizationPotential[Rhenium] ^= 
  7.88 ElectronVolt
SpecificHeat[Rhenium] ^= (0.137 Joule)/(Gram Kelvin)
 
AtomicNumber[Osmium] ^= 76
AtomicWeight[Osmium] ^= 190.23
MeltingPoint[Osmium] ^= 3327. Kelvin
BoilingPoint[Osmium] ^= 5300. Kelvin
HeatOfFusion[Osmium] ^= 29.3 Kilo Joule / Mole
HeatOfVaporization[Osmium] ^= 738.06 Kilo Joule / Mole
Density[Osmium] ^=  22590. Kilogram/(Meter)^3
ThermalConductivity[Osmium] ^=  87.6 Watt/(Meter Kelvin)
StableIsotopes[Osmium] ^= {184,186,187,188,189,190,192} 
ElectronConfiguration[Osmium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,6},{2}}
Abbreviation[Osmium] ^= "Os"
SolarSystemAbundance[Osmium] ^= 2.2 10^-11
EarthCrustAbundance[Osmium] ^= 1.5 10^-9
EarthOceanAbundance[Osmium] ^:= EarthOceanAbundanceNA[Osmium] 
IonizationPotential[Osmium] ^= 8.7 ElectronVolt
SpecificHeat[Osmium] ^= (0.13 Joule)/(Gram Kelvin)
 
AtomicNumber[Iridium] ^= 77
AtomicWeight[Iridium] ^= 192.217
MeltingPoint[Iridium] ^= 2683. Kelvin
BoilingPoint[Iridium] ^= 4403. Kelvin
HeatOfFusion[Iridium] ^=  26.4 Kilo Joule / Mole
HeatOfVaporization[Iridium] ^= 612.1 Kilo Joule / Mole
Density[Iridium] ^= 22420. Kilogram/(Meter)^3 
ThermalConductivity[Iridium] ^=  147. Watt/(Meter Kelvin)
StableIsotopes[Iridium] ^= {191,193} 
ElectronConfiguration[Iridium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,7},{2}}
Abbreviation[Iridium] ^= "Ir"
SolarSystemAbundance[Iridium] ^= 2.16 10^-11
EarthCrustAbundance[Iridium] ^= 1. 10^-9
EarthOceanAbundance[Iridium] ^:= EarthOceanAbundanceNA[Iridium] 
IonizationPotential[Iridium] ^= 
  9.1 ElectronVolt
SpecificHeat[Iridium] ^= (0.131 Joule)/(Gram Kelvin)
 
AtomicNumber[Platinum] ^= 78
AtomicWeight[Platinum] ^= 195.078
MeltingPoint[Platinum] ^= 2045. Kelvin
BoilingPoint[Platinum] ^= 4100. Kelvin 			(*  +- 100. Kelvin *)
HeatOfFusion[Platinum] ^= 19.7 Kilo Joule / Mole
HeatOfVaporization[Platinum] ^= 469. Kilo Joule / Mole 
Density[Platinum] ^= 21450. Kilogram/(Meter)^3 
ThermalConductivity[Platinum] ^= 71.6 Watt/(Meter Kelvin) 
StableIsotopes[Platinum] ^= {194,195,196,198} 
ElectronConfiguration[Platinum] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,9},{1}}
Abbreviation[Platinum] ^= "Pt"
SolarSystemAbundance[Platinum] ^= 4.4 10^-11
EarthCrustAbundance[Platinum] ^= 
  5. 10^-9
EarthOceanAbundance[Platinum] ^:= EarthOceanAbundanceNA[Platinum] 
IonizationPotential[Platinum] ^= 9. ElectronVolt
SpecificHeat[Platinum] ^= (0.133 Joule)/(Gram Kelvin)
 
AtomicNumber[Gold] ^= 79
AtomicWeight[Gold] ^= 196.96655
MeltingPoint[Gold] ^= 1337.58 Kelvin
BoilingPoint[Gold] ^= 3080. Kelvin
HeatOfFusion[Gold] ^= 12.7  Kilo Joule / Mole
HeatOfVaporization[Gold] ^= 343.1 Kilo Joule / Mole
Density[Gold] ^=  19320. Kilogram/(Meter)^3
ThermalConductivity[Gold] ^= 317. Watt/(Meter Kelvin) 
StableIsotopes[Gold] ^=  {197}
ElectronConfiguration[Gold] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{1}}
Abbreviation[Gold] ^= "Au"
SolarSystemAbundance[Gold] ^= 6.1 10^-12
EarthCrustAbundance[Gold] ^= 4. 10^-9
EarthOceanAbundance[Gold] ^= 4.0 10^-12
IonizationPotential[Gold] ^= 9.226 ElectronVolt
SpecificHeat[Gold] ^= (0.129 Joule)/(Gram Kelvin)
 
AtomicNumber[Mercury] ^= 80
AtomicWeight[Mercury] ^= 200.59
MeltingPoint[Mercury] ^= 234.28 Kelvin
BoilingPoint[Mercury] ^= 629.73 Kelvin
HeatOfFusion[Mercury] ^= 2.331  Kilo Joule / Mole
HeatOfVaporization[Mercury] ^= 59.11 Kilo Joule / Mole
Density[Mercury] ^:= DensityException[Mercury, 13546., "liquid"]
ThermalConductivity[Mercury] ^:= ThermalConductivityException[Mercury,
				8.34, "liquid"]
StableIsotopes[Mercury] ^= {196,198,199,200,201,202, 204} 
ElectronConfiguration[Mercury] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2}}
Abbreviation[Mercury] ^= "Hg"
SolarSystemAbundance[Mercury] ^= 1.11 10^-11
EarthCrustAbundance[Mercury] ^= 8.5 10^-8
EarthOceanAbundance[Mercury] ^= 3.0 10^-11
IonizationPotential[Mercury] ^= 10.438 ElectronVolt
SpecificHeat[Mercury] ^= (0.14 Joule)/(Gram Kelvin)
 
AtomicNumber[Thallium] ^= 81
AtomicWeight[Thallium] ^= 204.3833
MeltingPoint[Thallium] ^= 576.6 Kelvin
BoilingPoint[Thallium] ^= 1730. Kelvin
HeatOfFusion[Thallium] ^=  4.31 Kilo Joule / Mole
HeatOfVaporization[Thallium] ^= 166.1 Kilo Joule / Mole
Density[Thallium] ^= 11850. Kilogram/(Meter)^3 
ThermalConductivity[Thallium] ^=  46.1 Watt/(Meter Kelvin)
StableIsotopes[Thallium] ^= {203,205} 
ElectronConfiguration[Thallium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,1}}
Abbreviation[Thallium] ^= "Tl"
SolarSystemAbundance[Thallium] ^= 6.0 10^-12
EarthCrustAbundance[Thallium] ^= 8.5 10^-7
EarthOceanAbundance[Thallium] ^= 
  1.9 10^-11
IonizationPotential[Thallium] ^= 
  6.108 ElectronVolt
SpecificHeat[Thallium] ^= (0.129 Joule)/(Gram Kelvin)
 
AtomicNumber[Lead] ^= 82
AtomicWeight[Lead] ^= 207.2
MeltingPoint[Lead] ^= 600.65 Kelvin
BoilingPoint[Lead] ^= 2013. Kelvin 
HeatOfFusion[Lead] ^= 5.121 Kilo Joule / Mole
HeatOfVaporization[Lead] ^= 177.8 Kilo Joule / Mole
Density[Lead] ^= 11350. Kilogram/(Meter)^3 
ThermalConductivity[Lead] ^=  35.3  Watt/(Meter Kelvin)
StableIsotopes[Lead] ^= {204,206,207,208} 
ElectronConfiguration[Lead] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,2}}
Abbreviation[Lead] ^= "Pb"
SolarSystemAbundance[Lead] ^= 1.03 10^-8
EarthCrustAbundance[Lead] ^= 0.000014
EarthOceanAbundance[Lead] ^= 3.0 10^-11
IonizationPotential[Lead] ^= 7.417 ElectronVolt
SpecificHeat[Lead] ^= (0.129 Joule)/(Gram Kelvin)
 
AtomicNumber[Bismuth] ^= 83
AtomicWeight[Bismuth] ^= 208.98038
MeltingPoint[Bismuth] ^= 544.5 Kelvin
BoilingPoint[Bismuth] ^= 1883. Kelvin
HeatOfFusion[Bismuth] ^=  10.48 Kilo Joule / Mole
HeatOfVaporization[Bismuth] ^= 179.1 Kilo Joule / Mole
Density[Bismuth] ^:= DensityException[Bismuth, 9747., 293]
ThermalConductivity[Bismuth] ^=  7.87 Watt/(Meter Kelvin)
StableIsotopes[Bismuth] ^= {209} 
ElectronConfiguration[Bismuth] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,3}}
Abbreviation[Bismuth] ^= "Bi"
SolarSystemAbundance[Bismuth] ^= 4.7 10^-10
EarthCrustAbundance[Bismuth] ^= 
  8.5 10^-9
EarthOceanAbundance[Bismuth] ^= 2.0 10^-11
IonizationPotential[Bismuth] ^= 
  7.289 ElectronVolt
SpecificHeat[Bismuth] ^= 
  (0.122 Joule)/(Gram Kelvin)
 
AtomicNumber[Polonium] ^= 84
AtomicWeight[Polonium] ^:= AtomicWeightUnstable[Polonium, 209]
MeltingPoint[Polonium] ^= 527. Kelvin
BoilingPoint[Polonium] ^= 1235. Kelvin
HeatOfFusion[Polonium] ^= 10. Kilo Joule / Mole 
HeatOfVaporization[Polonium] ^= 100.8 Kilo Joule / Mole
Density[Polonium] ^= 9320. Kilogram/(Meter)^3 
ThermalConductivity[Polonium] ^= 20. Watt/(Meter Kelvin) 
StableIsotopes[Polonium] ^=  {}
ElectronConfiguration[Polonium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,4}}
Abbreviation[Polonium] ^= "Po"
SolarSystemAbundance[Polonium] ^:= SolarSystemAbundanceNA[Polonium] 
EarthCrustAbundance[Polonium] ^= 
  2.0 10^-16 
EarthOceanAbundance[Polonium] ^= 
  1.5 10^-20
IonizationPotential[Polonium] ^= 
  8.417 ElectronVolt
SpecificHeat[Polonium] ^:= SpecificHeatNA[Polonium] 
 
AtomicNumber[Astatine] ^= 85
AtomicWeight[Astatine] ^:= AtomicWeightUnstable[Astatine, 210]
MeltingPoint[Astatine] ^= 575. Kelvin 			(* "estimated" *)
BoilingPoint[Astatine] ^= 610. Kelvin 			(* "estimated" *)
HeatOfFusion[Astatine] ^= 23.8 Kilo Joule / Mole
HeatOfVaporization[Astatine] ^:= HeatOfVaporizationNA[Astatine]
Density[Astatine] ^:= DensityNA[Astatine] 
ThermalConductivity[Astatine] ^= 1.7 Watt/(Meter Kelvin)
StableIsotopes[Astatine] ^= {}
ElectronConfiguration[Astatine] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,5}}
Abbreviation[Astatine] ^= "At"
SolarSystemAbundance[Astatine] ^:= SolarSystemAbundanceNA[Astatine] 
EarthCrustAbundance[Astatine] ^:= EarthCrustAbundanceNA[Astatine]
EarthOceanAbundance[Astatine] ^:= EarthOceanAbundanceNA[Astatine]
IonizationPotential[Astatine] ^:= IonizationPotentialNA[Astatine] 
SpecificHeat[Astatine] ^:= SpecificHeatNA[Astatine]
 
AtomicNumber[Radon] ^= 86
AtomicWeight[Radon] ^:= AtomicWeightUnstable[Radon, 222]
MeltingPoint[Radon] ^= 202. Kelvin
BoilingPoint[Radon] ^= 211.4 Kelvin
HeatOfFusion[Radon] ^= 2.7 Kilo Joule / Mole		(* "estimated" *) 
HeatOfVaporization[Radon] ^= 18.1 Kilo Joule / Mole
Density[Radon] ^:= DensityException[Radon,4400.,211,"liquid"]
ThermalConductivity[Radon] ^:= ThermalConductivityException[Radon,
			0.00364, "gaseous"]		(* "estimated" *)
StableIsotopes[Radon] ^=  {}
ElectronConfiguration[Radon] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,6}}
Abbreviation[Radon] ^= "Rn"
SolarSystemAbundance[Radon] ^:= SolarSystemAbundanceNA[Radon]
EarthCrustAbundance[Radon] ^= 
  4.0 10^-19
EarthOceanAbundance[Radon] ^= 6.0 10^-22
IonizationPotential[Radon] ^= 10.749 ElectronVolt
SpecificHeat[Radon] ^= 
  (0.094 Joule)/(Gram Kelvin)
 
AtomicNumber[Francium] ^= 87
AtomicWeight[Francium] ^:= AtomicWeightUnstable[Francium, 223]
MeltingPoint[Francium] ^= 300. Kelvin
BoilingPoint[Francium] ^= 950. Kelvin
HeatOfFusion[Francium] ^:=  HeatOfFusionNA[Francium]
HeatOfVaporization[Francium] ^:= HeatOfVaporizationNA[Francium]
Density[Francium] ^:=  DensityNA[Francium]
ThermalConductivity[Francium] ^=  15. Watt/(Meter Kelvin)     (* "estimated" *)
StableIsotopes[Francium] ^=  {}
ElectronConfiguration[Francium] ^=  
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,6},{1}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 1} *)
Abbreviation[Francium] ^= "Fr"
SolarSystemAbundance[Francium] ^:= SolarSystemAbundanceNA[Francium]
EarthCrustAbundance[Francium] ^:= EarthCrustAbundanceNA[Francium] 
EarthOceanAbundance[Francium] ^:= EarthOceanAbundanceNA[Francium]
IonizationPotential[Francium] ^:= IonizationPotentialNA[Francium]
SpecificHeat[Francium] ^:= SpecificHeatNA[Francium] 
 
AtomicNumber[Radium] ^= 88
AtomicWeight[Radium] ^:= AtomicWeightUnstable[Radium, 226]
	 				(* NOTE: this used to be 226.0254 *)
MeltingPoint[Radium] ^= 973. Kelvin
BoilingPoint[Radium] ^= 1413. Kelvin
HeatOfFusion[Radium] ^= 7.15  Kilo Joule / Mole
HeatOfVaporization[Radium] ^= 136.7  Kilo Joule / Mole
Density[Radium] ^= 5000. Kilogram/(Meter)^3  		     (* "circa" *)
ThermalConductivity[Radium] ^=  18.6 Watt/(Meter Kelvin)      (* "estimated" *)
StableIsotopes[Radium] ^=  {}
ElectronConfiguration[Radium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,6},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Radium] ^= "Ra"
SolarSystemAbundance[Radium] ^:= SolarSystemAbundanceNA[Radium]
EarthCrustAbundance[Radium] ^= 
  9.0 10^-13
EarthOceanAbundance[Radium] ^= 8.9 10^-17
IonizationPotential[Radium] ^= 5.279 ElectronVolt
SpecificHeat[Radium] ^:= SpecificHeatNA[Radium]
 
AtomicNumber[Actinium] ^= 89
AtomicWeight[Actinium] ^:= AtomicWeightUnstable[Actinium, 227]
					(* NOTE: this used to be 227.0278 *)
MeltingPoint[Actinium] ^= 1320. Kelvin 
BoilingPoint[Actinium] ^= 3470. Kelvin
HeatOfFusion[Actinium] ^= 14.2  Kilo Joule / Mole
HeatOfVaporization[Actinium] ^= 293  Kilo Joule / Mole
Density[Actinium] ^=  10060. Kilogram/(Meter)^3 
ThermalConductivity[Actinium] ^=  12.  Watt/(Meter Kelvin)   
StableIsotopes[Actinium] ^=  {}
ElectronConfiguration[Actinium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,6,1},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2} *)
Abbreviation[Actinium] ^= "Ac"
SolarSystemAbundance[Actinium] ^:= SolarSystemAbundanceNA[Actinium] 
EarthCrustAbundance[Actinium] ^= 
  5.5 10^-16
EarthOceanAbundance[Actinium] ^:= EarthOceanAbundanceNA[Actinium] 
IonizationPotential[Actinium] ^= 
  5.17 ElectronVolt
SpecificHeat[Actinium] ^= (0.12 Joule)/(Gram Kelvin) 
 
AtomicNumber[Thorium] ^= 90
AtomicWeight[Thorium] ^= 232.0381
MeltingPoint[Thorium] ^= 2023. Kelvin
BoilingPoint[Thorium] ^= 5060. Kelvin
HeatOfFusion[Thorium] ^= 19.2 Kilo Joule / Mole        (* actually "< 19.2" *)
HeatOfVaporization[Thorium] ^=  513.67 Kilo Joule / Mole
Density[Thorium] ^=  11720. Kilogram/(Meter)^3
ThermalConductivity[Thorium] ^=  54.0 Watt/(Meter Kelvin)
StableIsotopes[Thorium] ^=  {}
ElectronConfiguration[Thorium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10},{2,6,2},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 2, 2} *)
Abbreviation[Thorium] ^= "Th"
SolarSystemAbundance[Thorium] ^= 1.09 10^-10
EarthCrustAbundance[Thorium] ^= 
  9.6 10^-6
EarthOceanAbundance[Thorium] ^= 1.0 10^-12
IonizationPotential[Thorium] ^= 6.08 ElectronVolt
SpecificHeat[Thorium] ^= (0.113 Joule)/(Gram Kelvin)
 
AtomicNumber[Protactinium] ^= 91
AtomicWeight[Protactinium] ^= 231.03588
MeltingPoint[Protactinium] ^= 2113. Kelvin
BoilingPoint[Protactinium] ^= 4300. Kelvin	      		(* "circa" *)
HeatOfFusion[Protactinium] ^= 16.7 Kilo Joule / Mole
HeatOfVaporization[Protactinium] ^= 481. Kilo Joule / Mole
Density[Protactinium] ^= 15370. Kilogram/(Meter)^3 	      (* "estimated" *)
ThermalConductivity[Protactinium] ^= 47. Watt/(Meter Kelvin)  (* "estimated" *)
StableIsotopes[Protactinium] ^= {}
ElectronConfiguration[Protactinium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,2},{2,6,1},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2} *)
Abbreviation[Protactinium] ^= "Pa"
SolarSystemAbundance[Protactinium] ^:= SolarSystemAbundanceNA[Protactinium] 
EarthCrustAbundance[Protactinium] ^= 
  1.4 10^-12
EarthOceanAbundance[Protactinium] ^= 
  5.0 10^-17
IonizationPotential[Protactinium] ^= 
  5.89 ElectronVolt
SpecificHeat[Protactinium] ^:= SpecificHeatNA[Protactinium]
 
AtomicNumber[Uranium] ^= 92
AtomicWeight[Uranium] ^= 238.0289
MeltingPoint[Uranium] ^= 1405.5 Kelvin
BoilingPoint[Uranium] ^= 4018. Kelvin
HeatOfFusion[Uranium] ^= 15.5 Kilo Joule / Mole
HeatOfVaporization[Uranium] ^= 417.1 Kilo Joule / Mole 
Density[Uranium] ^=  18950. Kilogram/(Meter)^3 
ThermalConductivity[Uranium] ^= 27.6  Watt/(Meter Kelvin) 
StableIsotopes[Uranium] ^=  {}
ElectronConfiguration[Uranium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,3},{2,6,1},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2} *)
Abbreviation[Uranium] ^= "U"
SolarSystemAbundance[Uranium] ^= 2.94 10^-11
EarthCrustAbundance[Uranium] ^= 2.7 10^-6
EarthOceanAbundance[Uranium] ^= 3.2 10^-9
IonizationPotential[Uranium] ^= 
  6.194 ElectronVolt
SpecificHeat[Uranium] ^= (0.116 Joule)/(Gram Kelvin)
 
AtomicNumber[Neptunium] ^= 93
AtomicWeight[Neptunium] ^:= AtomicWeightUnstable[Neptunium, 237]
					(* NOTE: this used to be 237.0482 *)
MeltingPoint[Neptunium] ^= 913. Kelvin
BoilingPoint[Neptunium] ^= 4175. Kelvin
HeatOfFusion[Neptunium] ^= 9.46 Kilo Joule / Mole
HeatOfVaporization[Neptunium] ^= 336.6 Kilo Joule / Mole
Density[Neptunium] ^=  20250. Kilogram/(Meter)^3
ThermalConductivity[Neptunium] ^=  6.3 Watt/(Meter Kelvin)
StableIsotopes[Neptunium] ^= {} 
ElectronConfiguration[Neptunium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,4},{2,6,1},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2} *)
Abbreviation[Neptunium] ^= "Np"
SolarSystemAbundance[Neptunium] ^:= SolarSystemAbundanceNA[Neptunium]
EarthCrustAbundance[Neptunium] ^:= EarthCrustAbundanceNA[Neptunium] 
EarthOceanAbundance[Neptunium] ^:= EarthOceanAbundanceNA[Neptunium]
IonizationPotential[Neptunium] ^= 6.266 ElectronVolt
SpecificHeat[Neptunium] ^:= SpecificHeatNA[Neptunium]
 
AtomicNumber[Plutonium] ^= 94
AtomicWeight[Plutonium] ^:= AtomicWeightUnstable[Plutonium, 244]
MeltingPoint[Plutonium] ^= 914. Kelvin
BoilingPoint[Plutonium] ^= 3505. Kelvin
HeatOfFusion[Plutonium] ^= 2.8 Kilo Joule / Mole
HeatOfVaporization[Plutonium] ^= 343.5 Kilo Joule / Mole
Density[Plutonium] ^=  19840. Kilogram/(Meter)^3
ThermalConductivity[Plutonium] ^= 6.74 Watt/(Meter Kelvin) 
StableIsotopes[Plutonium] ^=  {}
ElectronConfiguration[Plutonium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,6},{2,6},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Plutonium] ^= "Pu"
SolarSystemAbundance[Plutonium] ^:= SolarSystemAbundanceNA[Plutonium]
EarthCrustAbundance[Plutonium] ^:= EarthCrustAbundanceNA[Plutonium]
EarthOceanAbundance[Plutonium] ^:= EarthOceanAbundanceNA[Plutonium]
IonizationPotential[Plutonium] ^= 
  6.06 ElectronVolt
SpecificHeat[Plutonium] ^:= SpecificHeatNA[Plutonium]
 
AtomicNumber[Americium] ^= 95
AtomicWeight[Americium] ^:= AtomicWeightUnstable[Americium, 243]
MeltingPoint[Americium] ^= 1267. Kelvin
BoilingPoint[Americium] ^= 2880. Kelvin
HeatOfFusion[Americium] ^= 14.4 Kilo Joule / Mole
HeatOfVaporization[Americium] ^= 238.5 Kilo Joule / Mole
Density[Americium] ^=  13670. Kilogram/(Meter)^3
ThermalConductivity[Americium] ^=  10. Watt/(Meter Kelvin)    (* "estimated" *)
StableIsotopes[Americium] ^=  {}
ElectronConfiguration[Americium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,7},{2,6},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Americium] ^= "Am"
SolarSystemAbundance[Americium] ^:= SolarSystemAbundanceNA[Americium] 
EarthCrustAbundance[Americium] ^:= EarthCrustAbundanceNA[Americium]
EarthOceanAbundance[Americium] ^:= EarthOceanAbundanceNA[Americium] 
IonizationPotential[Americium] ^= 5.993 ElectronVolt
SpecificHeat[Americium] ^:= SpecificHeatNA[Americium] 
 
AtomicNumber[Curium] ^= 96
AtomicWeight[Curium] ^:= AtomicWeightUnstable[Curium, 247]
MeltingPoint[Curium] ^= 1610. Kelvin			(* +- 40. Kelvin *)
BoilingPoint[Curium] ^:= BoilingPointNA[Curium]
HeatOfFusion[Curium] ^:= HeatOfFusionNA[Curium]
HeatOfVaporization[Curium] ^:= HeatOfVaporizationNA[Curium]
Density[Curium] ^:= 13300. Kilogram/(Meter)^3
ThermalConductivity[Curium] ^= 10. Watt/(Meter Kelvin)
StableIsotopes[Curium] ^= {}
ElectronConfiguration[Curium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,7},{2,6,1},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2} *)
Abbreviation[Curium] ^= "Cm"
SolarSystemAbundance[Curium] ^:= SolarSystemAbundanceNA[Curium]
EarthCrustAbundance[Curium] ^:= EarthCrustAbundanceNA[Curium]
EarthOceanAbundance[Curium] ^:= EarthOceanAbundanceNA[Curium]
IonizationPotential[Curium] ^= 6.02 ElectronVolt
SpecificHeat[Curium] ^:= SpecificHeatNA[Curium] 
 
AtomicNumber[Berkelium] ^= 97
AtomicWeight[Berkelium] ^:= AtomicWeightUnstable[Berkelium, 247]
MeltingPoint[Berkelium] ^:= MeltingPointNA[Berkelium]
BoilingPoint[Berkelium] ^:= BoilingPointNA[Berkelium]
HeatOfFusion[Berkelium] ^:= HeatOfFusionNA[Berkelium]
HeatOfVaporization[Berkelium] ^:= HeatOfVaporizationNA[Berkelium]
Density[Berkelium] ^:= 14790. Kilogram/(Meter)^3
ThermalConductivity[Berkelium] ^= 10. Watt/(Meter Kelvin)
StableIsotopes[Berkelium] ^= {}
ElectronConfiguration[Berkelium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,9},{2,6},{2}}
	(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Berkelium] ^= "Bk"
SolarSystemAbundance[Berkelium] ^:= SolarSystemAbundanceNA[Berkelium] 
EarthCrustAbundance[Berkelium] ^:= EarthCrustAbundanceNA[Berkelium]
EarthOceanAbundance[Berkelium] ^:= EarthOceanAbundanceNA[Berkelium]
IonizationPotential[Berkelium] ^= 6.23 ElectronVolt
SpecificHeat[Berkelium] ^:= SpecificHeatNA[Berkelium] 
 
AtomicNumber[Californium] ^= 98
AtomicWeight[Californium] ^:= 
		AtomicWeightUnstable[Californium, 251]
MeltingPoint[Californium] ^:= MeltingPointNA[Californium]
BoilingPoint[Californium] ^:= BoilingPointNA[Californium]
HeatOfFusion[Californium] ^:= HeatOfFusionNA[Californium]
HeatOfVaporization[Californium] ^:=
		HeatOfVaporizationNA[Californium]
Density[Californium] ^:= DensityNA[Californium]	    
ThermalConductivity[Californium] ^= 10. Watt/(Meter Kelvin)
StableIsotopes[Californium] ^= {}
ElectronConfiguration[Californium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,10},{2,6},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Californium] ^= "Cf"
SolarSystemAbundance[Californium] ^:= SolarSystemAbundanceNA[Californium]
EarthCrustAbundance[Californium] ^:= EarthCrustAbundanceNA[Californium] 
EarthOceanAbundance[Californium] ^:= EarthOceanAbundanceNA[Californium]
IonizationPotential[Californium] ^= 
  6.3 ElectronVolt
SpecificHeat[Californium] ^:= SpecificHeatNA[Californium]
 
AtomicNumber[Einsteinium] ^= 99
AtomicWeight[Einsteinium] ^:= AtomicWeightUnstable[Einsteinium, 252]
MeltingPoint[Einsteinium] ^:= MeltingPointNA[Einsteinium]
BoilingPoint[Einsteinium] ^:= BoilingPointNA[Einsteinium]
HeatOfFusion[Einsteinium] ^:= HeatOfFusionNA[Einsteinium]
HeatOfVaporization[Einsteinium] ^:= HeatOfVaporizationNA[Einsteinium]
Density[Einsteinium] ^:= DensityNA[Einsteinium] 
ThermalConductivity[Einsteinium] ^= 10. Watt/(Meter Kelvin)
StableIsotopes[Einsteinium] ^= {}
ElectronConfiguration[Einsteinium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,11},{2,6},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Einsteinium] ^= "Es"
SolarSystemAbundance[Einsteinium] ^:= SolarSystemAbundanceNA[Einsteinium]
EarthCrustAbundance[Einsteinium] ^:= EarthCrustAbundanceNA[Einsteinium] 
EarthOceanAbundance[Einsteinium] ^:= EarthOceanAbundanceNA[Einsteinium] 
IonizationPotential[Einsteinium] ^= 
  6.42 ElectronVolt
SpecificHeat[Einsteinium] ^:= SpecificHeatNA[Einsteinium] 
 
AtomicNumber[Fermium] ^= 100
AtomicWeight[Fermium] ^:= AtomicWeightUnstable[Fermium, 257]
MeltingPoint[Fermium] ^:= MeltingPointNA[Fermium]
BoilingPoint[Fermium] ^:= BoilingPointNA[Fermium]
HeatOfFusion[Fermium] ^:= HeatOfFusionNA[Fermium]
HeatOfVaporization[Fermium] ^:= HeatOfVaporizationNA[Fermium]
Density[Fermium] ^:=  DensityNA[Fermium] 
ThermalConductivity[Fermium] ^=  10. Watt/(Meter Kelvin)
StableIsotopes[Fermium] ^=  {}
ElectronConfiguration[Fermium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,12},{2,6},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2} *)
Abbreviation[Fermium] ^= "Fm"
SolarSystemAbundance[Fermium] ^:= SolarSystemAbundanceNA[Fermium]
EarthCrustAbundance[Fermium] ^:= EarthCrustAbundanceNA[Fermium]
EarthOceanAbundance[Fermium] ^:= EarthOceanAbundanceNA[Fermium]
IonizationPotential[Fermium] ^= 6.5 ElectronVolt
SpecificHeat[Fermium] ^:= SpecificHeatNA[Fermium] 
 
AtomicNumber[Mendelevium] ^= 101
AtomicWeight[Mendelevium] ^:= 
		AtomicWeightUnstable[Mendelevium, 258]
MeltingPoint[Mendelevium] ^:= MeltingPointNA[Mendelevium]
BoilingPoint[Mendelevium] ^:= BoilingPointNA[Mendelevium]
HeatOfFusion[Mendelevium] ^:= HeatOfFusionNA[Mendelevium]
HeatOfVaporization[Mendelevium] ^:=
		HeatOfVaporizationNA[Mendelevium]
Density[Mendelevium] ^:= DensityNA[Mendelevium] 
ThermalConductivity[Mendelevium] ^:= 10. Watt/(Meter Kelvin)
StableIsotopes[Mendelevium] ^= {} 
ElectronConfiguration[Mendelevium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,13},{2,6},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2},
	but it also indicates that the final orbitals are
	arranged as 31-8-2 *)
Abbreviation[Mendelevium] ^= "Md"
SolarSystemAbundance[Mendelevium] ^:= SolarSystemAbundanceNA[Mendelevium]
EarthCrustAbundance[Mendelevium] ^:= EarthCrustAbundanceNA[Mendelevium]
EarthOceanAbundance[Mendelevium] ^:= EarthOceanAbundanceNA[Mendelevium] 
IonizationPotential[Mendelevium] ^= 6.58 ElectronVolt
SpecificHeat[Mendelevium] ^:= SpecificHeatNA[Mendelevium]
 
AtomicNumber[Nobelium] ^= 102
AtomicWeight[Nobelium] ^:= AtomicWeightUnstable[Nobelium, 259]
MeltingPoint[Nobelium] ^:= MeltingPointNA[Nobelium]
BoilingPoint[Nobelium] ^:= BoilingPointNA[Nobelium]
HeatOfFusion[Nobelium] ^:= HeatOfFusionNA[Nobelium]
HeatOfVaporization[Nobelium] ^:= HeatOfVaporizationNA[Nobelium]
Density[Nobelium] ^:=   DensityNA[Nobelium] 
ThermalConductivity[Nobelium] ^:=  10. Watt/(Meter Kelvin)
StableIsotopes[Nobelium] ^=  {}
ElectronConfiguration[Nobelium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 0, 2},
	but it also indicates that the final orbitals are
	arranged as 32-8-2 *)
Abbreviation[Nobelium] ^= "No"
SolarSystemAbundance[Nobelium] ^:= SolarSystemAbundanceNA[Nobelium]
EarthCrustAbundance[Nobelium] ^:= EarthCrustAbundanceNA[Nobelium]
EarthOceanAbundance[Nobelium] ^:= EarthOceanAbundanceNA[Nobelium]
IonizationPotential[Nobelium] ^= 6.65 ElectronVolt
SpecificHeat[Nobelium] ^:= SpecificHeatNA[Nobelium] 
 
AtomicNumber[Lawrencium] ^= 103
AtomicWeight[Lawrencium] ^:= AtomicWeightUnstable[Lawrencium, 262]
MeltingPoint[Lawrencium] ^:= MeltingPointNA[Lawrencium]
BoilingPoint[Lawrencium] ^:= BoilingPointNA[Lawrencium]
HeatOfFusion[Lawrencium] ^:= HeatOfFusionNA[Lawrencium]
HeatOfVaporization[Lawrencium] ^:= HeatOfVaporizationNA[Lawrencium]
Density[Lawrencium] ^:=  DensityNA[Lawrencium] 
ThermalConductivity[Lawrencium] ^=   10. Watt/(Meter Kelvin)
StableIsotopes[Lawrencium] ^=  {}
ElectronConfiguration[Lawrencium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,1},{2}}
(* CRC 79th Edition indicates that the last orbital is {2, 6, 1, 2},
        but it also indicates that the final orbitals are
        arranged as 32-9-2 *)
Abbreviation[Lawrencium] ^= "Lr"
SolarSystemAbundance[Lawrencium] ^:= SolarSystemAbundanceNA[Lawrencium]
EarthCrustAbundance[Lawrencium] ^:= EarthCrustAbundanceNA[Lawrencium]
EarthOceanAbundance[Lawrencium] ^:= EarthOceanAbundanceNA[Lawrencium]
IonizationPotential[Lawrencium] ^:= IonizationPotentialNA[Lawrencium] 
SpecificHeat[Lawrencium] ^:= SpecificHeatNA[Lawrencium]

Kurchatovium = Rutherfordium
Unnilquadium = Rutherfordium
AtomicNumber[Rutherfordium] ^= 104
AtomicWeight[Rutherfordium] ^:= AtomicWeightUnstable[Rutherfordium, 261]
MeltingPoint[Rutherfordium] ^:=  MeltingPointNA[Rutherfordium]
BoilingPoint[Rutherfordium] ^:=  BoilingPointNA[Rutherfordium]
HeatOfFusion[Rutherfordium] ^:=  HeatOfFusionNA[Rutherfordium]
HeatOfVaporization[Rutherfordium] ^:= HeatOfVaporizationNA[Rutherfordium]
Density[Rutherfordium] ^:=  DensityNA[Rutherfordium] 
ThermalConductivity[Rutherfordium] ^:=  ThermalConductivityNA[Rutherfordium]
StableIsotopes[Rutherfordium] ^= {}
ElectronConfiguration[Rutherfordium] ^= 
	{{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,2},{2}}
				(*  CRC, 69th Ed., 
				"Electronic Configuation of the Elements" *)
(* CRC 79th Edition indicates that the last orbital is {2, 6, 2, 2},
        but it also indicates that the final orbitals are
        arranged as 32-10-2 *)
Abbreviation[Rutherfordium] ^= "Rf"
SolarSystemAbundance[Rutherfordium] ^:= SolarSystemAbundanceNA[Rutherfordium] 
EarthCrustAbundance[Rutherfordium] ^:= EarthCrustAbundanceNA[Rutherfordium] 
EarthOceanAbundance[Rutherfordium] ^:= EarthOceanAbundanceNA[Rutherfordium] 
IonizationPotential[Rutherfordium] ^:= IonizationPotentialNA[Rutherfordium]
SpecificHeat[Rutherfordium] ^:= SpecificHeatNA[Rutherfordium] 
 
Hahnium = Dubnium
Unnilpentium = Dubnium
AtomicNumber[Dubnium] ^= 105
AtomicWeight[Dubnium] ^:= AtomicWeightUnstable[Dubnium, 262] 
MeltingPoint[Dubnium] ^:= MeltingPointNA[Dubnium]
BoilingPoint[Dubnium] ^:= BoilingPointNA[Dubnium]
HeatOfFusion[Dubnium] ^:= HeatOfFusionNA[Dubnium]
HeatOfVaporization[Dubnium] ^:= HeatOfVaporizationNA[Dubnium]
Density[Dubnium] ^:=  DensityNA[Dubnium]
ThermalConductivity[Dubnium] ^:= ThermalConductivityNA[Dubnium]
StableIsotopes[Dubnium] ^= {}
ElectronConfiguration[Dubnium] ^=
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,3},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as 32-11-2 *)
Abbreviation[Dubnium] ^= "Db"
SolarSystemAbundance[Dubnium] ^:= SolarSystemAbundanceNA[Dubnium]
EarthCrustAbundance[Dubnium] ^:= EarthCrustAbundanceNA[Dubnium] 
EarthOceanAbundance[Dubnium] ^:= EarthOceanAbundanceNA[Dubnium]
IonizationPotential[Dubnium] ^:= IonizationPotentialNA[Dubnium]
SpecificHeat[Dubnium] ^:= SpecificHeatNA[Dubnium]
 
Unnilhexium = Seaborgium
AtomicNumber[Seaborgium] ^= 106
AtomicWeight[Seaborgium] ^:= AtomicWeightUnstable[Seaborgium, 263]
MeltingPoint[Seaborgium] ^:= MeltingPointNA[Seaborgium]
BoilingPoint[Seaborgium] ^:= BoilingPointNA[Seaborgium]
HeatOfFusion[Seaborgium] ^:= HeatOfFusionNA[Seaborgium]
HeatOfVaporization[Seaborgium] ^:= HeatOfVaporizationNA[Seaborgium]
Density[Seaborgium] ^:= DensityNA[Seaborgium]
ThermalConductivity[Seaborgium] ^:=  ThermalConductivityNA[Seaborgium]
StableIsotopes[Seaborgium] ^= {}
ElectronConfiguration[Seaborgium] ^= 
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,4},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as 32-12-2 *)
Abbreviation[Seaborgium] ^= "Sg" 
SolarSystemAbundance[Seaborgium] ^:= SolarSystemAbundanceNA[Seaborgium]
EarthCrustAbundance[Seaborgium] ^:= EarthCrustAbundanceNA[Seaborgium]
EarthOceanAbundance[Seaborgium] ^:= EarthOceanAbundanceNA[Seaborgium]
IonizationPotential[Seaborgium] ^:= IonizationPotentialNA[Seaborgium]
SpecificHeat[Seaborgium] ^:= SpecificHeatNA[Seaborgium]

AtomicNumber[Bohrium] ^= 107
AtomicWeight[Bohrium] ^:= AtomicWeightUnstable[Bohrium, 262]
MeltingPoint[Bohrium] ^:= MeltingPointNA[Bohrium]
BoilingPoint[Bohrium] ^:= BoilingPointNA[Bohrium]
HeatOfFusion[Bohrium] ^:= HeatOfFusionNA[Bohrium]
HeatOfVaporization[Bohrium] ^:= HeatOfVaporizationNA[Bohrium]
Density[Bohrium] ^:= DensityNA[Bohrium]
ThermalConductivity[Bohrium] ^:=  ThermalConductivityNA[Bohrium]
StableIsotopes[Bohrium] ^= {}
ElectronConfiguration[Bohrium] ^= 
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,5},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as 32-13-2 *)
Abbreviation[Bohrium] ^= "Bh" 
SolarSystemAbundance[Bohrium] ^:= SolarSystemAbundanceNA[Bohrium]
EarthCrustAbundance[Bohrium] ^:= EarthCrustAbundanceNA[Bohrium]
EarthOceanAbundance[Bohrium] ^:= EarthOceanAbundanceNA[Bohrium]
IonizationPotential[Bohrium] ^:= IonizationPotentialNA[Bohrium]
SpecificHeat[Bohrium] ^:= SpecificHeatNA[Bohrium]

AtomicNumber[Hassium] ^= 108
AtomicWeight[Hassium] ^:= AtomicWeightUnstable[Hassium, 265]
MeltingPoint[Hassium] ^:= MeltingPointNA[Hassium]
BoilingPoint[Hassium] ^:= BoilingPointNA[Hassium]
HeatOfFusion[Hassium] ^:= HeatOfFusionNA[Hassium]
HeatOfVaporization[Hassium] ^:= HeatOfVaporizationNA[Hassium]
Density[Hassium] ^:= DensityNA[Hassium]
ThermalConductivity[Hassium] ^:=  ThermalConductivityNA[Hassium]
StableIsotopes[Hassium] ^= {}
ElectronConfiguration[Hassium] ^= 
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,6},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as 32-14-2 *)
Abbreviation[Hassium] ^= "Hs" 
SolarSystemAbundance[Hassium] ^:= SolarSystemAbundanceNA[Hassium]
EarthCrustAbundance[Hassium] ^:= EarthCrustAbundanceNA[Hassium]
EarthOceanAbundance[Hassium] ^:= EarthOceanAbundanceNA[Hassium]
IonizationPotential[Hassium] ^:= IonizationPotentialNA[Hassium]
SpecificHeat[Hassium] ^:= SpecificHeatNA[Hassium]

AtomicNumber[Meitnerium] ^= 109
AtomicWeight[Meitnerium] ^:= AtomicWeightUnstable[Meitnerium, 266]
MeltingPoint[Meitnerium] ^:= MeltingPointNA[Meitnerium]
BoilingPoint[Meitnerium] ^:= BoilingPointNA[Meitnerium]
HeatOfFusion[Meitnerium] ^:= HeatOfFusionNA[Meitnerium]
HeatOfVaporization[Meitnerium] ^:= HeatOfVaporizationNA[Meitnerium]
Density[Meitnerium] ^:= DensityNA[Meitnerium]
ThermalConductivity[Meitnerium] ^:=  ThermalConductivityNA[Meitnerium]
StableIsotopes[Meitnerium] ^= {}
ElectronConfiguration[Meitnerium] ^=
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,7},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as 32-15-2 *)
Abbreviation[Meitnerium] ^= "Mt" 
SolarSystemAbundance[Meitnerium] ^:= SolarSystemAbundanceNA[Meitnerium]
EarthCrustAbundance[Meitnerium] ^:= EarthCrustAbundanceNA[Meitnerium]
EarthOceanAbundance[Meitnerium] ^:= EarthOceanAbundanceNA[Meitnerium]
IonizationPotential[Meitnerium] ^:= IonizationPotentialNA[Meitnerium]
SpecificHeat[Meitnerium] ^:= SpecificHeatNA[Meitnerium]

AtomicNumber[Ununnilium] ^= 110 
AtomicWeight[Ununnilium] ^:= AtomicWeightUnstable[Ununnilium, 269]
MeltingPoint[Ununnilium] ^:= MeltingPointNA[Ununnilium]
BoilingPoint[Ununnilium] ^:= BoilingPointNA[Ununnilium]
HeatOfFusion[Ununnilium] ^:= HeatOfFusionNA[Ununnilium]
HeatOfVaporization[Ununnilium] ^:= HeatOfVaporizationNA[Ununnilium]
Density[Ununnilium] ^:= DensityNA[Ununnilium]
ThermalConductivity[Ununnilium] ^:=  ThermalConductivityNA[Ununnilium]
StableIsotopes[Ununnilium] ^= {}
ElectronConfiguration[Ununnilium] ^= 
        {{2},{2,6},{2,6,10},{2,6,10,14},{2,6,10,14},{2,6,8},{2}}
(* CRC 79th Edition indicates that the final orbitals are
        arranged as  32-16-2 *)
Abbreviation[Ununnilium] ^= "Uun" 
SolarSystemAbundance[Ununnilium] ^:= SolarSystemAbundanceNA[Ununnilium]
EarthCrustAbundance[Ununnilium] ^:= EarthCrustAbundanceNA[Ununnilium]
EarthOceanAbundance[Ununnilium] ^:= EarthOceanAbundanceNA[Ununnilium]
IonizationPotential[Ununnilium] ^:= IonizationPotentialNA[Ununnilium]
SpecificHeat[Ununnilium] ^:= SpecificHeatNA[Ununnilium]

AtomicNumber[Unununium] ^= 111 
AtomicWeight[Unununium] ^:= AtomicWeightUnstable[Unununium, 272]
MeltingPoint[Unununium] ^:= MeltingPointNA[Unununium]
BoilingPoint[Unununium] ^:= BoilingPointNA[Unununium]
HeatOfFusion[Unununium] ^:= HeatOfFusionNA[Unununium]
HeatOfVaporization[Unununium] ^:= HeatOfVaporizationNA[Unununium]
Density[Unununium] ^:= DensityNA[Unununium]
ThermalConductivity[Unununium] ^:=  ThermalConductivityNA[Unununium]
StableIsotopes[Unununium] ^= {}
ElectronConfiguration[Unununium] ^:= ElectronConfigurationNA[Unununium]
Abbreviation[Unununium] ^= "Uuu" 
SolarSystemAbundance[Unununium] ^:= SolarSystemAbundanceNA[Unununium]
EarthCrustAbundance[Unununium] ^:= EarthCrustAbundanceNA[Unununium]
EarthOceanAbundance[Unununium] ^:= EarthOceanAbundanceNA[Unununium]
IonizationPotential[Unununium] ^:= IonizationPotentialNA[Unununium]
SpecificHeat[Unununium] ^:= SpecificHeatNA[Unununium]

AtomicNumber[Ununbium] ^= 112 
AtomicWeight[Ununbium] ^:= AtomicWeightNA[Ununbium]
MeltingPoint[Ununbium] ^:= MeltingPointNA[Ununbium]
BoilingPoint[Ununbium] ^:= BoilingPointNA[Ununbium]
HeatOfFusion[Ununbium] ^:= HeatOfFusionNA[Ununbium]
HeatOfVaporization[Ununbium] ^:= HeatOfVaporizationNA[Ununbium]
Density[Ununbium] ^:= DensityNA[Ununbium]
ThermalConductivity[Ununbium] ^:=  ThermalConductivityNA[Ununbium]
StableIsotopes[Ununbium] ^= {}
ElectronConfiguration[Ununbium] ^:= ElectronConfigurationNA[Ununbium]
Abbreviation[Ununbium] ^= "Uub" 
SolarSystemAbundance[Ununbium] ^:= SolarSystemAbundanceNA[Ununbium]
EarthCrustAbundance[Ununbium] ^:= EarthCrustAbundanceNA[Ununbium]
EarthOceanAbundance[Ununbium] ^:= EarthOceanAbundanceNA[Ununbium]
IonizationPotential[Ununbium] ^:= IonizationPotentialNA[Ununbium]
SpecificHeat[Ununbium] ^:= SpecificHeatNA[Ununbium]


Scan[
 (#[x_] := Module[
		   {i=Flatten[Position[ElementAbbreviations,ToString[x]]]},
                   #[ Elements[[ i[[1]] ]] ] /;
		       !SameQ[i,{}]
                 ])&,
 {AtomicNumber,AtomicWeight,MeltingPoint,BoilingPoint,HeatOfFusion,
  HeatOfVaporization,Density,ThermalConductivity,StableIsotopes,
  ElectronConfiguration,ElectronConfigurationFormat,Abbreviation, 
  EarthOceanAbundance, EarthCrustAbundance, SolarSystemAbundance, 
  IonizationPotential, SpecificHeat}]
Scan[
 (#[x_] := Module[
		   {i=Flatten[Position[ElementAbbreviations,ToString[x]]]},
                   #[ Elements[[ i[[1]] ]] ] /;
		       !SameQ[i,{}]
                 ])&,
 {AtomicNumber,AtomicWeight,MeltingPoint,BoilingPoint,HeatOfFusion,
  HeatOfVaporization,Density,ThermalConductivity,StableIsotopes,
  ElectronConfiguration,ElectronConfigurationFormat,Abbreviation, 
    EarthOceanAbundance, EarthCrustAbundance, SolarSystemAbundance, 
    IonizationPotential, SpecificHeat}]
	 

End[]

EndPackage[]



