(* :Title: World Data *)

(* :Author: John M. Novak (based on public domain data; see Source.) *)

(* :Summary:
This package defines polygons corresponding to maps of countries
listed in the Miscellaneous`WorldNames` package, for use with the
Miscellaneous`WorldPlot` package.
Each country is made up of a list of lists of coordinates (of the
form {lat, long} in minutes) that form the borders of that country.
*)

(* :Context: Miscellaneous`WorldData` *)

(* :Package Version: 1.1 *)

(* :Copyright: The structure of this data file is Copyright 1990-2005,
	Wolfram Research, Inc.  The data is from public domain sources. *)

(* :History: V 1.0 by John M. Novak, November 1990 
             V 1.1 by Jeff Adams, January 1995.
                   - updated with new country names and data.
*)

(* :Keywords: nations, cartography *)

(* :Source:
	These data are derived from what was originally a CIA database,
	released into the public domain by the Freedom of Information
	Act.  I acquired a copy of these data, and rearranged them
	to the form below.  The copy I found was from the Amiga
	Fish disks, a public-domain distribution of software for the
	Amiga computer, under the name WorldDataBank.  Note that these
	data (in the original format) are available at higher resolution
	from this source.
*)

(* :Mathematica Version: 1.2 or 2.0 *)


BeginPackage["Miscellaneous`WorldData`","Miscellaneous`WorldNames`"]

WorldData::usage = 
"WorldData[country] returns the data for the polygon representing the country in a
map.  This function is to be used with the WorldPlot package."

WorldData::indep =
"Warning: `1` is now divided into separate countries. Using the data of the separate countries."

WorldData::united =
"Warning: `1` is now united with `2`. Use `3` for the proper united country name."


WorldData/: WorldData["Afghanistan"] = 
   {{{2136, 3677}, {2160, 3844}, {2254, 3947}, {2233, 4066}, 
     {2308, 4257}, {2201, 4301}, {2234, 4495}, {2222, 4474}, 
     {2188, 4298}, {2041, 4194}, {1916, 4160}, {1873, 4003}, 
     {1791, 3975}, {1792, 3652}, {1861, 3711}, {1984, 3635}, 
     {2136, 3677}}}
 
WorldData/: WorldData["Albania"] = 
   {{{2381, 1201}, {2511, 1162}, {2551, 1210}, {2451, 1259}, {2381, 1201}}}
 
WorldData/: WorldData["Algeria"] = 
   {{{2216, 517}, {2150, 0}, {2105, -133}, {1926, -71}, 
     {1902, -229}, {1858, -218}, {1723, -520},  {1660, -520}, 
     {1637, -520}, {1500, -288}, {1310, 0}, 
     {1149, 255},  {1411, 720}, {1569, 564}, 
     {1814, 572}, {2033, 450}, {2216, 517}}}
 
WorldData/: WorldData["Andorra"] = 
   {{{2550, 105}, {2556, 90}, {2556, 89}, {2550, 105}}}
 
WorldData/: WorldData["Angola"] = 
   {{{-352, 791}, {-366, 735}, {-708, 828}, {-1035, 706}, 
     {-1040, 831}, {-1058, 1409}, {-970, 1320}, {-780, 1320}, 
     {-780, 1441}, {-652, 1439}, {-652, 1330}, {-437, 1307}, 
     {-486, 1080}, {-435, 1019}, {-353, 991}, {-351, 791}}}
 
WorldData/: WorldData["Antarctica"] = 
   {{{-5101, -9929}, {-5098, -8322}, {-5056, -9119}, {-4868, -9403}, 
     {-4842, -8635}, {-4741, -9366}, {-4626, -9461}, {-4666, -8970}, 
     {-4608, -8732}, {-4567, -8900}, {-4469, -8129}, {-4430, -6842}, 
     {-4491, -6758}, {-4522, -6037}, {-4418, -6191}, {-4417, -5951}, 
     {-4365, -6195}, {-4356, -5364}, {-4381, -4812}, {-4423, -4436}, 
     {-4370, -4055}, {-4280, -4051},  {-4023, -4049}, 
     {-3842, -3659}, {-3813, -3403}, {-3974, -3654}, {-4044, -3931}, 
     {-4117, -3913}, {-4234, -3682}, {-4462, -3644}, {-4549, -4672}, 
     {-4611, -5094}, {-4705, -4637}, {-4749, -4901}, {-4763, -4569}, 
     {-4798, -4806}, {-4855, -4421}, {-4982, -3495}, 
     {-4827, -1826}, {-4671, -2113}, {-4437, -823}, {-4427, -1014}, 
     {-4277, -711}, {-4242, -355}, {-4285, -361}, {-4290, 0}, {-4204, 521}, 
     {-4232, 1404}, {-4110, 2038}, {-4200, 2309}, {-4111, 2466}, 
     {-4029, 3050}, {-3959, 3118}, {-4023, 3407}, {-4067, 4021}, 
      {-4066, 4179}, {-4213, 4080}, {-4277, 4284}, 
     {-4144, 4672}, {-4088, 4737}, {-3962, 5294}, {-4010, 5376}, 
     {-3939, 6002}, {-4015, 6532}, {-3943, 6804}, {-4019, 7066}, 
     {-4022, 7762}, {-3961, 8124}, {-4021, 8722}, {-4104, 8882}, 
     {-4158, 9541}, {-4300, 10248},  {-4490, 9735}, 
     {-4717, 10037}, {-4736, 9600}, {-4794, 9552}, {-5101, -9930}}, 
    {{-4352, -5761}, {-4329, -6138}, {-4296, -5930}, {-4352, -5761}}, 
    {{-4358, -4325}, {-4296, -4305}, {-4287, -4540}, {-4257, -4236}, 
     {-4144, -4324}, {-4298, -4085}, {-4358, -4325}}, 
    {{-4644, -4372}, {-4684, -4482}, {-4637, -4851}, {-4644, -4372}}, 
    {{-4795, -3578}, {-4862, -3612}, {-4820, -3967}, {-4795, -3578}}}
 
WorldData/: WorldData["Argentina"] = 
   {{{-1947, -3492}, {-2067, -3511}, {-2214, -3400}, {-2309, -3499}, 
     {-2328, -3743}, {-2449, -3738}, {-2451, -3908}, {-2557, -3815}, 
     {-2560, -3898}, {-2701, -3936}, {-2760, -4055}, {-2843, -3944}, 
     {-3001, -4141}, {-3143, -4106}, {-3119, -4316}, 
     {-2972, -4415}, {-2672, -4266}, {-2537, -4328}, {-2054, -4189}, 
     {-1879, -4234}, {-1623, -4099}, {-1369, -4031}, 
     {-1307, -3974}, {-1372, -3859}, {-1320, -3836}, {-1334, -3758}, 
     {-1511, -3465}, {-1639, -3516}, {-1637, -3335}, 
     {-1534, -3276}, {-1628, -3228}, {-1815, -3458}, {-1947, -3492}}, 
    {{-3159, -4117}, {-3279, -3908}, {-3287, -4118}, {-3293, -4118}, 
     {-3159, -4117}}}

WorldData/: WorldData["Armenia"] =
    {{{2435, 2590}, {2472, 2701}, {2419, 2781}, {2361, 2849}, {2344, 2806}, 
     {2374, 2770}, {2378, 2689}, {2435, 2590}}}

WorldData/: WorldData["Australia"] = 
   {{{-2300, 8678}, {-2332, 8613}, {-2277, 8422}, {-2155, 8369}, 
     {-2049, 8285}, {-2118, 8210}, {-1951, 8266}, {-2099, 8157}, 
     {-1933, 8027}, {-1889, 7872}, {-1939, 7559}, {-2036, 7201}, 
     {-2109, 7077}, {-2063, 6907}, {-1914, 6944}, {-1569, 6789}, 
     {-1568, 6854}, {-1450, 6804},  {-1309, 6842}, 
     {-1230, 7009}, {-1172, 7262}, {-982, 7376}, {-1073, 7426}, {-968, 7417}, 
     {-935, 7515}, {-834, 7561}, {-837, 7645}, {-936, 7670}, {-886, 7709}, 
     {-923, 7812}, {-833, 7782}, {-730, 7860}, {-731, 7961}, {-672, 7919}, 
     {-738, 8143},  {-744, 8219}, {-899, 8127}, {-1063, 8406}, 
     {-914, 8503}, {-662, 8528}, {-732, 8586}, {-863, 8627}, {-898, 8722}, 
     {-1133, 8777}, {-1215, 8927}, {-1353, 8978}, {-1352, 9048}, 
     {-1518, 9175}, {-1727, 9217}, {-2144, 9013}, 
     {-2263, 8977}, {-2349, 8782}, {-2300, 8678}}, 
    {{-1546, 9182}, {-1482, 9196}, {-1546, 9182}}, 
    {{-2590, 8746}, {-2441, 8682}, {-2458, 8901}, {-2591, 8879}, 
     {-2589, 8746}}, {{-716, 7856}, {-672, 7823}, {-678, 7886}, {-716, 7856}}}
 
WorldData/: WorldData["Austria"] = 
   {{{2852, 584}, {2850, 783}, {2926, 830},  
     {2881, 1030}, {2812, 967},  {2792, 823},  
     {2812, 628}, {2824, 576}, {2836, 572}, {2852, 584}}}

WorldData/: WorldData["Azerbaijan"] =
    {{{2378, 2689}, {2374, 2770}, {2344, 2806}, {2330, 2770}, {2378, 2689}}, 
     {{2361, 2849}, {2419, 2781}, {2472, 2701}, {2445, 2823}, {2509, 2778}, 
     {2473, 2871}, {2510, 2915}, {2414, 3024}, {2307, 2933}, {2376, 2888}, {2361, 2849}}}

WorldData/: WorldData["Bahrain"] = {{{1574, 3030}}}
 
WorldData/: WorldData["Bangladesh"] = 
   {{{1269, 5532}, {1375, 5489}, {1437, 5355}, {1342, 5437}, {1315, 5342}, 
     {1517, 5341}, {1598, 5305}, {1517, 5391}, {1502, 5544}, 
     {1418, 5470}, {1377, 5497}, {1418, 5537}, {1319, 5556}, 
     {1269, 5532}}}

WorldData/: WorldData["Belarus"] =
    {{{3237, 1410}, {3249, 1519}, {3332, 1580}, {3361, 1686}, {3350, 1799}, 
     {3222, 1903}, {3124, 1898}, {3095, 1640}, {3092, 1416}, {3237, 1410}}}

WorldData/: WorldData["Belgium"] = 
   {{{3065, 153}, {3083, 203}, {3046, 361}, 
     {3008, 368},  {2972, 349}, {3065, 153}}}
 
WorldData/: WorldData["Belize"] = 
   {{{1109, -5297}, {954, -5335}, {1069, -5348}, {1109, -5297}}}
 
WorldData/: WorldData["Benin"] = 
   {{{373, 98}, {382, 164}, {705, 214}, {713, 144}, {660, 58}, {373, 98}}}
 
WorldData/: WorldData["Bermuda"] = {{{1939, -3885}}}
 
WorldData/: WorldData["Bhutan"] = 
   {{{1639, 5335}, {1699, 5400}, {1666, 5499}, {1639, 5335}}}
 
WorldData/: WorldData["Bolivia"] = 
   {{{-657, -4174}, {-750, -4120}, {-1050, -4170},  
     {-1369, -4031},  {-1307, -3974}, {-1372, -3859}, 
     {-1320, -3836}, {-1334, -3758}, {-1179, -3705}, 
     {-1210, -3490}, {-1092, -3447}, {-977, -3500}, 
     {-976, -3610}, {-828, -3628}, {-748, -3864}, {-655, -3919}, 
     {-581, -3926}, {-657, -4174}}}

WorldData/: WorldData["Bosnia and Herzegovina"] =
    {{{2550, 1118}, {2651, 982}, {2674, 966}, {2710, 957}, 
     {2703, 1108}, {2693, 1128}, {2550, 1118}}}

WorldData/: WorldData["Botswana"] = 
   {{{-1068, 1516}, {-1099, 1260}, {-1320, 1200},  
     {-1486, 1200}, {-1610, 1239}, {-1539, 1368}, 
     {-1540, 1531}, {-1332, 1763},  {-1068, 1516}}}
 
WorldData/: WorldData["Brazil"] = 
   {{{-2024, -3202}, {-1801, -3077}, {-1827, -3034}, {-1930, -3124}, 
     {-1712, -2925}, {-1521, -2925}, {-1319, -2458}, {-1061, -2348}, 
     {-823, -2339}, {-764, -2335},  {-764, -2289}, 
     {-501, -2096}, {-313, -2125}, {-171, -2400}, {-198, -2687}, 
     {-105, -2676}, {-43, -2809}, {-107, -2936}, 
     {-161, -2981}, {-107, -2965}, {-107, -3043}, {-55, -3051}, {-96, -3163}, 
     {70, -2993}, {242, -3101},  {140, -3164}, {140, -3277}, 
     {117, -3388},  {83, -3555}, {263, -3580}, 
     {312, -3644},  {256, -3691}, {214, -3773}, {257, -3888}, 
     {129, -3804}, {73, -4012}, {103, -4191}, {39, -4147}, 
     {35, -4203}, {-253, -4197},  {-308, -4374}, {-452, -4441}, 
     {-564, -4392}, {-566, -4231}, {-661, -4238}, {-657, -4174}, 
     {-581, -3926}, {-655, -3919}, {-748, -3864}, 
     {-828, -3628}, {-976, -3610}, {-977, -3500}, {-1092, -3447}, 
     {-1210, -3490},  {-1325, -3479}, {-1337, -3351}, 
     {-1438, -3324}, {-1443, -3255},  {-1534, -3276}, 
     {-1628, -3228}, {-1815, -3458}, {-1923, -3225}, 
     {-2024, -3202}}, {{-16, -2904}, {-110, -3023}, {-12, -3035}, 
     {-16, -2904}}}
 
WorldData/: WorldData["Brunei"] = 
   {{{275, 6845}, {292, 6901}, {293, 6901}, {275, 6845}}, 
    {{294, 6909}, {289, 6902}, {294, 6909}}}
 
WorldData/: WorldData["Bulgaria"] = 
   {{{2625, 1715}, {2519, 1681},  {2503, 1581}, 
     {2481, 1376}, {2654, 1361},  {2625, 1715}}}
 
WorldData/: WorldData["Burkina" | "Burkina Faso"] = 
   {{{569, -161},  {626, -331}, {810, -238}, {900, 0}, {900, 14}, 
     {714, 144},  {660, 58},  {668, -10}, {660, -170}, {569, -161}}}
 
WorldData/: WorldData["Burma" | "Myanmar"] = 
   {{{621, 5925}, {994, 5864}, {1047, 5813}, {957, 5740}, {958, 5655}, 
     {1163, 5645}, {1268, 5532},  {1319, 5556}, 
     {1597, 5708}, {1625, 5828}, {1693, 5841},  {1688, 5889}, 
      {1552, 5923}, {1436, 5852}, {1446, 5934}, {1294, 6069}, 
     {1221, 6005}, {1294, 6069}, {1221, 6005}, {1113, 5841}, {983, 5936}, 
     {906, 5891}, {710, 5980}, {621, 5925}}}
 
WorldData/: WorldData["Burundi"] = 
   {{{-165, 1741}, {-267, 1765}, {-144, 1834}, {-165, 1741}}}
 
WorldData/: WorldData["CAR"] = 
   {{{452, 930}, {482, 1115}, {657, 1366}, {655, 1373}, {523, 1411}, 
     {301, 1647}, {248, 1340}, {298, 1152},  
     {209, 1118}, {209, 995}, {133, 972}, {363, 865}, {452, 930}}}
 
WorldData/: WorldData["Cabinda"] = 
   {{{-301, 722}, {-347, 732},  {-278, 785}, {-301, 722}}}
 
WorldData/: WorldData["Cambodia"] = 
   {{{625, 6267}, {698, 6175}, {813, 6140}, {861, 6313}, 
     {882, 6453},  {741, 6453}, {700, 6351}, {625, 6267}}}
 
WorldData/: WorldData["Cameroon"] = 
   {{{141, 589}, {248, 585}, {289, 516}, {408, 588}, {425, 713}, 
     {694, 879}, {785, 844}, {599, 941}, {578, 837}, {452, 930}, 
     {363, 865}, {133, 972}, {130, 798}, {130, 680}, {141, 589}}}
 
WorldData/: WorldData["Canada"] = 
   {{{2940, -7364}, {2940, -7022}, {2940, -6962}, 
     {2940, -6240}, {2940, -5832}, {2880, -5375}, {2792, -5048}, 
     {2581, -4945}, {2553, -4959},  
     {2501, -4962}, {2575, -4735}, {2586, -4744}, {2596, -4744}, 
     {2669, -4548}, {2701, -4401}, {2701, -4290},  {2718, -4265}, 
     {2848, -4154}, {2711, -4032}, 
     {2716, -3943}, {2765, -3885}, {2721, -3896}, 
     {2722, -3802}, {2665, -3971}, {2608, -3928}, {2716, -3658}, 
     {2753, -3715}, {2750, -3825}, {2864, -3881}, {2880, -4011}, 
     {2910, -3853}, {2956, -3927}, {2805, -4278}, {2887, -4184}, 
     {2906, -4264}, {2888, -4181}, {3019, -3868}, 
     {3012, -3608}, {3133, -3337}, {3191, -3346}, {3254, -3482}, 
     {3196, -3625}, {3254, -3502}, {3275, -3441}, {3300, -3641}, 
     {3403, -3732}, {3622, -3890}, {3499, -3964}, 
     {3515, -4080}, {3466, -4162}, {3513, -4092}, {3604, -4262}, 
     {3654, -4162}, {3704, -4325}, {3740, -4686}, 
     {3517, -4714}, {3434, -4595}, {3339, -4628}, {3279, -4786}, 
     {3043, -4760}, {3131, -4913}, {3309, -4938}, {3299, -5125}, 
     {3410, -5330}, {3414, -5573},  {3509, -5658}, 
     {3630, -5688}, {3784, -5438}, {3851, -5627}, {3851, -5279}, 
     {3914, -5222}, {3957, -5490}, {3919, -5244}, {3971, -5022}, 
     {4015, -5113}, {4020, -4890}, {4151, -4880}, {4192, -5134}, 
     {4035, -5223}, {4154, -5337}, {4094, -5415}, 
     {4180, -5575}, {4320, -5674}, {4208, -5792}, {4118, -5602}, 
     {4058, -5728}, {3995, -5749}, {4064, -5771}, {4113, -5871}, 
     {4057, -5832}, {4061, -6134}, {4136, -6373}, {4091, -6507}, 
     {4118, -6339}, {3981, -6435}, {4080, -6605}, 
     {4068, -6906}, {4132, -6899}, {4161, -7460}, {4209, -7473}, 
     {4159, -7526}, {4235, -7680}, {4161, -7848}, {4133, -8008}, 
     {4216, -7780}, {4168, -7983}, {4119, -8110}, {4179, -8461}, 
     {3618, -8460}, {3588, -8128}, {3355, -7801}, 
     {3233, -7803}, {3213, -7738}, {3142, -7700}, 
     {3142, -7604}, {3100, -7673}, {3070, -7667}, {3055, -7489}, 
     {2939, -7365}}, {{3247, -7900}, {3195, -7962}, {3250, -7984}, 
     {3247, -7900}}, {{3042, -7648}, {2898, -7413}, {3042, -7649}}, 
    {{4442, -7184}, {4409, -6919}, {4265, -7368}, {4318, -7560}, 
     {4460, -7487}, {4442, -7185}}, 
    {{4356, -6876}, {4351, -6516}, {4296, -6471}, {4401, -6485}, 
     {4343, -6312}, {4209, -6058}, {4128, -6174}, {4170, -6395}, 
     {4108, -6799}, {4200, -7046}, {4220, -6689}, {4236, -7054}, 
     {4292, -6904}, {4301, -7146}, {4375, -7041}, {4356, -6876}}, 
    {{3895, -4981}, {3807, -4858}, {3787, -5114}, {3815, -5231}, 
     {3955, -5137}, {3895, -4981}}, 
    {{4412, -6269}, {4371, -6317}, {4409, -6422}, {4412, -6269}}, 
    {{4193, -5872}, {4120, -5732}, {4133, -5966}, {4193, -5872}}, 
    {{3097, -3337}, {2973, -3411}, {2942, -3323}, {2957, -3209}, 
     {2851, -3220}, {2886, -3170}, {2799, -3185}, {2874, -3255}, 
     {2812, -3323}, {2877, -3346}, {2857, -3558}, {2875, -3564}, 
     {2990, -3470}, {3096, -3337}}, 
    {{2811, -3619}, {2742, -3669}, {2756, -3588}, {2756, -3691}, 
     {2811, -3619}}, {{4097, -4034}, {3998, -3676}, {3893, -3813}, 
     {3981, -3861}, {3978, -4112}, {3932, -4043}, 
     {3774, -3878}, {3825, -4140}, {3712, -3964}, {3783, -4284}, 
     {3867, -4512},  {3865, -4680}, {3928, -4646}, 
     {3883, -4523}, {3927, -4409}, {3957, -4275}, {3994, -4153}, 
     {4005, -4379}, {4122, -4599}, {4241, -4741}, {4183, -4857}, 
     {4214, -5275}, {4265, -5374},  {4313, -5401}, 
     {4432, -5169}, {4259, -5208}, {4374, -5141}, {4424, -4918}, 
     {4313, -4858}, {4365, -4624}, {4271, -4505}, {4259, -4403}, 
     {4294, -4315}, {4238, -4357}, {4201, -4292}, {4253, -4192}, 
     {4200, -4140}, {4172, -4202}, {4160, -4006}, 
     {4132, -4164}, {4097, -4034}}, 
    {{4441, -5705}, {4434, -5412}, {4319, -5644}, {4441, -5705}}, 
    {{4642, -7119}, {4601, -6954}, {4548, -7220}, {4565, -7382}, 
     {4642, -7119}}, {{4761, -6222}, {4670, -5951}, {4701, -6289}, 
     {4761, -6222}}, {{4598, -6023}, {4507, -5857}, {4533, -6167}, 
     {4587, -6112}, {4598, -6023}}}
 
WorldData/: WorldData["Chad"] = 
   {{{1170, 1440}, {1410, 960}, {1380, 900}, {1221, 960}, 
     {1014, 929}, {823, 818}, {785, 844}, {599, 941}, 
     {578, 837}, {452, 930}, {482, 1115},  
     {657, 1366}, {768, 1310}, {933, 1376}, {942, 1440}, {1170, 1440}}}
 
WorldData/: WorldData["Chile"] = 
   {{{-1101, -4224}, {-1959, -4287}, {-2240, -4421}, {-2483, -4338}, 
     {-2670, -4356}, {-2792, -4424}, {-2748, -4450}, {-2798, -4542}, 
     {-2808, -4471}, {-2880, -4393}, {-2881, -4479}, 
     {-3012, -4482}, {-3123, -4413}, {-3107, -4348}, {-3180, -4407}, 
     {-3158, -4288}, {-3194, -4393}, {-3171, -4267}, {-3203, -4328}, 
     {-3143, -4106}, {-3119, -4316}, 
     {-2972, -4415}, {-2672, -4266}, {-2537, -4328}, {-2054, -4189}, 
     {-1879, -4234}, {-1623, -4099}, {-1369, -4031}, 
     {-1050, -4170}, {-1101, -4224}}, 
    {{-3306, -4023}, {-3297, -4102}, {-3306, -4023}}, 
    {{-3298, -4104}, {-3343, -4085}, {-3310, -4202}, {-3298, -4103}}, 
    {{-3293, -4118}, {-3271, -4321}, {-3268, -4140}, {-3166, -4225}, 
     {-3159, -4117},  {-3293, -4118}}, 
    {{-2603, -4434}, {-2511, -4410}, {-2603, -4434}}}
 
WorldData/: WorldData["China"] = 
   {{{2406, 7462}, {2323, 7271}, {2400, 7313}, {2459, 7312}, {2399, 7192}, 
     {2297, 7067}, {2241, 7136}, {2244, 7362}, 
     {2104, 7157}, {1904, 7314}, {1941, 7177}, {1858, 7313}, 
     {1812, 7209}, {1793, 7328}, {1633, 7226}, {1569, 7146}, 
     {1417, 7031}, {1336, 6857}, {1386, 6804}, {1331, 6813}, 
     {1291, 6698}, {1286, 6621}, {1214, 6595}, {1243, 6591}, {1289, 6597}, 
     {1293, 6479}, {1287, 6487}, {1391, 6331},  
     {1344, 6128}, {1294, 6069}, {1446, 5934}, {1436, 5852}, 
     {1552, 5923}, {1688, 5889}, {1693, 5841},  
     {1768, 5765}, {1666, 5499}, {1699, 5400}, {1639, 5335}, 
     {1672, 5289}, {1812, 4861},  
     {1952, 4704}, {1959, 4770}, {2038, 4734}, {2106, 4812}, 
     {2130, 4669}, {2222, 4474}, {2234, 4494}, 
     {2403, 4440}, {2523, 4816}, {2694, 4792}, {2708, 4954}, 
     {2833, 4982}, {2832, 5141}, {2946, 5240}, {2950, 5270}, 
     {2817, 5454}, {2719, 5442}, {2656, 5724}, {2564, 5783}, 
     {2565, 5783}, {2504, 6314}, {2559, 6608}, {2622, 6718}, 
     {2706, 6720}, {2800, 7194}, {2880, 7106}, {2873, 6935}, 
     {2990, 7003}, {3001, 7154}, {3166, 7202}, {3205, 7322}, 
     {3183, 7537}, {2988, 7652}, {2932, 7840}, {2862, 7859}, 
     {2896, 8084}, {2707, 7987}, {2690, 7857}, {2545, 7836}, 
     {2406, 7462}}, {{1199, 6574}, {1198, 6657}, {1127, 6632}, {1110, 6521}, 
     {1199, 6575}}}
 
WorldData/: WorldData["Colombia"] = 
   {{{711, -4279}, {668, -4492}, {475, -4605}, {520, -4642}, 
     {434, -4673}, {235, -4622}, {86, -4729},  
     {-6, -4517},  {-146, -4376}, {-158, -4206}, {-227, -4243}, 
     {-253, -4197}, {35, -4203}, {39, -4147}, {103, -4191}, 
     {73, -4012}, {167, -4071}, {372, -4047}, {421, -4320}, 
     {551, -4403}, {711, -4279}}}
 
WorldData/: WorldData["Congo"] = 
   {{{-237, 670}, {-301, 722}, {-278, 785}, 
     {-235, 954}, {209, 1118}, {209, 995}, 
     {133, 972}, {130, 798},  {74, 790}, {56, 870}, {-114, 867}, 
     {-140, 695}, {-237, 670}}}
 
WorldData/: WorldData["Costa Rica"] = 
   {{{655, -5019}, {574, -4954}, {482, -4974}, 
     {594, -5140}, {664, -5141}, {655, -5019}}}
 
WorldData/: WorldData["Croatia"] =
    {{{2542, 1119}, {2718, 874}, {2730, 839}, {2748, 943}, {2796, 992}, 
     {2746, 1072}, {2755, 1131}, {2693, 1128}, {2703, 1108}, {2710, 957}, 
     {2674, 966}, {2651, 982}, {2550, 1118}, {2542, 1119}}}

WorldData/: WorldData["Cuba"] = 
   {{{1315, -5095}, {1393, -4868}, {1305, -4638}, {1211, -4448}, 
     {1209, -4648}, {1361, -4913}, {1315, -5095}}}
 
WorldData/: WorldData["Cyprus"] = 
   {{{2140, 2075}, {2074, 1982}, {2104, 1936}, {2140, 2075}}}

WorldData/: WorldData["Czechoslovakia"] := 
	( Message[ WorldData::indep, "Czechoslovakia"];
		Join[ WorldData["Czech Republic"], WorldData["Slovakia"] ] )

WorldData/: WorldData["Czech Republic"] = 
   {{{2890, 991}, {2965, 1066}, {2993, 1147}, {3052, 890}, 
    {3019, 726}, {2926, 830}, {2890, 991}}}

WorldData/: WorldData["Denmark"] = 
   {{{3294, 520}, {3395, 488}, {3423, 597}, {3409, 495}, {3464, 633}, 
     {3289, 567}, {3295, 520}}}
 
WorldData/: WorldData["Djibouti"] = 
   {{{691, 2597}, {692, 2551}, {763, 2587}, {691, 2506}, {660, 2577},  {691, 2597}}}
 
WorldData/: WorldData["Dominican Republic"] = 
   {{{1182, -4306}, {1116, -4099}, {1083, -4307}, {1182, -4306}}}
 
WorldData/: WorldData["Ecuador"] = 
   {{{-202, -4820}, {-121, -4786}, {-139, -4855}, {-62, -4855}, {86, -4729}, 
     {-6, -4517}, {-300, -4742}, {-266, -4829}, {-202, -4820}}}
 
WorldData/: WorldData["Egypt"] = 
   {{{1388, 2137}, {1776, 1940}, {1664, 2055}, {1773, 2098}, {1769, 2094}, 
     {1873, 2056}, {1880, 2053}, {1899, 1509}, 
     {1320, 1500}, {1305, 2036}, {1388, 2137}}}
 
WorldData/: WorldData["El Salvador"] = 
   {{{806, -5270}, {825, -5406}, {866, -5361}, {806, -5270}}}
 
WorldData/: WorldData["Equatorial Guinea"] = 
   {{{60, 588}, {141, 589}, {130, 680}, {60, 681}, {60, 588}}}

WorldData/: WorldData["Eritrea"] =
   {{{862, 2188}, {1024, 2220}, {1082, 2317}, {905, 2383}, {763, 2587}, 
     {739, 2560}, {878, 2330}, {862, 2188}}}

WorldData/: WorldData["Estonia"] =
    {{{3569, 1682}, {3469, 1656}, {3472, 1446}, {3553, 1408}, {3569, 1682}}}

WorldData/: WorldData["Ethiopia"] = 
   {{{862, 2188}, {878, 2330}, {739, 2560}, {691, 2506}, {660, 2577}, 
    {540, 2641}, {480, 2879}, {294, 2697}, {239, 2514}, {206, 2371}, 
    {277, 2156}, {476, 1979}, {633, 2057}, {762, 2169}, {862, 2188}}}
 
WorldData/: WorldData["Falkland Islands"] = 
   {{{-3121, -3662}, {-3084, -3553}, {-3121, -3662}}, 
    {{-3127, -3583}, {-3102, -3464}, {-3127, -3583}}}
 
WorldData/: WorldData["Fiji"] = 
   {{{-1042, 10670}, {-1080, 10722}, {-1096, 10680}}}
 
WorldData/: WorldData["Finland"] = 
   {{{3949, 1450}, {3897, 1527}, {3766, 1266}, {3635, 1285}, {3588, 1373}, 
     {3634, 1672}, {3775, 1895}, {4142, 1741}, {4143, 1738}, 
     {4205, 1674}, {4139, 1545}, {4143, 1236}, {3949, 1450}}}
 
WorldData/: WorldData["France"] = 
   {{{2627, 452}, {2625, 446}, {2624, 444}, 
     {2613, 233}, {2546, 193}, {2550, 105}, 
     {2556, 90}, {2604, -106}, {2779, -67}, 
     {2911, -287}, {2918, -81}, {2984, -116}, {2960, 0}, 
     {3065, 153}, {2972, 349}, {2968, 382},  
     {2938, 494}, {2855, 455}, {2773, 358}, {2756, 422}, {2627, 452}}}
 
WorldData/: WorldData["French Guiana"] = 
   {{{321, -3250}, {327, -3176}, {242, -3101}, {140, -3164}, 
     {140, -3277}, {321, -3250}}}
 
WorldData/: WorldData["Gabon"] = 
   {{{60, 588}, {-38, 523}, {-237, 670}, {-140, 695}, 
     {-114, 867}, {56, 870}, {74, 790}, {130, 798}, {130, 680}, 
     {60, 681}, {60, 588}}}
 
WorldData/: WorldData["Gambia" | "The Gambia"] = 
   {{{815, -994}, {807, -918}, {782, -1005}, {784, -1005}, {803, -828}, 
     {815, -994}}}

WorldData/: WorldData["Georgia"] =
    {{{2603, 2400}, {2563, 2637}, {2509, 2778}, {2445, 2823}, {2472, 2701}, 
     {2435, 2590}, {2491, 2492}, {2603, 2400}}}

WorldData/: WorldData["Germany"] = 
   {{{3194, 432}, {3212, 590}, {3295, 520}, {3289, 567}, 
     {3290, 567}, {3237, 654}, {3224, 856}, {3236, 854}, {3052, 890},
     {3019, 726}, {2926, 830}, {2850, 783}, {2852, 584}, 
     {2852, 574}, {2855, 455}, {2938, 494},  
     {2968, 382}, {3008, 368}, {3045, 361}, {3194, 432}}}
   
WorldData/: WorldData["Ghana"] = 
   {{{305, -186}, {337, 0}, {366, 72}, {636, 0}, {642, -5}, 
     {666, 0}, {668, -9}, {660, -170}, {569, -161}, {305, -186}}}
 
WorldData/: WorldData["Gibraltar"] = 
   {{{2170, -322}, {2169, -323}, {2169, -322}}}
 
WorldData/: WorldData["Greece"] = 
   {{{2444, 1562}, {2443, 1422}, {2428, 1355}, {2331, 1351}, {2292, 1444}, 
     {2254, 1363}, {2186, 1392}, {2271, 1266}, {2291, 1333}, {2289, 1394}, 
     {2357, 1223}, {2381, 1201}, {2451, 1259}, 
     {2481, 1376}, {2503, 1582}, {2444, 1562}}}
 
WorldData/: WorldData["Greenland"] = 
   {{{3968, -2149}, {4117, -1949}, {4112, -1633}, {4200, -1338}, 
     {4224, -1752}, {4276, -1525}, {4328, -1716}, {4228, -1294}, 
     {4344, -1533}, {4407, -1647}, {4407, -1230}, 
     {4475, -1327}, {4469, -1139}, {4510, -1346}, {4524, -1160}, 
     {4610, -1348}, {4617, -1087}, {4661, -1322}, {4830, -967}, 
     {4834, -1275}, {4896, -729}, {4832, -1468}, {4925, -1338}, 
     {4920, -1514}, {4901, -1959}, {4956, -1279}, 
     {4965, -2129}, {4997, -1540}, {5017, -1981}, {4981, -2801}, 
     {4960, -2394}, {4966, -2745}, {4905, -2678}, {4897, -2992}, 
     {4926, -3515}, {4880, -3387}, {4916, -3608}, {4748, -3948}, 
     {4710, -4358}, {4650, -3964}, {4623, -4282}, {4565, -4110}, 
     {4561, -3605}, {4301, -3354}, {4341, -3214}, 
     {4231, -3029}, {4239, -3278}, {4201, -3013}, {4100, -3203}, 
     {4076, -3009}, {4052, -3224}, {4074, -3058}, 
     {4034, -3232}, {4030, -3022}, {4024, -3238}, {3966, -3209}, 
     {4019, -3000}, {3959, -3208}, {3943, -3033}, {3919, -3154}, 
     {3852, -3120}, {3892, -3000}, {3803, -3028},  
     {3649, -2894}, {3672, -2713}, {3604, -2707}, {3606, -2586}, 
     {3704, -2548}, {3765, -2589}, {3907, -2467}, 
     {3968, -2149}}, {{4219, -3258}, {4178, -3110}, {4154, -3214}, 
     {4219, -3258}}}
 
WorldData/: WorldData["Guatemala"] = 
   {{{954, -5335}, {943, -5293}, {866, -5361}, 
     {825, -5406}, {873, -5535}, {964, -5504}, 
     {966, -5426}, {1069, -5459}, {1069, -5348}, {954, -5335}}}
 
WorldData/: WorldData["Guinea"] = 
   {{{542, -798}, {657, -901}, {761, -823}, 
     {748, -681}, {744, -682}, {744, -539}, {609, -479}, {610, -478}, 
     {454, -508}, {509, -616}, {600, -673}, 
     {542, -798}}}
 
WorldData/: WorldData["Guinea Bissau" | "Guinea-Bissau"] = 
   {{{739, -1003}, {718, -900}, {680, -930}, {657, -901}, 
     {761, -823}, {739, -1003}}}
 
WorldData/: WorldData["Guyana"] = 
   {{{329, -3435}, {383, -3521}, {442, -3509}, {512, -3599}, 
     {357, -3683}, {312, -3644}, {263, -3580}, {83, -3555}, 
     {117, -3388}, {240, -3483}, {329, -3435}}}
 
WorldData/: WorldData["Haiti"] = 
   {{{1082, -4306}, {1106, -4469}, {1112, -4341}, {1185, -4407}, 
     {1182, -4305}, {1082, -4306}}}
 
WorldData/: WorldData["Honduras"] = 
   {{{943, -5293}, {900, -4988}, {779, -5238}, 
     {806, -5270}, {866, -5361}, {943, -5293}}}
 
WorldData/: WorldData["Hungary"] = 
   {{{2877, 1374}, {2905, 1329}, {2881, 1030}, 
     {2812, 967}, {2746, 1072}, {2767, 1216}, 
     {2877, 1374}}}
 
WorldData/: WorldData["Iceland"] = 
   {{{3831, -1282}, {3892, -1443}, {3927, -1302}, {3930, -1472}, 
     {3986, -1387}, {3910, -1265}, {3992, -970}, {3928, -818}, 
     {3808, -1090}, {3831, -1282}}}
 
WorldData/: WorldData["India"] = 
   {{{1315, 5342}, {1345, 5274}, {1199, 5185}, {942, 4817}, {670, 4791}, 
     {549, 4767}, {504, 4618}, {991, 4398}, {1336, 4375}, 
     {1272, 4327}, {1243, 4244}, {1338, 4137}, {1386, 4230}, {1426, 4092}, 
     {1465, 4266}, {1620, 4171}, {1678, 4314}, {1864, 4481}, 
     {1937, 4523}, {2079, 4437}, {2130, 4669}, {2106, 4812}, 
     {2038, 4734}, {1959, 4770}, {1952, 4704}, {1812, 4862}, 
     {1730, 4804}, {1650, 4966}, {1581, 5240}, {1672, 5289}, 
     {1639, 5335}, {1666, 5499}, 
     {1768, 5765}, {1693, 5841}, {1625, 5828}, {1597, 5708}, 
     {1319, 5556}, {1418, 5537}, {1377, 5497}, {1418, 5470}, 
     {1502, 5544}, {1517, 5391}, {1598, 5305}, {1517, 5341}, {1315, 5342}}}
 
WorldData/: WorldData["Indonesia"] = 
   {{{334, 5714}, {315, 5851}, {42, 6176}, {15, 6146}, {17, 6224}, 
     {-62, 6263}, {-166, 6272}, {-194, 6365}, {-350, 6347}, {-356, 6280}, 
     {-193, 6096}, {334, 5714}}, {{-106, 5952}, {-55, 5933}, {-106, 5952}}, 
    {{-89, 6353}, {-186, 6391}, {-124, 6308}, {-90, 6354}}, 
    {{124, 6579}, {-30, 6547}, {-179, 6614}, {-214, 6712}, {-251, 6878}, 
     {-213, 6960}, {-107, 6973}, {67, 7074}, {59, 7141}, {216, 7022}, 
     {250, 7055}, {262, 6952}, {124, 6893}, {51, 6634}, 
     {124, 6579}}, {{-470, 6594}, {-527, 6874}, {-468, 6866}, {-386, 6661}, 
     {-355, 6422}, {-406, 6313}, {-470, 6594}}, 
    {{-414, 6776}, {-417, 6848}, {-414, 6777}}, 
    {{-486, 6869}, {-504, 6943}, {-485, 6869}}, 
    {{-493, 6980}, {-535, 6961}, {-493, 6981}}, 
    {{-503, 7033}, {-503, 7140}, {-547, 7023}, {-503, 7033}}, 
    {{-566, 7142}, {-599, 7249}, {-566, 7142}}, 
    {{-496, 7216}, {-495, 7380}, {-537, 7261}, {-532, 7194}, {-496, 7217}}, 
    {{-488, 7468}, {-500, 7508}, {-488, 7468}}, 
    {{-568, 7506}, {-504, 7638}, {-518, 7509}, {-622, 7408}, {-568, 7506}}, 
    {{-427, 7899}, {-481, 7868}, {-427, 7899}}, 
    {{-337, 7175}, {-186, 7126}, {43, 7202}, {90, 7515}, {22, 7455}, 
     {21, 7215}, {-84, 7240}, {-46, 7407}, {-108, 7278}, {-264, 7374}, 
     {-264, 7292}, {-157, 7246}, {-263, 7221}, {-335, 7204}}, 
    {{-78, 7413}, {-87, 7368}, {-79, 7413}}, 
    {{-99, 7465}, {-113, 7519}, {-99, 7465}}, 
    {{-107, 7525}, {-109, 7581}, {-107, 7525}}, 
    {{-277, 7363}, {-323, 7337}, {-277, 7363}}, 
    {{-264, 7385}, {-341, 7369}, {-264, 7384}}, 
    {{-187, 7568}, {-202, 7635}, {-187, 7568}}, 
    {{-172, 7691}, {-188, 7835}, {-232, 7850}, {-214, 7675}, {-172, 7691}}, 
    {{-326, 8071}, {-374, 8060}, {-326, 8071}}, 
    {{12, 7734}, {12, 7675}, {-55, 7706}, {25, 7654}, {132, 7683}, 
     {55, 7658}, {12, 7734}}, {{-443, 8313}, {-485, 8334}, {-506, 8259}, 
     {-443, 8314}}, {{-548, 8460}, {-473, 8409}, {-498, 8335}, {-324, 8284}, 
     {-231, 8038}, {-176, 8032}, {-245, 7974}, {-167, 7917}, {-126, 8036}, 
     {-86, 7857}, {-21, 7946}, {-86, 8054}, {-202, 8107}, {-88, 8271}, 
     {-156, 8460}, {-156, 8460}, {-548, 8460}}}
 
WorldData/: WorldData["Iran"] = 
   {{{1512, 3697}, {1538, 3467}, {1602, 3223}, {1797, 3008}, {1798, 2912}, 
     {2038, 2724}, {2148, 2781}, {2230, 2688}, 
     {2379, 2688}, {2330, 2770}, {2376, 2888}, {2307, 2933}, 
     {2204, 3066}, {2241, 3234}, {2297, 3432}, 
     {2233, 3574}, {2136, 3677}, {1984, 3635}, {1861, 3711}, 
     {1792, 3652}, {1628, 3800}, {1512, 3697}}}
 
WorldData/: WorldData["Iraq"] = 
   {{{1801, 2877}, {1798, 2912}, {2038, 2724}, {2148, 2781}, 
     {2230, 2688}, {2226, 2541}, {2065, 2460}, 
     {2002, 2328}, {1934, 2358}, {1752, 2683}, 
     {1746, 2793}, {1801, 2877}}}
 
WorldData/: WorldData["Ireland"] = 
   {{{3245, -376}, {3131, -382}, {3087, -586}, {3131, -628}, {3160, -524}, 
     {3246, -607}, {3303, -435}, {3266, -489}, {3245, -376}}}
 
WorldData/: WorldData["Israel"] = 
   {{{1880, 2053}, {1986, 2106}, {1994, 2132}, 
     {1995, 2137}, {1995, 2140}, {1961, 2139}, {1890, 2129}, 
     {1773, 2098}, {1769, 2094}, {1873, 2056}, {1880, 2053}}}
 
WorldData/: WorldData["Italy"] = 
   {{{2736, 823}, {2727, 736}, {2653, 743}, {2408, 1111}, {2406, 997}, 
     {2275, 964}, {2402, 940}, {2544, 666}, {2641, 606}, {2627, 452}, 
     {2756, 422}, {2813, 628},  
     {2792, 823}, {2736, 823}}, 
    {{2295, 937}, {2199, 905}, {2268, 745}, {2295, 937}}, 
    {{2475, 555}, {2333, 531}, {2444, 488}, {2475, 555}}}
 
WorldData/: WorldData["Ivory Coast" | "Cote d'Ivoire"] = 
   {{{261, -452}, {305, -186}, {305, -186}, {569, -161}, {569, -161}, 
     {627, -331}, {626, -331}, {610, -478}, {610, -478}, {454, -508}, 
     {454, -508}, {261, -451}}}
 
WorldData/: WorldData["Jamaica"] = {{{1096, -4702}}}
 
WorldData/: WorldData["Japan"] = 
   {{{1947, 7899}, {2037, 7861}, {2001, 7775}, {1860, 7840}, {1947, 7899}}, 
    {{2045, 8076}, {1962, 7981}, {2000, 7921}, {2044, 8076}}, 
    {{2486, 8468}, {2373, 8524}, {2108, 8420}, {2076, 8331}, {2105, 8211}, 
     {2006, 8146}, {2083, 8120}, {2036, 7853}, {2135, 7985}, {2241, 8206}, 
     {2242, 8313}, {2486, 8468}}, 
    {{2731, 8517}, {2646, 8626}, {2602, 8749}, {2515, 8594}, {2554, 8428}, 
     {2508, 8472}, {2487, 8402}, {2597, 8484}, {2730, 8518}}, 
    {{1612, 7697}, {1565, 7659}, {1611, 7697}}}
 
WorldData/: WorldData["Jordan"] = 
   {{{1762, 2098}, {1773, 2099}, {1890, 2129}, {1961, 2139}, 
     {2003, 2328}, {1934, 2358}, {1890, 2220}, {1800, 2250}, {1762, 2098}}}

WorldData/: WorldData["Kazakhstan"] =
    {{{2782, 2953}, {2827, 3072}, {2777, 3190}, {2720, 3195}, {2671, 3094}, 
     {2674, 3013}, {2502, 3170}, {2490, 3360}, {2700, 3360}, {2730, 3520}, 
     {2610, 3730}, {2477, 4075}, {2480, 4170}, {2538, 4261}, {2592, 4461}, 
     {2568, 4695}, {2523, 4816}, {2694, 4792}, {2708, 4954}, {2833, 4982}, 
     {2832, 5141}, {2946, 5240}, {2948, 5269}, {3027, 5012}, {3086, 4768}, 
     {3262, 4638}, {3227, 4471}, {3308, 4238}, {3246, 3695}, {3047, 3653}, 
     {3067, 3247}, {3097, 3064}, {2965, 2811}, {2782, 2953}}}

WorldData/: WorldData["Kenya"] = 
   {{{-100, 2494}, {-280, 2352}, {-60, 2035}, 
     {115, 2102}, {253, 2040}, {277, 2156},  
     {206, 2371}, {239, 2515}, {170, 2459}, {-100, 2494}}}

WorldData/: WorldData["Kyrgyzstan"] =
    {{{2372, 4450}, {2354, 4291}, {2396, 4174}, {2456, 4380}, {2467, 4274}, 
     {2538, 4261}, {2592, 4461}, {2568, 4695}, {2523, 4816}, {2403, 4440}, {2372, 4450}}}

WorldData/: WorldData["Kuwait"] = 
   {{{1713, 2905}, {1801, 2877}, {1746, 2793}, {1713, 2905}}}
 
WorldData/: WorldData["Laos"] = 
   {{{1221, 6005}, {1294, 6069}, {1344, 6128}, 
     {1251, 6190}, {1240, 6279}, {1158, 6233}, {965, 6448}, {882, 6453}, 
     {861, 6313}, {1050, 6283}, {1106, 6204}, 
     {1054, 6055}, {1174, 6077}, {1221, 6005}}}

WorldData/: WorldData["Latvia"] =
    {{{3472, 1446}, {3469, 1656}, {3361, 1686}, {3332, 1580}, {3383, 1479}, 
     {3363, 1280}, {3432, 1285}, {3434, 1464}, {3472, 1446}}}

WorldData/: WorldData["Lebanon"] = 
   {{{2079, 2158}, {1986, 2106}, {1994, 2132}, 
     {1995, 2137}, {2079, 2158}}}
 
WorldData/: WorldData["Liberia"] = 
   {{{416, -690}, {261, -452}, {454, -508}, 
     {509, -616}, {416, -690}}}
 
WorldData/: WorldData["Libya"] = 
   {{{1990, 692}, {1879, 955}, {1817, 1160}, {1977, 1303}, {1899, 1509}, 
     {1320, 1500}, {1320, 1500}, {1200, 1500}, {1170, 1440}, 
     {1410, 960}, {1379, 898}, 
     {1411, 720}, {1569, 564}, {1814, 572}, {1990, 692}}}
 
WorldData/: WorldData["Liechtenstein"] = 
   {{{2824, 576}, {2837, 572}, {2836, 572}, {2824, 576}}}

WorldData/: WorldData["Lithuania"] =
    {{{3363, 1280}, {3383, 1479}, {3332, 1580}, {3249, 1519}, {3237, 1410}, 
     {3242, 1369}, {3286, 1373}, {3320, 1277}, {3363, 1280}}}

WorldData/: WorldData["Luxembourg"] = 
   {{{3008, 368}, {2968, 382}, {2973, 349}, {3008, 368}}}
 
WorldData/: WorldData["Macedonia"] =
    {{{2506, 1232}, {2533, 1274}, {2534, 1371}, {2480, 1376}, {2451, 1259}, {2506, 1232}}}

WorldData/: WorldData["Madagascar"] = 
   {{{-1312, 2599}, {-1197, 2671}, {-971, 2668}, {-886, 2880}, {-717, 2957}, 
     {-919, 3030}, {-933, 2978}, {-1499, 2828}, {-1497, 2644}, {-1312, 2599}}}
 
WorldData/: WorldData["Malawi"] = 
   {{{-569, 1979}, {-584, 2060}, {-694, 2097}, {-801, 2074}, 
     {-893, 2155}, {-1028, 2117}, {-875, 2072}, {-841, 1993}, 
     {-569, 1979}}}
 
WorldData/: WorldData["Malaysia"] = 
   {{{124, 6579}, {51, 6634}, {124, 6893}, {262, 6952}, {250, 7056}, 
     {321, 7157}, {352, 7048}, {420, 7029}, {294, 6909}, 
     {289, 6902}, {292, 6901}, 
     {275, 6845}, {161, 6687}, {81, 6683}, {124, 6579}}, 
    {{385, 6008}, {170, 6077}, {76, 6211}, {87, 6258}, {288, 6207}, 
     {374, 6126}, {338, 6069}, {385, 6008}}}
 
WorldData/: WorldData["Mali"] = 
   {{{1149, 255}, {1310, 0}, {1500, -288}, {1500, -395}, 
     {930, -330}, {939, -690}, {884, -723}, {886, -735}, {748, -681}, 
     {744, -682}, {744, -539}, {610, -478}, {626, -331}, 
     {810, -238}, {900, 0}, {900, 14}, {943, 233}, {1149, 255}}}
 
WorldData/: WorldData["Mauritania"] = 
   {{{1246, -1023}, {1213, -972}, {963, -992}, {962, -803}, 
     {883, -723}, {939, -690}, {930, -330}, {1500, -395}, 
     {1500, -288}, {1638, -520}, {1560, -520}, 
     {1560, -720}, {1374, -786}, {1280, -780}, {1246, -1023}}}
 
WorldData/: WorldData["Mexico"] = 
   {{{1558, -5828}, {1356, -5873}, {1123, -5752}, {1120, -5471}, 
     {1132, -5482}, {1262, -5420}, {1297, -5227}, {1109, -5298}, 
     {1069, -5348}, {1069, -5459}, 
     {966, -5426}, {964, -5504}, {873, -5535}, {905, -5567}, {873, -5535}, 
     {986, -5692}, {958, -5867}, {1100, -6209}, {1195, -6327}, {1352, -6343}, 
     {1538, -6564}, {1738, -6730}, {1874, -6785}, {1917, -6902}, 
     {1697, -6772}, {1388, -6571}, {1484, -6726}, 
     {1664, -6901}, {1711, -6844}, {1952, -7027}, 
     {1963, -6883}, {1880, -6543}, 
     {1907, -6391}, {1760, -6243}, {1757, -6050}, 
     {1584, -5945}, {1558, -5828}}}

WorldData/: WorldData["Moldova"] =
    {{{2895, 1600}, {2889, 1697}, {2811, 1778}, {2728, 1690}, {2895, 1600}}}

WorldData/: WorldData["Monaco"] = 
   {{{2625, 447}, {2623, 443}, {2624, 443}, {2625, 446}}}
 
WorldData/: WorldData["Mongolia"] = 
   {{{2950, 5271}, {2817, 5454}, {2719, 5442}, {2656, 5724}, {2564, 5783}, 
     {2504, 6315}, {2559, 6608}, {2622, 6718}, 
     {2706, 6720}, {2800, 7194}, {2880, 7106}, {2873, 6935}, 
     {2990, 7003}, {2960, 6513}, {3012, 6199}, {3129, 5936}, 
     {2984, 5840}, {3052, 5543}, {2950, 5271}}}
 
WorldData/: WorldData["Morocco"] = 
   {{{2106, -132}, {2155, -325}, {2101, -374}, {1953, -557}, {1759, -614}, 
     {1660, -791}, {1660, -520}, {1723, -520}, 
     {1858, -218}, {1902, -229}, {1926, -71}, {2106, -132}}}
 
WorldData/: WorldData["Mozambique"] = 
   {{{-629, 2426}, {-917, 2440}, {-1177, 2074}, {-1446, 2130}, {-1536, 1969}, 
     {-1611, 1974}, {-1610, 1928},  
     {-1557, 1918},  {-1345, 1878},  
     {-1202, 1982}, {-1003, 1979}, {-938, 1825}, {-841, 1993}, 
     {-875, 2072}, {-1028, 2117}, {-893, 2155}, {-801, 2074}, 
     {-694, 2097}, {-629, 2426}}}

WorldData/: WorldData["Namibia"] = 
   {{{-1035, 706}, {-1361, 872}, {-1388, 867},  
     {-1715, 989}, {-1706, 1200}, {-1486, 1200},  
     {-1320, 1200}, {-1099, 1260}, {-1068, 1516}, 
     {-1058, 1409}, {-1040, 831}, {-1035, 706}}}
 
WorldData/: WorldData["Nepal"] = 
   {{{1812, 4861}, {1672, 5289}, {1581, 5240}, {1650, 4966}, 
     {1730, 4804}, {1812, 4861}}}
 
WorldData/: WorldData["Netherlands"] = 
   {{{3082, 202}, {3178, 284}, {3195, 433}, {3045, 361}, 
    {3082, 202}}}
 
WorldData/: WorldData["New Zealand"] = 
   {{{-2501, 10441}, {-2430, 10363}, {-2505, 10288}, {-2773, 10004}, 
     {-2753, 10247}, {-2685, 10229}, {-2634, 10332}, {-2598, 10257}, 
     {-2596, 10365}, {-2501, 10441}}, 
    {{-2381, 10616}, {-2497, 10514}, {-2412, 10521}, {-2365, 10428}, 
     {-2224, 10497}, {-2065, 10362}, {-2258, 10560}, {-2262, 10713}, 
     {-2381, 10616}}}
 
WorldData/: WorldData["Nicaragua"] = 
   {{{900, -4988}, {655, -5019}, {664, -5141}, 
     {778, -5240}, {900, -4988}}}
 
WorldData/: WorldData["Niger"] = 
   {{{1379, 898}, {1411, 719}, {1149, 255},  
     {943, 233}, {900, 14}, {714, 144}, {705, 214}, 
     {702, 216}, {827, 293}, {768, 578}, {823, 818}, {1014, 929}, 
     {1221, 960}, {1379, 898}}}
 
WorldData/: WorldData["Nigeria"] = 
   {{{288, 515}, {256, 366}, {354, 317}, {382, 163}, {705, 214}, 
     {702, 216}, {827, 293}, {768, 578}, {823, 818}, {784, 845}, 
     {694, 879}, {425, 713}, {408, 588}, {288, 515}}}
 
WorldData/: WorldData["North Korea"] = 
   {{{2317, 7702}, {2383, 7650}, {2538, 7841},
     {2545, 7836}, {2406, 7462}, {2318, 7539}, {2287, 7480}, 
     {2270, 7602}, {2317, 7702}}}
 
WorldData/: WorldData["Norway"] = 
   {{{4187, 1847}, {4217, 1864}, {4252, 1746}, {4267, 1659}, {4258, 1522}, 
     {4159, 1086}, {4000, 791}, {3826, 573}, {3827, 687}, 
     {3775, 413}, {3731, 305}, {3712, 409}, {3666, 312}, {3674, 462}, 
     {3639, 311}, {3630, 426}, {3570, 311}, {3484, 396}, 
     {3593, 634}, {3539, 686}, {3815, 728}, {4143, 1235}, 
     {4139, 1545}, {4205, 1674}, {4143, 1738}, {4142, 1741}, 
     {4188, 1851}}, {{4821, 1491}, {4751, 1435}, {4808, 1067}, {4821, 1491}}, 
    {{4772, 641}, {4692, 780}, {4705, 1038}, {4683, 815}, {4594, 981}, 
     {4726, 1293}, {4804, 978}, {4734, 987}, {4772, 641}}}
 
WorldData/: WorldData["Oman"] = 
   {{{998, 3186}, {1138, 3468}, {1333, 3588}, {1470, 3396}, 
     {1390, 3350}, {1320, 3400}, {1140, 3119}, 
     {998, 3186}}}
 
WorldData/: WorldData["Pakistan"] = 
   {{{1512, 3697}, {1507, 4002}, {1426, 4092}, {1465, 4266}, 
     {1620, 4171}, {1678, 4314}, {1864, 4481}, {1937, 4523}, {2079, 4437}, 
     {2130, 4669}, {2222, 4474}, {2188, 4298}, 
     {2041, 4194}, {1916, 4160}, {1873, 4003}, {1791, 3975}, {1792, 3652}, 
     {1628, 3800}, {1512, 3697}}}
 
WorldData/: WorldData["Panama"] = 
   {{{574, -4954}, {556, -4802}, {562, -4790}, 
     {521, -4642}, {434, -4673}, {513, -4687}, 
     {536, -4772}, {533, -4776}, {435, -4826}, 
     {482, -4974}, {574, -4954}}}
 
WorldData/: WorldData["Papua-New Guinea" | "Papua New Guinea"] = 
   {{{-156, 8460}, {-359, 8849}, {-448, 8831}, {-614, 9053}, {-606, 8863}, 
     {-450, 8671}, {-494, 8528}, {-542, 8600}, {-548, 8460}, {-156, 8460}}, 
    {{-249, 9131}, {-363, 9062}, {-329, 8902}, {-329, 9056}, {-249, 9131}}}
 
WorldData/: WorldData["Paraguay"] = 
   {{{-1534, -3276}, {-1637, -3335}, {-1639, -3516}, {-1511, -3465}, 
     {-1334, -3759}, {-1179, -3705}, {-1210, -3490}, 
     {-1325, -3479}, {-1337, -3351}, {-1438, -3324}, 
     {-1443, -3255}, {-1534, -3276}}}
 
WorldData/: WorldData["Peru"] = 
   {{{-1101, -4224}, {-850, -4577}, {-515, -4726}, {-360, -4871}, 
     {-202, -4820}, {-266, -4829}, {-300, -4742}, {-6, -4517}, 
     {-146, -4376}, {-158, -4206}, {-227, -4243}, {-253, -4197}, 
     {-308, -4374}, {-452, -4441}, {-564, -4392}, 
     {-566, -4231}, {-661, -4238}, {-657, -4174}, 
     {-750, -4120}, {-1050, -4170}, {-1101, -4224}}}
 
WorldData/: WorldData["Philippines"] = 
   {{{1104, 7337}, {917, 7283}, {839, 7317}, {827, 7436}, {753, 7445}, 
     {838, 7352}, {790, 7356}, {828, 7238}, {979, 7187}, {1112, 7236}, 
     {1104, 7338}}, {{755, 7396}, {704, 7444}, {755, 7396}}, 
    {{811, 7222}, {787, 7293}, {734, 7267}, {811, 7223}}, 
    {{588, 7527}, {436, 7597}, {376, 7572}, {435, 7540}, {335, 7524}, 
     {469, 7421}, {426, 7313}, {523, 7404}, {478, 7421}, {540, 7531}, 
     {589, 7527}}, {{606, 7466}, {585, 7428}, {606, 7467}}, 
    {{653, 7412}, {546, 7393}, {590, 7344}, {653, 7412}}, 
    {{714, 7322}, {688, 7390}, {626, 7316}, {714, 7322}}, 
    {{685, 7171}, {630, 7183}, {500, 7031}, {684, 7171}}}
 
WorldData/: WorldData["Poland"] = 
   {{{3236, 854}, {3268, 1177}, {3237, 1410}, {3091, 1416}, 
     {2945, 1353}, {3052, 890}, {3236, 854}}}
 
WorldData/: WorldData["Portugal"] = 
   {{{2231, -445}, {2231, -535}, {2323, -570}, {2518, -525}, 
     {2495, -371}, {2231, -445}}}
 
WorldData/: WorldData["Puerto Rico"] = {{{1103, -4036}}}
 
WorldData/: WorldData["Qatar"] = 
   {{{1477, 3073}, {1569, 3075}, {1485, 3050}, {1477, 3073}}}
 
WorldData/: WorldData["Romania"] = 
   {{{2713, 1780}, {2625, 1715}, {2653, 1361}, 
     {2767, 1216}, {2877, 1374}, {2895, 1600}, 
     {2728, 1690}, {2713, 1780}}}

WorldData/: WorldData["Russia"] =
    {{{2537, 7842}, {2600, 7909}, {2610, 8108}, {2907, 8411}, 
     {3197, 8485}, {3259, 8385}, {3212, 8239}, {3259, 8264}, 
     {3296, 8113}, {3554, 8555}, {3586, 8977}, {3534, 9174}, 
     {3550, 9311}, {3593, 9254}, {3708, 9449}, {3706, 9622}, 
     {3635, 9608}, {3668, 9686}, {3753, 9796}, {3747, 9939}, 
     {3400, 9357}, {3053, 9399}, {3286, 9727}, {3372, 9801}, 
     {3484, 9716}, {3602, 9818}, {3637, 10153}, {3595, 10215}, 
     {3772, 10619}, {3737, 10746}, {3867, 10691}, {3881, 10466}, 
     {3889, 10769}, {3985, -10751}, {3855, -10391}, {3942, -10371},
     {3968, -10183}, {4028, -10421}, {3978, -10468}, {4193, 10567}, 
     {4188, 10233}, {4125, 10237}, {4180, 9746}, {4112, 9663}, 
     {4249, 9555}, {4247, 9152}, {4339, 8896}, {4300, 8693}, {4339, 8816}, 
     {4341, 8811}, {4374, 8439}, {4288, 8402}, {4244, 7861}, {4414, 7118}, 
     {4410, 6809}, {4357, 6817}, {4432, 6678}, {4367, 6304}, {4524, 6823}, 
     {4605, 6666}, {4664, 6244}, {4583, 6135}, {4590, 5929}, {4510, 6011}, 
     {4576, 5928}, {4483, 5152}, {4379, 5207}, {4434, 5207}, {4414, 4831}, 
     {4205, 4990}, {4303, 4996}, {4344, 4711}, {4295, 4575}, {4253, 4707}, 
     {4294, 4530}, {4373, 4500}, {4225, 4450}, {4148, 4428}, {4134, 4658}, 
     {4054, 4743}, {4139, 4593}, {3973, 4320}, {4008, 4138}, {4063, 4389}, 
     {4362, 4370}, {4360, 4139}, {4263, 3997}, {4099, 4119}, {4156, 3903},
     {4100, 3579}, {4132, 3193}, {4012, 2854}, {4041, 2695}, {4108, 2760}, 
     {4077, 2648}, {3952, 2651}, {3937, 2391}, {3872, 2430}, {3911, 2211}, 
     {3829, 2252}, {3860, 2120}, {3873, 2085}, {3972, 2050}, {4030, 1913}, 
     {3965, 2278}, {4032, 2482}, {4187, 1847}, {4188, 1851}, {4142, 1741}, 
     {3775, 1895}, {3634, 1671}, {3598, 1815}, {3569, 1682}, {3469, 1656}, 
     {3361, 1686}, {3350, 1799}, {3222, 1903}, {3124, 1898}, {3101, 2033}, 
     {2971, 2374}, {2867, 2396}, {2827, 2296}, {2827, 2363}, {2800, 2264}, 
     {2762, 2314}, {2711, 2195}, {2603, 2400}, {2563, 2637}, {2509, 2778}, 
     {2473, 2871}, {2510, 2915}, {2671, 2801}, {2782, 2953}, {2965, 2811}, 
     {3097, 3064}, {3067, 3247}, {3047, 3653}, {3246, 3695}, {3308, 4238}, 
     {3227, 4471}, {3262, 4638}, {3086, 4768}, {3027, 5012}, {2946, 5240}, 
     {2950, 5271}, {3052, 5543}, {2984, 5840}, {3129, 5936}, {3012, 6199}, 
     {2960, 6513}, {2990, 7003}, {3001, 7154}, {3166, 7202}, {3205, 7322}, 
     {3183, 7537}, {2988, 7652}, {2932, 7840},  {2862, 7859}, {2896, 8084}, 
     {2707, 7987}, {2690, 7857}, {2545, 7836}, {2537, 7842}},
    {{3320, 1277}, {3286, 1373}, {3242, 1369}, {3268, 1178}, {3297, 1275}, {3320, 1277}},
	{{3189, 8510}, {2753, 8525}, {2782, 8617}, {2963, 8595}, {2918, 8685}, 
     {3248, 8581}, {3189, 8510}}, {{3003, 9314}, {3047, 9364}, {3003, 9314}}, 
    {{4433, 8469}, {4399, 8609}, {4433, 8470}}, 
    {{4535, 8790}, {4508, 9057}, {4535, 8790}}, 
    {{4488, 8390}, {4496, 8622}, {4531, 8724}, {4573, 8324}, {4516, 8215}, 
     {4489, 8404}}, {{4677, 5968}, {4710, 6322}, {4766, 6142}, {4677, 5967}}, 
    {{4755, 5968}, {4810, 5852}, {4773, 5571}, {4755, 5967}}, 
    {{4307, 3084}, {4398, 3226}, {4390, 3394}, {4244, 3458}, {4308, 3084}}, 
    {{4481, 3419}, {4402, 3243}, {4617, 4110}, {4481, 3418}}
    }

WorldData/: WorldData["Rwanda"] = 
   {{{-83, 1776}, {-165, 1741}, {-144, 1834}, {-64, 1829}, {-83, 1776}}}
 
WorldData/: WorldData["San Marino"] = {{{2635, 744}, {2635, 744}}}
 
WorldData/: WorldData["Saudi Arabia"] = 
   {{{1762, 2098}, {1685, 2074}, {1266, 2351}, {983, 2567}, 
     {1044, 2670}, {906, 2735}, {1140, 3119},  
     {1320, 3400}, {1390, 3350}, {1360, 3290}, {1430, 3070}, 
     {1477, 3073}, {1485, 3050}, {1713, 2905}, 
     {1746, 2793}, {1752, 2683}, 
     {1934, 2358}, {1890, 2220}, {1800, 2250}, {1762, 2098}}}
 
WorldData/: WorldData["Senegal"] = 
   {{{964, -992}, {885, -1052}, {815, -994}, {803, -828}, 
     {784, -1005}, {770, -923}, {739, -1003}, 
     {761, -823}, {748, -681}, {886, -735}, 
     {883, -723}, {962, -803}, {963, -992}}}

WorldData/: WorldData["Serbia and Montenegro"] =
    {{{2542, 1119}, {2550, 1118}, {2693, 1128}, {2755, 1131}, {2767, 1216}, 
     {2653, 1361}, {2534, 1371}, {2533, 1274}, {2506, 1232}, {2551, 1210}, 
     {2511, 1162}, {2542, 1119}}}

WorldData/: WorldData["Sierra Leone"] = 
   {{{542, -798}, {416, -689}, {509, -615}, {600, -673}, {542, -798}}}
 
WorldData/: WorldData["Singapore"] = {{{83, 6239}, {83, 6239}}}

WorldData/: WorldData["Slovakia"] = 
   {{{2945, 1353}, {2993, 1147}, {2965, 1066}, {2890, 991}, {2881, 1030},
    {2905, 1329}, {2945, 1353}}}

WorldData/: WorldData["Somalia"] = 
   {{{-100, 2493}, {268, 2877}, {627, 3085}, {720, 3047}, {625, 2695}, 
     {691, 2597}, {689, 2595}, {660, 2577}, {540, 2641}, 
     {480, 2879}, {294, 2697}, {239, 2514}, {170, 2459}, {-100, 2493}}}
 
WorldData/: WorldData["South Africa"] = 
   {{{-1715, 989}, {-2050, 1099}, {-2089, 1200}, {-1982, 1674}, 
     {-1611, 1974}, {-1610, 1928}, 
     {-1608, 1849}, {-1558, 1918}, {-1345, 1878}, 
     {-1331, 1762}, {-1540, 1531}, 
     {-1539, 1368}, {-1610, 1239}, {-1486, 1200}, 
     {-1706, 1200}, {-1715, 989}}}
 
WorldData/: WorldData["Lesotho"] = 
   {{{-1717, 1722}, {-1778, 1621}, {-1840, 1686}, {-1761, 1767}, 
     {-1717, 1722}}}

WorldData/: WorldData["Slovenia"] =
    {{{2730, 839}, {2736, 823}, {2792, 823}, {2812, 967}, {2796, 992}, 
      {2748, 943}, {2730, 839}}}

WorldData/: WorldData["South Korea"] = 
   {{{2270, 7602}, {2057, 7591}, {2129, 7766}, {2317, 7702}, 
     {2270, 7602}}}
 
WorldData/: WorldData["Spain"] = 
   {{{2546, 193}, {2397, 0}, {2336, -7}, {2317, 0}, {2204, -128}, 
     {2169, -322}, {2231, -445}, 
     {2495, -371}, {2518, -525}, {2584, -557}, 
     {2627, -474}, {2604, -106}, {2556, 89}, {2550, 105}, {2546, 193}}}
 
WorldData/: WorldData["Sri Lanka"] = 
   {{{456, 4788}, {356, 4828}, {437, 4913}, {586, 4795}, {567, 4837}, 
     {456, 4788}}}
 
WorldData/: WorldData["Sudan"] = 
   {{{1082, 2317}, {1322, 2214}, {1388, 2137}, {1305, 2036}, 
     {1320, 1500}, {1200, 1500}, {1170, 1440}, 
     {942, 1440}, {933, 1376}, {768, 1310}, {657, 1366}, {655, 1373}, 
     {523, 1411}, {301, 1647}, {209, 1852},  
     {253, 2040}, {277, 2156}, {476, 1979}, 
     {633, 2057}, {762, 2169}, {1024, 2220}, {1082, 2317}}}
 
WorldData/: WorldData["Suriname"] = 
   {{{321, -3250}, {329, -3435}, {240, -3483}, {117, -3388}, 
     {140, -3276},{321, -3250}}}
 
WorldData/: WorldData["Swaziland"] = 
   {{{-1610, 1928}, {-1557, 1918}, {-1608, 1849}, 
     {-1610, 1928}}}
 
WorldData/: WorldData["Sweden"] = 
   {{{3539, 686}, {3320, 798}, {3365, 951}, {3518, 971}, {3548, 1105}, 
     {3570, 961}, {3584, 1145}, {3749, 1040}, {3949, 1450}, 
     {4143, 1236}, {3815, 728}, {3539, 686}}}
 
WorldData/: WorldData["Switzerland"] = 
   {{{2756, 422}, {2813, 628}, {2824, 576}, 
     {2837, 572}, {2852, 574}, {2855, 455}, 
     {2773, 358}, {2756, 422}}}
 
WorldData/: WorldData["Syria"] = 
   {{{2156, 2155}, {2079, 2158}, {1995, 2140}, 
     {1961, 2139}, {2003, 2328}, {2065, 2460}, 
     {2227, 2541}, {2210, 2200}, {2156, 2155}}}
 
WorldData/: WorldData["Taiwan"] = 
   {{{1507, 7277}, {1500, 7320}, {1316, 7243}, {1417, 7207}, {1507, 7277}}}

WorldData/: WorldData["Tajikistan"] =
    {{{2232, 4066}, {2282, 4093}, {2369, 4055}, {2438, 4175}, {2396, 4174}, 
     {2354, 4291}, {2372, 4450}, {2234, 4494}, {2201, 4301}, {2308, 4257},{2232, 4066}}}

WorldData/: WorldData["Tanzania"] = 
   {{{-629, 2426}, {-363, 2327}, {-280, 2352}, {-60, 2035}, 
     {-64, 1829}, {-144, 1834}, 
     {-267, 1765}, {-491, 1846}, {-565, 1979}, 
     {-584, 2060}, {-694, 2098}, {-629, 2426}}}
 
WorldData/: WorldData["Thailand"] = 
   {{{698, 6175}, {759, 6052}, {811, 6016}, {622, 5949}, {430, 6025}, 
     {374, 6126}, {338, 6069}, {385, 6008}, 
     {493, 5897}, {621, 5925}, {710, 5980}, {906, 5891}, 
     {983, 5936}, {1113, 5841}, {1221, 6005}, {1174, 6077}, 
     {1054, 6055}, {1106, 6204}, {1050, 6283}, {861, 6313}, 
     {813, 6140}, {698, 6175}}}
 
WorldData/: WorldData["Togo"] = 
   {{{366, 72}, {373, 99}, {660, 55}, {668, -10}, 
     {666, 0}, {642, -5}, {636, 0}, {366, 72}}}
 
WorldData/: WorldData["Tunisia"] = 
   {{{2216, 517}, {2225, 663}, {2050, 600}, {1990, 692}, 
     {1814, 572}, {2033, 450}, {2216, 517}}}
 
WorldData/: WorldData["Turkey"] = 
   {{{2156, 2155}, {2216, 2161}, {2161, 1968}, {2167, 1781}, {2201, 1642}, 
     {2222, 1700}, {2296, 1574}, {2307, 1630}, {2397, 1569}, {2471, 1777}, 
     {2526, 2097}, {2491, 2492}, {2378, 2689}, 
     {2230, 2688}, {2226, 2541}, {2210, 2200}, {2156, 2155}}, 
    {{2444, 1562}, {2403, 1571}, {2460, 1739}, {2519, 1681}, {2503, 1581}, 
     {2444, 1562}}}

WorldData/: WorldData["Turkmenistan"] =
    {{{2502, 3170}, {2462, 3286}, {2427, 3163}, {2241, 3234}, {2297, 3432}, 
     {2233, 3574}, {2136, 3677}, {2160, 3844}, {2254, 3947}, {2346, 3814}, 
     {2484, 3622}, {2570, 3520}, {2490, 3360}, {2502, 3170}}}

WorldData/: WorldData["UAE"] = 
   {{{1477, 3073}, {1448, 3247}, {1583, 3382}, {1470, 3396}, 
     {1390, 3350}, {1360, 3290}, {1430, 3070}, {1477, 3073}}}

WorldData/: WorldData["Ukraine"] =
    {{{3092, 1416}, {3095, 1640}, {3124, 1898}, {3101, 2033}, {2971, 2374}, 
     {2867, 2396}, {2827, 2296}, {2812, 2222}, {2771, 2089}, {2702, 2182}, 
     {2666, 2018}, {2724, 1949}, {2756, 2026}, {2777, 1908}, {2835, 1905}, 
     {2713, 1780}, {2728, 1690}, {2811, 1778}, {2889, 1697}, {2895, 1600}, 
     {2877, 1374}, {2905, 1329}, {2945, 1353}, {3092, 1416}}}

WorldData/: WorldData["United Kingdom"] = 
   {{{3518, -300}, {3518, -181}, {3450, -255}, {3456, -109}, {3361, -221}, 
     {3226, 0}, {3215, 8}, {3212, 0}, {3149, 106}, {3087, 23}, {3044, 14}, 
     {3035, -295}, {3105, -143}, {3100, -258}, {3171, -283}, 
     {3200, -164}, {3299, -181}, {3291, -308}, {3376, -296}, {3318, -349}, 
     {3518, -300}}, {{3304, -435}, {3245, -375}, {3246, -376}, {3266, -489}, 
     {3304, -435}}}
 
WorldData/: WorldData["USA"] = 
   {{{1952, -7027}, {1980, -7037}, {2074, -7237}, 
     {2251, -7349}, {2281, -7285}, {2336, -7422}, {2520, -7451}, 
     {2777, -7444}, {2904, -7483}, 
     {2890, -7365}, {2827, -7384}, {2939, -7365}, {2940, -7364}, 
     {2940, -7022}, {2940, -6962}, {2940, -6240}, {2940, -5832},  
     {2880, -5375}, {2792, -5048}, {2581, -4945}, 
     {2553, -4959}, {2501, -4962}, 
     {2575, -4735}, {2586, -4744}, {2596, -4744}, 
     {2669, -4548}, {2701, -4401}, {2701, -4290}, {2718, -4265},  
     {2848, -4154}, {2711, -4032}, {2584, -4245}, 
     {2572, -4249}, {2491, -4267}, 
     {2490, -4268}, {2491, -4267}, {2479, -4311}, 
     {2460, -4419}, {2431, -4455}, {2337, -4491}, {2400, -4502}, 
     {2388, -4526}, {2307, -4502},  
     {2281, -4522}, {2228, -4557}, {2278, -4539}, 
     {2311, -4574}, {2375, -4550}, {2283, -4579}, 
     {2328, -4621}, {2334, -4624},  
     {2290, -4628}, {2225, -4576}, {2239, -4637}, {2193, -4551}, 
     {2147, -4531}, {2203, -4556}, {2157, -4604}, 
     {2137, -4545}, {2132, -4623}, {2093, -4580}, {2031, -4712}, 
     {1921, -4853}, {1843, -4890}, 
     {1607, -4802}, {1515, -4825}, {1507, -4865}, {1675, -4970}, {1720, -4960},
	 {1795, -5021}, {1806, -5040}, {1780, -5119}, {1824, -5200}, {1817, -5251},
	 {1823, -5304}, {1811, -5371}, {1803, -5362}, 
     {1755, -5352}, {1746, -5452}, {1782, -5630}, {1672, -5851}, 
     {1558, -5828}, {1584, -5945}, {1757, -6050}, 
     {1760, -6243}, {1907, -6391}, {1880, -6543}, 
     {1963, -6883}, {1952, -7027}}, 
    {{3355, -7800}, {3286, -7841}, {3366, -7861}, {3345, -7936}, 
     {3568, -8120}, {3492, -8109}, {3603, -8370}, {3677, -8863}, 
     {3591, -8979}, {3557, -9119}, {3690, -8955}, 
     {3523, -9207}, {3360, -9506}, {3288, -9800}, {3450, -9444}, 
     {3549, -9407}, {3533, -9609}, {3519, -9730}, 
     {3642, -9713}, {3588, -9820}, {3654, -9813}, {3687, -9901}, 
     {3695, -9972}, {3793, -9865}, {3811, -9669}, {3896, -9659}, 
     {3939, -10088}, {3996, -9857}, {3959, -9708}, {4017, -9749}, 
     {3984, -9614}, {4101, -10010}, {4212, -9704}, 
     {4284, -9388}, {4179, -8461}, {3618, -8460}, 
     {3588, -8128}, {3355, -7801}}, 
    {{1216, -9349}, {1172, -9288}, {1135, -9340}, {1216, -9349}}, 
    {{3480, -9196}, {3446, -9140}, {3441, -9288}, {3480, -9196}}}

WorldData/: WorldData["USSR"] := 
   ( Message[ WorldData::indep, "USSR"];
		Join @@ ( WorldData /@ {"Armenia", "Azerbaijan", "Belarus", 
         "Estonia", "Georgia", "Kazakhstan", "Kyrgyzstan",
         "Latvia", "Lithuania", "Moldova", "Russia", "Tajikistan",
         "Turkmenistan", "Ukraine", "Uzbekistan"}) )

WorldData/: WorldData["Uganda"] = 
   {{{209, 1852}, {127, 1878}, {-83, 1776}, {-64, 1829}, 
     {-60, 2035}, {115, 2102}, {253, 2040}, {209, 1852}}}

WorldData/: WorldData["Uruguay"] = 
   {{{-2025, -3202}, {-2070, -3470}, {-1947, -3492}, 
     {-1814, -3458}, {-1923, -3225}, {-2025, -3202}}}

WorldData/: WorldData["Uzbekistan"] =
    {{{2490, 3360}, {2700, 3360}, {2730, 3520}, {2610, 3730}, 
     {2477, 4075}, {2480, 4170}, {2538, 4261}, {2467, 4274}, 
     {2456, 4380}, {2396, 4174}, {2438, 4175}, {2369, 4055}, 
     {2282, 4093}, {2232, 4066}, {2254, 3947}, {2346, 3814}, 
     {2484, 3622}, {2570, 3520}, {2490, 3360}}}

WorldData/: WorldData["Venezuela"] = 
   {{{512, -3599}, {514, -3697}, {566, -3651}, {606, -3781}, {645, -3711}, 
     {602, -3905}, {641, -4098}, {732, -4201}, {686, -4188}, {648, -4295}, 
     {543, -4298}, {711, -4279}, {551, -4403}, {421, -4320}, 
     {372, -4047}, {167, -4071}, {73, -4012}, {129, -3804}, 
     {257, -3888}, {214, -3773}, {256, -3691}, {312, -3644}, 
     {357, -3683}, {512, -3599}}}
 
WorldData/: WorldData["Vietnam"] = 
   {{{622, 6267}, {514, 6291}, {600, 6349}, {575, 6393}, {703, 6555}, 
     {915, 6537}, {1054, 6395}, {1176, 6347}, {1293, 6479}, 
     {1287, 6487}, {1391, 6331}, {1344, 6128}, 
     {1251, 6190}, {1240, 6279}, {1158, 6233}, {965, 6448}, {882, 6453}, 
     {741, 6453}, {700, 6351}, {622, 6267}}}
 
WorldData/: WorldData["Western Sahara"] = 
   {{{1660, -791}, {1246, -1023}, {1280, -780}, {1374, -786}, 
     {1560, -720}, {1560, -520}, {1638, -520}, {1660, -520}, 
     {1660, -791}}}
 
WorldData/: WorldData["Yemen AR"] := 
  ( Message[WorldData::united, "Yemen AR", "Yemen PR", "Yemen" ];
	 {{{984, 2567}, {762, 2613}, {849, 2754}, {906, 2735}, 
     {1044, 2670}, {984, 2567}}}
	)
 
WorldData/: WorldData["Yemen PR"] := 
   ( Message[WorldData::united, "Yemen PR", "Yemen AR", "Yemen" ];
    {{{761, 2612}, {842, 2922}, {999, 3187}, {1140, 3119}, 
     {906, 2735}, {849, 2754}, {761, 2612}}} )

WorldData/: WorldData["Yemen"] =
   {{ {1044, 2670}, {984, 2567}, {762, 2613}, {842, 2922}, {998, 3185},
     {1140, 3119}, {906, 2735}, {1044, 2670} }}

WorldData/: WorldData["Yugoslavia"] := 
   ( Message[ WorldData::indep, "Yugoslavia"];
		Join @@ ( WorldData /@ {"Bosnia and Herzegovina", "Croatia", 
          "Macedonia", "Slovenia", "Serbia and Montenegro"}) )

WorldData/: WorldData["Zaire"] = 
   {{{-352, 791}, {-347, 732}, {-278, 786}, 
     {-235, 954}, {209, 1118}, {298, 1152}, {248, 1340}, 
     {301, 1647}, {209, 1852}, {127, 1878}, 
     {-83, 1776}, {-164, 1742}, {-267, 1765}, 
     {-491, 1846}, {-556, 1702}, {-693, 1702}, 
     {-729, 1788}, {-807, 1788}, {-694, 1632}, {-652, 1439}, 
     {-652, 1330}, {-437, 1307}, {-486, 1080}, {-435, 1019}, {-353, 991}, 
     {-352, 791}}}
 
WorldData/: WorldData["Zambia"] = 
   {{{-491, 1846}, {-556, 1702}, {-693, 1702}, {-729, 1788}, {-807, 1788}, 
     {-694, 1632}, {-652, 1439}, {-780, 1441}, {-780, 1320}, 
     {-970, 1320}, {-1058, 1409},  
     {-1068, 1516}, {-1068, 1629}, {-938, 1825}, 
     {-841, 1993}, {-569, 1979}, {-565, 1979}, {-492, 1846}}}
 
WorldData/: WorldData["Zimbabwe"] = 
   {{{-1345, 1878}, {-1331, 1762}, {-1068, 1516}, 
     {-1068, 1629}, {-938, 1825}, {-1003, 1979}, 
     {-1202, 1982}, {-1345, 1878}}}
	 
	 
(* US States, added by David Terr, May-June, 2003 *)

WorldData/: WorldData["Alabama"] = 
   {{(*GA-TN*){2099, -5136}, 
     (*TN-MS*){2100, -5292}, {2094, -5286}, {1973, -5308}, 
	 (*MS-GOM*){1823, -5304}, 
	 (*GOM-FL*){1817, -5251}, {1860, -5256}, 
	 (*FL-GA*){1860, -5100}, {1950, -5100}, 
	 (*GA-TN*){2099, -5136}}}


(* I borrowed the data for Alaska from the USA data. - DTERR, 6-03 *)
	 
WorldData/: WorldData["Alaska"] = 
   {{{3355, -7800}, {3286, -7841}, {3366, -7861}, {3345, -7936}, 
     {3568, -8120}, {3492, -8109}, {3603, -8370}, {3677, -8863}, 
     {3591, -8979}, {3557, -9119}, {3690, -8955}, 
     {3523, -9207}, {3360, -9506}, {3288, -9800}, {3450, -9444}, 
     {3549, -9407}, {3533, -9609}, {3519, -9730}, 
     {3642, -9713}, {3588, -9820}, {3654, -9813}, {3687, -9901}, 
     {3695, -9972}, {3793, -9865}, {3811, -9669}, {3896, -9659}, 
     {3939, -10088}, {3996, -9857}, {3959, -9708}, {4017, -9749}, 
     {3984, -9614}, {4101, -10010}, {4212, -9704}, 
     {4284, -9388}, {4179, -8461}, {3618, -8460}, 
     {3588, -8128}, {3355, -7801}},
	 {{3480, -9196}, {3446, -9140}, {3441, -9288}, {3480, -9196}}}

WorldData/: WorldData["Arizona"] = 
   {{(*NM-CO-UT*){2220, -6543},
	 (*UT-NV*){2220, -6843}, {2172, -6843}, {2162, -6849}, {2162, -6856}, 
	 	{2169, -6863}, {2166, -6885},   
	 (*NV-CA*){2100, -6878}, {2067, -6863}, {2056, -6848}, {2036, -6872},
	 	{2013, -6871}, {2004, -6883}, {1983, -6880}, {1981, -6870},
		{1971, -6868}, 
	 (*CA-MEX*){1963, -6884}, {1949, -6889}, {1880, -6664}, 
	 (*MEX-NM*){1881, -6543},
     (*NM-CO-UT*){2220, -6543}}}
	 
WorldData/: WorldData["Arkansas"] = 
   {{(*TX-LA*){1981, -5643}, 
     (*LA-MS*){1980, -5470}, {2020, -5474}, 
     (*MS-TN*){2100, -5418}, 
	 (*TN-MO*){2160, -5384}, {2160, -5422}, {2180, -5404}, {2190, -5410}, 
	 (*MO-OK*){2190, -5677}, {2184, -5666}, 
	 (*OK-TX*){2018, -5669}, {2013, -5643}, 
	 (*TX-LA*){1981, -5643}}}

WorldData/: WorldData["California"] = 
   {{(*NV-AZ*){2100, -6878}, {2067, -6863}, {2056, -6848}, {2036, -6872},
	 	{2013, -6871}, {2004, -6883}, {1983, -6880}, {1981, -6870},
		{1971, -6868}, 
	 (*AZ-MEX*){1963, -6884},
     (*MEX-PO*){1952, -7027}, {1993, -7043}, {2026, -7090}, {2022, -7097},
	 	{2025, -7104}, {2042, -7112}, {2040, -7128}, {2049, -7153},
		{2056, -7156}, {2065, -7175}, {2067, -7228}, {2072, -7231},
		{2075, -7238}, {2108, -7238}, {2115, -7254}, {2123, -7252}, 
		{2127, -7254}, {2128, -7260}, {2179, -7314}, {2198, -7316}, 
		{2196, -7313}, {2202, -7308}, {2208, -7307}, {2219, -7316},
		{2217, -7323}, {2219, -7329}, {2231, -7344}, {2241, -7344},
		{2249, -7347}, {2250, -7350}, {2256, -7351}, {2258, -7349}, 
		{2267, -7351}, 
	 (*PO-SFB*){2268, -7349}, {2268, -7344}, {2263, -7342}, {2263, -7343},
	 	{2256, -7343}, {2247, -7325}, {2248, -7317}, {2248, -7323},
		{2250, -7323}, {2250, -7326}, {2262, -7331}, {2267, -7340}, 
		{2268, -7340}, {2270, -7339}, {2275, -7339}, {2275, -7343},
		{2278, -7346}, {2277, -7344}, {2281, -7342}, {2281, -7338}, 
		{2283, -7335}, {2283, -7331}, {2281, -7329}, {2283, -7324},
		{2283, -7313}, {2283, -7315}, {2285, -7314}, {2285, -7322},
		{2287, -7319}, {2288, -7320}, {2288, -7324}, {2283, -7328},
		{2283, -7330}, {2284, -7336}, {2287, -7339}, {2289, -7344},
		{2286, -7349}, {2281, -7350}, {2279, -7347}, {2278, -7351}, 
		{2272, -7346}, {2274, -7350}, {2270, -7349},
	 (*SFB-PO*){2269, -7352}, {2274, -7363}, {2282, -7374}, {2282, -7377},
	 	{2280, -7379}, {2280, -7381}, {2290, -7377}, {2337, -7424},
		{2341, -7421}, {2361, -7429}, {2373, -7426}, {2390, -7431},
		{2416, -7462}, {2419, -7461}, {2426, -7465}, {2462, -7447},
		{2464, -7449}, {2468, -7449}, {2486, -7444}, {2504, -7449},
		{2507, -7455}, {2514, -7452},
	 (*PO-OR*){2520, -7453},
	 (*OR-NV*){2520, -7200}, {2340, -7200},
	 (*NV-AZ*){2100, -6878}}} 

WorldData/: WorldData["Colorado"] = 
   {{(*NM-AZ-UT*){2220, -6543},
     (*UT-WY*){2460, -6543},
	 (*WY-NE*){2460, -6243}, {2460, -6123},
	 (*NE-KS*){2400, -6123},
	 (*KS-OK*){2220, -6123},
	 (*OK-NM*){2220, -6180},
     (*NM-AZ-UT*){2220, -6543}}}

WorldData/: WorldData["Connecticut"] = 
   {{(*MA-NY*){2523, -4410}, {2478, -4413}, {2473, -4409}, {2466, -4423}, 
	 (*NY-AO*){2478, -4375}, 
	 (*AO-RI*){2479, -4312},
	 (*RI-MA*){2521, -4308},
	 (*MA-NY*){2523, -4410}}} 

WorldData/: WorldData["Delaware"] = 
   {{(*PA-MD*){2383, -4547}, {2308, -4542},
	 (*MD-AO-NCM*){2307, -4503}, {2327, -4505}, {2368, -4535}, 
	 (*NCM-NJ*){2377, -4535}, 
	 (*NJ-PA*){2388, -4526}, {2390, -4538},  
	 (*PA-MD*){2383, -4547}}} 

WorldData/: WorldData["Florida"] = 
   {{(*GOM-AL*){1817, -5251}, {1860, -5256}, 
     (*AL-GA*){1860, -5100}, {1843, -5095}, 
	 (*GA-AO*){1843, -4890}, {1607, -4802}, {1515, -4825}, 
	 (*AO-GOM*){1507, -4865}, {1675, -4970}, {1720, -4960},
	 	{1795, -5021}, {1806, -5040}, {1780, -5119}, {1824, -5200}, 
	 (*GOM-AL*){1817, -5251}}}

WorldData/: WorldData["Georgia"] = 
   {{(*FL-AL*){1860, -5100}, {1950, -5100}, 
     (*AL-TN*){2099, -5136}, 
     (*TN-NC*){2099, -5060}, 
	 (*NC-SC*){2100, -4987}, {2083, -5001},
     (*SC-AO*){1921, -4853}, 
	 (*AO-FL*){1843, -4890}, {1843, -5095}, 
	 (*FL-AL*){1860, -5100}}}

WorldData/: WorldData["Hawaii"] = 
	{(*Hawaii*){{1216, -9351}, {1191, -9305}, {1184, -9305}, {1184, -9300}, 
		{1179, -9299}, {1172, -9289}, {1169, -9289}, {1156, -9310},
		{1156, -9318}, {1145, -9333}, {1135, -9341}, {1145, -9353},
		{1149, -9355}, {1159, -9353}, {1154, -9364}, {1157, -9363},
		{1211, -9354}, {1216, -9349}},
     (*Maui*){{1262, -9396}, {1254, -9389}, {1257, -9379}, {1247, -9359},
	    {1241, -9360}, {1236, -9385}, {1247, -9388}, {1248, -9391},
		{1247, -9392}, {1249, -9398}, {1256, -9402}, {1261, -9400},
		{1262, -9396}},
	 (*Kalamao*){{1270, -9403}, {1271, -9417}, {1273, -9418}, {1271, -9420},
	 	{1274, -9435}, {1268, -9438}, {1265, -9421}, {1263, -9414},
		{1265, -9407}, {1270, -9403}},
	 (*Oahu*){{1279, -9459}, {1287, -9465}, {1285, -9467}, {1303, -9479},
	 	{1295, -9487}, {1295, -9497}, {1278, -9487}, {1280, -9474},
		{1276, -9469}, {1277, -9463}, {1276, -9463}, {1279, -9459}}, 
	 (*Kauai*){{1312, -9567}, {1314, -9576}, {1320, -9587}, {1324, -9587},
	 	{1330, -9583}, {1334, -9575}, {1333, -9561}, {1328, -9558},
		{1322, -9560}, {1317, -9560}, {1312, -9567}}}
	
WorldData/: WorldData["Idaho"] = 
   {{(*WY-MT*){2669, -6663}, {2684, -6680}, {2682, -6689}, {2673, -6689},
	 	{2672, -6741}, {2667, -6743}, {2670, -6763}, {2662, -6771},
		{2669, -6795}, {2667, -6800}, {2742, -6838}, {2728, -6860},
		{2734, -6873}, {2800, -6860}, {2800, -6878}, {2846, -6945},
		{2862, -6944}, {2879, -6963}, 
	 (*MT-ALB*){2940, -6963},
	 (*ALB-BC*){2940, -6990},
	 (*BC-WA*){2940, -7022}, {2786, -7022}, {2781, -7024}, {2770, -7015},
	 	{2765, -7019}, 
	 (*WA-OR*){2760, -7015}, {2737, -6988}, {2729, -6973}, {2718, -6973},
	 	{2658, -7034}, {2649, -7014}, {2628, -7022},
	 (*OR-NV*){2520, -7022},
	 (*NV-UT*){2520, -6843},
	 (*UT-WY*){2520, -6663},
     (*WY-MT*){2669, -6663}}}

WorldData/: WorldData["Illinois"] = 
   {{(*LM-IN*){2503, -5251}, {2361, -5252}, {2348, -5259}, {2321, -5251},
	 	{2295, -5279},
	 (*IN-KY*){2268, -5281}, {2224, -5310}, {2233, -5338}, 
	 (*KY-MO*){2219, -5348}, {2223, -5363}, {2238, -5371}, {2262, -5371},
	 	{2297, -5422}, {2331, -5407}, {2337, -5440}, {2345, -5444}, 
		{2383, -5481}, {2410, -5491}, 
	 (*MO-IA*){2423, -5486}, {2461, -5456}, {2480, -5494}, {2491, -5431},
	 	{2520, -5408}, 
	 (*IA-WI*){2550, -5439},
	 (*WI-LM*){2550, -5268},
	 (*LM-IN*){2503, -5251}}}

WorldData/: WorldData["Indiana"] = 
   {{(*KY-OH*){2346, -5089},
	 (*OH-MI*){2502, -5088}, {2506, -5088},
	 (*MI-LM*){2506, -5209}, {2497, -5235},
	 (*LM-IL*){2503, -5251}, {2361, -5252}, {2348, -5259}, {2321, -5251},
	 	{2295, -5279},
	 (*IL-KY*){2268, -5281}, {2280, -5160}, 
	 (*KY-OH*){2346, -5089}}}

WorldData/: WorldData["Iowa"] = 
   {{(*MO-IL*){2423, -5486}, {2461, -5456}, {2480, -5494}, {2491, -5431},
	 	{2520, -5408}, 
	 (*IL-WI*){2550, -5439}, {2558, -5443}, {2565, -5464},
	 (*WI-MN*){2610, -5473}, 
	 (*MN-SD*){2610, -5787}, {2610, -5795}, 
	 (*SD-NE*){2549, -5788},
	 (*NE-MO*){2435, -5746}, {2437, -5503},
	 (*MO-IL*){2423, -5486}}}

WorldData/: WorldData["Kansas"] = 
   {{(*NE-MO*){2400, -5719}, {2385, -5691}, {2374, -5707}, {2353, -5690},
	 	{2349, -5676}, 
	 (*MO-OK*){2220, -5677},
	 (*OK-CO*){2220, -6123},
	 (*CO-NE*){2400, -6123},
	 (*NE-MO*){2400, -5719}}}
	 
WorldData/: WorldData["Kentucky"] = 
   {{(*VA-TN*){2196, -5020}, {2199, -5283}, {2190, -5282}, 
	 (*TN-MO*){2190, -5370}, {2200, -5350}, 
	 (*MO-IL*){2219, -5348}, {2233, -5338}, {2224, -5310}, 
	 (*IL-IN*){2268, -5281}, {2280, -5160},
	 (*IN-OH*){2346, -5089},
	 (*OH-WV*){2305, -4956}, {2290, -4959}, {2261, -4938},
	 (*WV-VA*){2252, -4918},  
	 (*VA-TN*){2196, -5020}}}

WorldData/: WorldData["Louisiana"] = 
   {{(*GOM-MS*){1811, -5371}, {1860, -5383}, {1860, -5496}, {1940, -5455}, 
     (*MS-AR*){1980, -5470},
     (*AR-TX*){1981, -5643}, {1920, -5643}, {1860, -5615}, 
	 (*TX-GOM*){1782, -5630}, {1746, -5452}, {1755, -5352}, {1803, -5362}, 
	 (*GOM-MS*){1811, -5371}}}

WorldData/: WorldData["Maine"] = 
   {{(*QUE-NH*){2718, -4265}, {2603, -4259}, 
	 (*NH-AO*){2583, -4243}, {2620, -4215}, 
	 (*AO-NB*){2689, -4017}, {2740, -4067}, {2824, -4067}, 
	 (*NB-QUE*){2847, -4180}, {2802, -4200}, {2744, -4223}, 
	 (*QUE-NH*){2718, -4265}}} 

WorldData/: WorldData["Maryland"] = 
   {{(*VA-WV*){2359, -4664}, {2382, -4690}, {2352, -4769}, 
	 (*WV-PA*){2383, -4769}, 
	 (*PA-DE*){2383, -4547}, {2308, -4542},
	 (*DE-AO*){2307, -4503}, 
	 (*AO-VA*){2281, -4522}, 
	 (*VA-CB*){2275, -4545}, {2310, -4580}, {2349, -4576}, {2370, -4560},
	 	{2352, -4585}, {2310, -4590},
	 (*CB-VA*){2273, -4574}, {2300, -4635},
	 
	 (* For now, Washington D.C. is displayed as part of Maryland.  DTERR, 6-03 *)
	 (*VA-DC 1*){2327, -4622}, {2333, -4622},
	 (*DC-VA 2*){2336, -4627},
	 (*VA-WV*){2359, -4664}}} 

WorldData/: WorldData["Massachusetts"] = 
   {{(*CN-RI*){2521, -4308}, {2521, -4283}, {2510, -4280}, {2500, -4268},
     (*RI-AO*){2490, -4267}, {2499, -4196}, {2515, -4198}, 
	 (*AO-CCB 1*){2524, -4215}, {2515, -4200}, {2508, -4200}, {2504, -4216},
	 (*CCB-AO 2*){2508, -4232}, {2541, -4264},
	 (*AO-NH*){2572, -4249}, {2562, -4278}, 
	 (*NH-VT*){2564, -4348},
	 (*VT-NY*){2565, -4396},  
	 (*NY-CN*){2523, -4410},
	 (*CN-RI*){2521, -4308}}} 

WorldData/: WorldData["Michigan"] = 
   {{(*OH-LE*){2504, -5008},
     (*LE-ONT*){2525, -4989},
     (*ONT-LH*){2580, -4945}, {2628, -4957}, {2637, -4963}, {2644, -4977},
	 	{2638, -4995}, {2615, -5021}, {2620, -5033}, {2626, -5037}, {2637, -5033},
		{2644, -5015}, {2660, -5000}, {2683, -4997}, {2720, -5008}, 
	 (*LH-LM*){2747, -5085}, {2682, -5175}, {2618, -5192}, {2580, -5174}, 
	 	{2550, -5175}, 
	 (*LM-IN*){2506, -5209}, {2506, -5088},
	 (*IN-OH*){2502, -5088},
	 (*OH-LE*){2504, -5008}},
	 
	 (*Upper Peninsula*)
	 
	 (*LS-WI*){{2794, -5425}, {2790, -5413}, {2780, -5407}, {2753, -5285}, 
	 (*WI-LM*){2705, -5256}, {2741, -5218}, {2766, -5130},
	 	{2751, -5083}, {2763, -5079}, 
	 (*LM-ONT*){2760, -5074}, 
	 (*ONT-LH*){2791, -5048}, {2806, -5097}, {2790, -5240}, {2813, -5270}, 
	 	{2818, -5288}, 
	 (*LH-LS*){2805, -5308}, {2833, -5290}, {2846, -5263}, {2848, -5290}, 
	 (*LS-WI*){2794, -5425}}}

WorldData/: WorldData["Minnesota"] = 
   {{(*LS-WI*){2806, -5526}, {2800, -5537}, {2765, -5537}, {2740, -5573},
	 	{2725, -5559}, {2685, -5568}, {2627, -5474}, 
	 (*WI-IA*){2610, -5473}, 
	 (*IA-SD*){2610, -5787}, {2718, -5787}, {2737, -5811}, {2744, -5800},
	 (*SD-ND*){2756, -5794}, 
	 (*ND-MAN*){2940, -5834}, {2940, -5719},
	 (*MAN-LS*){2880, -5375}, 
	 (*LS-WI*){2806, -5526}}}

WorldData/: WorldData["Mississippi"] = 
   {{(*TN-AL*){2100, -5292}, {2094, -5286}, {1973, -5308}, 
     (*AL-GOM*){1823, -5304}, 
	 (*GOM-LA*){1811, -5371},{1860, -5383}, {1860, -5496}, {1940, -5455}, 
	 (*LA-AR*){1980, -5470}, {2020, -5474}, 
	 (*AR-TN*){2100, -5418}, 
	 (*TN-AL*){2100, -5292}}}

WorldData/: WorldData["Missouri"] = 
   {{(*KY-IL*){2219, -5348}, {2223, -5363}, {2238, -5371}, {2262, -5371},
	 	{2297, -5422}, {2331, -5407}, {2337, -5440}, {2345, -5444}, 
		{2383, -5481}, {2410, -5491}, 
	 (*IL-IA*){2423, -5486}, {2437, -5503}, 
	 (*IA-NE*){2435, -5746},
	 (*NE-KS*){2400, -5719}, {2385, -5691}, {2374, -5707}, {2353, -5690},
	 	{2349, -5676}, 
	 (*KS-OK*){2220, -5677},
	 (*OK-AR*){2190, -5677}, {2190, -5410}, {2180, -5404}, {2160, -5422}, 
	 (*AR-TN*){2160, -5384},
	 (*TN-KY*){2190, -5370}, {2200, -5350}, 
	 (*KY-IL*){2219, -5348}}}

WorldData/: WorldData["Montana"] = 
   {{(*SD-WY*){2700, -6243}, {2700, -6663}, 
	 (*WY-ID*){2669, -6663}, {2684, -6680}, {2682, -6689}, {2673, -6689},
	 	{2672, -6741}, {2667, -6743}, {2670, -6763}, {2662, -6771},
		{2669, -6795}, {2667, -6800}, {2742, -6838}, {2728, -6860},
		{2734, -6873}, {2800, -6860}, {2800, -6878}, {2846, -6945},
		{2862, -6944}, {2879, -6963}, 
	 (*ID-ALB*){2940, -6963},
	 (*ALB-SAS*){2940, -6600},
	 (*SAS-SD*){2940, -6243},
     (*SD-WY*){2700, -6243}}}

WorldData/: WorldData["Nebraska"] = 
   {{(*WY-SD*){2580, -6243}, {2580, -5910}, {2566, -5880}, {2572, -5870},
	 	{2571, -5835}, 
	 (*SD-IA*){2549, -5788}, 
	 (*IA-MO*){2435, -5746}, 
	 (*MO-KS*){2400, -5719}, 	 
	 (*KS-CO*){2400, -6123}, {2460, -6123},
	 (*CO-WY*){2460, -6243},
	 (*WY-SD*){2580, -6243}}}

WorldData/: WorldData["Nevada"] = 
   {{(*UT-AZ*){2220, -6843}, {2172, -6843}, {2162, -6849}, {2162, -6856}, 
	 	{2169, -6863}, {2166, -6885},   
	 (*AZ-CA*){2100, -6878}, {2340, -7200},
	 (*CA-OR*){2520, -7200},
	 (*OR-ID*){2520, -7022}, 
	 (*ID-UT*){2520, -6843},     
	 (*UT-AZ*){2220, -6843}}}

WorldData/: WorldData["New Hampshire"] = 
   {{(*MA-VT*){2564, -4348}, {2659, -4322}, {2669, -4290}, 
	 (*VT-QUE*){2701, -4290}, {2715, -4284}, 
	 (*QUE-ME*){2718, -4265}, {2603, -4259}, 
	 (*ME-AO*){2583, -4243},
	 (*AO-MA*){2572, -4249}, {2562, -4278},
	 (*MA-VT*){2564, -4348}}} 

WorldData/: WorldData["New Jersey"] = 
   {{(*DE-PA*){2388, -4526}, {2408, -4484}, {2436, -4511},
     (*PA-NY*){2482, -4481}, 
	 (*NY-AO*){2460, -4434},{2430, -4456}, {2425, -4439}, {2370, -4460},
	 (*AO-NCM*){2340, -4490}, 
	 (*NCM-DE*){2377, -4535}, 
	 (*DE-PA*){2388, -4526}}} 

WorldData/: WorldData["New Mexico"] = 
   {{(*OK-TX*){2190, -6180}, {2190, -6183}, {1920, -6184}, {1920, -6397},
	 	{1909, -6396},
	 (*TX-MEX*){1907, -6392}, {1907, -6492}, {1881, -6492}, 
	 (*MEX-AZ*){1881, -6543},
	 (*AZ-UT-CO*){2220, -6543},
	 (*CO-OK*){2220, -6180},
	 (*OK-TX*){2190, -6180}}}

WorldData/: WorldData["New York"] = 
   {{(*NJ-PA*){2482, -4481}, {2520, -4521}, {2520, -4786},
	 (*PA-LE*){2536, -4786}, {2567, -4731},
	 (*LE-ONT*){2573, -4734}, 
	 (*ONT-LO*){2596, -4743}, {2596, -4620}, {2612, -4574}, 
	 (*LO-QUE*){2646, -4582}, {2700, -4500}, 
	 (*QUE-VT*){2701, -4401},
	 (*VT-MA*){2565, -4396},
	 (*MA-CN*){2523, -4410}, {2478, -4413}, {2473, -4409}, {2466, -4423}, 
	 (*CN-AO*){2460, -4419}, {2442, -4441},
	 (*AO-NJ*){2460, -4434},
	 (*NJ-PA*){2482, -4481}},
	 (*Long Island*){{2442, -4441}, {2436, -4440}, {2444, -4380}, {2464, -4312},
	 	{2460, -4419}, {2442, -4441}}} 

WorldData/: WorldData["North Carolina"] = 
   {{(*AO-SC*){2031, -4712}, {2108, -4780}, {2089, -4848}, {2109, -4860}, 
   		{2112, -4940}, 
     (*SC-GA*){2100, -4987}, 
	 (*GA-TN*){2099, -5060}, 
	 (*TN-VA*){2196, -4900}, 
	 (*VA-AO*){2193, -4560}, {2100, -4590}, 
	 (*AO-SC*){2031, -4712}}}

WorldData/: WorldData["North Dakota"] = 
   {{(*SD-MN*){2756, -5794}, 
	 (*MN-MAN*){2940, -5834},
	 (*MAN-SAS*){2940, -6082},
	 (*SAS-MT*){2940, -6243}, 
	 (*MT-SD*){2757, -6243}, 
	 (*SD-MN*){2756, -5794}}}
	 
WorldData/: WorldData["Ohio"] = 
   {{(*WV-KY*){2305, -4956}, 
	 (*KY-IN*){2346, -5089},
	 (*IN-MI*){2502, -5088}, 
	 (*MI-LE*){2504, -5008}, {2483, -4950}, {2489, -4904}, {2506, -4887},
	 (*LE-PA*){2519, -4831},
	 (*PA-WV*){2438, -4831}, {2383, -4850},
	 (*WV-KY*){2305, -4956}}}

WorldData/: WorldData["Oklahoma"] = 
   {{(*MO-AR*){2190, -5677}, {2184, -5666}, 
	 (*AR-TX*){2018, -5669}, {2038, -5714}, {2022, -5780}, {2074, -6000},
	 	{2190, -6000}, 
	 (*TX-NM*){2190, -6180}, 
	 (*NM-CO*){2220, -6180},
	 (*CO-KS*){2220, -6123},
	 (*KS-MO*){2220, -5677},
	 (*MO-AR*){2190, -5677}}}
	 
WorldData/: WorldData["Oregon"] = 
   {{(*CA-PO*){2520, -7453}, {2526, -7461}, {2539, -7466}, {2547, -7466},
     	{2544, -7463}, {2550, -7465}, {2560, -7474}, {2587, -7466},
	 	{2598, -7464}, {2601, -7460}, {2613, -7454}, {2640, -7448}, 
	 	{2681, -7444}, {2690, -7444}, {2726, -7439}, {2780, -7436},
	 (*PO-WA*){2795, -7444}, {2793, -7432}, {2796, -7410}, {2794, -7406},
	 	{2791, -7406}, {2764, -7373}, {2740, -7366}, {2733, -7335},
		{2742, -7309}, {2742, -7220}, {2760, -7139}, 
	 (*WA-ID*){2760, -7015}, {2737, -6988}, {2729, -6973}, {2718, -6973},
	 	{2658, -7034}, {2649, -7014}, {2628, -7022},
	 (*ID-NV*){2520, -7022},
	 (*NV-CA*){2520, -7200},
	 (*CA-PO*){2520, -7453}}}

WorldData/: WorldData["Pennsylvania"] = 
   {{(*WV-MD*){2383, -4769}, 
	 (*MD-DE*){2383, -4547}, {2390, -4538},
	 (*DE-NJ*){2388, -4526}, {2408, -4484}, {2436, -4511},
	 (*NJ-NY*){2482, -4481}, {2520, -4521}, {2520, -4786},
	 (*NY-LE*){2536, -4786}, 
	 (*LE-OH*){2519, -4831},
	 (*OH-WV*){2438, -4831}, {2383, -4831}, 
	 (*WV-MD*){2383, -4769}}} 

WorldData/: WorldData["Rhode Island"] = 
   {{(*AO-CN*){2479, -4312},
	 (*CN-MA*){2521, -4308}, {2521, -4283}, {2510, -4280}, {2500, -4268},
	 (*MA-AO*){2490, -4267}, 
	 (*AO-CN*){2479, -4312}}} 

WorldData/: WorldData["South Carolina"] = 
   {{(*NC-GA*){2100, -4987}, {2083, -5001}, 
     (*GA-AO*){1921, -4853}, 
   	 (*AO-NC*){2031, -4712}, {2108, -4780}, 
     	{2089, -4848}, {2109, -4860}, {2112, -4940}, 
	 (*NC-GA*){2100, -4987}}}

WorldData/: WorldData["South Dakota"] = 
   {{(*IA-MN*){2610, -5787}, {2718, -5787}, {2737, -5811}, {2744, -5800},
	 (*MN-ND*){2756, -5794}, 
	 (*ND-MT*){2757, -6243},
	 (*MT-WY*){2700, -6243},
	 (*WY-NE*){2580, -6243}, {2580, -5910}, {2566, -5880}, {2572, -5870},
	 	{2571, -5835}, 
	 (*NE-IA*){2549, -5788}, {2610, -5795},
	 (*IA-MN*){2610, -5787}}}

WorldData/: WorldData["Tennessee"] = 
   {{(*GA-NC*){2099, -5060}, 
     (*NC-VA*){2196, -4900}, 
	 (*VA-KY*){2196, -5020}, {2199, -5283}, {2190, -5282}, 
	 (*KY-MO*){2190, -5370}, 
	 (*MO-AR*){2160, -5384}, 
	 (*AR-MS*){2100, -5418}, 
	 (*MS-AL*){2100, -5292}, 
	 (*AL-GA*){2099, -5136}, 
	 (*GA-NC*){2099, -5060}}}

WorldData/: WorldData["Texas"] = 
   {{(*GOM-LA*){1782, -5630}, {1860, -5615}, {1920, -5643}, 
     (*LA-AR*){1981, -5643},   
	 (*AR-OK*){2013, -5643}, {2018, -5669}, {2038, -5714}, {2022, -5780}, 
	 	{2074, -6000}, {2190, -6000}, 
	 (*OK-NM*){2190, -6180}, {2190, -6183}, {1920, -6184}, {1920, -6397},
	 	{1909, -6396},
	 (*NM-MEX*){1907, -6392}, {1838, -6299}, {1794, -6280}, {1774, -6265},
	 	{1740, -6199}, {1738, -6190}, {1784, -6160}, {1786, -6143},
	 	{1793, -6140}, {1786, -6085}, {1776, -6048}, {1651, -5971}, 
		{1586, -5947}, {1566, -5896}, {1561, -5859}, {1550, -5843},
		{1555, -5842}, 
	 (*MEX-GOM*){1558, -5830}, {1575, -5842}, {1625, -5850}, {1662, -5860},
	   	{1665, -5853}, {1670, -5853}, {1673, -5856}, {1669, -5851}, 
		{1699, -5808}, 
	 (*GOM-LA*){1782, -5630}}}

WorldData/: WorldData["Utah"] = 
   {{(*WY-CO*){2460, -6543},
     (*CO-NM-AZ*){2220, -6543},
	 (*AZ-NV*){2220, -6843},
	 (*NV-ID*){2520, -6843},
	 (*ID-WY*){2520, -6663}, {2460, -6663},
     (*WY-CO*){2460, -6543}}}

WorldData/: WorldData["Vermont"] = 
   {{(*NY-MA*){2565, -4396},
   	 (*MA-NH*){2564, -4348}, {2659, -4322}, {2669, -4290}, 
	 (*NH-QUE*){2701, -4290}, 
	 (*QUE-NY*){2701, -4401},
	 (*NY-MA*){2565, -4396}}} 

WorldData/: WorldData["Virginia"] = 
   {{(*TN-NC*){2196, -4900}, 
     (*NC-AO*){2193, -4560},
	 (*AO-CB*){2215, -4560}, 
	 (*CB-MD*){2273, -4574}, {2300, -4635}, 
	 (*MD-DC 1*){2327, -4622}, {2333, -4622}, 
	 (*DC-MD 2*){2336, -4627}, 
	 (*MD-WV*){2359, -4664}, {2348, -4650}, {2368, -4701},
	 	{2315, -4780}, {2250, -4820}, {2232, -4837}, 
	 (*WV-KY*){2252, -4918}, 
	 (*KY-TN*){2196, -5020},
	 (*TN-NC*){2196, -4900}},
	 (*CB-NJ*){{2278, -4539}, 
	 (*NJ-AO*){2281, -4522}, 
	 (*AO-CB*){2228, -4555},
	 (*CB-NJ*){2278, -4539}}}
	 
WorldData/: WorldData["Washington"] = 
   {{(*PO-OR*){2795, -7444}, {2793, -7432}, {2796, -7410}, {2794, -7406},
	 	{2791, -7406}, {2764, -7373}, {2740, -7366}, {2733, -7335},
		{2742, -7309}, {2742, -7220}, {2760, -7139}, 
	 (*OR-ID*){2760, -7015}, {2765, -7019},	{2770, -7015}, {2781, -7024},
	 	{2786, -7022},  
	 (*ID-BC*){2940, -7022}, 
	 (*BC-PS*){2940, -7366}, {2887, -7342}, {2880, -7333}, {2868, -7343},
	 	{2863, -7342}, {2860, -7346}, {2856, -7340}, {2855, -7345}, 
		{2875, -7352}, {2888, -7365}, {2890, -7440}, 
	 (*PS-PO*){2903, -7484}, {2874, -7478}, {2865, -7466}, 
	 (*PO-OR*){2795, -7444}}}

WorldData/: WorldData["West Virginia"] = 
   {{(*MD-VA*){2359, -4664}, {2348, -4650}, {2368, -4701},
	 	{2315, -4780}, {2250, -4820}, {2232, -4837},
     (*VA-KY*){2252, -4918}, {2261, -4938}, {2290, -4959}, 
	 (*KY-OH*){2305, -4956}, {2383, -4850},
	 (*OH-PA*){2438, -4831}, {2383, -4831}, 
	 (*PA-MD*){2383, -4769}, {2352, -4769}, {2382, -4690},
	 (*MD-VA*){2359, -4664}}}

WorldData/: WorldData["Wisconsin"] = 
   {{(*IA-IL*){2550, -5439},
	 (*IL-LM*){2550, -5268}, {2567, -5266}, {2582, -5274}, {2645, -5260}, 
	 	{2717, -5222}, {2672, -5280}, 
	 (*LM-ONT*){2706, -5255}, {2755, -5287}, {2780, -5407}, {2790, -5412},
	 (*ONT-LS*){2794, -5425}, {2800, -5443}, {2795, -5457}, {2817, -5450},
	 (*LS-MN*){2806, -5526}, {2800, -5537}, {2765, -5537}, {2740, -5573},
	 	{2725, -5559}, {2685, -5568}, {2627, -5474}, 
	 (*MN-IA*){2610, -5473}, {2565, -5464}, {2558, -5443},
	 (*IA-IL*){2550, -5439}}}

WorldData/: WorldData["Wyoming"] = 
   {{(*UT-CO*){2460, -6543},
     (*CO-NE*){2460, -6243},
	 (*NE-SD*){2580, -6243},
	 (*SD-MT*){2700, -6243}, {2700, -6663}, 
	 (*MT-ID*){2669, -6663},
	 (*ID-UT*){2520, -6663}, {2460, -6663},
     (*UT-CO*){2460, -6543}}}

EndPackage[]
