(*:Copyright: Copyright 1988-2005 Wolfram Research, Inc. *)

(*:Mathematica Version: 5.0 *)

(*:Package Version: 1.0 *)

(*:Name: NumberTheory`Frobenius` *)

(*:Title: Functions for solving the Frobenius equation and computing
          Frobenius numbers.
*) 

(*:Authors: David Einstein, Daniel Lichtblau, Adam Strzebonski, Stan Wagon *)

(*:Keywords: Frobenius equation, Frobenius number *)

(*:Sources: David Einstein, Daniel Lichtblau, Adam Strzebonski, Stan Wagon,
"Frobenius Numbers by Integer-Linear Programming"
*)

(*:Summary:
The Frobenius equation has the from a1 x1+...+an xn==b, where a1, ..., an
are positive integers, b is an integer, and the solutions {x1, ..., xn} are
required to have nonnegative integer coordinates. The package introduces
the following functions:

FrobeniusInstance[{a1, ..., an}, b] gives a solution instance or {},
FrobeniusSolve[{a1, ..., an}, b] gives a list of all solutions,
FrobeniusF[{a1, ..., an}] gives the Frobenius number, the largest b for
   which the Frobenius equation a1 x1+...+an xn==b has no solutions.
*)

BeginPackage["NumberTheory`Frobenius`"]

FrobeniusInstance::usage = "FrobeniusInstance[{a1, ..., an}, b] gives a solution of the Frobenius equation a1 x1+...+an xn==b. If there are no solutions gives an empty list." 
FrobeniusSolve::usage = "FrobeniusSolve[{a1, ..., an}, b] gives a list of all solutions of the Frobenius equation a1 x1+...+an xn==b." 
FrobeniusF::usage = "FrobeniusF[{a1, ..., an}] gives the Frobenius number of {a1, ..., an}." 
FrobeniusInstance::frcf = FrobeniusSolve::frcf = FrobeniusF::frcf = "`1` is not a list of at least two positive integers." 

Unprotect[FrobeniusInstance]
Unprotect[FrobeniusSolve]
Unprotect[FrobeniusF]

Begin["`Private`"]

FrobeniusCoeffsQ[aa_, head_] :=
   If[ListQ[aa] && Length[aa]>=2 && And@@(IntegerQ/@aa) && And@@(Positive/@aa),
      True,
      Message[head::frcf, aa];
      False]

(* FrobeniusInstance with aa sorted and relatively prime, and b positive. *)

finst0[aa_, b_] :=
   Module[{tmp, sol, nulls},
      If[aa[[1]]<=("MaxFrobeniusGraph"/.Internal`ReduceOptions[]),
         sol=Reduce`FrobeniusInstance[aa, b];
         If[!ListQ[sol], Return[$Failed], Return[sol]]];
      tmp=Reduce`LinearDiophantineSolve[{aa}, {b}];
      If[!ListQ[tmp], Return[$Failed]];
      {sol, nulls}=tmp;
      If[sol==={}, Return[{}]];
      Internal`DeactivateMessages[
         System`DiophantineDump`LatticeNNInstance[sol, nulls]]]

finst[aa_, b_] :=
   Module[{g, sol, tmp},
      If[b<0, Return[{}]];
      If[b==0, Return[Table[0, {Length[aa]}]]];   
      g=GCD@@aa;
      If[!IntegerQ[b/g], Return[{}]]; 
      tmp=Transpose[Sort[Transpose[{aa, Range[Length[aa]]}]]];
      sol=finst0[tmp[[1]]/g, b/g];
      If[!ListQ[sol], Return[$Failed]];
      If[sol==={}, Return[{}]];
      Transpose[Sort[Transpose[{tmp[[2]], sol}]]][[2]]]
         
FrobeniusInstance[aa_?(FrobeniusCoeffsQ[#, FrobeniusInstance]&), b_Integer] :=
   With[{ans=finst[aa, b]}, ans /; ListQ[ans]]

fsolve[aa_, b_] :=
   Module[{g, sol, nulls, tmp, op},
      If[b<0, Return[{}]];
      If[b==0, Return[{Table[0, {Length[aa]}]}]];   
      g=GCD@@aa;
      If[!IntegerQ[b/g], Return[{}]]; 
      tmp=Reduce`LinearDiophantineSolve[{aa/g}, {b/g}];
      If[!ListQ[tmp], Return[$Failed]];
      {sol, nulls}=tmp;
      If[sol==={}, Return[{}]];
      Internal`WithLocalSettings[
         op="ReduceOptions"/.Developer`SystemOptions[];
         Developer`SetSystemOptions["ReduceOptions"->{
            "DiscreteSolutionBound"->Infinity}],
         tmp=System`DiophantineDump`ICIntLinReduce[Transpose[nulls], -sol],
         Developer`SetSystemOptions["ReduceOptions"->op]];
      If[tmp===$Failed, Return[$Failed]];
      Sort[(sol+#.nulls)&/@tmp]]
        
FrobeniusSolve[aa_?(FrobeniusCoeffsQ[#, FrobeniusSolve]&), b_Integer] :=
   With[{ans=Internal`DeactivateMessages[fsolve[aa, b]]}, ans /; ListQ[ans]]

HomogeneousBasisReduced[A_] :=
   With[{tmp=Reduce`LinearDiophantineSolve[{A}, {0}]},
      If[!ListQ[tmp] || Length[tmp]!=2, 
         $Failed, 
         LatticeReduce[Drop[#, 1]&/@tmp[[2]]]]]

BoundingBoxILP[A_, V_, j_]:=
   Module[{bt, mat, b, ans, B}, 
      B=RotateLeft[Rest[A], j];
      bt=Transpose[V];
      mat=Join[{bt[[1]], -bt[[1]], Total[B bt], 
                Total[bt B]-Total[Take[bt,-j]]}, -Rest[bt]];
      b=Join[{1, -A[[1]], 0, 1}, Table[0, {Length[bt]-1}]];
      Reduce`LinearDiophantineSolve;
      ans=System`DiophantineDump`BBIntLinMinimize[bt[[1]], mat, b];
      If[!ListQ[ans] || Length[ans]!=2, Return[$Failed]];
      RotateRight[ans[[2]].V, j]];

AxialProtoElbows[A_, V_] := 
   Table[BoundingBoxILP[A, Nest[RotateLeft, #, j]&/@V, j], 
      {j, 0, Length[A]-2}]

AxialElbows[A_, V_] :=
   Table[BoundingBoxILP[A, Nest[RotateLeft, #, j]&/@V, j][[j+1]],
      {j, 0, Length[A]-2}]

sumgood[v_List] := Positive[Plus@@v] || 
   With[{ss=Select[v, #!=0&, 1]}, ss=!={} && Negative[ss[[1]]]]

ClearNegsAndDeleteZeroVector[{}] := {}

ClearNegsAndDeleteZeroVector[vecs_] := 
   Union[DeleteCases[(vecs+Abs[vecs])/2, Table[0, {Length[vecs[[1]]]}]]] 

SetAttributes[ProtoToPre, HoldFirst]
ProtoToPre[biglist_, B_, zero_, abds_] :=
   Module[{i, v},
      Do[v=biglist[[i]];
         If[v===zero || !sumgood[v],
            biglist[[i]]=abds,
            v=(v+Abs[v])/2;
            If[v===zero,
               biglist[[i]]=abds,
               biglist[[i]]=v/B]],
         {i, Length[biglist]}]]

SetAttributes[HeavyToPre, HoldFirst]
HeavyToPre[biglist_, B_, bd_, zero_, abds_] :=
   Module[{i, v},
      Do[v=biglist[[i]];
         If[v.B>bd,
            biglist[[i]]=abds,
            v=(v+Abs[v])/2;
            If[v===zero,
               biglist[[i]]=abds,
               biglist[[i]]=v]],
         {i, Length[biglist]}]]

farthest[corns_, A_] := 
   Fold[If[#2.Rest[A]>#1[[2]], {#2, #2.Rest[A]}, #1]&, 
        {Table[0, {Length[A]-1}], 0}, corns][[1]]

FarthestCornerSub[A_, elbows_, currfar_, backdata_] /; Length[elbows[[1]]]>2 := 
   Module[{p1, p2, B, maxes},
      Flatten[Map[(
         {p1,p2}={#[[1]], Rest[#]};
         B=Rest/@Select[elbows, First[#]<p1&];
         B=Sort[Internal`ListMin[ClearNegsAndDeleteZeroVector[#-p2&/@B]]];
         If[B==={}, 
            {},
            maxes=Prepend[Max/@Transpose[B], 0]; 
            If[(PadLeft[maxes+#, Length[A]-1]+backdata).Rest[A]>currfar,
               (Prepend[#+p2, p1]&/@FarthestCornerSub[
                   A, B, currfar, PadLeft[#, Length[A]-1]+backdata]),
               {}]])&,
         Select[elbows, First[#]>0&]], 1]]

FarthestCornerSub[_, elbows_, ___]  /; Length[elbows[[1]]]==2 := 
   Partition[Take[Flatten[Reverse/@Reverse[elbows]], {2, -2}], 2]

FarthestCorner[A_, elbows_] /; Length[elbows[[1]]]>2 :=  
   Module[{pts, p1, p2, B, cc, farvertex, maxes, A1, currfar},
      A1=Rest[A];
      currfar=farthest[elbows, A].A1;
      pts=Select[elbows, #[[1]]>0&];
      Scan[(
         {p1, p2}={First[#], Rest[#]};
         B=Rest/@Select[elbows, #[[1]]<p1&];
         B=Sort[Internal`ListMin[ClearNegsAndDeleteZeroVector[#-p2&/@B]]];
         maxes=Prepend[Max/@Transpose[B], 0];
         If[(maxes+#).A1>currfar,
            cc=FarthestCornerSub[A, B, currfar, #]; 
            farvertex=farthest[Prepend[#+p2, p1]&/@cc, A];
            currfar=Max[currfar, farvertex.A1]])&,
         pts]; 
      currfar-Total[A]]

FarthestCorner[A_, elbows_] /; Length[elbows[[1]]]==2 := 
   farthest[FarthestCornerSub[A, elbows], A].Rest[A]-Total[A]
   
MinWeightLattice[A_, V_] := 
   Module[{B, VT, b, mat, c, ans},
      B=Rest[A];
      VT=Transpose[V];
      b=Join[0&/@B, {1, -A[[1]] Total[B]}];
      mat=Join[VT, {Total/@V, -Total[B VT]}];
      c=Total[B VT];
      Reduce`LinearDiophantineSolve;
      ans=System`DiophantineDump`BBIntLinMinimize[c, mat, b];
      If[ListQ[ans] && Length[ans]==2, ans[[1]], $Failed]]

FrobeniusFElbows[{a_, b_}] := a b - a - b;

FrobeniusFElbows[{a_, b_, c_}] := 
   Module[{x1, x2, y1, y2, V, tmp},
      V=HomogeneousBasisReduced[{a, b, c}];
      If[!ListQ[V], Return[$Failed]];
      tmp=AxialProtoElbows[{a, b, c}, V];
      If[!FreeQ[tmp, $Failed], Return[$Failed]];
      {{x1, y1}, {x2, y2}}=tmp;
      Max[y1 c, x2 b]+(x1-1) b+(y2-1) c-a]
     
FrobeniusFElbows[A_] /; Length[A]>3 := 
   Module[{V, VT, wt, B=Rest[A], mat, b, proto, elbows, op, ans},
      V=HomogeneousBasisReduced[A];
      If[!ListQ[V], Return[$Failed]];
      abds=AxialElbows[A, V];
      If[!FreeQ[abds, $Failed], Return[$Failed]];
      wt=MinWeightLattice[A, V]/A[[1]];
      VT=Transpose[V];
      mat=Join[VT, -VT, {V.B/A[[1]], -V.B/A[[1]]}];
      b=Join[-abds, -abds+1, {0, -wt}];
      Internal`WithLocalSettings[
         op="ReduceOptions"/.Developer`SystemOptions[];
         Developer`SetSystemOptions["ReduceOptions"->{
            "BranchLinearDiophantine"->False,
            "DiscreteSolutionBound"->Infinity}],
         proto=(B (VT.#))&/@Reduce`PolyhedronIntegerPoints[mat, b],
         Developer`SetSystemOptions["ReduceOptions"->op]];
      ProtoToPre[proto, B, Table[0, {Length[B]}], abds];
      elbows=Union[DiagonalMatrix[abds], Internal`ListMin[proto]];
      proto=.;
      ans=FarthestCorner[A, elbows];  
      b=finst0[A, ans];
      If[b==={}, Return[ans]];
      mat=Join[VT, -VT, {V.B/A[[1]]}];
      b=Join[-abds, -abds+1, {wt+1}];
      Internal`WithLocalSettings[
         op="ReduceOptions"/.Developer`SystemOptions[];
         Developer`SetSystemOptions["ReduceOptions"->{
            "BranchLinearDiophantine"->False,
            "DiscreteSolutionBound"->Infinity}],
         proto=(VT.#)&/@Reduce`PolyhedronIntegerPoints[mat, b],
         Developer`SetSystemOptions["ReduceOptions"->op]];
      HeavyToPre[proto, B, ans+A[[1]], Table[0, {Length[B]}], abds];
      elbows=Sort[Internal`ListMin[Join[elbows, proto]]];
      proto=.;
      ans=FarthestCorner[A, elbows];        
      ans]

frobeniusf[A_] :=
   If[(GCD@@A)=!=1,
      Infinity,
      If[A[[1]]<=("MaxFrobeniusGraph"/.Internal`ReduceOptions[]),
         Reduce`FrobeniusF[A],
         Internal`DeactivateMessages[FrobeniusFElbows[A]]]]

FrobeniusF[aa_?(FrobeniusCoeffsQ[#, FrobeniusF]&)] :=
   With[{ans=frobeniusf[Sort[aa]]}, ans /; ans=!=$Failed]
         
End[] (* `Private` *)

Attributes[FrobeniusInstance] = {Protected, ReadProtected}
Attributes[FrobeniusSolve] = {Protected, ReadProtected}
Attributes[FrobeniusF] = {Protected, ReadProtected}

EndPackage[]  (* NumberTheory`Frobenius` *)

            
