(*:Mathematica Version: 5.0 *)

(*:Copyright: Copyright 1992-2005, Wolfram Research, Inc. *)

(*:Name: NumberTheory`NumberTheoryFunctions` *)

(*:Title: Number Theory Functions *)

(*:History:
Original package by Ilan Vardi, Wolfram Research, Inc., 1992.
Support added for cases where PrimeQ returns True for composites,
	ECM, Wolfram Research, Inc., 1993.
SumOfSquaresR and SumOfSquaresRepresentations by Stan Wagon,
	Macalester College, 1995.
Enhancements to QuadraticRepresentation, ECM,
	Wolfram Research, Inc., March 1997.
ChineseRemainderTheorem by Stan Wagon, Macalester College, and
        Daniel Lichtblau, Wolfram Research, Inc., Jan. 1998.  Code is
        based in part on earlier code by Mike McGeachie and Craig Ortner,
        students at Macalester College.
Renamed ChineseRemainderTheorem to be ChineseRemainder, retaining support
	for older symbol, ECM, WRI, Feb. 1998.
Reimplemented PrimitiveRoot as a kernel function - Mark Sofroniou,
	Wolfram Research, Inc., April. 1999.
Moved ChineseRemainder code and parts of SqrtMod code to StartUp/Algebra/
	RNumberTheory.m, so that they can be used by Reduce -
	Adam Strzebonski, Wolfram Research, Inc., March 2000.
KroneckerSymbol by David Terr, Wolfram Research, Inc., Jan. 2001.
ClassNumber modified by David Terr, Wolfram Research, Inc., Jan. 2001. 
Moved SumOfSquaresRepresentations code to StartUp/Algebra/RNumberTheory.m, 
	so that it can be used by Reduce -
	Adam Strzebonski, Wolfram Research, Inc., January 2001.
SqrtMod modified by David Terr, Wolfram Research, Inc., Feb. 2001.	
SqrtModList by David Terr, Wolfram Research, Inc., Feb. 2001.
FundamentalDiscriminantQ by David Terr, Wolfram Research, Inc., June 2001.
PreviousPrime by David Terr, Wolfram Research, Inc., Sept. 2001.	
Random[Prime,{a,b}] by David Terr, Wolfram Research, Inc., Oct. 2001.
WhichRootOfUnity[a] by David Terr, Wolfram Research, Inc., Dec. 2001.	
AliquotSequence by David Terr, Wolfram Research, Inc., Jan. 2002.	
AliquotCycle by David Terr, Wolfram Research, Inc., Jan. 2002.
LeastPrimeFactor by	David Terr, Wolfram Research, Inc., Feb. 2002.
PrimePowerQ by David Terr, Wolfram Research, Inc., Feb. 2002.
I removed the restriction that the discriminant be fundamental in ClassList.
	David Terr, Wolfram Research, Inc., Aug. 2002.
PrimeFactorList by David Terr, Wolfram Research, Inc., Aug. 2002.
OrderedSumOfSquaresRepresentations (former SumOfSquaresRepresentations with
	order of terms reversed) by David Terr, Wolfram Research, Inc., 
	Jan. 2003
SumOfSquaresRepresentations modified by David Terr, Wolfram Research, Inc.,
	Jan. 2003 
Removed PrimitiveRoot, which is now in numbertheory.mc - David Terr.
	Wolfram Research, Inc., Feb. 2003. 
Modified ClassNumber to work with positive as well as negative discriminants 
	David Terr, Wolfram Research, Inc., Mar. 2003.
Sped up ClassNumber for positive discriminants by a factor of 3 by using my own 
	regulator.  David Terr, Wolfram Research, Inc., Mar. 2003. *)

(*:Keywords:
	number theory, Chinese Remainder Theorem, class number, primitive roots,
	Fundamental discriminant, Aliquot sequence 
*)

(*:Requirements: None. *)

(*:Warnings:
QuadraticRepresentation needs to have -d
a quadratic residue only for primes appearing to odd powers.

For SumOfSquaresR[d, n], the cases d = 2, 4, 6, 8 can handle large
integer values of n, so long as n can be factored. The
other cases use recursion, calling (d-1, m) where m takes
on Sqrt[n] many values. Thus only modest size values of
m and d can be used.
*)

(*:Limitations:
SquareFreeQ, SqrtMod, SqrtModList, and QuadraticRepresentation 
depend on obtaining the factorization of n. 
       
For a discussion of when QuadraticRepresentation returns
an answer, see Cox's book.

ClassList and also ClassNumber have only been implemented
for negative discriminants. The implementation uses
the simplest algorithm and is slow for large inputs.

For SumOfSquaresR[d, n], d = 2, 4, 6, or 8, some simple formulas are called,
so n can be quite large (its factorization will be used).  For other
values of d a recursion is used that calls r[d-1, n] approximately Sqrt[n]
times, so n must not be too large.
*)

(*:Sources: 

D.A. Buell, Binary Quadratic Forms, Springer-Verlag, 1989.

D.A. Cox, Primes of the Form x^2 + n y^2, Wiley, 1989.  

H. Cohen, A Course in Computational Algebraic Number Theory,  Springer-Verlag,
1993.

Emil Grosswald, Representations of Integers as
  Sums of Squares, Springer, New York, 1985.

G.H. Hardy and E.M. Wright, An Introduction to the Theory Of
  Numbers, Oxford University Press, 1988.

K. Hardy, J.B. Muskat, and K.S. Williams, A determinisitic
  algorithm for solving n = fu^2 + gv^2 in coprime integers
  u and v, Math. Comp. #55 (1990) 327-343.

D.E. Knuth, Seminumerical Algorithms, Addison-Wesley, 1981.

Stan Wagon, Mathematica in Action, Freeman, 1991.

Stan Wagon, The Euclidean Algorithm Strikes Again,
  American Math. Monthly # 97, (1990), 125-124. 

Stan Wagon, Complex Factoring, Mathematica in Education and Research, 1996,
  to appear.
*)

(*:Summary:
This package implements some standard functions from number theory.
*)


BeginPackage["NumberTheory`NumberTheoryFunctions`",
	"NumberTheory`FactorIntegerECM`"]
	
AliquotCycle::usage =
"AliquotCycle[n] gives the repeating cycle at the end of the Aliquot \
sequence of n." 	
	
AliquotSequence::usage =
"AliquotSequence[n] gives the Aliquot sequence {s_0, s_1, s_2, ..., s_k} \
of n, where s_0 = n and s_(i+1) is the sum of factors of s_i other than s_i. \
The sequence either terminates with s_k = 0, with s_k = s_j for some j<k, or \
tends to infinity, in which case it will run forever. (Although it has not been \
proven that the Aliquot sequence of n ever grows to infinity, it is strongly \
suspected that it does for certain n. The smallest number for which the \
fate of the Aliquot sequence is unknown is n=276.)" 	

Automatic::usage = "Automatic is the default value of the SumOfFactorsType option \
to AliquotSequence, AliquotCycle, and SumOfFactors, specifying to sum all factors \
of the number except the number itself."

Biunitary::usage = "Biunitary is an option value for the SumOfFactorsType option \
of AliquotSequence, AliquotCycle, and SumOfFactors, specifying that only divisors \
d of n except n are to be added such that if p^(2e) is the largest power of p \
dividing n for some prime p, then p^e is not the largest power of p dividing d."

ChineseRemainder::usage =
"ChineseRemainder[list1, list2] gives the minimal \
nonnegative integer solution of Mod[r, list2] == Mod[list1, list2], \
for list2 positive. The solution is unique modulo the LCM of list2.";

ChineseRemainderTheorem::usage =
"ChineseRemainderTheorem[list1, list2] is an obsolete function, \
superseded by ChineseRemainder[list1, list2].";

ClassList::usage = "ClassList[d] gives a list of inequivalent quadratic \
forms of discriminant d, where d < 0 is a fundamental discriminant, \
and a quadratic form a x^2 + b x y + c y^2 is represented as {a, b, c}."

ClassNumber::usage = "ClassNumber[d] gives the number of inequivalent \
quadratic forms of discriminant d, where d is a fundamental discriminant."

Exponential::usage = "Exponential is an option value for the SumOfFactorsType \
option of AliquotSequence, AliquotCycle, and SumOfFactors, specifying that only \
divisors d of n except n are to be added such that if p^c is the largest power \
of p dividing n and p^a is the largest power of p dividing d, then either a=0 \
or a divides c."

FundamentalDiscriminantQ::usage = "FundamentalDiscriminantQ[d] returns True \
if d is the discriminant of a quadratic number field, otherwise it returns \
False."

Infinitary::usage = "Infinitary is an option value for the SumOfFactorsType \
option of AliquotSequence, AliquotCycle, and SumOfFactors, specifying that \
only divisors d of n except n are to be added such that if p^c is the largest \
power of p dividing n and p^a is the largest power of p dividing d, then all \
corresponding binary digits of c that are zero are also zero for a."

KroneckerSymbol::usage = "KroneckerSymbol[a,b] gives the Kronecker (or \
Kronecker-Jacobi) symbol (a/b), which is a generalization of the Jacobi symbol \
in which b as well as a is an arbitrary integer."

LeastPrimeFactor::usage = "LeastPrimeFactor[n] returns the least prime \
factor of n for integer n with |n| > 1."

MaxTerms::usage = "MaxTerms specifies the maximum size of terms computed in \
the Aliquot sequence in AliquotSequence or AliquotCycle. Its default \
value is Infinity."

ModifiedExponential::usage = "ModifiedExponential is an option value for the \
SumOfFactorsType option of AliquotSequence, AliquotCycle, and SumOfFactors, \
specifying that only divisors d of n except n are to be added such that if p^c \
is the largest power of p dividing n and p^a is the largest power of p dividing d, \
then either a=0 or both a divides c and a+1 divides c+1."

NextPrime::usage = "NextPrime[n] gives the smallest prime p such that p > n."

OrderedSumOfSquaresRepresentations::usage = 
"OrderedSumOfSquaresRepresentations[d, n] gives \
the list of all representations of the positive integer n as a sum of d \
squares of nonnegative nonincreasing integers."

PreviousPrime::usage = "PreviousPrime[n] gives the largest prime p such that p < n."

PrimeFactorList::usage = "PrimeFactorList[n] returns the list of prime
factors of n for rational n, including -1 if n is negative."

PrimePowerQ::usage = "PrimePowerQ[q] yields True if \
q is a positive integer and a positive integer power of a prime, False otherwise."

PrimitiveRoot::usage = "For n an odd prime power, twice an odd prime power, or 4, \
PrimitiveRoot[n] returns the least positive integer a \
such that the multiplicative order of a mod n is EulerPhi[n]."

QuadraticRepresentation::usage = "QuadraticRepresentation[d, n] gives \
{x, y}, where x^2 + d y^2 = n for positive d and odd n."

ShowProgress::usage = "ShowProgress is an option to AliquotSequence and AliquotCycle. When \
set to True, the result of each iteration is printed out before the final sequence \
or cycle is output. The default value is False."

SqrtMod::usage = "SqrtMod[n, m] returns the least nonnegative integer \
from 0 to m-1 whose square is congruent to n modulo m, if such an integer \
exists, otherwise it returns unevaluated with an error message." 

SqrtModList::usage = "SqrtModList[n, m] gives the complete list of integers \
from 0 to m-1 whose squares are congruent to n modulo m." 

SquareFreeQ::usage = "SquareFreeQ[n] is True if n is a square-free \
integer, False otherwise."

SumOfFactors::usage = "SumOfFactors[n] returns the sum of the factors of n other \
than n itself. This is equal to DivisorSigma[1,n]-n. The option SumOfFactorsType may \
be used with this function, specifying the type of factor sum to use."

SumOfFactorsType::usage = "SumOfFactorsType is an option to SumOfFactors which specifies \
which type of divisor sum to use. The default type is \
Automatic, which is just the usual divisor sum. The other types are \
Unitary, Biunitary, Infinitary, Exponential, and ModifiedExponential."

SumOfSquaresR::usage = "SumOfSquaresR[d, n] gives the number of \
representations of a nonnegative integer n as a sum of d squares of integers; \
standard notation for this is r[d, n]."

SumOfSquaresRepresentations::usage = 
"SumOfSquaresRepresentations[d, n] gives \
the list of all representations of the positive integer n as a sum of d squares \
of integers."

TermIncrement::usage = "TermIncrement is an option for AliquotSequence \
and AliquotCycle which specifies the amount by which each term in the Aliquot \
sequence is to be incremented from its standard value (the sum of factors of the \
previous term). The value of this option may be any integer. The default value is 0. "

Unitary::usage = "Unitary is an option value for the SumOfFactorsType option of \
AliquotSequence, AliquotCycle, and SumOfFactors, specifying that only divisors \
d of n except n are to be added such that d and n/d are coprime."

WhichRootOfUnity::usage = "WhichRootOfUnity[a] returns {n,k} if \
a = Exp(2 Pi I k / n) for a (unique) pair of nonnegative coprime integers k and n with k<n, \
otherwise returns unevaluated."


(* Error messages *)

AliquotCycle::open = "No cycle was found. You may wish to enlarge your search by
increasing MaxIterations or MaxTerms.";

AliquotSequence::badarg = "The argument `1` is not a nonnegative integer.";

AliquotSequence::increment = "The value `1` of TermIncrement is not an integer.";

AliquotSequence::maxit = "The value `1` of MaxIterations is not a positive integer or Infinity.";

AliquotSequence::maxterms = "The value `1` of MaxTerms is not a positive integer or Infinity.";

AliquotSequence::progress = "The value `1` of ShowProgress is not True or False.";

ClassNumber::baddisc = "Argument `1` is not a fundamental discriminant.";

LeastPrimeFactor::badarg = "The argument `1` is not an integer greater \
than 1.";

LeastPrimeFactor::nargs = "LeastPrimeFactor called with `1` arguments, 1 is
expected.";

PrimeFactorList::badarg = "The argument `1` is not a rational number.";

Random::noprime = "There are no primes in the specified interval.";

SqrtMod::arg1 = "The value `1` of the first argument of SqrtMod is not an integer."

SqrtMod::arg2 = "The value `1` of the second argument of SqrtMod is not a positive \
integer."

SumOfFactors::badarg = "The argument `1` is not a nonnegative integer.";
SumOfFactors::type = "`1` is not a valid SumOfFactorsType.";

WhichRootOfUnity::nonroot = "The argument `1` is not a root of unity.";


Unprotect[AliquotCycle, AliquotSequence, Automatic, Biunitary, ChineseRemainder, 
	ChineseRemainderTheorem, ClassNumber, ClassList, Exponential,
	FundamentalDiscriminantQ, Infinitary, 
	KroneckerSymbol, LeastPrimeFactor, MaxIterations, MaxTerms, 
	ModifiedExponential, NextPrime, PreviousPrime, PrimePowerQ, PrimitiveRoot,
	PrimeFactorList, QuadraticRepresentation, Random, 
	ShowProgress, SquareFreeQ, SqrtMod, SqrtModList, 
	SumOfFactors, SumOfFactorsType, TermIncrement, Unitary, WhichRootOfUnity]

(* Set default options. *)

Options[AliquotCycle] =
  {SumOfFactorsType->Automatic, TermIncrement->0, ShowProgress->False,
  MaxIterations->Infinity, MaxTerms->Infinity};

Options[AliquotSequence] =
  {SumOfFactorsType->Automatic, TermIncrement->0, ShowProgress->False,
  MaxIterations->Infinity, MaxTerms->Infinity};

Options[SumOfFactors] =
  {SumOfFactorsType->Automatic};


Begin["`Private`"]


(* Functions to check for valid parameter values *)

ValidSumOfDivisorsQ[Automatic] = True;
ValidSumOfDivisorsQ[Unitary] = True;
ValidSumOfDivisorsQ[Biunitary] = True;
ValidSumOfDivisorsQ[Infinitary] = True;
ValidSumOfDivisorsQ[Exponential] = True;
ValidSumOfDivisorsQ[ModifiedExponential] = True;
ValidSumOfDivisorsQ[_] = False;


(* LeastPrimeFactor *)

lpf[n_] :=
    Module[{i, p, found = False},
    
      	If[n <= 1 || ! IntegerQ[n],
        	Message[LeastPrimeFactor::badarg, n];
        	Return[$Failed];
      	];
        
      	If[PrimeQ[n], 
			Return[n]
		];
      
      	For[i = 1, !found, i++,
        	p = Prime[i];
			
         	If[Mod[n, p] == 0,
          		found = True;
          	];
        ];
      	p
	];
	
LeastPrimeFactor[] := Null /; (Message[LeastPrimeFactor::nargs, 0]; False)
	
LeastPrimeFactor[a_, b__] := Null /; 
	(Message[LeastPrimeFactor::nargs, Length[{b}] + 1]; False)
	
LeastPrimeFactor[n_] :=
	With[{res = lpf[n]},
		res /; res =!= $Failed
	];


(* PrimeFactorList *)

PrimeFactorList[0 | 1] := {}

PrimeFactorList[n:(_Integer | _Rational)] :=
    FactorInteger[n][[All, 1]]

PrimeFactorList[any_] :=
    Null/;(Message[PrimeFactorList::badarg, any]; False)


(* PrimePowerQ *)

PrimePowerQ[q_] :=

    Module[{p, q1},
	
		If[ !IntegerQ[q], Return[False] ];
      	If[ q < 2, Return[False] ];
      	If[ PrimeQ[q], Return[True] ];
      	p = LeastPrimeFactor[q];
      	q1 = q/p;
      	While[ Mod[q1, p] == 0, q1 /= p ];
      	q1 == 1
		
	];


(* KroneckerSymbol *)

(* Reference: Cohen, A Course In Computational Algebraic Number Theory,
Springer-Verlag (1993), p. 28 *)

KroneckerSymbol[a_Integer, b_Integer] :=
	Module[{res},	
		If[ b == 0,		
			If[ Abs[a] == 1,				
				res = 1,				
				res = 0;				
			],			
			If[ b == 2,			
				If[ EvenQ[a],				
					res = 0,					
					res = (-1)^((a^2-1)/8);					
				],				
				If[ b == -1,				
					If[ a >= 0,					
					 	res = 1,					 	
					 	res = -1;					 	
					],					 
					If[ EvenQ[b],					 
					 	res = KroneckerSymbol[a,2] KroneckerSymbol[a,b/2],					 	
					 	If[ b < 0,					 	
					 		res = KroneckerSymbol[a,-1] KroneckerSymbol[a,-b],					 		
					 		res = JacobiSymbol[a,b];					 		
					 	];					 	
				 	];					 
				];				
			];			
		];		
		res		
	];

	
(* The source code for SqrtMod and SqrtModList is in powermod.mc. *)

sm[n_, m_] := 

	Module[{sml, res},
	
		If[ !IntegerQ[n],
			Message[PowerMod::arg1, n];
			Return[$Failed];
		];
	
		If[ !IntegerQ[m],
			Message[PowerMod::arg2, m];
			Return[$Failed];
		];
	
		If[ m<=0,
			Message[PowerMod::arg2, m];
			Return[$Failed];
		];

		sml = Internal`SqrtModList[n,m];
		
		If[ sml == {},
			Message[PowerMod::root, 2, n, m];
			res = $Failed,
			
			res = Internal`SqrtMod[n,m];
		];
		
		res
	];
	
SqrtMod[n_, m_] :=
	With[{res = sm[n,m]},
		res /; res =!= $Failed
	];
		
SqrtModList[n_Integer, m_Integer] := Internal`SqrtModList[n,m];
							

(* 
SquareFreeQ modified by David Terr, Aug. 2001. Works much faster than before
simply by checking whether MoebiusMu[d] = 0.

Further modifications and addition of GaussianIntegers option by Eric
Weisstein with assistance from Brett Champion. 2003-10-04.
*)

Options[SquareFreeQ]:={GaussianIntegers->False}

SquareFreeQ[d_]:=
  If[Options[SquareFreeQ][[1,2]],SquareFreeGaussianQ[d],MoebiusMu[d]=!=0]

ee:(SquareFreeQ[d_,opts__?OptionQ])/;optCheck[HoldForm[ee],{opts}]:=
  Block[{gauss,fullopts},fullopts=Flatten[{opts,Options[SquareFreeQ]}];
    gaussian=GaussianIntegers/.fullopts;
    
If[gaussian||Head[d]===Complex,SquareFreeGaussianQ[d],MoebiusMu[d]=!=0]]

SquareFreeGaussianQ[c_]:=If[
    MatchQ[c,Complex[_Integer,_Integer]|_Integer],
    Times@@(Last/@FactorInteger[c,GaussianIntegers->True])===1,
    True
    ]

SquareFreeQ[args___/;Length[{args}]<1,
    opts___?OptionQ/;
      Length[{opts}]<1||SameQ[GaussianIntegers,Sequence@@(First/@{opts})]]:=
  Null/;(Message[SquareFreeQ::argx,SquareFreeQ,0];False)

ee:(SquareFreeQ[args__/;Length[{args}]>1&&!OptionQ[Last[{args}]],
        opts___?OptionQ]):=
  Null/;(Message[SquareFreeQ::nonopt,
        First[DeleteCases[Rest[{args}],_?OptionQ]],1,HoldForm[ee]];False)

optCheck[ee_,opts_]:=
  Block[{bad,good},
    bad=Cases[Flatten[{opts}],x_/;First[x]=!=GaussianIntegers];
    good=GaussianIntegers/.Flatten[{opts}];
    If[Length[bad]>0,Message[SquareFreeQ::optx,bad[[1,1]],ee];
      False,
      Switch[good,True|False|GaussianIntegers,True,_,
        Message[SquareFreeQ::opttf,GaussianIntegers,good];
        False]]]

SetAttributes[SquareFreeQ,{Listable}];


(*
Determine whether a given integer is a fundamental discriminant of a quadratic
number field.  - DTERR
*) 
      
FundamentalDiscriminantQ[d_Integer]:=
	Module[{m, mod=Mod[d,4]},
		If[mod>1, Return[False]];
		If[mod==1, Return[SquareFreeQ[d]&&d!=1]];
		m = d/4;
		Return[SquareFreeQ[m]&&Mod[m,4]>1];
	];      


(* NextPrime and PreviousPrime *)

NextPrime[-3] := -2
NextPrime[-2] := 2 
NextPrime[-1] := 2 
NextPrime[0] := 2 
NextPrime[1] := 2

NextPrime[r_?(!IntegerQ[#]&)] := 	
  Block[{$MaxExtraPrecision = Max[$MaxExtraPrecision, 1 + Ceiling[Log[10., Abs[N[r]]]]]},
	With[{x = Floor[r]},
		NextPrime[x] /; IntegerQ[x]
	]
  ];

NextPrime[n_Integer] :=    
 Block[{i = n + 1 + Mod[n, 2]},
   If[Abs[n] < 10^20,
      While[Not[PrimeQ[i]], i += 2],
      prod = Apply[Times, Prime[Range[Log[ Abs[n] ]]]];
      While[True,
        If[GCD[ i, prod] == 1, If[PrimeQ[i], Break[]]];
        i += 2
      ]
   ];
   i
 ]
 
PreviousPrime[x_] := - NextPrime[ -x ]
 

(* Note: As currently implemented, Random[Prime,{a,b}] and Random[Prime,n] favor primes
with large gaps preceding them.  - DTERR, 1/03 *)
 
rp[range_List] :=
	Module[{a, b, t, len = Length[range], p, q},
	
		If[ len != 2, 
			Message[Random::randn,range];
			Return[$Failed] 
		];
		
		a = range[[1]];
		b = range[[2]];
		
		If[ Element[a,Reals] =!= True || Element[b,Reals] =!= True, 
			Message[Random::randn,range];
			Return[$Failed] 
		];
		
		
		(* Swap endpoints if necessary *)
		
		If[ a > b,
			t = a;
			a = b;
			b = t;
		];

		p = b + 1;
		q = NextPrime[a-1];
		
		
		(* See if interval contains any primes *)
		
		If[ q > b, 
			Message[Random::noprime];
			Return[$Failed];
		];
		
		If[ q == b, Return[ b ] ];
		
		While[ p > b,
			p = NextPrime[Random[Integer,{a-1,b-2}]];
		];
		
		p
		
	] 
	
Random[Prime, range_List] :=
	With[{res = rp[range]},
		res /; res =!= $Failed
	];
	
Random[Prime, n_] := Random[Prime, {1,n}] /; !ListQ[n]

ChineseRemainder::pilist =
"The arguments to ChineseRemainder must be two lists of integers of identical
length, with the second list only containing positive integers.";

ChineseRemainder[a:{__Integer}, m:{__Integer}] := 
   Reduce`RChineseRemainder[a, m] /;
       Length[a] == Length[m] && (And @@ Positive[m])

ChineseRemainder[{___Integer},{___Integer}] := $Dummy/;
     (Message[ChineseRemainder::pilist]; False)

ChineseRemainderTheorem = ChineseRemainder;

(* Uses generalization of Cornacchia's algorithm, see Stan Wagon's article 
   and Hardy, Muskat, and Williams. See Cox's book for the theory of when
   a representation x^2 + d y^2 = n exists.    *)

QuadraticRepresentation::norep :=
"Warning: `1` splits into factors not in the principal ideal class of
Q(Sqrt[-`2`]), so a representation of `1` as x^2 + `2` y^2 may be missed."

QuadraticRepresentation[d_, n_]:= 
Block[{x, y, result, root1, quadrep, roots, scan},
  (
	result
  ) /; ( root1 = SqrtMod[-d, n];
         {x, y} = {root1, n};
         (* first try to find representation using root provided by SqrtMod *)
	 quadrep = quadraticRepresentation[d, n, x, y];
         If[quadrep === $Failed, 
	    If[FreeQ[roots = Roots[z^2 == -d, z, Modulus -> n], Roots] &&
			Head[roots] === Or,
		roots = Union[  Apply[List, Map[Sort[{#[[2]], n-#[[2]]}]&,
			 roots]]  ];
		roots = Select[Map[#[[1]]&, roots],
			 (# =!= root1 && # =!= (n-root1))&];
	        scan = Scan[ (
		              {x, y} = {#, n};
	        	      (* try to find representation using a root
					provided by Roots *)
			      quadrep = quadraticRepresentation[d, n, x, y];
			      If[quadrep =!= $Failed, Return[OK]]
			     )&, roots];
		If[scan =!= OK,
                   Message[QuadraticRepresentation::norep, n, d]; False,
	           result = quadrep;  True
		],
		(* ! (FreeQ[roots, Roots] && Head[roots] === Or) *)
		Message[QuadraticRepresentation::norep, n, d]; False
	    ],
	    (* quadrep =!= $Failed *)
	    result = quadrep;  True
	 ] (* end If quadrep === $Failed *)
       )
]  /; n > 0 && OddQ[n] && d > 0 && JacobiSymbol[-d, n] == 1 

quadraticRepresentation[d_, n_, x_, y_] :=
  Module[{xx = x, yy = y},
         If[2 xx > n, xx = n - xx];
         While[xx^2 >= n, {xx, yy} = {Mod[yy, xx], xx}];
         yy = Sqrt[(n - xx^2)/d];
	 If[IntegerQ[yy],
	    {xx, yy},
	    $Failed
	 ]
  ]


(* The following method suggested by Dan Lichtblau. *)
QuadraticRepresentation[d_, n_]:=
Module[{g, sqrt, quadrep, x, y, xx, yy},
  (
  {xx, yy}
  ) /; (g = GCD[d, n];  IntegerQ[g] && g =!= 1) &&
       (sqrt = Sqrt[d/g];
	IntegerQ[sqrt]) &&
       FreeQ[quadrep = QuadraticRepresentation[g, n/g], 
		QuadraticRepresentation] && 
       ({x, y} = quadrep;  xx = g y;
	IntegerQ[yy = x/sqrt])
] /; n > 0 && OddQ[n] && d > 0


(* ClassList only works for negative d, and is inefficient for large d. *)

ClassList[d_Integer] := 
     Block[{a,b,c,list},
           a = 1; list = {};
           While[a <= N[Sqrt[-d/3]],
                 b = 1 - a;
                 While[b <= a, 
                       c= (b^2 - d)/(4 a);
                       If[Mod[c,1] == 0 && c >= a && GCD[a,b,c] == 1 &&
                            Not[a == c && b < 0],
                          AppendTo[list,{a,b,c}]
                         ];
                        b++
                       ];
                   a++
                  ];
             Return[list]
            ]     /;           d < 0 

ClassList[n_] := Null/;(Message[ClassNumber::baddisc, n]; False)


(* Fast class number computation for imaginary quadratic number fields
provided by Stan Wagon. *)

P[n_] := Times @@ ( (({p, b} = #; 
   1 + Sum[p^(-j), {j, 1, b - 1}] + 
   1/(p^b*(1 - JacobiSymbol[-(n/p^(2*b)), p]/p))) & ) /@ 
   (Rest[FactorInteger[First[Sqrt[4E n]]]])); 

q[n_] := Module[{nn}, a = IntegerExponent[n, 4]; 
     nn = Mod[n/4^a, 8]; 
     Which[nn == 7, 0., nn == 3, 2^(-a), True, 3*2^(-a-1)]]; 

cn[-3] = 1;
cn[-4] = 1;

cn[n_] :=  SumOfSquaresR[3, -n]/(24 P[-n]) /;
    n < 0 &&  Mod[n, 8] == 5 && FundamentalDiscriminantQ[n];

cn[n_] :=  SumOfSquaresR[3, -n]/(16 q[-n] P[-n]) /; 
    n < 0 &&  Mod[n, 4] == 0 && FundamentalDiscriminantQ[n];
	
cn[n_] := $Failed /; n <= 0 && !FundamentalDiscriminantQ[n];


(* Algorithm for d = 1 (mod 8) given in Cohen, p.233 *)

cn[d_Integer] :=
	
	Module[{n, nmax, term, dabs, sum, res},
	
		If[ !FundamentalDiscriminantQ[d],
			(* Message[ClassNumber::baddisc, d]; *)
			Return[$Failed];
		];
		
		If[ d == -3 || d == -4, Return[1] ];
		
		dabs = Abs[d];
		nmax = Floor[ Sqrt[ dabs Log[ dabs ] / ( 2 Pi ) ] ];
		sum = 0.0;
		
		For[ n = 1, n <= nmax, n ++,
			term = N[ Erfc[ n * Sqrt[ Pi / dabs ] ] 
				+ Sqrt[ Abs[ d ] ] / ( Pi n ) * Exp[ -Pi n^2 / dabs ] ];
			sum += KroneckerSymbol[ d, n ] * term;
		];

		Return[ Round[ sum ] ];
		
	]; /; d < 0 && Mod[d, 8] == 1;
	
    
(* Positive discriminant *)

cn[d_Integer] :=
  
  	Module[{n, nmax, term, sum = 0., sqrtd = Sqrt[d], sqrt2 = Sqrt[Pi/d], res},

  		If[!FundamentalDiscriminantQ[d], 
			(* Message[ClassNumber::baddisc, d]; *)
			Return[$Failed]
		];
		
      	nmax = Floor[Sqrt[d*Log[d]/(2*Pi)]];
      
      	For[n = 1, n <= nmax, n++,
        	term = N[sqrtd/n*Erfc[n*sqrt2] - ExpIntegralEi[-Pi*n^2/d]];
        	sum += KroneckerSymbol[d, n]*term;
        ];
		
      	res = Round[sum/
			NumberTheory`AlgebraicNumberFields`NumberFieldRegulator[#^2-Mod[d,4]#-Floor[d/4]&]];
		If[ !EvenQ[res], Return[$Failed] ];
		res/2
		
	] /; d > 0 


ClassNumber[d_Integer] :=
  	With[{res = cn[d]},
    	res /; res =!= $Failed
  	];

ClassNumber[n_] := Null/;(Message[ClassNumber::baddisc, n]; False)


(* Written by Stan Wagon, see Complex Factoring,
	Mathematica in Education and Research, 1996  *)

SetAttributes[{SumOfSquaresR, OrderedSumOfSquaresRepresentations, 
	SumOfSquaresRepresentations, NextPrime, PreviousPrime}, Listable]
	
	
(* Modified by David Terr, 2003 *)

OrderedSumOfSquaresRepresentations[d_Integer?Positive, n_Integer] :=

	Module[{sosr = Reduce`SumOfSquaresReps[d,n],i,len,res={}},
		len = Length[sosr];	
		
		For[i=1,i<=len,i++,
			res = Append[res,Reverse[sosr[[i]]]];
		];	
		
  		res
	];


SignedPermutations[a_List] :=
    
    Module[{bd, perm0, perm, perms = Permutations[a], i, alen, bdlen, plen, 
        res = {}},
      alen = Length[a];
      plen = Length[perms];
      
      For[i = 1, i <= plen, i++,
        perm0 = perms[[i]];
        res = Append[res, perm0];
        
        For[j = 1, j < 2^alen, j++,
          bd = IntegerDigits[j, 2];
          bdlen = Length[bd];
		  bd = Flatten[Prepend[bd,Table[0,{k,alen-bdlen}]]];
		  perm = perm0;
          
          For[k = 1, k <= alen, k++,
            If[bd[[k]] == 1,
                perm[[k]] = -perm[[k]];
                ];
            ];
			
          res = Append[res, perm];
          ];
        
        ];
		
      Sort[Union[res]]
      ];

	  
SumOfSquaresRepresentations[d_Integer?Positive, n_Integer] :=

	Module[{len,preres=OrderedSumOfSquaresRepresentations[d,n],sp,splen,res={}},
		len = Length[preres];
		
		For[i=1,i<=len,i++,
			sp = SignedPermutations[preres[[i]]];
			splen = Length[sp];
			
			For[j=1, j<=splen, j++,
				res = Append[res,sp[[j]]];
			];
			
		];
		
  		Sort[Union[res]]
	]	
	
		
FourFreePart[n_] :=
  n / ( 4 ^ IntegerExponent[n,4] );

SumOddDivs[n_] := Plus @@ Select[Divisors[n], OddQ]

OddPart[n_] := Module[{m = n}, While[EvenQ[m], m /= 2]; m]

(* DivisorSigma utilities, that consider only primes 1 mod 4 or
   3 mod 4 but are otherwise like DivisorSigma; also
   AlternatingDivisorSigma[k,n] gives the sum of (-1)^d
   d^k over the divisors d of n; all from Stan Wagon *)
DivisorSigma1[k_, {}] := 1; 

DivisorSigma1[k_, {{p_, e_}, x___}] := 
   (1 + (-1)^e*p^(k*(e + 1)))/(1 + p^k)*
     DivisorSigma1[k, {x}] + 
    p^k*(1 - p^(k*(2*(e - 2 + Mod[e, 2])/2 + 2)))/
      (1 - p^(2*k))*DivisorSigma[k, 
      Times @@ Apply[Power, {x}, {1}]]

DivisorSigma1[k_, n_Integer] := (fi = FactorInteger[n]; 
    n1 = Times @@ 
   Apply[Power, Cases[fi, {p_ /; Mod[p, 4] == 1, _}], 
     {1}];
   n3 = Cases[fi, {p_ /; Mod[p, 4] == 3, _}]; 
  DivisorSigma[k, n1]*DivisorSigma1[k, n3])

DivisorSigma3[k_, n_] := DivisorSigma[k, OddPart[n]] - DivisorSigma1[k, n]

AlternatingDivisorSigma[k_, {p_, e_}] := 
  -(1-p^(k(e+1)))/(1-p^k) /; OddQ[p]

AlternatingDivisorSigma[k_, {2, e_}] := 
  (1-2^(k(e+1)))/(1-2^k) - 2

AlternatingDivisorSigma[k_, n_Integer] := 
  (fi = FactorInteger[n]; 
    -(-1)^Length[fi] * Times @@ 
      (AlternatingDivisorSigma[k, #]&) /@ fi)

(* SumOfSquaresR definitions *)

SumOfSquaresR[_, _?Negative] := 0;
SumOfSquaresR[_, 0] := 1;
SumOfSquaresR[0, _] := 0;

(* Following definition for SumOfSquaresR[2, _] is due to
   Stan Wagon. *)
SumOfSquaresR[2, n_Integer?Positive] :=
    SumOfSquaresR[2, n] =
        If[MemberQ[fi = FactorInteger[n],
                {p_ /; Mod[p,4] == 3, _?OddQ}],
            0,
            4 Apply[Times, Cases[fi,{p_ /; Mod[p,4] == 1, a_} :> a+1 ]]
        ]

SumOfSquaresR[3, n_Integer?Positive] := (SumOfSquaresR[3, n] = 0)/; 
	Mod[FourFreePart[n], 8] == 7;
SumOfSquaresR[3, n_Integer?Positive] := (SumOfSquaresR[3, n] = SumOfSquaresR[3, n/4])/;
       Mod[n, 4] == 0;

SumOfSquaresR[3, n_Integer?Positive] := (SumOfSquaresR[3, n] = 
Module[{n1 = OddPart[Sqrt[n]], ss},
  ss = Select[FactorInteger[n1], Mod[#[[1]], 4] == 3 &];
  If[ss == {}, 6 n1,
    {q, a} = Transpose[ss];
    6 n1 / Times @@(q^a) (Apply[Times,
     q^a + 2 (q^a - 1)/(q-1)])]]) /; IntegerQ[Sqrt[n]]

rprim[n_?OddQ] := (rprim[n] =
  24 Sum[JacobiSymbol[s,n], {s, n/4}]) /; Mod[n, 4] == 1

rprim[n_?OddQ] := (rprim[n] =
  8 Sum[JacobiSymbol[s,n], {s, n/2}]) /; Mod[n, 8] == 3

rprim[n_?OddQ] := 0 /; Mod[n, 8] == 7

(* odd square-free case; because rprim sums to n/4, we
restrcit this case to small n *)

SumOfSquaresR[3, n_?OddQ] := (SumOfSquaresR[3,n] =
  Plus @@ (rprim /@ (n / Select[Divisors[n], IntegerQ[Sqrt[#]]&]))) /;
     SquareFreeQ[n] && n < 500

SumOfSquaresR[3, n_?EvenQ] := SumOfSquaresR[3, n] =
    SumOfSquaresR[2, n] + 2 *
        Apply[Plus, SumOfSquaresR[2, n - Range[Sqrt[N[n]]]^2]]

SumOfSquaresR[4, n_Integer?Positive] := SumOfSquaresR[4,n] =
  If[EvenQ[n], 24, 8] * DivisorSigma[1, OddPart[n]]

SumOfSquaresR[5, n_?((IntegerQ[Sqrt[#]] && (# > 1) &&
                        !IntegerQ[Log[2, #]])&)] :=
  SumOfSquares[5, n] =
    Module[{nn, n1, p, b},
        nn = Sqrt[n];
        n1 = OddPart[nn];
        {p,b} = Transpose @ FactorInteger[n1];
        10 (8^(1+Log[2, nn/n1]) - 1)/7 *
            Times @@ (   (p^(3 b + 3) - p^(3 b+1) + p -1)/(p^3 - 1))
    ]

SumOfSquaresR[6, n_Integer?Positive] := SumOfSquares[6, n] =
  16*n^2*(DivisorSigma1[-2, n] - DivisorSigma3[-2, n]) -
    4*(DivisorSigma1[2, n] - DivisorSigma3[2, n])

SumOfSquaresR[8, n_Integer?Positive] := SumOfSquaresR[8,n] =
  16 (-1)^n AlternatingDivisorSigma[3, n]

SumOfSquaresR[d_Integer, n_Integer?Positive] :=

	If[ d>0,
      	SumOfSquaresR[d-1, n] + 2 *
      		Apply[Plus, SumOfSquaresR[d-1, n - Range[Sqrt[N[n]]]^2]],
		0
	]


(*
TraditionalForm notation for SumOfSquaresR
*)


SumOfSquaresR /:
  MakeBoxes[
    SumOfSquaresR[d_, n_],
    TraditionalForm
  ] :=
  RowBox[{
    TagBox[SubscriptBox["r", BoxForm`ToTrad[d]],
      SumOfSquaresR],
    "(", BoxForm`ToTrad[n], ")"
  }]


wasProt = Unprotect[TraditionalForm]

TraditionalForm /:
  MakeExpression[
    RowBox[{
      TagBox[SubscriptBox["r", d_],
        SumOfSquaresR],
      "(", n_, ")"
    }],
    TraditionalForm
  ] :=
  MakeExpression[
    RowBox[{"SumOfSquaresR",
      "[", BoxForm`MakeCommaSepArgs[d, n], "]"
    }],
    TraditionalForm
  ]

Protect @@ wasProt



(* Which root of unity *)

wrou[x_] :=

	Module[{n,d,k,x1=x,y},
	
		(* First make sure input is an explicit algebraic number. If not,
		return an error message. *)
	
		If[!System`Private`AlgebraicNumberQ[x],
			x1 = Simplify[x];
			
			If[!System`Private`AlgebraicNumberQ[x1],
				Message[RootOfUnityQ::nalg, x];	  
	  			Return[$Failed];
			];
		];		
	
   		If[RootOfUnityQ[x1],
	
      		If[x1 == 1, 
				Return[{1, 1}]
			];
		
			y = ToRadicals[RootReduce[x1]];
			k = FullSimplify[Log[y]/(2*Pi*I)];
			d = Denominator[k];
			n = Mod[Numerator[k],d];
			Return[{d, n}],

			Message[WhichRootOfUnity::nonroot, x];	  
	  		Return[$Failed];
			
		];
		
	];
	
WhichRootOfUnity[x_] :=
	With[{res = wrou[x]},
		res /; res =!= $Failed
	];
	

(* Aliquot and related divisor sum sequences by David Terr, 1/02 *)

(* AliquotSequence, AliquotCycle, and SumOfFactors compute various integer
sequences based on sums of divisors. The definitions were taken from David
Moews' website at http://xraysgi.ims.uconn.edu:8080/amicable.html *)
      
sof[n_Integer, opts___?OptionQ]:= 

	Module[{type, fi, p, d, e, q, i, len, mult=1, j, res=1},
	
		If[ n == 0, Return[0] ];
	
      	If[n < 0 || !(IntegerQ[n] && n>=0),
        	Message[SumOfFactors::badarg, n];
        	Return[$Failed];
        ];

		type = SumOfFactorsType /. Flatten[{opts, Options[SumOfFactors]}]; 		
		If[ type == Automatic, Return[ DivisorSigma[1,n] - n ] ];
		
		If [ !MemberQ[ {Unitary,Biunitary,Exponential,ModifiedExponential,Infinitary},
			type], 
			Message[SumOfFactors::type, type];
			Return[$Failed] 
		];
		
		fi = FactorInteger[n]; 
		len = Length[fi]; 
		
		For[ i=1, i<=len, i++,
			p = fi[[i,1]];
			e = fi[[i,2]];
			q = p^e;
			res *= Switch[type,
				Unitary, q+1,
				Biunitary, (p*q-1)/(p-1) - If[EvenQ[e],p^(e/2),0],
				Exponential, 1+p +
					If[e>1,p^e + Sum[p^d*If[Mod[e,d]==0,1,0],{d,2,e-1}], 0],
				ModifiedExponential, 1+p^e + 
					Sum[p^d*If[Mod[e,d]==0&&Mod[e+1,d+1]==0,1,0],{d,1,e-1}],
				Infinitary,	1+p^e + Sum[p^d*If[BitAnd[e,d]==d,1,0],{d,1,e-1}]
			];
		];  
		
		res - n
	]; 
	
SumOfFactors[n_Integer, opts___?OptionQ]:= 
	With[{res = sof[n,opts]},
		res /; res =!= $Failed
	];
	
      
as[n_, opts___?OptionQ] := 

	Module[{type, increment, printQ, maxlen, maxsize, i, l, m, res = {n}, 
			done = False},

	  	If[ n < 0 || !IntegerQ[n], 
			Message[AliquotSequence::badarg, n];
	  		Return[$Failed];
	  	];
			
		{type, increment, printQ, maxlen, maxsize} = 
			{SumOfFactorsType, TermIncrement, ShowProgress, 
			MaxIterations, MaxTerms} /. 
			Flatten[{opts,Options[AliquotSequence]}]; 		
		
		If [ !MemberQ[ {Automatic,Unitary,Biunitary,Exponential,ModifiedExponential,Infinitary},
			type], 
			Message[SumOfFactors::type, type];
			Return[$Failed] 
		];
		
		If[ !IntegerQ[increment],
			Message[AliquotSequence::increment, increment];
			Return[$Failed] 
		];
		
		If[ !MemberQ[{True,False},printQ],
			Message[AliquotSequence::progress, printQ];
			Return[$Failed] 
		];
		
		If[ (!IntegerQ[maxlen] || maxlen < 1) && maxlen =!= Infinity,
			Message[AliquotSequence::maxit, maxlen];
			Return[$Failed] 
		];
		
		If[ (!IntegerQ[maxsize] || maxsize < 1) && maxsize =!= Infinity,
			Message[AliquotSequence::maxterms, maxsize];
			Return[$Failed] 
		];

		If[ printQ, Print[{0,n}]//TableForm ]; 		

      	For[i = 1, i <= maxlen && !done, i++, 
      		l = If[i > 1, res[[i]], n];
        	m = SumOfFactors[l,SumOfFactorsType->type] + increment;
			If[ printQ, Print[{i,m}]//TableForm ]; 		
        	If[m <= 0 || MemberQ[res, m] || m > maxsize, done = True];
        	res = Append[res, m];
        ];
		
      	res
	];
 	
AliquotSequence[n_, opts___?OptionQ]:= 
	With[{res = as[n,opts]},
		res /; res =!= $Failed
	];
     
NormalizeList[l_List] :=
    
    Module[{min = Min[l], res = l, leadingTerm = l[[1]]},
	
      	While[leadingTerm != min,
		
        	res = Append[res, leadingTerm];
        	res = Delete[res, 1];
        	leadingTerm = res[[1]];
			
        ];
		
	res
	];

ac[n_, opts___?OptionQ] :=

    Module[{as = as[n, opts], increment, maxlen, maxsize, k, 
		len, t},
			
		If[ as === $Failed, Return[$Failed]];
	
		{increment, maxlen, maxsize} = 
			{TermIncrement, MaxIterations, MaxTerms} /. 
			Flatten[{opts,Options[AliquotCycle]}];
			 		
      	len = Length[as];
      	t = as[[len]];

      	If[ len == maxlen + 1 || t > maxsize,
			Message[AliquotCycle::open];
	  		Return[$Failed];
	  	];
      	
      	If[t == 0 && increment == 0, Return[{0}]];
       	For[k = len - 1, as[[k]] != t, k--];
     	NormalizeList[Take[as, k - len]]
    ];
 	
AliquotCycle[n_Integer, opts___?OptionQ]:= 
	With[{res = ac[n,opts]},
		res /; res =!= $Failed
	];

	
pr[n_Integer]:=

	Module[{res = System`Private`PrimitiveRoot[n]},
		If[ !IntegerQ[res], Return[$Failed], Return[res] ]
	];
	
PrimitiveRoot[n_Integer] :=
	With[{res = pr[n]},
		res /; res =!= $Failed
	];
          
      
End[]   (* NumberTheory`NumberTheoryFunctions`Private` *)

SetAttributes[ {AliquotCycle, AliquotSequence, Automatic, Biunitary, ChineseRemainder, 
	ChineseRemainderTheorem, ClassNumber, ClassList, Exponential,
	FundamentalDiscriminantQ, Infinitary, 
	KroneckerSymbol, LeastPrimeFactor, MaxIterations, MaxTerms, 
	ModifiedExponential, NextPrime, PreviousPrime, PrimePowerQ, PrimitiveRoot,
	PrimeFactorList, QuadraticRepresentation, Random, 
	ShowProgress, SquareFreeQ, SqrtMod, SqrtModList,  
	SumOfFactors, SumOfFactorsType, TermIncrement, Unitary, WhichRootOfUnity},
ReadProtected ]

Protect[AliquotCycle, AliquotSequence, Automatic, Biunitary, ChineseRemainder, 
	ChineseRemainderTheorem, ClassNumber, ClassList, Exponential,
	FundamentalDiscriminantQ, Infinitary, 
	KroneckerSymbol, LeastPrimeFactor, MaxIterations, MaxTerms, 
	ModifiedExponential, NextPrime, PreviousPrime, PrimePowerQ, PrimitiveRoot,
	PrimeFactorList, QuadraticRepresentation, Random, 
	ShowProgress, SquareFreeQ, SqrtMod, SqrtModList, 
	SumOfFactors, SumOfFactorsType, TermIncrement, Unitary, WhichRootOfUnity]



EndPackage[] (* NumberTheory`NumberTheoryFunctions` *)

(* :Examples:

SumOfSquaresRepresentations[2, 100]

SumOfSquaresRepresentations[3, 100]

Apply[Plus, (%^2), 2]	(* check *)

SumOfSquaresRepresentations[5, 100]

SumOfSquaresRepresentations[{1, 2, 3, 4, 5}, 64]

SumOfSquaresR[5, 100]

SumOfSquaresR[2, 10^30]

SumOfSquaresR[4, 10^30]

SumOfSquaresR[6, 10^30]

SumOfSquaresR[8, 10^30]

SumOfSquaresR[Range[10], 100]

(* The asymptotic average value of r2 is Pi *)
Sum[N[SumOfSquaresR[2, k]], {k, 200}] / 200

(* A curious alternating sum equals - Pi Log[2] *)
Sum[N[(-1)^n SumOfSquaresR[2, n] / n], {n, 1, 200}]

- Pi Log[2] //N

QuadraticRepresentation[14,65]
	{3, 2}

QuadraticRepresentation[12,21]
	{3, 1}

QuadraticRepresentation[2, 9]
	{1, 2}

QuadraticRepresentation[3, 49]
	{1, 4}

QuadraticRepresentation[15, 991]
	{16, 7}

QuadraticRepresentation[2, 81]
	{7, 4}

QuadraticRepresentation[49, 277]
	{9, 2}

QuadraticRepresentation[48, 147]
	unevaluated
	
*)


(*
Reference: [Cohen] A Course in Computational Algebraic Number Theory, Springer-Verlag, 1993
*)
