(* :Copyright: Copyright 2004-2005, Wolfram Research, Inc.  *)

(* :Name: ClusterAnalysis *)

(* :Title: ClusterAnalysis *)

(* :Author: Andrew A. de Laix *)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 5.1 *)

(*:Summary:
	Provides functionality for cluster creation, plotting,
	and verification.
*)

(* :History:
	Created April 2004 by Andrew A. de Laix
	Modified per design changes September 2004 by Darren Glosemeyer
*)

BeginPackage["Statistics`ClusterAnalysis`", 
   {"Utilities`FilterOptions`"}]; 

CanberraDistance::usage = "CanberraDistance[a,b] computes the Canberra distance between \
numerical vectors a and b.";

BrayCurtisDistance::usage = "BrayCurtisDistance[a,b] computes the Bray-Curtis distance \
between numerical vectors a and b.";

EuclideanDistance::usage = "EuclideanDistance[a,b] computes the Euclidean distance between \
numerical vectors a and b.";

EditDistance::usage = "EditDistance[a,b] computes the number of edits \
required to convert string a into string b."

SquaredEuclideanDistance::usage = "SquaredEuclideanDistance[a,b] computes the square of the \
Euclidean distance between numerical vectors a and b.";

ManhattanDistance::usage = "ManhattanDistance[a,b] computes the Manhattan (a.k.a. taxi cab) \
distance between numerical vectors a and b.";

SupDistance::usage = 
  "SupDistance[a,b] computes the sup distance between numerical vectors a and b.";

CorrelationDissimilarity::usage = "CorrelationDissimilarity[a,b] computes the \
correlation coefficient dissimilarity between numerical vectors a and b.";

CosineAngleDissimilarity::usage = "CosineAngleDissimilarity[a,b] computes the angular cosine \
dissimilarity between numerical vectors a and b.";

JaccardDissimilarity::usage = "JaccardDissimilarity[a,b] computes the Jaccard dissimilarity between \
Boolean vectors a and b.";

MatchingDissimilarity::usage = "MatchingDissimilarity[a,b] computes the matching dissimilarity between \
Boolean vectors a and b.";

RussleRaoDissimilarity::usage = "RussleRaoDissimilarity[a,b] computes the Russle-Rao dissimilarity \
between Boolean vectors a and b.";

SokalSneathDissimilarity::usage = "SokalSneathDissimilarity[a,b] computes the Sokal-Sneath \
dissimilarity between Boolean vectors a and b.";

RogersTanimotoDissimilarity::usage = "RogersTanimotoDissimilarity[a,b] computes the Rogers-Tanimoto \
dissimilarity between Boolean vectors a and b.";

DiceDissimilarity::usage = 
  "Dice[a,b] computes the Dice dissimilarity between Boolean vectors a and \
b.";

YuleDissimilarity::usage = 
  "Yule[a,b] computes the Yule dissimilarity between Boolean vectors a and \
b.";


FindClusters::usage="FindClusters[{e1, e2, ... }] partitions the ei into \
clusters of similar elements. FindClusters[{e1 -> v1, e2->v2, ... }] returns \
the vi corresponding to the ei in each cluster. \
FindClusters[{e1, e2, ... } -> {v1, v2, ... }] gives the same result. \
FindClusters[{e1, e2, ... }, n] partitions the ei into exactly n clusters.";

DistanceFunction::usage="Distance Function is an option to FindClusters, Agglomerate, \
DistanceMatrix, and DendrogramPlot that specifies the distance or dissimilarity measure.";

RandomSeed::usage="RandomSeed is an option to FindClusters that specifies the starting \
value for the random number generator within FindClusters.";

Agglomerate::usage = "Agglomerate[list] collects the elements of list \
into a hierarchy of clusters.";

Cluster::usage = "Cluster[c1,c2,d,n1,n2] represents a merger in the \
cluster hierarchy where the elements c1 and c2 are the subclusters \
merged with distance or dissimilarity value d and the subclusters contain n1 and \
n2 data elements respectively."

DirectAgglomerate::usage = "DirectAgglomerate[m] constructs a cluster \
hierarchy based on the square distance or dissimilarity matrix m.  \
DirectAgglomerate[m,list] associates the elements of list with the rows \
of the distance or dissimilarity matrix m inserting them into the cluster \
hierarchy."; 

DendrogramPlot::usage = "DendrogramPlot[{y1, y2, ... }] plots the \
clustering dendrogram derived from the list of input values. \
DendrogramPlot[c] plots the clustering dendrogram for the Cluster object c.";

LeafLabels::usage="LeafLabels is an option to DendrogramPlot that specifies labels for the dendogram leaves.";

TruncateDendrogram::usage="TruncateDendrogram is an option to DendrogramPlot that specifies the fusion \
levels at which to truncate the dendrogram.  With the default setting of All, all levels of the \
dendrogram are shown.";

HighlightLevel::usage="HighlightLevel is an option to DendrogramPlot that specifies the level at which \
to highlight the dendrogram.";

HighlightStyle::usage="HighlightStyle is n option to DendrogramPlot that specifies the style for \
highlighted clusters.";

Orientation::usage="Orientation is an option to DendrogramPlot that specifies the orientation of the \
dendrogram. Possible settings are Top, Bottom, Left, and Right.";

Linkage::usage="Linkage is an option to Agglomerate and DendrogramPlot that specifies the linkage \
method to be used for agglomerative clustering.";

DistanceMatrix::usage = "DistanceMatrix[list] computes the \
symmetric matrix of distance or dissimilarity coefficients between the elements of list."; 

ClusterFlatten::usage = "ClusterFlatten[c] flattens the cluster c \
returning a list of the original input data."; 

ClusterSplit::usage = "ClusterSplit[c,n] splits the cluster c into n \
clusters by repeatedly dividing at the largest remaining fusion level.";

Agglomerate::bdlink = "Value of option Linkage->`1` is not \
\"Single\", \"Complete\", \"Average\", \"WeightedAverage\", \"Centroid\", \"Median\", \
\"Ward\" or a pure function.";

Agglomerate::amtd = "Agglomerate is unable to automatically select an \
appropriate distance or dissimilarity function for the input data `1`."

Agglomerate::xnum = "A nonnumeric, negative or complex distance or dissimilarity \
value was computed; distances and dissimilarities must be nonnegative and real valued.";

Agglomerate::ties = "`1` ties have been detected; reordering input may \
produce a different result.";

Agglomerate::nelist = "Argument `1` at position `2` is not a nonempty list.";

DirectAgglomerate::bdlink = "Value of option Linkage->`1` is not \
\"Single\", \"Complete\", \"Average\", \"WeightedAverage\", \"Centroid\", \"Median\", \
\"Ward\" or a pure function.";

DirectAgglomerate::ties = "`1` ties have been detected; reordering \
input may produce a different result.";

DirectAgglomerate::dcbd = "The input distance or dissimilarity matrix `1` contains \
non-numeric, negative, or complex values; the distance or dissimilarity matrix must \
be non-negative and real valued.";

DirectAgglomerate::xnum = "A nonnumeric, negative or complex \
distance or dissimilarity value was computed; distances and dissimilarities must be nonnegative and \
real valued.";

DirectAgglomerate::bdat = "The length of the data list `1` at position \
2 does not match the dimensions of the distance or dissimilarity matrix `2` at \
position 1.";

ClusterSplit::dep = 
   "Cannot split a cluster of `1` elements into `2` clusters."; 
   
ClusterSplit::notcl = ClusterFlatten::notcl = "`1` is not a valid cluster.";

DendrogramPlot::labl = "The number of labels (`1`) does not match the \
number of data points (`2`)."; 

DendrogramPlot::labo = "`1` is not a valid LeafLabels option."; 

DendrogramPlot::labm = "Cannot map the labels in `1` to the data \
elements of the cluster `2`; the mapping is ambiguous."; 

DendrogramPlot::trnc = "`1` is not a valid truncation range."; 

DendrogramPlot::hltr = 
   "The highlight value `1` should be between 1 and `2`."; 

DendrogramPlot::hlt = "`1` is not a valid highlight specification."; 

DendrogramPlot::orn = "The orientation value `1` should be either Top, \
Bottom, Left, or Right."; 

DendrogramPlot::arg1 =" `1` is neither a list of elements nor a Cluster object.";

DendrogramPlot::cldist = "Distance information is contained within the input Cluster object.
The DistanceFunction option will be ignored.";

DistanceMatrix::amtd = "DistanceMatrix is unable to automatically select an appropriate dissimilarity function for the input data `1`.";

DistanceMatrix::xnum = "A nonnumeric, negative or complex distance or dissimilarity \
value was computed; distances and dissimilarities must be nonnegative and real valued.";

FindClusters::rseed = "Value of option RandomSeed -> `1` is not an integer or Automatic.";
FindClusters::bdmtd = "The value of option Method->`1` is not Automatic, \"Optimize\" or \"Agglomerate\"";
FindClusters::bdst = "The value of method option SignificanceTest->`1` is not Automatic, \"Gap\" or \"Silhouette\"";
FindClusters::amtd = "FindClusters is unable to automatically select an appropriate dissimilarity function for the input data `1`.";
FindClusters::bditr = "The value of option \"Iterations\"->`1` should be a positive integer.";
FindClusters::bdnul = "The value of option \"NullSets\"->`1` should be a positive integer greater than 1.";
FindClusters::bdtol = "The value of option \"Tolerance\"->`1` should be a real number.";
FindClusters::nclst = "The number of clusters requested (`1`) is larger than the number of elements to cluster (`2`).";
FindClusters::gap = "The data to be clustered must consist of vectors of real machine numbers to use the \"Gap\" method.  Try Method->\"Silhouette\" instead.";
FindClusters::bdlink = "Value of Agglomerate method option \"Linkage\"->`1` is not \"Single\", \"Complete\", \"Average\", \"Weighted\", \"Centroid\", \"Median\" or \"Ward\".";
FindClusters::xnum = "A nonnumeric, negative or complex dissimilarity value was computed; dissimilarities must be nonnegative and real valued.";

Begin["`Private`"]; 

ClusterSplit[args___] := Module[{res = iClusterSplit[args]}, 
    res /; res =!= $Failed]; 

(*
   iClusterSplit
   
   Description
   	seprarates a Cluster expression into multiple clusters by
   	repeatedly splitting at the largest fusion value
   	
   Arguments
   	clust_ : a cluster object
   	n_ : the number of clusterst to be produced
   	
   Returns
   	a list of cluster objects
*)

iClusterSplit[clust_Cluster, n_] := Module[{mx, res},
    If[Length[Apply[List,clust]] =!= 5
    	, 
    	Message[ClusterSplit::notcl, clust]; Return[$Failed]
    	,
      	mx = clust[[4]] + clust[[5]];
      	Which[! IntegerQ[n] || n <= 0,
        	Message[ClusterSplit::intpm, HoldForm[ClusterSplit[clust, n]], 2]; 
        	Return[$Failed],
        	n > mx,
        	Message[ClusterSplit::dep, mx, n]; Return[$Failed],
        	True,
        	Null];
      	Catch[Nest[(res = With[{p = Replace[#1, cl_ :> Switch[cl, _Cluster, If[Length[Apply[List, cl]] =!= 5, 
      			Message[ClusterSplit::notcl, clust]; $Failed, cl[[3]]], _, -1.], {1}]}, 
      			If[! FreeQ[p, $Failed], $Failed, 
      		  	MapAt[Sequence @@ Take[#1, 2] &, #1, Ordering[p, -1]]]]; 
      		  	If[res === $Failed, Throw[$Failed], res]) &
      		  , {clust}, n - 1]]]]
      		  

iClusterSplit[args___] := If[Length[{args}] != 2, 
   Message[ClusterSplit::argrx, ClusterSplit, Length[{args}], 2]; 
    Return[$Failed], Return[$Failed]]


ClusterFlatten[args___] := Module[{res = iClusterFlatten[args]}, 
   res /; res =!= $Failed]

(*
   iClusterFlatten[cl_Cluster]
   
   Description
   	Flattens out a Cluster expression returning the data
   	elements in the cluster
   	
   Arguments
   	cl_ : a Cluster expression
   
   Returns
   	a list of data elements
*)

iClusterFlatten[cl_Cluster] := Module[{foo, res, aa, bb, cc, dd, ee},
   If[FreeQ[res = cl //. Cluster[aa_, bb_, cc_, dd_, ee_] -> foo[aa, bb], Cluster]
   	,
      	List @@ Flatten[res, Infinity, foo]
      	,
      	Message[ClusterFlatten::notcl, cl]; $Failed]]

iClusterFlatten[args___] := If[Length[{args}] != 1, 
   Message[ClusterFlatten::argx, ClusterFlatten, Length[{args}]]; 
    Return[$Failed], Return[$Failed]]
    

(* ProcessOptionNames allows for option names to be strings or symbols *)    
SetAttributes[ProcessOptionNames, Listable];

ProcessOptionNames[(r : (Rule | RuleDelayed))[name_Symbol, val_]] :=
    r[SymbolName[name], val];

ProcessOptionNames[opt_] := opt;


Options[DendrogramPlot] = {DistanceFunction->Automatic, LeafLabels -> None, 
    TruncateDendrogram -> All, HighlightLevel -> None, 
    HighlightStyle -> Automatic, Orientation -> Top, 
    PlotStyle -> Automatic, Linkage -> Automatic};  

DendrogramPlot[args___] := Module[{res = iDendrogramPlot[args]}, 
   res /; res =!= $Failed]

(*
   iDendrogramPlot (v1)
   	This is the main argument parsing function for plotting
   	dendrograms with data (as compared to Cluster expression) input.
   	
   Arguments
   	data_List : a list of data elements
   	opts___ : options for this function
   	
   Returns
   	a Graphics object
*)

iDendrogramPlot[data:(_List|_Rule), (opts___)?OptionQ]:= 
  Block[{i = 0, $SowLabel, $lotruncate, $hitruncate, 
    	$hi, $Orientation, $LabelFunction = First}
    	, 
   	Module[{c, lab, mydata, lines, lablist, dcs, 
     	  trunc, hilite, rects, txtfxn, hs, ps, link, distfun, processedoptions}
     	  , 
     	  (* use ProcessOptionNames to convert all option names to strings;
     	     also include opts so Graphics options are included *)
     	  processedoptions = Join[ProcessOptionNames[Flatten[{opts, Options[DendrogramPlot]}]],{opts}];
     	  lab = "LeafLabels"/.processedoptions;
    	  Switch[lab
    	    ,
    	    None | False
    	    ,
    	    mydata = data; $SowLabel = False;
    	    ,
    	    _Function
    	    ,
    	    $LabelFunction = lab;
    	    mydata = data;
    	    $SowLabel = True;
    	    ,
    	    Automatic | _List
    	    ,
    	    If[Head[data]===List, 
    	    	If[lab === Automatic,lab = Range[Length[data]]];
    	    	If[Length[lab] != Length[data]
       		  , 
       		  Message[DendrogramPlot::labl, Length[lab], Length[data]]; 
       		  $SowLabel = False; 
         	  mydata = data; 
         	  , 
         	  $SowLabel = True; 
         	  If[ArrayDepth[data] < 2
         		, 
          		mydata = Partition[data, 1]
          		, 
          		mydata = data]; 
         	  mydata = MapThread[Rule[#1, #2] &, {mydata, lab}];
         	  $LabelFunction = (#&);
         	  ]];
    	    If[Head[data]===Rule, 
    	    	If[lab===Automatic, lab = Range[Length[data[[2]]]]];
    	    	If[Length[lab] != Length[data[[2]]]
		  , 
		  Message[DendrogramPlot::labl, Length[lab], Length[data]];
		  $SowLabel = False;
		  mydata = data;
		  , 
		  $SowLabel = True;
		  mydata = MapThread[Rule[#1, #2] &, {data[[1]], lab}];
		  $LabelFunction = (#&);
         	  ]];
           ,
           _
           ,
           Message[DendrogramPlot::labo, lab];
           mydata = data; $SowLabel = False; ];
         link = "Linkage" /. processedoptions;
         distfun = "DistanceFunction" /. processedoptions;
         Check[c = Agglomerate[mydata, "Linkage" -> link, "DistanceFunction" -> distfun]
     		, 
     		Return[$Failed]
     		,  
      		Agglomerate::bmtd, Agglomerate::amtd, Agglomerate::xnum];
      	 iDendrogramFromCluster[c, processedoptions]
      ]]
   
   

(*
   iDendrogramPlot (v2)
   	This is the main argument parsing function for plotting
   	dendrograms with a Cluster expression as input.
   	
   Arguments
   	c_Cluster : a Cluster expression
   	opts___ : options for this function
   	
   Returns
   	a Graphics object
*)

iDendrogramPlot[c_Cluster, (opts___)?OptionQ] := 
  Block[{i = 0, $SowLabel, $lotruncate, $hitruncate, $hi, $Orientation, 
    $LabelFunction}
    , 
    Module[{lab, lines, lablist, dcs, trunc, hilite, rects, txtfxn, hs, ps, distfun, 
    	processedoptions}
    	,
    	(* use ProcessOptionNames to convert all option names to strings;
     	     also include opts so Graphics options are included *)
    	processedoptions = Join[ProcessOptionNames[Flatten[{opts, Options[DendrogramPlot]}]],{opts}];
    	(* check against opts so a message is only issued for user-defined "DistanceFunction" *)
    	If[(distfun="DistanceFunction"/.ProcessOptionNames[{opts}])=!="DistanceFunction"
    		,
    		Message[DendrogramPlot::cldist, distfun]];
    	lab = "LeafLabels" /. processedoptions; 
     	Switch[lab
     		, 
     		None | False
     		, 
     		$SowLabel = False
     		, 
     		Automatic
     		, 
      		$SowLabel = True; $LabelFunction = Short; 
      		, 
      		_Function
      		, 
      		$SowLabel = True; $LabelFunction = lab; 
      		, 
      		_List
      		, 
      		Message[DendrogramPlot::labm, lab, c]; $SowLabel = False
      		, 
      		_
      		, 
      		Message[DendrogramPlot::labo, lab]; $SowLabel = False; 
      		]; 
     	iDendrogramFromCluster[c, processedoptions]
     ]]

(*
   iDendrogramFromCluster
   
   Description
   	This builds up the dendrogram graphics from a Cluster expression.
   
   Arguments
   	c_ : a Cluster expression
   	opts___ : options for this function
*)

iDendrogramFromCluster[c_, opts___] := 
  Module[{lines, lablist, dcs, trunc, hilite, rects, txtfxn, hs, ps}
  	, 
   	dcs = Reverse[Sort[Cases[{c}, cc_Cluster :> cc[[3]], Infinity]]]; 
    	trunc = "TruncateDendrogram" /. opts; 
    	Switch[trunc
    	  , 
    	  All | Infinity
    	  , 
    	  {$lotruncate, $hitruncate} = {-Infinity, Infinity}
    	  , 
    	  n_Integer /; Inequality[0, Less, n, LessEqual, Length[dcs]]
    	  , 
    	  {$lotruncate, $hitruncate} = {dcs[[trunc]], First[dcs]}
    	  , 
    	  {n_Integer, m_Integer} /; m > n && n > 0 && m < Length[dcs]
    	  , 
    	  {$lotruncate, $hitruncate} = {dcs[[Last[trunc]]], dcs[[First[trunc]]]}
    	  , 
    	  {n_Integer /; Inequality[0, Less, n, LessEqual, Length[dcs]], Infinity}
    	  , 
    	  {$lotruncate, $hitruncate} = {-Infinity, dcs[[First[trunc]]]}
    	  , 
    	  _
    	  , 
    	  Message[DendrogramPlot::trnc, trunc]; Return[$Failed]
    	  ];
    	hilite = "HighlightLevel" /. opts;
    	$hi = Switch[hilite
    	  , 
    	  False | None
    	  , 
    	  -Infinity
    	  , 
    	  n_Integer /; Inequality[0, Less, n, LessEqual, Length[dcs]]
    	  , 
    	  dcs[[hilite]]
    	  , 
    	  _Integer
    	  , 
    	  Message[DendrogramPlot::hltr, hilite, Length[dcs]]; 
    	  Return[$Failed]
    	  , 
    	  _
    	  , 
    	  Message[DendrogramPlot::hlt, hilite]; 
    	  Return[$Failed]
    	  ]; 
    	$Orientation = "Orientation" /. opts; 
    	txtfxn = Switch[$Orientation
    	  , 
    	  Left
    	  , 
    	  Text[#2, Offset[{4, 0}, {0, #1}], {-1, 0}] & 
    	  , 
    	  Right
    	  , 
    	  Text[#2, Offset[{-4, 0}, {0, #1}], {1, 0}] & 
    	  , 
    	  Bottom
    	  , 
    	  Text[#2, Offset[{0, 4}, {#1, 0}], {0, -1}] & 
    	  , 
    	  Top
    	  , 
    	  Text[#2, Offset[{0, -4}, {#1, 0}], {0, 1}] & 
    	  , 
    	  _
    	  , 
    	  Message[DendrogramPlot::orn, $Orientation]; 
    	  Return[$Failed]
    	  ];
    	{lines, lablist, rects} = Last[Reap[iGenerateDendrogram[c], 
       		{"lines", "labels", "rectangles"}]]; 
       	ps = "PlotStyle" /. opts; 
       	hs = "HighlightStyle" /. opts; 
       	Show[
       	  Graphics[{
       	    	{Switch[hs,Automatic,RGBColor[0, 1, 0],_List,Sequence @@ hs,_,hs], rects}
       	    	, 
       	    	{Switch[ps,Automatic,{},_List,Sequence @@ ps,_,ps], lines}
       	    	, 
       	    	If[$SowLabel, Apply[txtfxn, First[lablist], {1}], {}]}
      	   , 
      	   FilterOptions[Graphics, Apply[Sequence,opts]], PlotRange -> All, AspectRatio -> 1/GoldenRatio]
     	]]

iDendrogramPlot[arg_] := (Message[DendrogramPlot::arg1, arg]; 
   $Failed)/;!MemberQ[{List,Cluster}, Head[arg]]

iDendrogramPlot[args___] := (Message[DendrogramPlot::argx, 
    DendrogramPlot, Length[Select[{args},  !OptionQ[#1] & ]]]; 
   $Failed)
   
(*
   iGenerateDendrogram
   
   Description
   	This function calls itself recursively sowing the graphics elements
   	that make up the dendrogram until the recursion terminates on a
   	leaf (a data element)
*)
   

iGenerateDendrogram[v_] := 
  (i++; If[$SowLabel, Sow[{i, $LabelFunction[v]}, "labels"]; i, i, i])

iGenerateDendrogram[c_Cluster] := Block[
	{x0, xl, xr, yl, yr, $RecursionLimit=Infinity}
	, 
   	If[c[[3]] > $hitruncate
   	  , 
   	  iGenerateDendrogram[c[[1]]]; 
   	  iGenerateDendrogram[c[[2]]]; 
   	  Return[]]; 
	If[c[[3]] <= $lotruncate
	  , 
	  i++; 
	  Return[If[$SowLabel
	  	, 
        	Sow[{i, DisplayForm[FrameBox[ToBoxes[c[[-2]] + c[[-1]]]]]},"labels"]; 
        	i
        	,
        	i, i]]
          ];
    	xl = -If[c[[-2]] == 1 || c[[1,3]] <= $lotruncate, 0, c[[1,3]]]; 
    	xr = -If[c[[-1]] == 1 || c[[2,3]] <= $lotruncate, 0, c[[2,3]]]; 
    	x0 = -c[[-3]]; 
    	yl = iGenerateDendrogram[c[[1]]]; 
    	yr = iGenerateDendrogram[c[[2]]]; 
    	Switch[$Orientation
    		, 
    		Left
    		, 
     		Sow[Line[{{xl, yl}, {x0, yl}, {x0, yr}, {xr, yr}}], "lines"]; 
      		If[c[[3]] <= $hi
      		  , 
      		  Sow[Rectangle[{x0, yl}, {0, yr}], "rectangles"]
      		  ]
      		, 
      		Right
      		, 
     		Sow[Line[{{-xl, yl}, {-x0, yl}, {-x0, yr}, {-xr, yr}}], "lines"]; 
      		If[c[[3]] <= $hi
      		  , 
      		  Sow[Rectangle[{0, yl}, {-x0, yr}], "rectangles"]
      		  ]
        	, 
        	Bottom
        	, 
     		Sow[Line[{{yl, xl}, {yl, x0}, {yr, x0}, {yr, xr}}], "lines"]; 
      		If[c[[3]] <= $hi
      		  , 
      		  Sow[Rectangle[{yl, x0}, {yr, 0}], "rectangles"]
      		  ]
        	, 
        	Top
        	, 
     		Sow[Line[{{yl, -xl}, {yl, -x0}, {yr, -x0}, {yr, -xr}}], "lines"]; 
      		If[c[[3]] <= $hi
      		  , 
      		  Sow[Rectangle[{yl, 0}, {yr, -x0}], "rectangles"]
      		  ]
      		]; 
      	(yl + yr)/2.]

End[]; 

EndPackage[]; 
