(* :Name: Statistics`DataSmoothing` *)

(* :Title: DataSmoothing *)

(* :Author: Yu He *)

(* :Summary:
This package does both univariate and multivariate data 
smoothing through applications of moving average, moving median, 
linear filter, or exponential smoothing. *)

(* :Context: Statistics`DataSmoothing` *)

(* :Package Version: 1.1 *)

(* :Copyright: Copyright 1993-2005 by Wolfram Research, Inc. *)

(* :History: 
	Original version by Yu He, November 1993.
	Version 1.1, Darren Glosemeyer and Daniel Lichtblau;
	  improved memory efficiency and speed for MovingAverage, 
	  MovingMedian, and LinearFilter, 2004.
*)

(* :Keywords: moving average, moving median, linear filter,
	 exponential smoothing *)

(* :Source:
	For smoothing by moving median see "Exploratory Data Analysis",
	J. W. Tukey, 1977, Addison-Wesley Publishing Co., Inc., Reading, MA
*)

(* :Warnings: 
	In Version 2.2 and earlier, MovingAverage[datalist, 0] would give
	the input datalist.  In this package, the definition of the
	second argument has changed.  Now, MovingAverage[datalist, 1] gives
	the input datalist.  In general, MovingAverage[datalist, n] in
	V2.2 now corresponds to MovingAverage[datalist, n+1].
	MovingAverage[datalist, n] is an n-term moving average.
*)

(* :Limitations: None *)

(* :Mathematica Version: 2.0 *)

BeginPackage["Statistics`DataSmoothing`"]

MovingAverage::usage =
"MovingAverage[datalist, r] smooths datalist using a simple r-term moving
average.  Each element in the result is the average of r elements from
datalist."

MovingMedian::usage =
"MovingMedian[datalist, r] smooths datalist using a moving median of span r.
Each element in the result is the median of r elements from datalist.
MovingMedian[datalist, r, RepeatedSmoothing->True] does repeated smoothing for
odd r until the result no longer changes."

LinearFilter::usage =
"LinearFilter[datalist, {c0, c1, c2, ...}] passes the data in datalist through 
the linear filter with weights {c0, c1, c2, ...}.  Each element in the
resulting list is given by Sum[cj datalist[[t-j]], {j, 0, r-1}], for t = r,
r+1, ... n, where n = Length[datalist] and r is the number of weights."

ExponentialSmoothing::usage = 
"ExponentialSmoothing[datalist, a] smooths datalist using an exponentially
weighted average with smoothing constant a, taking the first entry of datalist
as the starting value.  ExponentialSmoothing[datalist, a, y0] smooths datalist
taking the starting value to be y0.  If yt is the smoothed result at time t,
then the result at time t+1 is given by yt + a*(datalist[[t+1]] - yt),
where a is the smoothing constant (0 < a < 1)."

RepeatedSmoothing::usage =
"RepeatedSmoothing is an option to MovingMedian. It specifies whether the
data are to be smoothed repeatedly."


Options[MovingMedian] = {RepeatedSmoothing -> False}

Begin["`Private`"]

(* for vectors and matrices, use ListConvolve for efficiency *)

MovingAverage[data_?(VectorQ[#,NumericQ]&),(n_Integer)?Positive] := ListConvolve[Table[1/n,{n}], data]

(* if the data are non-numeric, use explicit division by n to get simpler form *)

MovingAverage[data_?VectorQ,(n_Integer)?Positive] := ListConvolve[Table[1,{n}], data]/n

MovingAverage[data_?(MatrixQ[#,NumericQ]&),(n_Integer)?Positive] := 
	Transpose[Map[ListConvolve[Table[1/n,{n}], #]&, Transpose[data]]]

(* if the data are non-numeric, use explicit division by n to get simpler form *)

MovingAverage[data_?MatrixQ,(n_Integer)?Positive] := 
	Transpose[Map[ListConvolve[Table[1,{n}], #]&, Transpose[data]]]/n

(* for general lists, compute sums recursively then divide *)

MovingAverage[data_List,(n_Integer)?Positive] := Module[
	{sumval=Apply[Plus,data[[Range[n]]]]},
	Join[{sumval},Table[sumval=sumval-data[[i]]+data[[i+n]],
	  {i,1,Length[data]-n}]]/n
	]


MovingMedian[data_List, n_Integer?Positive, opts___Rule] := Module[
	{result=iMovingMedian[data,n,opts]},
	result/;FreeQ[result,MovingMedianR]
	]

iMovingMedian[data_, n_, opts___] := Module[
	{repeatedsmoothing},
	repeatedsmoothing = RepeatedSmoothing/. {opts}/.Options[MovingMedian];
	If[repeatedsmoothing === True, MovingMedianR[data,n], movmed[data,n]]
	]

movmed[data_, n_] := Developer`PartitionMap[Median, data, n, 1]


MovingMedianR[data_,n_] := Module[
	{olddata = data, newdata = MovingMedianC[data,n]}, 
	While[!SameQ[newdata,olddata],
		olddata = newdata;
        newdata = MovingMedianC[olddata,n]
		];
	Drop[Drop[newdata,(n-1)/2],-(n-1)/2]
	] /; If[OddQ[n],True,Message[MovingMedian::even];False]

MovingMedianC[data_,n_] :=
	Join[Take[data,Floor[(n-1)/2]], movmed[data,n], 
	Take[data,-Ceiling[(n-1)/2]]]

MovingMedian::even =
"The span of the moving median should be odd for repeated smoothing to converge."


(* for vectors and matrices, use ListConvolve for efficiency *)

LinearFilter[data_?VectorQ, (theta_)?VectorQ] := ListConvolve[theta, data]

LinearFilter[data_?MatrixQ,(theta_)?VectorQ] := Transpose[Map[ListConvolve[theta, #]&, 
	Transpose[data]]]

(* the general list case *)	

LinearFilter[data_List, (theta_)?VectorQ] := Table[
	Reverse[theta].Take[data, {i, i + Length[theta] - 1}]
		, 
		{i, Length[data] - Length[theta] + 1}]
	
LinearFilter[data_List, theta:{_?MatrixQ..}] := Module[
	{l1 = Length[theta]-1, temp = Apply[Join,Transpose[theta],{1}]}, 
   	Transpose[temp.Transpose[Drop[
	  Flatten /@ Transpose[NestList[RotateRight, data, l1]], l1]]]
	] /; ListQ[data[[1]]]

LinearFilter[(data_)?VectorQ, (theta_)?MatrixQ] :=
	Transpose[Map[ListConvolve[#, data] &, Transpose[theta]]]

ExponentialSmoothing[data_?MatrixQ,a_?MatrixQ, xstart_:Automatic]:=
        Module[{x0=If[xstart===Automatic,data[[1]],xstart]},
        Drop[FoldList[Function[{x,y},x+a.(y-x)],x0,data],1]]

ExponentialSmoothing[data_List, a_, xstart_:Automatic]:=
        Module[{x0=If[xstart===Automatic,data[[1]],xstart]},
        Drop[FoldList[Function[{x,y},x+a(y-x)],x0,data],1]
        ]

End[ ]
EndPackage[ ]

