(*:Mathematica Version: 3.0 *)

(*:Package Version: 1.2 *)

(*:Name: Statistics`DiscreteDistributions` *)

(*:Context: Statistics`DiscreteDistributions` *)

(*:Title: Discrete Statistical Distributions *)

(*:Author: David Withoff *)

(*:History:
  David Withoff (Wolfram Research), April 1990.
  Revised March 1991.
  Improved parameter checks for all distributions, added ExpectedValue and
	 RandomArray, speeded up some random number generation,
	 redefined Domain in terms of Range, ECM (Wolfram Research), 1995.
*)

(*:Copyright: Copyright 1990-2005, Wolfram Research, Inc. *)

(*:Reference: Usage messages only. *)

(*:Summary:
This package provides properties and functionals of discrete probability
distributions used in statistical computations.
*)

(*:Keywords: discrete distribution *)

(*:Requirements: No special system requirements. *)

(*:Limitations:
  Currently Sum does not accept the option Assumptions, as does Integrate.
  When Sum is revised to handle assumptions, the ExpectedValue rules 
  should be updated to make the needed assumptions about parameters whereever
  doing so will result in a simplified result.
*)

(*:Warning:
  This package extends the definition of several descriptive
  statistics functions and the definition of Random.  If the original
  usage message for such a function is reloaded, this change will
  not be reflected in the usage message, although the extended
  functionality will remain.
*)

(*:Sources:
  Norman L. Johnson and Samuel Kotz, Discrete Distributions, 1969.
  Stephen Kokoska and Christopher Nevison, Statistical Tables and
    Formulae, 1989.
*)

BeginPackage["Statistics`DiscreteDistributions`",
             "Statistics`DescriptiveStatistics`",
	     "Statistics`Common`DistributionsCommon`"]

(* Extend the usage of Random.  Look for the indicated phrase
        to determine if this has already been done.
*)

If[StringQ[Random::usage] &&
     StringPosition[Random::usage, "specified statistical distribution"] === {},

Random::usage = Random::usage <> " " <>
"Random[distribution] gives a random number with the specified
statistical distribution."

]


$NewMessage[Random, "randt"];
If[StringQ[Random::randt] &&
  StringPosition[Random::randt, "statistical"] === {},
    Random::randt = Random::randt <>
       " It may also be a valid statistical distribution with numeric parameters."
];

 	(* Distributions *)

BernoulliDistribution::usage =
"BernoulliDistribution[p] represents the Bernoulli distribution with
probability p.  A BernoulliDistribution[p] random variable takes on the
values 1 and 0 with probabilities p and 1-p, respectively."

BinomialDistribution::usage =
"BinomialDistribution[n, p] represents the binomial distribution
for n trials with probability p.  A BinomialDistribution[n, p] random
variable describes the number of successes occurring in n trials,
where the probability of success in each trial is p."

DiscreteUniformDistribution::usage =
"DiscreteUniformDistribution[n] represents the discrete uniform
distribution with n possible outcomes. A DiscreteUniformDistribution[n]
random variable takes on the values 1 through n with equal probability." 

GeometricDistribution::usage =
"GeometricDistribution[p] represents the geometric distribution
with probability p.  A GeometricDistribution[p] random variable describes
the number of trials occurring before the first success occurs, where
the probability of success in each trial is p."

HypergeometricDistribution::usage =
"HypergeometricDistribution[n, nsucc, ntot] represents the
hypergeometric distribution for a sample of size n, drawn without
replacement from a population with nsucc successes and total size
ntot.  A HypergeometricDistribution[n, nsucc, ntot] random variable
describes the number of successes occurring in the sample."

LogSeriesDistribution::usage =
"LogSeriesDistribution[theta] represents the logarithmic series
distribution with parameter theta."

NegativeBinomialDistribution::usage =
"NegativeBinomialDistribution[n, p] represents the negative binomial
distribution with parameters n and p. If n is an integer, a
NegativeBinomialDistribution[n, p] random variable describes the number
of failures that occur before n successes occur, where the probability
of success in each trial is p."

PoissonDistribution::usage =
"PoissonDistribution[mu] represents the Poisson distribution with mean mu."

(*
  Extend usage messages for functions defined in DescriptiveStatistics.m.
  Look for the indicated phrase to determine if this has already been done.
*)

If[StringQ[Mean::usage] &&
   StringPosition[Mean::usage, "specified statistical distribution"] === {},

If[StringQ[ExpectedValue::usage],
ExpectedValue::usage = ExpectedValue::usage <> " " <>
"ExpectedValue[f, distribution] gives the expected value of the pure function
f with respect to the specified statistical distribution.
ExpectedValue[f, distribution, x] gives the expected value of the
function f of x with respect to the specified univariate statistical
distribution for x."];
 
Mean::usage = Mean::usage <> " " <>
"Mean[distribution] gives the mean of the specified statistical distribution.";
 
If[StringQ[Variance::usage],
Variance::usage = Variance::usage <> " " <>
"Variance[distribution] gives the variance of the specified statistical
distribution."];
 
If[StringQ[StandardDeviation::usage],
StandardDeviation::usage = StandardDeviation::usage <> " " <>
"StandardDeviation[distribution] gives the standard deviation of the
specified statistical distribution."];
 
If[StringQ[Skewness::usage],
Skewness::usage = Skewness::usage <> " " <>
"Skewness[distribution] gives the coefficient of skewness of the
specified statistical distribution."];
 
If[StringQ[Kurtosis::usage],
Kurtosis::usage = Kurtosis::usage <> " " <>
"Kurtosis[distribution] gives the coefficient of kurtosis of the
specified statistical distribution."];
 
If[StringQ[KurtosisExcess::usage],
KurtosisExcess::usage = KurtosisExcess::usage <> " " <>
"KurtosisExcess[distribution] gives the kurtosis excess for the
specified statistical distribution."];
 
If[StringQ[Quantile::usage],
Quantile::usage = Quantile::usage <> " " <>
"Quantile[distribution, q] gives the q-th quantile of the specified
statistical distribution."]
]

(* Unprotect symbols that definitions will be attached to *)
Unprotect[BernoulliDistribution, BinomialDistribution,
	DiscreteUniformDistribution, GeometricDistribution,
	HypergeometricDistribution, LogSeriesDistribution,
	NegativeBinomialDistribution, PoissonDistribution];

Begin["`Private`"]

(* Someday Sum will take the Assumptions option, so allow ExpectedValue to
	take this option as well, even though it is not effective at doing
	anything yet! *)
Options[ExpectedValue] = {Assumptions -> {}}

ExpectedValue::sum =
"In the case of discrete distributions, the Assumptions option of
ExpectedValue is ignored.  Sum does not currently accept Assumptions."


(* ========================================================================= *)
(* Bernoulli Distribution *)

BernoulliDistribution/: ParameterQ[BernoulliDistribution[p_], warnflag_:True] := And[
        If[FreeQ[N[p], Complex], True,
           If[warnflag, Message[BernoulliDistribution::realparm, p]]; False],
        If[0 <= N[p] <= 1, True,
           If[warnflag, Message[BernoulliDistribution::probparm, p]]; False, True]
]

BernoulliDistribution::probparm =
"Parameter `1` is expected to be a probability between 0 and 1."

BernoulliDistribution::realparm = "Parameter `` is expected to be real."

BernoulliDistribution/: DomainQ[BernoulliDistribution[p_], list_?VectorQ] :=
	(Scan[If[!TrueQ[#==0 || #==1], Return[False]]&, list] =!= False)/;
	ParameterQ[BernoulliDistribution[p]]
BernoulliDistribution/: DomainQ[BernoulliDistribution[p_], x_] :=
	TrueQ[x==0 || x==1]/;
	ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: Domain[BernoulliDistribution[p_]]:={0, 1} /;
	ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: PDF[BernoulliDistribution[p_], x_] :=
   Piecewise[{{1 - p, x == 0}, {p, x == 1}}] /;
        ParameterQ[BernoulliDistribution[p]] && Head[x] =!= List

BernoulliDistribution/: CDF[BernoulliDistribution[p_], x_] := 
	Piecewise[{{1 - p, 0 <= x < 1}, {1, x >= 1}}] /;
        ParameterQ[BernoulliDistribution[p]] && Head[x] =!= List

BernoulliDistribution/:
    Mean[BernoulliDistribution[p_]] := p /; ParameterQ[
		BernoulliDistribution[p]]

BernoulliDistribution/:
    Variance[BernoulliDistribution[p_]] := p (1-p) /; ParameterQ[
		BernoulliDistribution[p]]

BernoulliDistribution/: StandardDeviation[BernoulliDistribution[p_]] :=
    Sqrt[p (1-p)] /; ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: Skewness[BernoulliDistribution[p_]] :=
    (1 - 2 p)/Sqrt[p (1 - p)] /; ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: Kurtosis[BernoulliDistribution[p_]] :=
    3 + (1 - 6 p (1-p))/(p (1-p)) /; ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: KurtosisExcess[BernoulliDistribution[p_]] :=
    (1 - 6 p (1-p))/(p (1-p)) /; ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: CharacteristicFunction[
        BernoulliDistribution[p_], t_] := 1 - p + p Exp[I t] /; ParameterQ[
				BernoulliDistribution[p]]

BernoulliDistribution/: Quantile[BernoulliDistribution[p_], q_] :=
    Which[q <= 1-p, 0, q > 1-p, 1] /; QuantileQ[q] && ParameterQ[
				BernoulliDistribution[p]]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

bernoulli = Compile[{{p, _Real}}, If[Random[] > p, 0, 1]]

BernoulliDistribution/:
    Random[BernoulliDistribution[p_?NumericQ]?ParameterQ] :=
        bernoulli[p]
        
bernoulliarray = Compile[{{p, _Real}, {len, _Integer}}, 
          Table[If[Random[] > p, 0, 1], {len}]]

BernoulliDistribution/:
  RandomArray[BernoulliDistribution[p_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = bernoulliarray[p, m];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])
  		
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: Someday add Assumptions here regarding parameter p *)
BernoulliDistribution/: ExpectedValue[f_Function, BernoulliDistribution[p_],
	opts___?OptionQ] :=
  (If[{opts} =!= {}, Message[ExpectedValue::sum]];
   Sum[f[x] PDF[BernoulliDistribution[p], x], {x, 0, 1}]
  )  /; ParameterQ[BernoulliDistribution[p]]

BernoulliDistribution/: ExpectedValue[f_, BernoulliDistribution[p_], x_Symbol,
	opts___?OptionQ] :=
  (If[{opts} =!= {}, Message[ExpectedValue::sum]];
   Sum[f PDF[BernoulliDistribution[p], x], {x, 0, 1}]
  )  /;		ParameterQ[BernoulliDistribution[p]]


(* ========================================================================= *)
(* Binomial Distribution, K & J, DD, Chap. 3 *)

BinomialDistribution/: ParameterQ[BinomialDistribution[n_, p_]] := And[
    If[FreeQ[N[n], Complex], True,
       Message[BinomialDistribution::realparm, n]; False],
    If[FreeQ[N[p], Complex], True,
       Message[BinomialDistribution::realparm, p]; False],
    If[N[n] > 0, If[IntegerQ[n], True,
        Message[BinomialDistribution::posint, n]; False],
        Message[BinomialDistribution::posint, n]; False, True],
    If[0 <= N[p] <= 1, True,
        Message[BinomialDistribution::probparm, p]; False, True]
]

BinomialDistribution::posint =
"The parameter `1` describing the number of trials is expected to be a
positive integer."

BinomialDistribution::probparm =
"The parameter `1` is expected to be a probability between 0 and 1."

BinomialDistribution::realparm = "Parameter `` is expected to be real."

BinomialDistribution/: DomainQ[BinomialDistribution[n_, p_], list_?VectorQ] :=
 (Scan[If[!TrueQ[IntegerQ[#] && 0 <= # <= n], Return[False]]&, list] =!= False)/; 
 	ParameterQ[BinomialDistribution[n, p]]
BinomialDistribution/: DomainQ[BinomialDistribution[n_, p_], x_] :=
	IntegerQ[x] && TrueQ[0 <= x <= n]/; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: Domain[BinomialDistribution[n_, p_]]:=
	Module[{domain, range = (Head[Range::range] === $Off)},
	  Off[Range::range];
	  domain = Range[0, n];
	  If[!range, On[Range::range]];
	  domain
	] /; ParameterQ[BinomialDistribution[n, p]]

(* out of range or nonintegral *)
BinomialDistribution/: PDF[BinomialDistribution[n_, p_], x_] := 0 /;
        ParameterQ[BinomialDistribution[n, p]] && NumberQ[N[x]] &&
        (!IntegerQ[x] || (x < 0 || x > n))
(* simplification of special cases *)
BinomialDistribution/: PDF[BinomialDistribution[n_, 0], n_] := 0 /;
        ParameterQ[BinomialDistribution[n, 0]]
BinomialDistribution/: PDF[BinomialDistribution[n_, 1], 0] := 0 /;
        ParameterQ[BinomialDistribution[n, 1]]
(* general case *)
BinomialDistribution/: PDF[BinomialDistribution[n_, p_], x_] :=
   Binomial[n, x] p^x (1-p)^(n-x) /;
        ParameterQ[BinomialDistribution[n, p]] && Head[x] =!= List

(* zero sum *)
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], x_] := 0 /;
        ParameterQ[BinomialDistribution[n, p]] &&
                        ((NumericQ[x] && Negative[x]) || (x === -Infinity))
(* unit sum *)
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], x_] := 1 /;
        ParameterQ[BinomialDistribution[n, p]] &&
                        ((NumericQ[x] && x >= n) || (x === Infinity))
(* simplification of special cases *)
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], n_] := 1 /;
        ParameterQ[BinomialDistribution[n, p]]
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], 0] := (1-p)^n /;
        ParameterQ[BinomialDistribution[n, p]]
BinomialDistribution/: CDF[BinomialDistribution[n_, 1], 0] := 0 /;
        ParameterQ[BinomialDistribution[n, 1]]
(* general case *)
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], x_] :=
   BetaRegularized[1 - p, n-Floor[x], Floor[x]+1] /;
        ParameterQ[BinomialDistribution[n, p]] &&
		  Head[x]=!=List &&
		  ((p>.01 && p<.99) || Internal`EffectivePrecision[p]===Infinity)
BinomialDistribution/: CDF[BinomialDistribution[n_, p_], x_] :=
   BetaRegularized[p, 1, Floor[x]+1, n-Floor[x]] /;
        ParameterQ[BinomialDistribution[n, p]] && Head[x]=!=List

BinomialDistribution/: Mean[BinomialDistribution[n_, p_]] := n p /;
       	ParameterQ[BinomialDistribution[n, p]] 

BinomialDistribution/: Variance[BinomialDistribution[n_, p_]] :=
    n p (1-p) /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/:
        StandardDeviation[BinomialDistribution[n_, p_]] :=
    Sqrt[n p (1-p)] /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: Skewness[BinomialDistribution[n_, p_]] :=
    (1 - 2 p)/Sqrt[n p (1-p)] /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: Kurtosis[BinomialDistribution[n_, p_]] :=
    3 + (1 - 6 p (1-p))/(n p (1-p)) /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: KurtosisExcess[BinomialDistribution[n_, p_]] :=
    (1 - 6 p (1-p))/(n p (1-p)) /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: CharacteristicFunction[
        BinomialDistribution[n_, p_], t_] :=
    (1 - p + p Exp[I t])^n /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/:
        Quantile[BinomialDistribution[n_, p_], q_] :=
		n /; ParameterQ[BinomialDistribution[n, p]] && TrueQ[(q-1)==0]

BinomialDistribution/:
        Quantile[BinomialDistribution[n_, p_], q_] :=
		0 /; ParameterQ[BinomialDistribution[n, p]] && TrueQ[q==0]

BinomialDistribution/:
        Quantile[BinomialDistribution[n_, p_], q_] :=
    With[{result = iBinomialQuantile[n, p, q]},
        result /; result =!= Fail] /; 
    NumericQ[n] && NumberQ[N[p]] && ParameterQ[BinomialDistribution[n, p]] &&
								 QuantileQ[q]

iBinomialQuantile[n_, p_, q_] := Module[{low, high, mid},
  If[q==1, Return[n]];
  If[N[(1-p)^n] < q,
    low = 0; high = n;
    While[high - low > 1,
        mid = Floor[(high+low)/2];
        If[N[CDF[BinomialDistribution[n, p], mid]] < q,
            low = mid,
            high = mid,
            high = Fail; Break[]
        ]
    ]; high,
    0,
    Fail
  ]
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

binomial = Compile[{{n, _Real}, {p, _Real}},
    Apply[Plus, Table[If[Random[] > p, 0, 1], {n}]]	]

(* following boundary case is not really valid, but there was some code
   that depended on it *)
BinomialDistribution/:
  Random[BinomialDistribution[_?(#==0&), _]] := 0

BinomialDistribution/:
  Random[BinomialDistribution[n_?NumericQ, p_?NumericQ]?ParameterQ] :=
      binomial[n, p]


(* Next few routines written by Darren Glosemeyer and Daniel Lichtblau
for faster generation of arrays of Binomial randomly distributed variables.
2003-09 DARRENG *)

compiledBinom = 
  Compile[{{n, _Integer}, {p, _Real}, {tabmin, _Integer}, 
    {tabmax, _Integer}, {pdf, _Real}, {len, _Integer}},
      Module[{tabl, high, low, mid, res, q, cdf, minval, comppdf},
        minval = Max[0, tabmin - 1];
        If[tabmin > 1,
           cdf = N[BetaRegularized[1. - p, n - tabmin - 1, 2 + tabmin]],
           cdf = 0.
          ];
        comppdf = pdf;
        tabl = Join[Table[cdf = cdf + comppdf; 
              		comppdf = (n - j)/(j + 1)*p/(1 - p)*comppdf; 
              		cdf,
              		{j, tabmin, tabmax}]
              	   , 
              	   Table[1, {tabmax - tabmin}]
              	  ];
        res = Table[q = Random[];
            	If[tabl[[1]] < q, 
            	  high = 1;
              	  While[tabl[[high]] < q, high *= 2]; 
              	  low = Round[high/2];
              	  While[high - low > 1, mid = Round[(high + low)/2];
                  If[tabl[[mid]] < q, low = mid, high = mid]]; 
          	  high + tabmin - 1
          	  , minval], {len}];
        res]]  

Binom[n_Integer, p_?NumericQ, len_Integer] := 
  Module[{tabmin, tabmax, pdf},
    Which[p == 0, Table[0, {len}],
      	  p == 1, Table[n, {len}],
      	  True, 
      	  {tabmin, tabmax} = {Quantile[BinomialDistribution[n, p], 10.^-10], 
      		Quantile[BinomialDistribution[n, p], 1. - 10^-10]}; 
      	  pdf = Binomial[N[n], tabmin]*p^(tabmin)*(1 - p)^(n - tabmin); 
      	  compiledBinom[n, p, tabmin, tabmax, pdf, len]
      	]]

BinomialDistribution/:
  RandomArray[BinomialDistribution[n_?NumericQ, p_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[(n > 32 && m >= 150) || p == 0 || p == 1
    		, 
    		Binom[n, p, m]
    		, (* ELSE *) 
    		Table[binomial[n, p], {m}]
    		];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: Someday add Assumptions here regarding parameters n and p *)
BinomialDistribution/: ExpectedValue[f_Function,
		 BinomialDistribution[n_, p_], opts___?OptionQ] :=
   Module[{x, sum},
     (
     If[{opts} =!= {}, Message[ExpectedValue::sum]];
     sum
     ) /; (sum = Sum[ Evaluate[f[x] PDF[BinomialDistribution[n, p], x]],
		         Evaluate[{x, 0, n}] ];
	   FreeQ[sum, Sum])
   ] /; ParameterQ[BinomialDistribution[n, p]]

BinomialDistribution/: ExpectedValue[f_, BinomialDistribution[n_, p_],
		 x_Symbol, opts___?OptionQ] :=
  Module[{sum},
    (
    If[{opts} =!= {}, Message[ExpectedValue::sum]];
    sum
    ) /; (sum = Sum[ Evaluate[f PDF[BinomialDistribution[n, p], x]],
			Evaluate[{x, 0, n}] ];
          FreeQ[sum, Sum])
  ] /; ParameterQ[BinomialDistribution[n, p]]


(* ========================================================================= *)
(* Discrete Uniform Distribution, K & J, DD, Chap. 10, p. 239 *)

ParameterQ[DiscreteUniformDistribution[n_]] := And[
        If[FreeQ[N[n], Complex], True,
           Message[DiscreteUniformDistribution::realparm, n]; False],
        If[N[n] > 0, If[IntegerQ[n], True,
           Message[DiscreteUniformDistribution::posint, n]; False],
           Message[DiscreteUniformDistribution::posint, n]; False, True]
]

DiscreteUniformDistribution::posint =
"The parameter `1` is expected to be a positive integer."

DiscreteUniformDistribution::realparm = "Parameter `` is expected to be real."

DiscreteUniformDistribution/: DomainQ[DiscreteUniformDistribution[n_],
						list_?VectorQ]:=
 (Scan[If[!TrueQ[IntegerQ[#] && TrueQ[1 <= # <= n]], Return[False]]&, list] =!=
						False)/; ParameterQ[DiscreteUniformDistribution[n]]
DiscreteUniformDistribution/: DomainQ[DiscreteUniformDistribution[n_], x_]:=
	IntegerQ[x] && TrueQ[1 <= x <= n]/; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: Domain[DiscreteUniformDistribution[n_]]:=
	Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
	  domain = Range[n];
          If[!range, On[Range::range]];
          domain
	] /; ParameterQ[DiscreteUniformDistribution[n]]

(* out of range and nonintegral *)
DiscreteUniformDistribution/: PDF[DiscreteUniformDistribution[n_], x_] := 0 /;
        ParameterQ[DiscreteUniformDistribution[n]] &&
        NumberQ[N[x]] && (!IntegerQ[x] || (x < 1 || x > n))
(* general case *)
DiscreteUniformDistribution/: PDF[DiscreteUniformDistribution[n_], x_] :=
        1/n /;
        ParameterQ[DiscreteUniformDistribution[n]] && Head[x] =!= List

(* zero sum *)
DiscreteUniformDistribution/: CDF[DiscreteUniformDistribution[n_], x_] := 0 /;
        ParameterQ[DiscreteUniformDistribution[n]] &&
                ((NumericQ[x] && NonPositive[x]) || (x === -Infinity))
(* unit sum *)
DiscreteUniformDistribution/: CDF[DiscreteUniformDistribution[n_], x_] := 1 /;
        ParameterQ[DiscreteUniformDistribution[n]] &&
        ((NumericQ[x] && x >= n) || (x === Infinity))
(* general case *)
DiscreteUniformDistribution/: CDF[DiscreteUniformDistribution[n_], x_] :=
   Floor[x]/n /;
        ParameterQ[DiscreteUniformDistribution[n]] && Head[x] =!= List

DiscreteUniformDistribution/:
        Mean[DiscreteUniformDistribution[n_]] :=
    (n + 1) / 2 /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/:
        Variance[DiscreteUniformDistribution[n_]] :=
    (n^2 - 1)/12 /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/:
        StandardDeviation[DiscreteUniformDistribution[n_]] :=
    Sqrt[(n^2-1)/12] /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: Skewness[
        DiscreteUniformDistribution[n_]] := 0 /; ParameterQ[
					DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/:
        Kurtosis[DiscreteUniformDistribution[n_]] :=
    3/5 (3 - 4/(n^2-1)) /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: KurtosisExcess[
        DiscreteUniformDistribution[n_]] :=
    -(12/(n^2-1) + 6)/5 /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: CharacteristicFunction[
        DiscreteUniformDistribution[n_], t_] :=
  Module[{k, sum},
    If[FreeQ[sum = Sum[Exp[I t k],{k,1,n}]/n, Sum],
       sum,
       sum /. k -> Unique[System`K]
    ]
  ] /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: Quantile[
        DiscreteUniformDistribution[n_], q_] :=
    With[{nq = n q},
	     If[NumberQ[nq],
		Max[Ceiling[nq], 1],
		Max[Ceiling[N[nq]], 1]
	     ]
    ] /; ParameterQ[DiscreteUniformDistribution[n]] && QuantileQ[q]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

DiscreteUniformDistribution/:
  Random[DiscreteUniformDistribution[n_?NumericQ]?ParameterQ] :=
    Random[Integer, {1, n}]

DiscreteUniformDistribution/: RandomArray[
	DiscreteUniformDistribution[n_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Table[Random[Integer, {1, n}], {m}];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: someday add Assumptions here regarding parameter n *)
DiscreteUniformDistribution/: ExpectedValue[f_Function,
		DiscreteUniformDistribution[n_], opts___?OptionQ] :=
   Module[{x, sum},
    (
     If[{opts}=!={}, Message[ExpectedValue::sum]];
     sum
    ) /; (sum = Sum[ Evaluate[f[x] PDF[DiscreteUniformDistribution[n], x]],
	 		 Evaluate[{x, 1, n}] ];
	  FreeQ[sum, Sum])
   ] /; ParameterQ[DiscreteUniformDistribution[n]]

DiscreteUniformDistribution/: ExpectedValue[f_,
		DiscreteUniformDistribution[n_], x_Symbol, opts___?OptionQ] :=
  Module[{sum},
  (
	If[{opts}=!={}, Message[ExpectedValue::sum]];
	sum
  ) /; (sum = Sum[ Evaluate[f PDF[DiscreteUniformDistribution[n], x]],
	     Evaluate[{x, 1, n}] ];
        FreeQ[sum, Sum])
  ] /; ParameterQ[DiscreteUniformDistribution[n]]


(* ========================================================================= *)
(* Geometric Distribution, K & J, DD, Chap. 5, p. 122 *)

GeometricDistribution/: ParameterQ[GeometricDistribution[p_]] := And[
        If[FreeQ[N[p], Complex], True,
           Message[GeometricDistribution::realparm, p]; False],
        If[0 <= N[p] <= 1, True,
           Message[GeometricDistribution::probparm, p]; False, True]
]

GeometricDistribution::probparm =
"The parameter `1` is expected to be a probability between 0 and 1."

GeometricDistribution::realparm = "Parameter `` is expected to be real."

GeometricDistribution/: DomainQ[GeometricDistribution[p_], list_?VectorQ] :=
	(Scan[If[!TrueQ[IntegerQ[#] && # >= 0], Return[False]]&, list] =!= False)/;
        ParameterQ[GeometricDistribution[p]]
GeometricDistribution/: DomainQ[GeometricDistribution[p_], x_] :=
	IntegerQ[x] && x >= 0/;ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: Domain[GeometricDistribution[p_]]:=
        Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Range[0, Infinity];
          If[!range, On[Range::range]];
          domain
        ] /; ParameterQ[GeometricDistribution[p]]

(* out of range and nonintegral *)
GeometricDistribution/: PDF[GeometricDistribution[p_], x_] := 0 /;
        ParameterQ[GeometricDistribution[p]] &&
        (NumberQ[N[x]] && (!IntegerQ[x] || Negative[x]))
(* general case *)
GeometricDistribution/: PDF[GeometricDistribution[p_], x_] :=
   p (1-p)^x /;
        ParameterQ[GeometricDistribution[p]] && Head[x] =!= List

(* zero sum *)
GeometricDistribution/: CDF[GeometricDistribution[p_], x_] := 0 /;
        ParameterQ[GeometricDistribution[p]] &&
                ((NumericQ[x] && Negative[x]) || (x === -Infinity))
(* unit sum *)
GeometricDistribution/: CDF[GeometricDistribution[p_], Infinity] := 1 /;
        ParameterQ[GeometricDistribution[p]]
(* general case *)
GeometricDistribution/: CDF[GeometricDistribution[p_], x_] :=
   1 - (1-p)^Floor[x + 1] /;
        ParameterQ[GeometricDistribution[p]] && Head[x] =!= List

GeometricDistribution/:
    Mean[GeometricDistribution[p_]] := 1/p - 1 /; ParameterQ[
					GeometricDistribution[p]]

GeometricDistribution/: Variance[GeometricDistribution[p_]] :=
    (1 - p) / p^2 /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: StandardDeviation[GeometricDistribution[p_]] :=
    Sqrt[1 - p] / p /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: Skewness[GeometricDistribution[p_]] :=
    (2 - p)/Sqrt[1 - p] /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: Kurtosis[GeometricDistribution[p_]] :=
    (p^2 + 6 - 6 p) / (1 - p) + 3 /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: KurtosisExcess[GeometricDistribution[p_]] :=
    (p^2 + 6 - 6 p) / (1 - p) /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: CharacteristicFunction[
        GeometricDistribution[p_], t_] :=
    p / (1 - (1-p) Exp[I t]) /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: Quantile[GeometricDistribution[p_], q_] :=
    With[{result = iGeometricQuantile[p, q]},
        result /; result =!= Fail
    ] /; ParameterQ[GeometricDistribution[p]] && QuantileQ[q]

iGeometricQuantile[p_, q_] :=
   (If[q == 1, Return[Infinity]];
    With[{l = Log[1-q]/Log[1-p]},
      If[NumberQ[N[l, Precision[l]]],
	If[(NumericQ[p] && p < q) || (!NumericQ[p] && N[p] < q),
		Ceiling[N[l, Precision[l]]] - 1, 0, Fail],
	If[N[p < q], Ceiling[N[l]] - 1, 0, Fail]
      ]
    ])

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* The following code provided by Johannes Ludsteck is many times faster than
the previous code.  - DTERR, 6-02 *)

GeometricDistribution/:
  Random[GeometricDistribution[p_?NumericQ]?ParameterQ] := 
    Floor[Log[Random[]]/Log[1-p]]

GeometricDistribution/:
  RandomArray[GeometricDistribution[p_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = Floor[Log[Table[Random[],{m}]]/Log[1-p]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]
 
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: Someday add Assumptions here regarding parameter p *)
GeometricDistribution/: ExpectedValue[f_Function,
		 GeometricDistribution[p_], opts___?OptionQ] :=
   Module[{x, sum},
    (
     If[{opts}=!={}, Message[ExpectedValue::sum]];
     sum
    ) /; (sum = Sum[ Evaluate[f[x] PDF[GeometricDistribution[p], x]],
	 		 Evaluate[{x, 0, Infinity}] ];
	  FreeQ[sum, Sum])
   ] /; ParameterQ[GeometricDistribution[p]]

GeometricDistribution/: ExpectedValue[f_, GeometricDistribution[p_],
		 x_Symbol, opts___?OptionQ] :=
  Module[{sum},
  (
   If[{opts}=!={}, Message[ExpectedValue::sum]];
   sum
  ) /; (sum = Sum[ Evaluate[f PDF[GeometricDistribution[p], x]],
	     Evaluate[{x, 0, Infinity}] ];
        FreeQ[sum, Sum])
  ] /; ParameterQ[GeometricDistribution[p]]


(* ========================================================================= *)
(* Hypergeometric Distribution, K & J, DD, Chap. 6 *)

HypergeometricDistribution/: 
ParameterQ[HypergeometricDistribution[n_, nsucc_, ntot_]] := And[
If[FreeQ[N[n], Complex], True,
   Message[HypergeometricDistribution::realparm, n]; False],
If[FreeQ[N[nsucc], Complex], True,
   Message[HypergeometricDistribution::realparm, nsucc]; False],
If[FreeQ[N[ntot], Complex], True,
   Message[HypergeometricDistribution::realparm, ntot]; False],
If[N[n] >= 0,
    If[Floor[n]==n, True,
       Message[HypergeometricDistribution::nonnegint, n]; False],
    Message[HypergeometricDistribution::nonnegint, n]; False,
    True],
If[N[nsucc] >= 0,
    If[Floor[nsucc]==nsucc, True,
        Message[HypergeometricDistribution::nonnegint, nsucc]; False],
    Message[HypergeometricDistribution::nonnegint, nsucc]; False,
    True],
If[N[ntot] > 0,
    If[Floor[ntot]==ntot, True,
        Message[HypergeometricDistribution::posint, ntot]; False],
    Message[HypergeometricDistribution::posint, ntot]; False,
    True],
If[N[n] <= N[ntot], True,
    Message[HypergeometricDistribution::ntoobig, n, ntot]; False, True],
If[N[nsucc] <= N[ntot], True,
    Message[HypergeometricDistribution::nsucctoobig, nsucc, ntot]; False, True]
]

HypergeometricDistribution::nonnegint =
"The parameter `1` is expected to be a nonnegative integer."

HypergeometricDistribution::posint =
"The parameter `1` describing the size of the total population is expected to
be a positive integer."

HypergeometricDistribution::ntoobig =
"The number of trials `1` without replacement must be less than
the total number `2`."

HypergeometricDistribution::nsucctoobig =
"The number of possible successes `1` must be less than the total
number `2`."

HypergeometricDistribution::realparm = "Parameter `` is expected to be real."

HypergeometricDistribution/: DomainQ[
	HypergeometricDistribution[n_, nsucc_, ntot_], list_?VectorQ] :=
 (Scan[If[!TrueQ[
	IntegerQ[#] && TrueQ[Max[0,n-ntot+nsucc] <= # <= Min[n,nsucc]]
		 ], Return[False]]&, list] =!= False)/; 
		 ParameterQ[HypergeometricDistribution[n,nsucc,ntot]]
HypergeometricDistribution/: DomainQ[
	HypergeometricDistribution[n_, nsucc_, ntot_], x_] :=
	IntegerQ[x] && TrueQ[Max[0,n-ntot+nsucc] <= x <= Min[n,nsucc]]/; 
		ParameterQ[HypergeometricDistribution[n,nsucc,ntot]]

HypergeometricDistribution/:
   Domain[HypergeometricDistribution[n_, nsucc_, ntot_]]:=
	Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Range[Max[0,n-ntot+nsucc], Min[n,nsucc]];
          If[!range, On[Range::range]];
          domain
        ] /; ParameterQ[HypergeometricDistribution[n,nsucc,ntot]] &&
	!(MatchQ[{n,nsucc,ntot}, {ntot,ntot,ntot}] ||
	  MatchQ[{n,nsucc,ntot}, {ntot,0,ntot}] ||
	  MatchQ[{n,nsucc,ntot}, {0,ntot,ntot}] ||
	  MatchQ[{n,nsucc,ntot}, {0,0,ntot}])

HypergeometricDistribution/: Domain[HypergeometricDistribution[n_, n_, n_]]:=
	{n} /;	ParameterQ[HypergeometricDistribution[n,n,n]]

HypergeometricDistribution/: Domain[HypergeometricDistribution[l_, m_, n_]]:=
	{0} /;	ParameterQ[HypergeometricDistribution[l,m,n]] && 
	(MatchQ[{l,m}, {n,0}] || MatchQ[{l,m}, {0,n}] || MatchQ[{l,m}, {0,0}])

(* out of range and nonintegral *)
HypergeometricDistribution/: PDF[
   HypergeometricDistribution[n_, nsucc_, ntot_], x_] := 0 /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] &&
        (NumberQ[N[x]] &&
         (!IntegerQ[x] || (x < Max[0,n-ntot+nsucc] || x > Min[n,nsucc])))
(* special cases *)
HypergeometricDistribution/: PDF[
   HypergeometricDistribution[n_, n_, n_], 0] := 0 /;
        ParameterQ[HypergeometricDistribution[n, n, n]]
HypergeometricDistribution/: PDF[
   HypergeometricDistribution[l_, m_, n_], 0] := 1 /;
        ParameterQ[HypergeometricDistribution[l, m, n]] &&
        (MatchQ[{l,m}, {n,0}] || MatchQ[{l,m}, {0,n}] || MatchQ[{l,m}, {0,0}])
HypergeometricDistribution/: PDF[
   HypergeometricDistribution[l_, m_, n_], n_] := 0 /;
        ParameterQ[HypergeometricDistribution[l, m, n]] &&
        (MatchQ[{l,m}, {n,0}] || MatchQ[{l,m}, {0,n}] || MatchQ[{l,m}, {0,0}])
(* general case *)
HypergeometricDistribution/: PDF[
   HypergeometricDistribution[n_, nsucc_, ntot_], x_] :=
           Binomial[nsucc, x] Binomial[ntot-nsucc, n-x]/Binomial[ntot, n] /;
                ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] &&
                If[NumericQ[x],
                   IntegerQ[x] && Max[0,n-ntot+nsucc] <= x <= Min[n,nsucc],
                   Head[x] =!= List]

(* zero sum *)
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[n_, nsucc_, ntot_], x_] := 0 /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] &&
           ((NumericQ[x] && x < Max[0,n-ntot+nsucc]) || (x === -Infinity))
(* unit sum *)
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[n_, nsucc_, ntot_], x_] := 1 /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] &&
          ((NumericQ[x] && x >= Min[n,nsucc]) || (x === Infinity))
(* special cases *)
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[n_, n_, n_], 0] := 0  /;
                ParameterQ[HypergeometricDistribution[n, n, n]]
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[l_, m_, n_], 0] := 1  /;
        ParameterQ[HypergeometricDistribution[l, m, n]] &&
        (MatchQ[{l,m}, {n,0}] || MatchQ[{l,m}, {0,n}] || MatchQ[{l,m}, {0,0}])
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[l_, m_, n_], n_] := 1  /; 
   	ParameterQ[HypergeometricDistribution[l,m,n]]&&
        (l===n || l==0) && (m===n || m==0)
(* general case *)
HypergeometricDistribution/:
   CDF[HypergeometricDistribution[n_, nsucc_, ntot_], x_] :=
	Module[{k, sum, summand, iter,
	  prec=Precision[{n,nsucc,ntot,x}], infprec,
	  newn, newnsucc, newntot, newx},
		infprec = prec===Infinity;
		If [infprec,
			summand = Binomial[nsucc, k] *
			  Binomial[ntot-nsucc, n-k] / Binomial[ntot, n],
			{newn, newnsucc, newntot, newx} = N[{n,nsucc,ntot,x}, prec];
			summand = Binomial[newnsucc, k] *
			  Binomial[newntot-newnsucc, newn-k]
			 ];
		iter = {k, Max[0,n-ntot+nsucc], Min[Floor[x],n,nsucc]};
		If [infprec,
			If [FreeQ[sum = Sum[Evaluate[summand],Evaluate[iter]], Sum],
				sum, sum /. k->Unique[System`K]],
			If [FreeQ[sum = NSum[Evaluate[summand],Evaluate[iter],
			  Evaluate[WorkingPrecision->prec]], NSum],
				sum, sum /. k->Unique[System`K]] / Binomial[newntot, newn]
			]
	] /; ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] &&
          Head[x] =!= List

HypergeometricDistribution/:
        Mean[HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    n nsucc / ntot /; ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/:
        Variance[HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    (ntot-n)/(ntot-1) n nsucc/ntot (1-nsucc/ntot) /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: StandardDeviation[
        HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    Sqrt[(ntot-n)/(ntot-1) n nsucc (ntot-nsucc)]/ntot /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: Skewness[
        HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    (ntot-2 nsucc) (ntot-2 n) Sqrt[ntot-1]/
        ((ntot-2) Sqrt[n nsucc (ntot-nsucc) (ntot-n)]) /;
            ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: Kurtosis[
        HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    ntot^2 (ntot-1)/((ntot-2)(ntot-3) n nsucc (ntot-nsucc)(ntot-n))*
      (ntot (ntot+1) - 6 n (ntot-n) + 3 nsucc (ntot-nsucc)/ntot^2 *
        (ntot^2 (n-2) - ntot n^2 + 6 n (ntot-n))) /;
            ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: KurtosisExcess[
        HypergeometricDistribution[n_, nsucc_, ntot_]] :=
    Kurtosis[HypergeometricDistribution[n, nsucc, ntot]] - 3 /;
        ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: CharacteristicFunction[
        HypergeometricDistribution[n_, nsucc_, ntot_], t_] :=
	((ntot-n)! (ntot-nsucc)!/ntot!) *
	Hypergeometric2F1[-n, -nsucc, ntot-nsucc-n+1, E^(I t)] /;
		ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: Quantile[
        HypergeometricDistribution[n_, nsucc_, ntot_], q_] :=
    With[{result = iHypergeometricQuantile[n, nsucc, ntot, q]},
        result /; result =!= Fail
    ] /; ParameterQ[HypergeometricDistribution[n, nsucc, ntot]] && QuantileQ[q]

HypergeometricDistribution/: Quantile[
        HypergeometricDistribution[n_, nsucc_, ntot_], q_] :=
	 Min[n, nsucc] /; ParameterQ[
		HypergeometricDistribution[n, nsucc, ntot]] && TrueQ[(q-1)==0]

HypergeometricDistribution/: Quantile[
  HypergeometricDistribution[n_, nsucc_, ntot_], q_] :=
   Max[0, n + nsucc - ntot] /; ParameterQ[
		HypergeometricDistribution[n, nsucc, ntot]] && TrueQ[q==0]

iHypergeometricQuantile[n_, nsucc_, ntot_, q_] :=
Module[{kk = 0, sum, count = Max[0,n-ntot+nsucc]},
    If[q == 1, Return[Min[nsucc, n]]];
    sum = PDF[HypergeometricDistribution[n, nsucc, ntot], count];
    If[NumericQ[sum],
        While[sum < q,
            count++;
            sum += Binomial[nsucc, count] *
                    Binomial[ntot-nsucc, n-count]/
                        Binomial[ntot, n]
        ],
    (* else not a number *)
        count = Fail
    ];
    count
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

internalRandomHypergeometric[n_, nsucc_, ntot_] :=
    Quantile[HypergeometricDistribution[n,nsucc,ntot], Random[]]

HypergeometricDistribution/:
        Random[HypergeometricDistribution[
                 n_?NumericQ, nsucc_?NumericQ, ntot_?NumericQ]?ParameterQ] :=
		 internalRandomHypergeometric[n, nsucc, ntot]


(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of Hypergeometric randomly distributed variables. 2003-09 DARRENG *)


compiledHyperGeom = Compile[
  {{n, _Real}, {nsucc, _Real}, {ntot, _Real}, {pdf, _Real}, {cdf, _Real}, 
    {tabmin, _Integer}, {tabmax, _Integer}, {len, _Real}},
      Module[{tabl, high, low, mid, res, q, comppdf, compcdf},
        comppdf = pdf;
        compcdf = cdf;
	tabl = Join[Table[compcdf = compcdf + comppdf; 
      	  comppdf = ((n - x)*(nsucc - x))/((1 + x)*(1 - n - nsucc + ntot + x))*comppdf; 
      	  compcdf, {x, tabmin, tabmax}], Table[1, {tabmax - tabmin}]];
        res = Table[q = Random[];
            If[tabl[[1]] < q, high = 1;
              While[tabl[[high]] < q, high *= 2]; low = Round[high/2];
              While[high - low > 1, mid = Round[(high + low)/2];
              If[tabl[[mid]] < q, low = mid, high = mid]]; 
            high + tabmin - 1, tabmin], {len}];
        res]];
        
       
hyperGeom[n_, nsucc_, ntot_, m_] := Module[
  {pdf = Binomial[N[nsucc], Max[0, n - ntot + nsucc]] Binomial[N[ntot - nsucc], 
  	n - Max[0, n - ntot + nsucc]]/Binomial[N[ntot], n], cdf, tabmin, tabmax}, 
     If[MachineNumberQ[pdf],
      	{tabmin, tabmax} = {Max[0, n - ntot + nsucc], Min[n, nsucc]},
      	{tabmin, tabmax} = {Quantile[HypergeometricDistribution[n, nsucc, ntot], 
      		10^-10], Quantile[HypergeometricDistribution[n, nsucc, ntot], 1 - 10^-10]}];
     cdf = 0.; 
     Do[cdf = cdf + pdf; 
     	pdf = ((n - x)*(nsucc - x))/((1 + x)*(1 - n - nsucc + ntot + x))*pdf; 
     	cdf, 
     	{x, Max[0, n - ntot + nsucc], tabmin - 1}];
    compiledHyperGeom[n, nsucc, ntot, pdf, cdf, tabmin, tabmax, m]]
    
   
HypergeometricDistribution/:
	RandomArray[HypergeometricDistribution[
          n_?NumericQ, nsucc_?NumericQ, ntot_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    (* check special cases first *)
    array = Which[
    		n === ntot, (* sample is entire population *)
    		Table[nsucc, {m}]
    		, 
    		n===0 || nsucc===0, (* no chance for success *) 
    		Table[0, {m}]
    		, 
    		True,  (* general case *)
        	hyperGeom[n, nsucc, ntot, m]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* NOTE: Someday add Assumptions here regarding parameters n, nsucc, and ntot *)
HypergeometricDistribution/: ExpectedValue[f_Function,
	HypergeometricDistribution[n_, nsucc_, ntot_], opts___?OptionQ] :=
   Module[{x, sum},
    (
     If[{opts}=!={}, Message[ExpectedValue::sum]];
     sum
    ) /; (sum = Sum[ Evaluate[f[x] *
		 	  PDF[HypergeometricDistribution[n, nsucc, ntot], x]],
		 	 Evaluate[{x, Max[0,n-ntot+nsucc], Min[n,nsucc]}] ];
	  FreeQ[sum, Sum])
   ] /;	ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]

HypergeometricDistribution/: ExpectedValue[f_,
   HypergeometricDistribution[n_, nsucc_, ntot_], x_Symbol, opts___?OptionQ] :=
  Module[{sum},
   (
    If[{opts}=!={}, Message[ExpectedValue::sum]];
    sum
   ) /; (sum = Sum[
	     Evaluate[f PDF[HypergeometricDistribution[n, nsucc, ntot], x]],
	     Evaluate[{x, Max[0,n-ntot+nsucc], Min[n,nsucc]}] ];
         FreeQ[sum, Sum])
  ] /; ParameterQ[HypergeometricDistribution[n, nsucc, ntot]]


(* ========================================================================= *)
(* Logarithmic Series Distribution, K & J, DD, Chap. 7 *)

LogSeriesDistribution/: ParameterQ[LogSeriesDistribution[theta_]] := And[
    If[FreeQ[N[theta], Complex], True,
       Message[LogSeriesDistribution::realparm, theta]; False],
    If[0 <= N[theta] < 1, True,
        Message[LogSeriesDistribution::fracparm, theta]; False,
        True]
]

LogSeriesDistribution::fracparm =
"Parameter `1` is expected to be less than 1 and non-negative."

LogSeriesDistribution::realparm = "Parameter `` is expected to be real."


LogSeriesDistribution/: DomainQ[LogSeriesDistribution[theta_], list_?VectorQ] :=
	(Scan[If[!TrueQ[IntegerQ[#] && # >= 1], Return[False]]&, list] =!= False)/; 
		ParameterQ[LogSeriesDistribution[theta]]
LogSeriesDistribution/: DomainQ[LogSeriesDistribution[theta_], x_] :=
	IntegerQ[x] && x >= 1/; ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: Domain[LogSeriesDistribution[theta_]]:=
        Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Range[Infinity];
          If[!range, On[Range::range]];
          domain
        ] /; ParameterQ[LogSeriesDistribution[theta]]

(* out of range and nonintegral *)
LogSeriesDistribution/: PDF[LogSeriesDistribution[theta_], x_] := 0 /;
        ParameterQ[LogSeriesDistribution[theta]] &&
        (NumberQ[N[x]] && (!IntegerQ[x] || NonPositive[x]))
(* general case *)
LogSeriesDistribution/: PDF[LogSeriesDistribution[theta_], x_] :=
   theta^x / (-x Log[1-theta]) /;
        ParameterQ[LogSeriesDistribution[theta]] && Head[x] =!= List

(* zero sum *)
LogSeriesDistribution/: CDF[LogSeriesDistribution[theta_], x_] := 0 /;
        ParameterQ[LogSeriesDistribution[theta]] &&
          ((NumericQ[x] && NonPositive[x]) || (x === -Infinity))
(* unit sum *)
LogSeriesDistribution/: CDF[LogSeriesDistribution[theta_], Infinity] := 1 /;
        ParameterQ[LogSeriesDistribution[theta]]
(* general case *)	
LogSeriesDistribution/: CDF[LogSeriesDistribution[theta_], x_] :=
 Module[{k, sum},
   If[FreeQ[sum = Sum[ Evaluate[theta^k / k], Evaluate[{k, 1, x}] ], Sum],
      sum,
      sum /. k->Unique[System`K]
   ] / (-Log[1-theta])
 ] /; ParameterQ[LogSeriesDistribution[theta]] && Head[x] =!= List

LogSeriesDistribution/: Mean[LogSeriesDistribution[theta_]] :=
    theta/(1 - theta) / (-Log[1-theta]) /;
	 ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: Variance[LogSeriesDistribution[theta_]] :=
  Module[{alpha=-1/Log[1-theta]},
	alpha theta (1-alpha theta)/(1-theta)^2
  ]	/; ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: StandardDeviation[LogSeriesDistribution[theta_]] :=
  Module[{alpha=-1/Log[1-theta]},
	Sqrt[alpha theta (1-alpha theta)]/(1-theta)
  ]	/; ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: Skewness[LogSeriesDistribution[theta_]] :=
  Module[{alpha=-1/Log[1-theta]},
    (1 + theta - 3 alpha theta + 2 alpha^2 theta^2)/(1-alpha theta)/
	Sqrt[alpha theta(1-alpha theta)]
  ]	/; ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: Kurtosis[LogSeriesDistribution[theta_]] :=
  Module[{alpha=-1/Log[1-theta]},
    (1 + 4 theta + theta^2 - 4 alpha theta (1+theta) + 6 alpha^2 theta^2 -
	3 alpha^3 theta^3)/(alpha theta)/(1-alpha theta)^2
  ] 	/; ParameterQ[LogSeriesDistribution[theta]]
 
LogSeriesDistribution/: KurtosisExcess[LogSeriesDistribution[theta_]] :=
	Kurtosis[LogSeriesDistribution[theta]] - 3 /; ParameterQ[
					LogSeriesDistribution[theta]]
 
LogSeriesDistribution/:
        CharacteristicFunction[LogSeriesDistribution[theta_], t_] :=
	Log[1 - theta E^(t I)] / Log[1-theta]	/; ParameterQ[
                                        LogSeriesDistribution[theta]]
 
LogSeriesDistribution/:
        Quantile[LogSeriesDistribution[theta_], q_] :=
    With[{result = iLogSeriesQuantile[theta, q]},
        result /; result =!= Fail] /;
             ParameterQ[LogSeriesDistribution[theta]] && QuantileQ[q]

iLogSeriesQuantile[theta_, q_] := Module[{high, low, mid},
    If[q == 1, Return[Infinity]];
    If[N[theta/(-Log[1-theta])] < q,
        high = 2;
        While[N[CDF[LogSeriesDistribution[theta], high]] < q, high *= 2];
        low = high/2;
        While[high - low > 1,
            mid = (high+low)/2;
            If[N[CDF[LogSeriesDistribution[theta], mid]] < q,
                low = mid,
                high = mid,
                low = high
            ]
        ]; high,
    (* else *)
        1,
    (* indeterminate *)
        Fail
    ]
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

internalRandomLogSeries[theta_] :=
    Quantile[LogSeriesDistribution[theta],Random[]]

LogSeriesDistribution/:
   Random[LogSeriesDistribution[theta_?NumericQ]?ParameterQ] :=
       internalRandomLogSeries[theta]

(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of LogSeries randomly distributed variables. 2003-09 DARRENG *)


compiledLogSeries = Compile[
  {{theta, _Real}, {len, _Integer}}, 
    Module[{tabllen,  tabl, cdf, high, low, mid, res, q}, 
    	tabllen = 32;
      	cdf = 0.;
      	tabl = Table[cdf = cdf + theta^x/(-x Log[1 - theta]), {x, tabllen}];
      	res = Table[q = Random[];
          If[N[theta^2/(-2 Log[1 - theta])] < q, high = 1;
            While[tabl[[high]] < q, high *= 2;
              If[high > tabllen, tabllen *= 2;
                tabl = Join[tabl, 
                	Table[cdf = cdf + theta^x/(-x Log[1 - theta]), 
                		{x, tabllen/2 + 1, tabllen}]];]]; 
                low = Round[high/2];
            	While[high - low > 1, mid = Round[(high + low)/2];
             If[tabl[[mid]] < q, low = mid, high = mid]]; high, 1], {len}];
      res]]

logSeries[theta_?NumericQ, len_Integer] := 
  If[theta == 0, Table[1, {len}], 
    compiledLogSeries[theta, len]]

LogSeriesDistribution/:
  RandomArray[LogSeriesDistribution[theta_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[theta == 0 || m > 35, 
    		logSeries[theta, m]
    		, (* ELSE *)
    		Table[internalRandomLogSeries[theta], {m}]
    	];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
 
LogSeriesDistribution/: ExpectedValue[f_Function,
		LogSeriesDistribution[theta_], opts___?OptionQ] :=
   Module[{x, sum},
    (
     If[{opts} =!= {}, Message[ExpectedValue::sum]];
     sum
    ) /; (sum = Sum[ Evaluate[f[x] PDF[LogSeriesDistribution[theta], x]],
			 Evaluate[{x, 1, Infinity}] ];
	  FreeQ[sum, Sum])
   ] /; ParameterQ[LogSeriesDistribution[theta]]

LogSeriesDistribution/: ExpectedValue[f_, LogSeriesDistribution[theta_],
	 x_Symbol, opts___?OptionQ] :=
  Module[{sum},
  (
   If[{opts} =!= {}, Message[ExpectedValue::sum]];
   sum
  ) /; (sum = Sum[ Evaluate[ f PDF[LogSeriesDistribution[theta], x]],
       		Evaluate[ {x, 1, Infinity}]];
        FreeQ[sum, Sum])
  ] /; ParameterQ[LogSeriesDistribution[theta]]


(* ========================================================================= *)
(* Negative Binomial Distribution, K & J, DD, Chap. 5 *)

NegativeBinomialDistribution/: ParameterQ[
	NegativeBinomialDistribution[n_, p_]] := And[
    If[FreeQ[N[n], Complex], True,
       Message[NegativeBinomialDistribution::realparm, n]; False],
    If[FreeQ[N[p], Complex], True,
       Message[NegativeBinomialDistribution::realparm, p]; False],
    If[N[n] > 0, True,
            Message[NegativeBinomialDistribution::posparm, n]; False, True],
    If[0 < N[p] <= 1, True,
        Message[NegativeBinomialDistribution::probparm, p]; False, True]
]

NegativeBinomialDistribution::posparm =
"The parameter `1` is expected to be positive."

NegativeBinomialDistribution::probparm =
"The parameter `1` is expected to be a non-zero probability
between 0 and 1."

NegativeBinomialDistribution::realparm = "Parameter `` is expected to be real."

NegativeBinomialDistribution/: DomainQ[
	NegativeBinomialDistribution[n_, p_], list_?VectorQ] :=
	(Scan[If[!TrueQ[IntegerQ[#] && # >= 0], Return[False]]&, list] =!= False)/; 
		ParameterQ[NegativeBinomialDistribution[n, p]]
NegativeBinomialDistribution/: DomainQ[
	NegativeBinomialDistribution[n_, p_], x_] := IntegerQ[x] && x >= 0/; 
		ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/:
  Domain[NegativeBinomialDistribution[n_, p_]]:=
        Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Range[0, Infinity];
          If[!range, On[Range::range]];
          domain
        ] /; ParameterQ[NegativeBinomialDistribution[n, p]]

(* out of range and nonintegral *)
NegativeBinomialDistribution/: PDF[NegativeBinomialDistribution[n_, p_], x_] :=
   0 /; ParameterQ[NegativeBinomialDistribution[n, p]] &&
        (NumberQ[N[x]] && (!IntegerQ[x] || Negative[x]))
(* general case *)
NegativeBinomialDistribution/: PDF[NegativeBinomialDistribution[n_, p_], x_] :=
   Binomial[x+n-1, n-1] p^n (1-p)^x /;
        ParameterQ[NegativeBinomialDistribution[n, p]] && Head[x] =!= List

(* zero sum *)
NegativeBinomialDistribution/: CDF[NegativeBinomialDistribution[n_, p_], x_] :=
   0 /; ParameterQ[NegativeBinomialDistribution[n, p]] &&
                ((NumericQ[x] && Negative[x]) || (x === -Infinity))
(* unit sum *)
NegativeBinomialDistribution/: CDF[NegativeBinomialDistribution[n_, p_],
        Infinity] := 1 /; ParameterQ[NegativeBinomialDistribution[n, p]]
(* general case *)
NegativeBinomialDistribution/: CDF[NegativeBinomialDistribution[n_, p_], x_] :=
   BetaRegularized[p, n, Floor[x] + 1] /;
        ParameterQ[NegativeBinomialDistribution[n, p]] && Head[x]=!=List

NegativeBinomialDistribution/:
        Mean[NegativeBinomialDistribution[n_, p_]] :=
    n (1-p)/p /; ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/: Variance[
                NegativeBinomialDistribution[n_, p_]] :=
        n (1-p) / p^2 /; ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/: StandardDeviation[
                NegativeBinomialDistribution[n_, p_]] :=
        Sqrt[n (1-p)] / p /; ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/: Skewness[
                NegativeBinomialDistribution[n_, p_]] :=
        (2 - p) / Sqrt[n (1-p)] /; ParameterQ[
					NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/:
        Kurtosis[NegativeBinomialDistribution[n_, p_]] :=
    3 + (p^2 + 6 - 6 p) / (n (1-p)) /; ParameterQ[
					NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/:
        KurtosisExcess[NegativeBinomialDistribution[n_, p_]] :=
    (p^2 + 6 - 6 p) / (n (1-p)) /; ParameterQ[
					NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/: CharacteristicFunction[
        NegativeBinomialDistribution[n_, p_], t_] :=
    (p/(1 - (1-p) Exp[I t]))^n /; ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/:
        Quantile[NegativeBinomialDistribution[n_, p_], q_] :=
    With[{result = iNegativeBinomialQuantile[n, p, q]},
        result /; result =!= Fail] /;
            ParameterQ[NegativeBinomialDistribution[n, p]] && QuantileQ[q]

iNegativeBinomialQuantile[n_, p_, q_] := Module[{high, low, mid},
    If[q == 1, Return[Infinity]];
    If[N[p^n] < q,
        high = 1;
        While[N[CDF[NegativeBinomialDistribution[n, p], high]] < q, high *= 2];
        low = high/2;
        While[high - low > 1,
            mid = (high+low)/2;
            If[N[CDF[NegativeBinomialDistribution[n, p], mid]] < q,
                low = mid,
                high = mid,
                low = high
            ]
        ]; high,
    (* else *)
        0,
    (* indeterminate *)
        Fail
    ]
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

negativebinomial = Compile[{{r, _Real}, {p, _Real}}, 
	Module[{nsucc = 0, nfail = 0},
	  If[Random[] > p, nfail++, nsucc++];
	  While[nsucc < r, If[Random[] > p, nfail++, nsucc++]];
	  nfail
	] ]


(* special case of integer number of "successes" *)
NegativeBinomialDistribution/: Random[
  NegativeBinomialDistribution[r_Integer, p_?NumericQ]?ParameterQ] := negativebinomial[r, p]
  
(* general case, where r need only be positive *)  
NegativeBinomialDistribution/: Random[
  NegativeBinomialDistribution[r_?NumericQ, p_?NumericQ]?ParameterQ] := iNegativeBinomialQuantile[r, p, Random[]]

(* Next few routines written by Darren Glosemeyer for faster generation of 
arrays of NegativeBinomial randomly distributed variables. 2003-09 DARRENG *)


compiledNegBinom = 
  Compile[{{r, _Real}, {p, _Real}, {tabmin, _Integer}, 
    {tabmax, _Integer}, {pdf, _Real}, {len, _Integer}},
      Module[{tabl, high, low, mid, res, q, cdf, minval, comppdf},
        minval = Max[0, tabmin - 1];
        If[tabmin > 1,
          cdf = N[BetaRegularized[p, r, 2 + tabmin]],
          cdf = 0.
          ];
        comppdf = pdf;
        tabl = Join[Table[cdf = cdf + comppdf; 
        	   comppdf = ((j + r)*(1 - p))/(1 + j)*comppdf; cdf,
              	   {j, tabmin, tabmax}]
              	   , 
              	   Table[1, {tabmax - tabmin}]];
        res = Table[q = Random[];
            If[tabl[[1]] < q, high = 1;
              While[tabl[[high]] < q, high *= 2]; low = Round[high/2];
              While[high - low > 1, mid = Round[(high + low)/2];
                If[tabl[[mid]] <q, low = mid, high = mid]]; high + tabmin - 1, 
                  minval], {len}];
        res]]

NegBinom[r_, p_, len_] := 
   Module[{tabmin, tabmax, pdf},
     If[p == 1, 
        Table[0, {len}]
        , (* ELSE*)
       	{tabmin, tabmax} = {Quantile[NegativeBinomialDistribution[r, p], 10.^-10], 
       		Quantile[NegativeBinomialDistribution[r, p], 1. - 10^-10]};
       	pdf = Binomial[N[tabmin + r - 1], r - 1]*p^r*(1 - p)^(tabmin); 
       	compiledNegBinom[r, p, tabmin, tabmax, pdf, len]]]

NegativeBinomialDistribution/: RandomArray[
	NegativeBinomialDistribution[r_?NumericQ, p_?NumericQ]?ParameterQ, dim_] :=
  Module[{m, array},
    m = If[VectorQ[dim], Apply[Times, dim], dim];
    array = If[!IntegerQ[r]||(r > 32 && m >= 50) || (r > 5 && p < r*m/10000) || p == 1
    		, 
    		NegBinom[r, p, m]
    		, (* ELSE *) 
    		Table[negativebinomial[r, p], {m}]];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

NegativeBinomialDistribution/: ExpectedValue[f_Function,
	NegativeBinomialDistribution[r_, p_], opts___?OptionQ] :=
  Module[{x, sum},
   (
    If[{opts} =!= {}, Message[ExpectedValue::sum]];
    sum
   ) /; (sum =
	   Sum[ Evaluate[f[x] PDF[NegativeBinomialDistribution[r, p], x]],
	      Evaluate[{x, 0, Infinity}]];
	 FreeQ[sum, Sum])
  ] /; ParameterQ[NegativeBinomialDistribution[n, p]]

NegativeBinomialDistribution/: ExpectedValue[f_,
	 NegativeBinomialDistribution[r_, p_], x_Symbol, opts___?OptionQ] :=
  Module[{sum},
   (
    If[{opts} =!= {}, Message[ExpectedValue::sum]];
    sum
   ) /; (sum = Sum[ Evaluate[f PDF[NegativeBinomialDistribution[r, p], x]],
       		Evaluate[{x, 0, Infinity}] ];
         FreeQ[sum, Sum])
  ] /; ParameterQ[NegativeBinomialDistribution[n, p]]


(* ========================================================================= *)
(* Poisson Distribution, K & J, DD, Chap. 4 *)

PoissonDistribution/: ParameterQ[PoissonDistribution[mu_]] := And[
         If[FreeQ[N[mu], Complex], True,
            Message[PoissonDistribution::realparm, mu]; False],
         If[N[mu] > 0, True,
            Message[PoissonDistribution::posparm, mu]; False, True]
]

PoissonDistribution::posparm =
"Parameter `1` is expected to be greater than zero."

PoissonDistribution::realparm = "Parameter `` is expected to be real."

PoissonDistribution/: DomainQ[PoissonDistribution[mu_], list_?VectorQ] := 
	(Scan[If[!TrueQ[IntegerQ[#] && # >= 0], Return[False]]&, list] =!= False)/; 
		ParameterQ[PoissonDistribution[mu]]
PoissonDistribution/: DomainQ[PoissonDistribution[mu_], x_] := 
	IntegerQ[x] && x >= 0/; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: Domain[PoissonDistribution[mu_]]:=
        Module[{domain, range = (Head[Range::range] === $Off)},
          Off[Range::range];
          domain = Range[0, Infinity];
          If[!range, On[Range::range]];
          domain
        ] /; ParameterQ[PoissonDistribution[mu]]

(* out of range and nonintegral *)
PoissonDistribution/: PDF[PoissonDistribution[mu_], x_] := 0 /;
        ParameterQ[PoissonDistribution[mu]] &&
        (NumberQ[N[x]] && (!IntegerQ[x] || Negative[x]))
(* general case *)
PoissonDistribution/: PDF[PoissonDistribution[mu_], x_] :=
   Exp[-mu] mu^x / x! /;
        ParameterQ[PoissonDistribution[mu]] && Head[x] =!= List

(* zero sum *)
PoissonDistribution/: CDF[PoissonDistribution[mu_], x_] := 0 /;
        ParameterQ[PoissonDistribution[mu]] &&
                   ((NumericQ[x] && Negative[x]) || (x === -Infinity))
(* unit sum *)
PoissonDistribution/: CDF[PoissonDistribution[mu_], Infinity] := 1 /;
        ParameterQ[PoissonDistribution[mu]]
(* general case *)
PoissonDistribution/: CDF[PoissonDistribution[mu_], x_] :=
   GammaRegularized[Floor[x] + 1, mu] /;
        ParameterQ[PoissonDistribution[mu]] && Head[x] =!= List

PoissonDistribution/: Mean[PoissonDistribution[mu_]] :=
    mu /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: Variance[PoissonDistribution[mu_]] :=
    mu /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: StandardDeviation[PoissonDistribution[mu_]] :=
    Sqrt[mu] /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: Skewness[PoissonDistribution[mu_]] :=
    1/Sqrt[mu] /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: Kurtosis[PoissonDistribution[mu_]] :=
    3 + 1/mu /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: KurtosisExcess[PoissonDistribution[mu_]] :=
    1/mu /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/:
        CharacteristicFunction[PoissonDistribution[mu_], t_] :=
    Exp[mu (Exp[I t] - 1)] /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: Quantile[PoissonDistribution[mu_], q_] :=
    With[{result = iPoissonQuantile[mu, q]},
        result /; result =!= Fail
    ] /; ParameterQ[PoissonDistribution[mu]] && QuantileQ[q]
 
iPoissonQuantile[mu_, q_] := Module[{high, low, mid},
    If[q == 1, Return[Infinity]];
    If[N[Exp[-mu]] < q,
        high = 1;
        While[N[CDF[PoissonDistribution[mu], high]] < q, high *= 2];
        low = high/2;
        While[high - low > 1,
            mid = (high+low)/2;
            If[N[CDF[PoissonDistribution[mu], mid]] < q,
                low = mid,
                high = mid,
                low = high
            ]
        ]; high,
    (* else *)
        0,
    (* indeterminate *)
        Fail
    ]
]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

internalRandomPoisson[mu_] :=
    Quantile[PoissonDistribution[mu],Random[]]

PoissonDistribution/: Random[PoissonDistribution[mu_?NumericQ]?ParameterQ] :=
       internalRandomPoisson[mu]

(* Next few routines written by Darren Glosemeyer and Daniel Lichtblau
for faster generation of arrays of Poisson randomly distributed variables.
2003-08 DANL *)

compiledPRAC = 
    Compile[{{mu, _Real}, {tabmin, _Integer}, {tabmax, _Integer}, {len, _Integer, 1}}, 
      Module[ {tabl, high, low, mid, res, q, allrand, tabllen, minval}, 
      	allrand = Apply[Times, len]; 
        If[mu == 0, Return[Table[0, {allrand}]]]; 
        minval = Max[0, tabmin - 1];
        tabllen=tabmax-tabmin;
        tabl = Join[Table[GammaRegularized[1. + j, mu], {j, tabmin, tabmax}],Table[1,{tabllen}]]; 
        res = Table[ q = Random[]; 
            If [Exp[-mu] < q, high = 1; 
              While[tabl[[high]] < q, high *= 2]; 
              low = Round[high/2]; 
              While[high - low > 1, mid = Round[(high + low)/2]; 
                If[tabl[[mid]] < q, low = mid, high = mid ] ]; high+ tabmin - 1, 
              minval ], {allrand}];
	res ]];


uncompiledPRAC[mu_, tabmin_, tabmax_, {len__Integer}] := 
  Module[ {tabl, high, low, mid, res, q, allrand, tabllen, minval}, 
    allrand = Apply[Times, len]; 
        If[mu == 0, Return[Table[0, {allrand}]]]; 
        minval = Max[0, tabmin - 1];
        tabllen=tabmax-tabmin;
        tabl = Join[Table[GammaRegularized[1. + j, mu], {j, tabmin, tabmax}],Table[1,{tabllen}]]; 
        res = Table[ q = Random[]; 
            If [Exp[-mu] < q, high = 1; 
              While[tabl[[high]] < q, high *= 2]; 
              low = Round[high/2]; 
              While[high - low > 1, mid = Round[(high + low)/2]; 
                If[tabl[[mid]] < q, low = mid, high = mid ] ]; high+ tabmin - 1, 
              minval ], {allrand}];
	res ] 


PRAC[mu_, tabmin_, tabmax_, len_] := 
  If[MachineNumberQ[Exp[N[mu]]], 
      compiledPRAC[mu, tabmin, tabmax, {len}], uncompiledPRAC[mu, tabmin, tabmax, {len}]]


PoissonDistribution /:
  RandomArray[PoissonDistribution[mu_?NumericQ]?ParameterQ, dim_] :=
	Module[{m, array, tabmin, tabmax},
    m = If [VectorQ[dim], Apply[Times, dim], dim];
    {tabmin, tabmax} = {Quantile[PoissonDistribution[mu], 10.^-10], 
    	Quantile[PoissonDistribution[mu], 1. - 10^-10]};
    If [m<5||(mu>1000&&m<(tabmax-tabmin)/3),
	array = Table[internalRandomPoisson[mu], {m}]
	, (* else *)
	array=PRAC[mu, tabmin, tabmax, m]
	];
    If [VectorQ[dim] && Length[dim] > 1,
		Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
  		array
		]
  ] /; (IntegerQ[dim] && Developer`$MaxMachineInteger >= dim > 0) || 
  	(VectorQ[dim, (IntegerQ[#] && # > 0)&] && 
  		Developer`$MaxMachineInteger >= Apply[Times,dim])

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

PoissonDistribution/: ExpectedValue[f_Function,
		 PoissonDistribution[mu_], opts___?OptionQ] :=
  Module[{x, sum},
   (
    If[{opts} =!= {}, Message[ExpectedValue::sum]];
    sum
   ) /; (sum = Sum[ Evaluate[f[x] PDF[PoissonDistribution[mu], x]],
		        Evaluate[{x, 0, Infinity}]];
	 FreeQ[sum, Sum])
  ] /; ParameterQ[PoissonDistribution[mu]]

PoissonDistribution/: ExpectedValue[f_, PoissonDistribution[mu_],
	 x_Symbol, opts___?OptionQ] :=
 Module[{sum},
  (
   If[{opts} =!= {}, Message[ExpectedValue::sum]];
   sum
  ) /; (sum = Sum[ Evaluate[f PDF[PoissonDistribution[mu], x]],
       		Evaluate[{x, 0, Infinity}] ];
        FreeQ[sum, Sum])
 ] /; ParameterQ[PoissonDistribution[mu]]


End[]

(* Protect symbols that definitions are attached to *)
SetAttributes[{BernoulliDistribution, BinomialDistribution,
	DiscreteUniformDistribution, GeometricDistribution,
	HypergeometricDistribution, LogSeriesDistribution,
	NegativeBinomialDistribution, PoissonDistribution},
  {Protected, ReadProtected}];

EndPackage[]

