(*:Mathematica Version: 3.0 *)

(*:Package Version: 1.1 *)

(*:Name: Statistics`MultinormalDistribution` *)

(*:Context: Statistics`MultinormalDistribution` *)

(*:Title: Statistical Distributions Derived from the Multinormal Distribution *)

(*:Author: E. C. Martin *)

(*:Copyright: Copyright 1993-2005, Wolfram Research, Inc. *)

(* :History:
Original version by ECM (Wolfram Research), October 1993, May 1994.
Added error messages for various functions
	of the distributions MultinormalDistribution and
	MultivariateTDistribution;  added support for singular
	MultinormalDistribution: ECM, April 1997.
*)

(*:Summary:
This package provides properties and functionals of five standard
probability distributions derived from the multinormal (multivariate Gaussian)
distribution.  The five distributions are the multinormal distribution, the 
Wishart distribution, Hotelling's T-square distribution, the multivariate
Student t distribution, and the distribution of the quadratic form of a
multinormal variate.
*)

(*:Keywords: Continuous distribution, multinormal distribution,
	multivariate normal, multivariate Gaussian, Wishart, Hotelling,
	quadratic form, multivariate student t *)

(*:Requirements: No special system requirements. *)

(*:Warning:
This package extends the definition of several descriptive statistics
functions and the definition of Random.  If the original usage messages
are reloaded, this change will not be reflected in the usage message,
although the extended functionality will remain.
This package also adds rules to Series.
*)

(*:Discussion:
MultinormalDistribution[{mu}, {{sigma^2}}] is NormalDistribution[mu, sigma].
Random variable X ~ WishartDistribution[{{sigma^2}}, m] is the same as
	random variable (Y * sigma^2) where Y ~ ChiSquareDistribution[m].
Random variable X ~ HotellingTSquareDistribution[1, m] is the same as
	random variable Y^2 where Y ~ StudentTDistribution[m].
Random variable X ~ HotellingTSquareDistribution[p, m] is the same as
	random variable (Y * (m p/(m-p+1))) where
	Y ~ FRatioDistribution[p, m-p+1].
Mean[dist], where dist is the distribution of a random vector, yields a
vector of means, where each element is the mean of the corresponding
element in the random vector.  Similarly, Mean[dist], where dist is the
distribution of a random matrix, yields a matrix of means, where each
element is the mean of the corresponding element in the random matrix.
The same goes for Variance, Skewness, Kurtosis, and KurtosisExcess.
*)

(*:Reference:
  Norman L. Johnson and Samuel Kotz, Continuous Multivariate Distributions,
    Wiley, 1970.
  K. V. Mardia, J. T. Kent, and J. M. Bibby, Multivariate Analysis, 
    Academic Press, 1979.
  Y. L. Tong, The Multivariate Normal Distribution, Springer, 1990.
  A. M. Mathai and Serge B. Provost, Quadratic Forms in Random Variables:
    Theory and Applications, Marcel Dekker, Inc., 1992.
  J. Wishart, The generalized product moment distribution in samples from a
    normal multivariate population, Biometrika, v. 20A, 1928.
*)

(*:Limitations:
Singular multinormal distribution (i.e., determinant of covariance matrix is
zero) is not supported.

RegionProbability[MultinormalDistribution[], ellipsoid] is supported only
for ellipsoids corresponding to constant-probability contours of the 
multinormal distribution.

RegionProbability[MultivariateTDistribution[], ellipsoid] is supported only
for ellipsoids corresponding to constant-probability contours of the 
multivariate t distribution.

PDF[QuadraticFormDistribution[], x] and CDF[QuadraticFormDistribution[], x]
must be expanded using Series if a symbolic expression in x is sought.

Quantile is not implemented for
MultinormalDistribution, WishartDistribution, QuadraticFormDistribution, and
MultivariateTDistribution.

EllipsoidQuantile and RegionProbability are not implemented for 
singular multinormal distributions.  CDF is not implemented for singular
multinormal distributions of dimensionality greater than two.
*)

BeginPackage["Statistics`MultinormalDistribution`",
	     (* needed for MultivariateSkewness, MultivariateKurtosis,
		MultivariateKurtosisExcess *)
             "Statistics`MultiDescriptiveStatistics`",
	     (* needed for Mean, Variance, StandardDeviation, Skewness,
		Kurtosis, KurtosisExcess *)
             "Statistics`DescriptiveStatistics`",
	     (* needed for NormalDistribution and ChiSquareDistribution: *)
             "Statistics`NormalDistribution`",	
	     (* needed for CovarianceMatrix and CorrelationMatrix: *)
             "Statistics`Common`MultivariateCommon`",
	     (* needed for Domain, PDF, CDF, CharacteristicFunction,
		RegionProbability: *)
             "Statistics`Common`DistributionsCommon`",
	     (* needed for NoncentralChiSquareDistribution and
		ChiDistribution *)
             "Statistics`ContinuousDistributions`"]

(* vector generalization of NormalDistribution *)
MultinormalDistribution::usage =
"MultinormalDistribution[mu, sigma] represents the multivariate normal
(Gaussian) distribution with mean vector mu and covariance matrix sigma.
For a p-variate random vector to be distributed MultinormalDistribution[mu,
sigma], mu must be a p-variate vector, and sigma must be a p x p symmetric
positive definite matrix."

(* matrix generalization of ChiSquareDistribution *)
WishartDistribution::usage =
"WishartDistribution[sigma, m] represents the Wishart distribution with scale
matrix sigma and degrees of freedom parameter m.  For a p x p symmetric
positive definite random matrix to be distributed WishartDistribution[sigma, m],
sigma must be a p x p symmetric positive definite matrix, and m must be an
integer satisfying m >= p+1."

(* univariate generalization of StudentTDistribution, proportional to
	FRatioDistribution *)
HotellingTSquareDistribution::usage =
"HotellingTSquareDistribution[p, m] represents Hotelling's T-square
distribution with dimensionality parameter p and degrees of freedom
parameter m.  For the random variable m*(d.Inverse[M].d)  to be distributed
HotellingTSquareDistribution[p, m], d and M must be independently distributed,
with d distributed MultinormalDistribution[Table[0, {p}], IdentityMatrix[p]]
and M distributed WishartDistribution[IdentityMatrix[p], m], respectively."

QuadraticFormDistribution::usage =
"QuadraticFormDistribution[{a, b, c}, {mu, sigma}] represents the (univariate)
distribution of the quadratic form  z.a.z + b.z + c, where z is distributed
MultinormalDistribution[mu, sigma].  For the random variable z.a.z + b.z + c
to be distributed QuadraticFormDistribution[{a, b, c}, {mu, sigma}], a must be
a p x p symmetric matrix, b must be a p-variate vector, and c must be a scalar."

(* vector generalization of StudentTDistribution *)
MultivariateTDistribution::usage =
"MultivariateTDistribution[r, m] represents the multivariate T distribution
with correlation matrix r and degrees of freedom parameter m. For the
random vector {x1 Sqrt[m]/s, ..., xp Sqrt[m]/s} to be distributed
MultivariateTDistribution[r, m], the random vector {x1, ..., xp} and the
random value s must be independently distributed, with {x1, ..., xp}
distributed MultinormalDistribution[Table[0, {p}], r] and s distributed
ChiSquareDistribution[m], respectively."


(* ========================================================================
  Extend usage messages for functions defined in Common/DistributionsCommon.m
  Look for the indicated phrase to determine if this has already been done.
*)

If[StringQ[PDF::usage] && StringPosition[PDF::usage, "multivariate"] === {},

PDF::usage = PDF::usage <> " " <>
"The dimensionality of x must match the dimensionality of a multivariate
distribution.";
 
If[StringQ[CDF::usage],
CDF::usage = CDF::usage <> " " <>
"The dimensionality of x must match the dimensionality of a multivariate
distribution."];
 
If[StringQ[CharacteristicFunction::usage],
CharacteristicFunction::usage = CharacteristicFunction::usage <> " " <>
"When the distribution is multivariate, the dimensionality of t must match the
dimensionality of the distribution."];
 
If[StringQ[RegionProbability::usage],
RegionProbability::usage = RegionProbability::usage <> " " <>
"RegionProbability[dist, Ellipsoid[mu, radii, dir]] gives the cumulative
probability within Ellipsoid[mu, radii, dir], when the ellipsoid
corresponds to a constant-probability contour of the specified distribution."];

]


(* ========================================================================
  Extend usage messages for functions defined in DescriptiveStatistics.m.
  Look for the indicated phrase to determine if this has already been done.
*)
 
If[StringQ[Mean::usage] &&
     StringPosition[Mean::usage, "distribution is multivariate"] === {},

If[StringQ[ExpectedValue::usage],
ExpectedValue::usage = ExpectedValue::usage <> " " <>
"ExpectedValue[f, distribution, {x1, x2, ..., xp}] gives the expected value of
the function f of {x1, x2, ..., xp} with respect to the specified multivariate
statistical distribution for {x1, x2, ..., xp}."];
 
Mean::usage = Mean::usage <> " " <>
"Mean[distribution] gives the mean vector when the specified statistical
distribution is multivariate.";

If[StringQ[Variance::usage],
Variance::usage = Variance::usage <> " " <>
"Variance[distribution] gives the diagonal of the covariance matrix when the
specified statistical distribution is multivariate."];

If[StringQ[StandardDeviation::usage],
StandardDeviation::usage = StandardDeviation::usage <> " " <>
"StandardDeviation[distribution] gives the square root of the diagonal of the
covariance matrix when the specified statistical distribution is multivariate."];

If[StringQ[Skewness::usage],
Skewness::usage = Skewness::usage <> " " <>
"Skewness[distribution] gives the skewness for each of the marginal
distributions when the specified statistical distribution is multivariate."];

If[StringQ[Kurtosis::usage],
Kurtosis::usage = Kurtosis::usage <> " " <>
"Kurtosis[distribution] gives the kurtosis for each of the marginal
distributions when the specified statistical distribution is multivariate."];

If[StringQ[KurtosisExcess::usage],
KurtosisExcess::usage = KurtosisExcess::usage <> " " <>
"KurtosisExcess[distribution] gives the kurtosis excess for each of the
marginal distributions when the specified statistical distribution is
multivariate."];

]


(* =========================================================================
  Extend usage messages for functions defined in Common/MultivariateCommon.m
  and MultiDescriptiveStatistics.m.  Look for the indicated phrase to
  determine if this has already been done.
*)

If[StringQ[Mean::usage] && StringPosition[Mean::usage,
 "specified multivariate statistical distribution"] === {},

If[StringQ[CovarianceMatrix::usage],
CovarianceMatrix::usage = CovarianceMatrix::usage <> " " <>
"CovarianceMatrix[distribution] gives the covariance matrix of the specified
multivariate statistical distribution."];
 
If[StringQ[CorrelationMatrix::usage],
CorrelationMatrix::usage = CorrelationMatrix::usage <> " " <>
"CorrelationMatrix[distribution] gives the correlation matrix of the specified
multivariate statistical distribution."];
 
If[StringQ[MultivariateSkewness::usage],
MultivariateSkewness::usage = MultivariateSkewness::usage <> " " <>
"MultivariateSkewness[distribution] gives the coefficient of skewness of the
specified multivariate statistical distribution."];
 
If[StringQ[MultivariateKurtosis::usage],
MultivariateKurtosis::usage = MultivariateKurtosis::usage <> " " <>
"MultivariateKurtosis[distribution] gives the coefficient of kurtosis of the
specified multivariate statistical distribution."];
 
If[StringQ[MultivariateKurtosisExcess::usage],
MultivariateKurtosisExcess::usage = MultivariateKurtosisExcess::usage <> " " <>
"MultivariateKurtosisExcess[distribution] gives the kurtosis excess for the
specified multivariate statistical distribution."];

If[StringQ[EllipsoidQuantile::usage],
EllipsoidQuantile::usage = EllipsoidQuantile::usage <> " " <>
"EllipsoidQuantile[distribution, q] gives the ellipsoid centered on
Mean[distribution] containing a fraction q of the specified distribution."]

]


(* Unprotect MultinormalDistribution.m symbols. *)
Unprotect[MultinormalDistribution, WishartDistribution,
	  HotellingTSquareDistribution, QuadraticFormDistribution,
	  MultivariateTDistribution]

(* Unprotect DescriptiveStatistics.m symbols. *)
Unprotect[Mean, Variance, StandardDeviation, Skewness, Kurtosis,
         KurtosisExcess]

(* ======================================================================= *)
Begin["`Private`"]

(* Gives output similar to old SingularValues command *)
CompactSVD[mat_, opts___] := 
Module[{sv, U, W, V, n},
    (* Compute SVD *)
    sv = SingularValueDecomposition[mat, Min[Dimensions[mat]], opts];
    If[!ListQ[sv], Return[$Failed]];
    {U, W, V} = sv;
    (*extract the diagonal vector*)
    sv = Tr[W, List];
    (* determine the number of positive singular values *)
    n = Length[sv] - Count[sv, _?(#==0&)];
    If [n == 0,
        {U, W, V},
        {Take[U, All, n], Take[sv, n], Take[V, All, n]}
    ]
]

CholDecomp = CholeskyDecomposition;

Delta::usage =
"Delta[num, accgoal, precgoal] uses the number num, the accuracy goal accgoal,
and the precision goal precgoal to determine the delta to use in 
Chop[num, delta].  The complete result is a list {delta, accuracyGoalWarningQ,
precisionGoalWarningQ}, where the final two elements of the list indicate
whether the Automatic setting of AccuracyGoal or PrecisionGoal was used in
place of accgoal or precgoal."

Delta[num_, accgoal_, precgoal_] :=
  Module[{acc, prec},
  	   acc = Max[Accuracy[num], $MachinePrecision]-3;
	   prec = Max[Precision[num], $MachinePrecision]-3;
	   Which[accgoal === Automatic,
		 Null,
		 validGoalQ[accgoal],
		 acc = accgoal,
		 True,
		 AccuracyGoalWarning = True];
	   Which[precgoal === Automatic,
		 Null,
		 validGoalQ[precgoal],
		 prec = precgoal,
		 True,
		 PrecisionGoalWarning = True];
	   (* use the largest delta *)
	   {Max[10^(-acc), Abs[num] 10^(-prec)],
		TrueQ[AccuracyGoalWarning], TrueQ[PrecisionGoalWarning]}
  ] 

validGoalQ[goal_] := (goal === Infinity ||
	(NumberQ[goal] && FreeQ[goal, Complex]))



(* routine to find sign of determinant, with proper
   adjustments for machine-precision input in zero-testing
   per Rob Knapp advice.
   Det may be precomputed as second arg if needed elsewhere. *)
(* use of CholeskyDecomposition in borderline cases is an addition 
to the testing suggested by Rob *)

detSign[mat_, idet_:None] :=
    Module[{det = If[idet === None, Det[mat], idet], posdef},
       Which[
         (* case zero *)
           If[MachineNumberQ[det],
             (* if machine prec, use test based on matrix size *)
             (* the inequality test alone can fail to detect positive definiteness
                for high dimensional matrices, so use CholeskyDecomposition as a 
                second check if the inequality is met *)
               Abs[det] < 2 * $MachineEpsilon * matrixnorm[mat]&&
               (Not[posdef=FreeQ[
               	Internal`DeactivateMessages[CholeskyDecomposition[mat]]
               	, 
               	CholeskyDecomposition]]),
             (* otherwise if bignum, just use ZeroQ *)
               Developer`ZeroQ[det]
           ]
         , 
         0
         ,
         (* positive definiteness determined by Cholesky *)
         TrueQ[posdef]
         ,
         1
         ,
         (* case non-zero *)
         True
         , 
         Sign[det]
        ]
    ]

(* core definition of matrix norm, borrowed
   from LinearAlgebra`MatrixManipulation` *)
matrixnorm[mat_] := Re[N[Max[Apply[Plus, Abs[mat], 1]]]]


(* assumes a square matrix *)
DiagonalElements[matrix_] :=
    MapIndexed[Part[#1, #2[[1]]]&, matrix]

AboveDiagonalElements[matrix_] :=
    Flatten[MapIndexed[Drop[#1, #2[[1]]]&, matrix]]

ZeroVectorQ[v_] := Apply[And, Map[TrueQ[#==0]&, v]]

ZeroMatrixQ[m_] := Apply[And, Map[TrueQ[#==0]&, Flatten[m]]]

trace[m_] := Apply[Plus, DiagonalElements[m]]

ellipsoidalLocus[mu_, sigma_] :=
  Module[{esystem, esystemT, acc, prec, delta, sorted, r, dir},
    (* (x-mu).Inverse[sigma].(x-mu) == 1 *)
    If[Apply[And, Map[(Head[#] === DirectedInfinity)&, Flatten[sigma] ]],
       Return[Ellipsoid[mu, Table[Infinity, {Length[mu]}]]] ];	
    If[!FreeQ[esystem = Eigensystem[sigma], Eigensystem], Return[$Failed]];
    acc = Max[Accuracy[esystem], $MachinePrecision]-1;
    prec = Max[Precision[esystem], $MachinePrecision]-1;
    delta = Max[10^(-acc), Apply[Times, Abs[esystem[[1]]]] 10^(-prec)];
    (* radii are square roots of eigenvalues *)
    esystemT = Transpose[MapAt[Sqrt[Chop[#, delta]]&, esystem, 1]];
    (* sort semi-axes from largest to smallest *)
    sorted = Sort[esystemT, #1[[1]] > #2[[1]]&];
    {r, dir} = Transpose[sorted];
    Ellipsoid[mu, r, dir]
  ]

(* ======================== Multinormal Distribution ====================== *)

(* - - - - - - - - -  ParameterQ[MultinormalDistribution[]] - - - - - - - - - *)
(* Issue a warning message if MultinormalDistribution is singular.
	Don't bother checking additional parameters. *)
MultinormalDistribution/: ParameterQ[
	MultinormalDistribution[mu_, sigma_, ___]] :=
  (
   MatrixQ[sigma] && VectorQ[mu] &&
   Dimensions[sigma] === {Length[mu], Length[mu]} &&
   sigma === Transpose[sigma] &&
  Module[{nsigma, dets, acc, prec}, 
   (nsigma = If[Precision[sigma] === Infinity, N[sigma], sigma];
    If[Apply[And, Map[NumberQ, Flatten[nsigma] ]],
       Apply[And, Map[TrueQ[# > 0]&, DiagonalElements[sigma] ]] &&
       (dets = detSign[nsigma];
	    Which[dets > 0, True,
	     dets == 0, Message[ParameterQ::mnorm];  True,
	     dets < 0, False,
	     True, False]),
      (* If sigma isn't numeric, just assume it is OK. *)
      True])
  ]
  )

ParameterQ::mnorm =
"Warning: Singular multinormal distribution encountered."

(* Internal function; same as ParameterQ except that check is for
	 Det[sigma] > 0 .
*)
nonsingularMultinormalQ[mu_, sigma_, warnflag_:False] :=
  (
   MatrixQ[sigma] && VectorQ[mu] &&
   Dimensions[sigma] === {Length[mu], Length[mu]} &&
   If[sigma === Transpose[sigma], True,
      If[warnflag, Message[MultinormalDistribution::cmsym]];False] &&
  Module[{nsigma, dets, acc, prec},
   (
    nsigma = If[Precision[sigma] === Infinity, N[sigma], sigma];
    If[Apply[And, Map[NumberQ, Flatten[nsigma] ]],
       Apply[And, Map[TrueQ[# > 0]&, DiagonalElements[sigma] ]] &&
       (
	    dets = detSign[nsigma];
        If[dets > 0, True,
            If[TrueQ[warnflag], Message[MultinormalDistribution::cmdet]];False]),
       (* If sigma isn't numeric, just assume it is OK. *)
      True]) 
  ]
  )

MultinormalDistribution::cmsym =
"The covariance matrix must be symmetric.";

MultinormalDistribution::cmdet =
"The covariance matrix must have a positive determinant.";

(* Internal function; same as ParameterQ except that check is for
         Det[sigma] == 0 and no message is issued for a singular distribution.
*)
singularMultinormalQ[mu_, sigma_] :=
 (
  MatrixQ[sigma] && VectorQ[mu] &&
   Dimensions[sigma] === {Length[mu], Length[mu]} &&
   sigma === Transpose[sigma] &&
  Module[{nsigma, acc, prec},
   (
    nsigma = If[Precision[sigma] === Infinity, N[sigma], sigma];
    If[Apply[And, Map[NumberQ, Flatten[nsigma] ]],
       Apply[And, Map[TrueQ[# > 0]&, DiagonalElements[sigma] ]] &&
       (detSign[nsigma] === 0),
       (* If sigma isn't numeric, just assume it is OK. *)
      True])
  ] 
 )

(* Internal function; same as ParameterQ except that check is for
         Det[sigma] >= 0 and no message is issued for a singular distribution.
*)
validMultinormalQ[mu_, sigma_] :=
  (
   MatrixQ[sigma] && VectorQ[mu] &&
   Dimensions[sigma] === {Length[mu], Length[mu]} &&
   sigma === Transpose[sigma] &&
  Module[{nsigma, acc, prec},
   (nsigma = If[Precision[sigma] === Infinity, N[sigma], sigma];
    If[Apply[And, Map[NumberQ, Flatten[nsigma] ]],
       Apply[And, Map[TrueQ[# > 0]&, DiagonalElements[sigma] ]] &&
       TrueQ[detSign[nsigma] >= 0],
       (* If sigma isn't numeric, just assume it is OK. *)
      True])
  ]
  ) 


(* - - - -  finding auxiliery parameters of MultinormalDistribution[]] - - - *)
Options[MultinormalDistribution] = {Tolerance -> Automatic}

(* Compute lower triangular matrix for use in calculations involving
	nonsingular MultinormalDistribution. *)
MultinormalDistribution[mu_:{0}, sigma_:{{1}}, opts___?OptionQ] :=
  Module[{u},
    (
    MultinormalDistribution[mu, sigma, Transpose[u]]	
    ) /;  FreeQ[u = CholDecomp[sigma], CholDecomp]
  ] /; MatrixQ[sigma] && nonsingularMultinormalQ[mu, sigma, True]

(* Hide lower triangular matrix using Format. *)
MultinormalDistribution /: Format[MultinormalDistribution[mu_, sigma_, l_]] :=
    (
	MultinormalDistribution[mu, Short[sigma]]
    )

(* special rule for sigma = IdentityMatrix[p] *)
MultinormalDistribution[mu_?VectorQ, sigma_?MatrixQ, opts___?OptionQ] :=
  Module[{p},
    (
  	MultinormalDistribution[mu, sigma, sigma]
    ) /; (p = Length[mu];  sigma === IdentityMatrix[p])
  ] 

(* Compute singular value decomposition of sigma for
	use in calculations involving singular MultinormalDistribution. *)
MultinormalDistribution[mu_, sigma_, opts___?OptionQ] :=
  Module[{tol, nsigma, svd, u, w, v},
   (
	Message[MultinormalDistribution::sing];
	{u, w, v} = svd;
	MultinormalDistribution[mu, sigma, w, Transpose[v]]
   ) /; (tol = Tolerance /. Options[MultinormalDistribution] /. {opts};
	 nsigma = If[Precision[sigma] === Infinity, N[sigma], sigma];
	 svd = CompactSVD[nsigma, Tolerance -> tol];
	 If[And[ListQ[svd], Length[svd] == 3],
	    True,
	    Message[MultinormalDistribution::svd, nsigma, tol];  False]
         )
  ] /; MatrixQ[sigma] && singularMultinormalQ[mu, sigma]

MultinormalDistribution::sing =
"Warning: Singular multinormal distribution encountered."

MultinormalDistribution::svd = 
"Unable to find singular value decomposition of the covariance matrix using
SingularValueDecomposition[`1`, Tolerance -> `2`]."

(* Hide singular value decomposition using Format. *)
MultinormalDistribution /: Format[MultinormalDistribution[mu_, sigma_,
	 w_?VectorQ, v_?MatrixQ]] :=
    (
	MultinormalDistribution[mu, Short[sigma]]
    )


(* - - - - - - - - -  DomainQ[MultinormalDistribution[]] - - - - - - - - - - *)
(* NOTE: DomainQ should be fast so that a large data set, expected to
	follow a particular distribution, can be quickly checked 
	to see whether each point falls in the prescribed domain.
	Unfortunately, it is not possible to ignore the parameters
	of the multinormal (and avoid checking them), because the
	domain depends on whether the multinormal distribution is
	singular or not.

	To make the domain check of a data set as efficient as possible, 
		use DomainQ[data], rather than Map[DomainQ, data].
	The latter will check the distribution validity and singularity
	for every single data point, which is very wasteful.
*)
MultinormalDistribution/: DomainQ[MultinormalDistribution[mu_, sigma_, l___],
        list_?MatrixQ] :=
   (
   MatchQ[Dimensions[list], {_, Length[mu]}] && FreeQ[N[list], Complex]
   ) /; (Length[{l}] <= 1 || nonsingularMultinormalQ[mu, sigma]   ) 

MultinormalDistribution/: DomainQ[MultinormalDistribution[mu_, sigma_, l___],
        x_?VectorQ] :=
   (
   TrueQ[Length[x] == Length[mu] && FreeQ[N[x], Complex]]
   ) /; (Length[{l}] <= 1 || nonsingularMultinormalQ[mu, sigma]   ) 

(* p x p is the dimensionality of sigma *)
(* w is a k-vector giving the "nonzero" singular values of sigma *)
(* v is a k x p matrix; a list of k orthonormal vectors *) 
(* nullspace is a p-k x p matrix *)
MultinormalDistribution/: DomainQ[MultinormalDistribution[mu_, sigma_,
		 w_?VectorQ, v_?MatrixQ], list_?MatrixQ] :=
   Module[{nullspace, k = Length[w], zeros, acc, prec, delta},
     ( 
	MatchQ[Dimensions[list], {_, Length[mu]}] && 
                (
                (* nullspace is a p-k by p matrix *)
                (* #-mu   is a p-vector *)
                (* Map[List, #-mu] is a p by 1 matrix *)
                zeros = Flatten[Map[(nullspace.Map[List, (#-mu)])&, list]];
	        acc = Max[Accuracy[{v, list}], $MachinePrecision]-1;
		prec = Max[Precision[{v, list}], $MachinePrecision]-1;
		delta = Max[10^(-acc),
               Max[Abs[Cases[{v, list}, _Integer | _Real, Infinity]]] 10^(-prec)];
		zeros = Chop[zeros, delta];
                Apply[And, Map[TrueQ[# == 0]&, zeros]]
                )
     ) /; (
	   nullspace = NullSpace[v];
           If[FreeQ[nullspace, NullSpace],      
                True,   
                Message[DomainQ::mnorm, v];  False]     
          )
   ] (* 2.5.14 (p. 41) of Mardia, Kent, Bibby *)

(* p x p is the dimensionality of sigma *)
(* w is a k-vector giving the "nonzero" singular values of sigma *)
(* v is a k x p matrix; a list of k orthonormal vectors *) 
(* nullspace is a p-k x p matrix *)
MultinormalDistribution/: DomainQ[MultinormalDistribution[mu_, sigma_,
		w_?VectorQ, v_?MatrixQ], x_?VectorQ] :=
   Module[{nullspace, k = Length[w], zeros, acc, prec, delta},
     (
	TrueQ[Length[x] == Length[mu]] && 
		(	
		(* nullspace is a p-k by p matrix *)
		(* x-mu   is a p-vector *)   
 		(* Map[List, x-mu] is a p by 1 matrix *)
		zeros = Flatten[nullspace.Map[List, (x-mu)]];
	        acc = Max[Accuracy[{v, x}], $MachinePrecision]-1;
                prec = Max[Precision[{v, x}], $MachinePrecision]-1;
                delta = Max[10^(-acc),
                     Max[Abs[Cases[{v, x}, _Integer | _Real, Infinity]]] 10^(-prec)];
                zeros = Chop[zeros, delta];
		Apply[And, Map[TrueQ[# == 0]&, zeros]]
		)
     ) /; (
	   nullspace = NullSpace[v];
	   If[FreeQ[nullspace, NullSpace],
		True,
		Message[DomainQ::mnorm, v];  False]	
	  )
   ] (* 2.5.14 (p. 41) of Mardia, Kent, Bibby *)

DomainQ::mnorm =
"Unable to find NullSpace of ``."


(* - - - - - - - - - -  Domain[MultinormalDistribution[]] - - - - - - - - - - *)

(* NOTE: need to check for nonsingular sigma because a singular sigma
	implies that the domain is a hyperplane *)
MultinormalDistribution/: Domain[MultinormalDistribution[mu_?VectorQ, sigma_,
	 l___]] :=
    (
	Table[Interval[{-Infinity, Infinity}], {Length[mu]}]
    ) /; (Length[{l}] <= 1 || nonsingularMultinormalQ[mu, sigma]   )

MultinormalDistribution/: Domain[MultinormalDistribution[mu_, sigma_,
		 w_?VectorQ, v_?MatrixQ]] :=
 Module[{nullspace,
	 k = Length[w], p = Length[sigma], zeroMatrix, u, xVector, slotVector,
		vars, soln, fcn, rules},
     (
	zeroMatrix = Table[0, {p-k}]; (* 1 x (p-k) *)
	xVector = Table[Unique[u], {p}];
	slotVector = Table[Slot[i], {i, p}];
	vars = Drop[xVector, k];
	soln = Simplify[ Solve[
		Thread[(xVector - mu).Transpose[nullspace] == zeroMatrix],
		vars][[1]] ];
	fcn = Map[function, soln];
	rules = Thread[RuleDelayed[xVector, Evaluate[slotVector]]];
	fcn = (fcn /. rules)  /. {function :> Function};
	(* Define the domain of each of the first k variables to be
		Interval[{-Infinity, Infinity}].
	   The remaining p-k variables are dependent on the first k. *)
	Join[	Table[Interval[{-Infinity, Infinity}], {k}],
		fcn	]
     ) /; (
	   nullspace = NullSpace[v];
           If[FreeQ[nullspace, NullSpace],
                True,   
                Message[Domain::mnorm, v];  False]     
          )
 ] (* 2.5.14 (p. 41) of Mardia, Kent, Bibby *)
 
Domain::mnorm =
"Unable to find NullSpace of ``."

(* - - - - - - - - - -  PDF[MultinormalDistribution[], ] - - - - - - - - - - *)

MultinormalDistribution/: PDF[MultinormalDistribution[mu_, sigma_, l___],
	 x_?VectorQ] :=
 	(
	  1/Sqrt[2 Pi]^Length[x] * 1/Sqrt[Det[sigma]] *
	  Exp[-1/2 (x-mu).Inverse[sigma].(x-mu)]
 	) /;  (Length[{l}] <= 1  || nonsingularMultinormalQ[mu, sigma]   ) &&
 		TrueQ[Length[x] == Length[mu]]


(* p x p is the dimensionality of sigma *)
(* w is a k-vector giving the "nonzero" singular values of sigma *)
(* v is a k x p matrix; a list of k orthonormal vectors *) 
(* NOTE: we don't use "If" to restrict the domain for symbolic x.  It is
	assumed that the user will use the function Domain to determine the
	support of this PDF. *)
MultinormalDistribution/: PDF[MultinormalDistribution[mu_, sigma_, 
	w_?VectorQ, v_?MatrixQ], x_?VectorQ] :=
   Module[{nullspace, zeros, acc, prec, delta,
		 k = Length[w], product = Apply[Times, w],
		pseudoinverse = Transpose[v].DiagonalMatrix[1/w].v},
 	(
	  zeros = Flatten[nullspace.(x-mu)];
	  acc = Max[Accuracy[{v, x}], $MachinePrecision]-1;
          prec = Max[Precision[{v, x}], $MachinePrecision]-1;
          delta = Max[10^(-acc),
              Max[Abs[Cases[{v, x}, _Integer | _Real, Infinity]]] 10^(-prec)];
	  zeros = Chop[zeros, delta];
	  If[Apply[And, Map[NumberQ, x]] &&
		 !Apply[And, Map[TrueQ[# == 0]&, zeros]],
	     (* It is manifestly certain that the vector x falls outside of
			the region of support of the PDF. *)
	     0,
	     1/Sqrt[2 Pi]^k * 1/Sqrt[ product ] *
	  	Exp[-1/2 (x-mu).pseudoinverse.(x-mu)]
	  ]
 	) /; (
              nullspace = NullSpace[v];
              If[FreeQ[nullspace, NullSpace],
                True,
                Message[PDF::mnorm, v];  False]
             )
   ] (* 2.5.13 (p. 41) of Mardia, Kent, Bibby *)

PDF::mnorm =
"Unable to find NullSpace of ``."

(* - - - - - - - - - -  CDF[MultinormalDistribution[], ] - - - - - - - - - - *)
Options[CDF] = Join[Options[CDF], Options[NIntegrate]]



(* special case code for diagonal sigma.
   if sigma is diagonal, the CDF is a product of univariate normal CDFs 
*)

MultinormalDistribution/: CDF[MultinormalDistribution[mu_, sigma_, 
   lowertri___], x_?VectorQ, opts___] := 
      Apply[Times,
      	Table[(1/2)*(1 + Erf[(-mu[[i]] + x[[i]])/(Sqrt[2*sigma[[i, i]]])]), {i, Length[x]}]
      	]/; TrueQ[diagonalmatrixQ[sigma]] && (Length[{lowertri}] <= 1 || 
            nonsingularMultinormalQ[mu, sigma]) && Length[mu] >= 2 &&
            TrueQ[Length[x] == Length[mu]] && 
            Apply[And, Map[((NumberQ[#] || Head[#] === DirectedInfinity) && 
            	FreeQ[#, Complex]) &, N[x]]]
            	

(* p-dimensional case, p >= 2, special structure sigma:
	sigma = {{s1^2, s1 s2 l1 l2, s1 s3 l1 l3, s1 s4 l1 l4, ...},
		 {s2 s1 l2 l1, s2^2, s2 s3 l2 l3, s2 s4 l2 l4, ...},
		 {s3 s1 l3 l1, s3 s2 l3 l2, s3^2, s3 s4 l3 l4, ...}, ...},
   Reference:
   Y. L. Tong, The Multivariate Normal Distribution, Springer, 1990,
	Sect. 8.2.5, p. 192;
   NOTE: equation (8.2.13) of the reference has sign errors. 

   this method works for p==2 and often for p==3,
   but fails almost surely for p>3
*)


tryTong[mu_, sigma_, x_, lowertri_List, opts___] := 
   Module[{p, lambda, sqrtdiag, l1, l2, product, res},
     p = Length[x];
     sqrtdiag = Sqrt[DiagonalElements[sigma]];
     lambda = If[p == 2,
        Module[{l1 = sigma[[1, 2]], l2 = Sqrt[(Times @@ sqrtdiag)]}, 
          If[l1 >= 0, 
             l1 = Sqrt[l1]; {l1, l1}, 
             l1 = Sqrt[-l1]; {-l1, l1}]/l2],
             (*p > 2*)
        Module[{lam, l, temp, soln, zeros, acc, prec, delta, i, j},
          lam = Array[l, p];
          temp = lam sqrtdiag;
          Internal`DeactivateMessages[soln = Solve[Take[Flatten[Table[sigma[[i, j]] == Times @@ temp[[{i, j}]], 
                 {i, p - 1}, {j, i + 1, p}], 1], p], lam]];
          If[FreeQ[soln, Solve] && Length[soln] == 2,
          (*can only solve up to a sign, so Length[soln] == 2*)
                 lam = (lam /. soln)[[1]];
                 temp = lam sqrtdiag;
                 zeros = Flatten[Table[sigma[[i, j]] - 
                     (Times @@ temp[[{i, j}]]), {i, p - 1}, {j, i + 1, p}]];
                 acc = Max[Accuracy[sigma], $MachinePrecision] - 1;
                 prec = Max[Precision[sigma], $MachinePrecision] - 1;
                 delta = Max[10^(-acc), Abs[Det[sigma]] 10^(-prec)];
                 zeros = Chop[zeros, delta];
                 If[(And @@ Map[N[Abs[#]] <= 1 &, lam]) && (And @@ Map[(# == 0) &, zeros]), 
                     lam,
                     $Failed],
            $Failed] (*end If[FreeQ[soln, Solve] && ...]*)
            ] (*end Module*)
            ] ;(*end If[p == 2,]*)
    If[lambda =!= $Failed, 
        product = hh[(x - mu)/sqrtdiag, lambda, z]; 
        Internal`DeactivateMessages[
        	If[FreeQ[res = NIntegrate[Evaluate[product/(E^(z^2/2)*(2*Pi)^(1/2))], 
                	{z, -Infinity, Infinity}, opts], NIntegrate], 
           	res, 
           	$Failed],NIntegrate::inum],
       $Failed]]
              
              
(* create polynomial approximation phiInverse to quantile of NormalDistribution[0,1]. 
multiple derivatives are used in creating the InterpolatingFunction to avoid numeric 
instability at the edges of the interpolation.
phiInverse is used in tryGenz to avoid expensive calls to InverseErf *)              


phiInverse = Interpolation[
   Map[With[{inverf = InverseErf[0, -1 + 2#]}, 
      {#, {Sqrt[2]*inverf, 
          E^inverf^2*Sqrt[2*Pi], 
          2*Sqrt[2]*E^(2*inverf^2)*inverf*Pi,
          Sqrt[2]*(2*E^(3*inverf^2)*Pi^(3/2) + 8*E^(3*inverf^2)*inverf^2*Pi^(3/2)), 
          Sqrt[2]*(28*E^(4*inverf^2)*inverf*Pi^2 + 48*E^(4*inverf^2)*inverf^3*Pi^2), 
          Sqrt[2]*(28*E^(5*inverf^2)*Pi^(5/2) + 368*E^(5*inverf^2)*inverf^2*Pi^(5/2) + 
              384*E^(5*inverf^2)*inverf^4*Pi^(5/2)), 
          Sqrt[2]*(1016*E^(6*inverf^2)*inverf*Pi^3 + 5216*E^(6*inverf^2)*inverf^3*Pi^3 + 
              3840*E^(6*inverf^2)*inverf^5*Pi^3)}}] &
          ,
          (* Sqrt[2]*InverseErf[0, -1 + 2x] changes rapidly near x==0 and x==1 
             so more points are needed in the tails to get a good interpolation *)
             
          Join[Flatten[Table[Range[9]*10.^zz, {zz, -10, -3}]], 
              Table[zz, {zz, .01, .99, .01}], 
              Flatten[Table[1 - Reverse[Range[9]]*10.^zz, {zz, -3, -10, -1}]]]
          ]];
                        
(* the following method is for higher dimensional multivariate CDFs.
   the idea is to transform the CDF integral into an equivalent multiple 
   integral in standard normal variables and then transform to an integral 
   over the p dimensional unit box to simplify the integration.
   
   literature suggests this method is reasonably precise and fast upto about 
   dimension 20.  keep in mind that "reasonably fast" for a p-dimensional 
   Multinormal means fast for a p-1 dimensional numeric integration.

   Reference:
   Alan Genz, "Numerical Computation of Multivariate Normal Probabilities,"
   revised version published in J. Comp. Graph Stat. 1 (1992), pp. 141-149. 
   
   also available through Genz's Research link on
   http://www.math.wsu.edu/math/faculty/genz/homepage
*)
   
tryGenz[mu_, sigma_, x_, lowertri_List, opts___] := 
   Module[{wvec, wvars, p, newbounds, integrand, intRange, evals, xshift, 
   	wp = WorkingPrecision /. {opts}, res, phiInv},
      p = Length[x];
      If[wp=!=MachinePrecision && Precision[{mu, sigma, x, lowertri}]=!=MachinePrecision
         ,(* for efficiency, only use exact form if bignums are used *)
         phiInv=(Sqrt[2]*InverseErf[0, -1 + 2#])&
         , (* for MachinePrecision, use interpolation of InverseErf *)
         phiInv=phiInverse];
      wvars = Table[Unique[w], {p}];
      xshift = x - mu;
      evals = Table[0, {p}];
      evals[[1]] = (1/2)*(1 + Erf[(xshift[[1]]/lowertri[[1, 1]])/Sqrt[2]]); 
      Do[evals[[i]] = (1/2)*(1 + Erf[((xshift[[i]] - Apply[Plus, 
         Table[lowertri[[i, j]]*phiInv[wvars[[j]]*evals[[j]]], 
             {j, i - 1}]])/lowertri[[i, i]])/Sqrt[2]]), {i, 2, p}];
      integrand = Apply[Times, evals];
      intRange = Map[{wvars[[#]], 0, 1} &, Range[p - 1]];
      Apply[NIntegrate[integrand, ##, opts] &, intRange]
     ]
    

(* CDF definition for dimension 2 or higher Multinormals such that sigma is 
   neither diagonal nor singular.  Tong's method and Genz's method are called 
   when appropriate.
*)

MultinormalDistribution/: CDF[MultinormalDistribution[mu_, sigma_, 
   lowertri___], x_?VectorQ, opts___] := 
      Module[{p, nint, delta, agWarningQ, pgWarningQ, accgoal, compiled, gpoints, 
         mxpoints, mxrec, meth, mnrec, precgoal, sdepth, wprec, tongres, ok},
      p = Length[mu];
      {accgoal, compiled, gpoints, mxpoints, mxrec, meth, mnrec, precgoal, 
        sdepth, wprec} = {AccuracyGoal, Compiled, GaussPoints, MaxPoints, 
          MaxRecursion, Method, MinRecursion, PrecisionGoal, SingularityDepth, 
          WorkingPrecision} /. {opts} /. Options[CDF];
      If[! (accgoal === Automatic || validGoalQ[accgoal]), 
          Message[CDF::accg, accgoal]; accgoal = Automatic];
      If[! (precgoal === Automatic || validGoalQ[precgoal]), 
          Message[CDF::precg, precgoal]; precgoal = Automatic];
      (If[! FreeQ[nint, Complex],
      
      (*Note : agWarningQ and pgWarningQ aren't used here because they are checked 
        for prior to NIntegrate.The purpose of these flags is to signal whether 
        appropriate warning Messages (i.e., CDF::accg and CDF::precg) should be issued.*)
        
        {delta, agWarningQ, pgWarningQ} = Delta[nint, accgoal, precgoal];
        nint = Chop[nint, delta]];
        Re[nint]) /; (

          (* try Tong's method first and try Genz's method if Tong fails;
             for p>3, Tong's method seldom works but it will either work quickly or fail quickly,
             so it's worth trying *)
             
          ok=FreeQ[nint = If[(tongres = tryTong[mu, sigma, x, lowertri, AccuracyGoal -> accgoal, 
                         Compiled -> compiled, GaussPoints -> gpoints, MaxPoints -> mxpoints, 
                         MaxRecursion -> mxrec, Method -> meth, MinRecursion -> mnrec, 
                         PrecisionGoal -> precgoal, SingularityDepth ->sdepth, 
                         WorkingPrecision -> wprec]) =!= $Failed,
                   tongres,
                   tryGenz[mu, sigma, x, lowertri, AccuracyGoal -> accgoal, 
                         Compiled -> compiled, GaussPoints -> gpoints, 
                         MaxPoints -> mxpoints, MaxRecursion -> mxrec, 
                         Method -> meth, MinRecursion -> mnrec, 
                         PrecisionGoal -> precgoal, SingularityDepth -> sdepth, 
                         WorkingPrecision -> wprec]]
                    ,NIntegrate];
          If[ok,
             True,
             Message[CDF::nint]; False])
             ] /; (Length[{lowertri}] <= 1 || nonsingularMultinormalQ[mu, sigma]) && 
                  Length[mu] >= 2 && TrueQ[Length[x] == Length[mu]] && 
                  Apply[And, Map[((NumberQ[#] || Head[#] === DirectedInfinity) && 
                      FreeQ[#, Complex]) &, N[Flatten[{x,mu,sigma}]]]]


(*
Note: special case code to deal with cases where the absolute value of
	one of the li's is equal to 1.	

Sign[(v-l z)]		(Erf[Infinity (v-l z)] + 1)/2     (Sign[(v-l z)]+1)/2

1			1					1
0			1/2					1/2
-1			0					0
*)

hh[vector_, lambda_, z_] :=
   Module[{v, l},
	Apply[Times, Map[
		({v, l} = #;
		 (If[TrueQ[Abs[l] - 1 == 0],
			Sign[(v-l z)],
			Erf[(v-l z)/Sqrt[1-l^2]/Sqrt[2]]
		  ] + 1)/2
		)&,
			Transpose[{vector, lambda}]	
	]]
   ]

(* utility function to identify diagonal matrix; assume
   matrix is square (or at least, has more columns than rows *)
diagonalmatrixQ[mat_] :=
  Module[{tmat = mat, i},
    Do[tmat[[i, i]] = 0, {i, Length[mat]}];
    MatchQ[tmat, {{(_?(#==0&)) ..} ..}]]

(* 1-dimensional case *) 
MultinormalDistribution/: CDF[MultinormalDistribution[mu_, sigma_, lowertri___],
	 x_?VectorQ] :=
  (
  (Erf[(x - mu)[[1]]/(Sqrt[2] PowerExpand[Sqrt[sigma[[1, 1]]]])] + 1)/2
  ) /; (Length[{lowertri}] <= 1  || nonsingularMultinormalQ[mu, sigma] ) &&
       (Length[mu] == 1 == Length[x]) && FreeQ[x[[1]], Complex]

CDF::nint =
"Unable to evaluate cumulative distribution function using NIntegrate."

CDF::accg =
"Warning: Value of AccuracyGoal -> `` is not Automatic, Infinity, or a
real number.  Using AccuracyGoal -> Automatic instead."

CDF::precg =
"Warning: Value of PrecisionGoal -> `` is not Automatic, Infinity, or a
real number.  Using PrecisionGoal -> Automatic instead."


(* 2-dimensional singular case;
	 bivariate density is concentrated on a line. *)
MultinormalDistribution/: CDF[MultinormalDistribution[mu_, sigma_, 
	w_?VectorQ, v_?MatrixQ], z_?VectorQ] :=
   Module[{domain, x, y, rule, x1, y1, a, b, distx, cdf, if},
 	(
	  {x, y} = z;
	  rule = (domain[[2]])[x1, y1];
	  {b, a} = CoefficientList[rule[[2]], x1];
	  distx = NormalDistribution[mu[[1]], sigma[[1, 1]]];
	  cdf = If[a > 0,
		   if[y >= a x + b,
			CDF[distx, x],
			CDF[distx, (y-b)/a]],
		   if[y >= a x + b,
			CDF[distx, x] - CDF[distx, (y-b)/a],
			0]
		];
	  (* Simplify needed to eliminate zero-valued complex component from
	 	CDF[NormalDistribution[1, 4], 0] -
		CDF[NormalDistribution[1, 4], -0.3333333333333333] *)
	  Simplify[cdf /. {if :> If}]
 	) /; (domain = Domain[MultinormalDistribution[mu, sigma, w, v]];
	      If[FreeQ[domain, Domain], True,
		 Message[CDF::mnormdom]; False])
   ] /; If[Length[mu] == 2, True, Message[CDF::mnormsing];  False]

CDF::mnormsing =
"The cumulative distribution function of a singular multinormal distribution is
not implemented for distributions of dimensionality greater than two."

CDF::mnormdom =
"Unable to find Domain of the multinormal distribution."

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

MultinormalDistribution/: Mean[MultinormalDistribution[mu_, sigma_, l___]] :=
   mu

(* StandardDeviation yields a vector in multivar case *)
MultinormalDistribution/: StandardDeviation[MultinormalDistribution[
	mu_, sigma_, l___]] :=
   Sqrt[DiagonalElements[sigma]]  

(* Variance yields a vector in multivar case *)
MultinormalDistribution/: Variance[MultinormalDistribution[
	mu_, sigma_, l___]] := 
   DiagonalElements[sigma]

MultinormalDistribution/: CovarianceMatrix[MultinormalDistribution[
	mu_, sigma_, l___]] :=
   sigma

MultinormalDistribution/: CorrelationMatrix[MultinormalDistribution[
	mu_, sigma_, l___]] :=
   Module[{sdvector},
	  (
	  DiagonalMatrix[1/sdvector].sigma.DiagonalMatrix[1/sdvector]
	  ) /; FreeQ[sdvector = StandardDeviation[MultinormalDistribution[
		mu, sigma, l]], StandardDeviation] 
   ]

MultinormalDistribution/: Skewness[MultinormalDistribution[
	mu_?VectorQ, sigma_, l___]] := Table[0, {Length[mu]}] 

MultinormalDistribution/: MultivariateSkewness[MultinormalDistribution[
	mu_, sigma_, l___]] := 0

MultinormalDistribution/: Kurtosis[MultinormalDistribution[
	mu_?VectorQ, sigma_, l___]] := Table[3, {Length[mu]}] 

MultinormalDistribution/: MultivariateKurtosis[MultinormalDistribution[
	mu_?VectorQ, sigma_, l___]] :=
	  With[{p = Length[mu]}, p(p + 2)]
	
MultinormalDistribution/: KurtosisExcess[MultinormalDistribution[
	mu_?VectorQ, sigma_, l___]] := Table[0, {Length[mu]}] 

MultinormalDistribution/: MultivariateKurtosisExcess[MultinormalDistribution[
	mu_, sigma_, l___]] := 0
 
(* NOTE:
  CharacteristicFunction[dist, t] for a random vector x is defined as the
	expected value of Exp[I t.x], where Dimensions[t] == Dimensions[x].
*)
MultinormalDistribution/: CharacteristicFunction[
                MultinormalDistribution[mu_, sigma_, l___], t_?VectorQ] :=
	(
	  Exp[I t.mu - 1/2 t.sigma.t]
 	) /; TrueQ[Length[t] == Length[mu]]
 
(* - - - - - - - - -  Quantile[MultinormalDistribution[], q] - - - - - - - - *)
(* NOTE:
Quantile is not implemented for MultinormalDistribution[mu, sigma] for p > 1.
Actually since there is no unique vector x such that
q = CDF[MultinormalDistribution[mu, sigma], x], it is difficult to even
define Quantile[MultinormalDistribution[mu, sigma], q].  One possibility is
the so-called "equicoordinate one-sided percentage points", i.e., the
quantile is {c, c, ..., c} such that CDF[MultinormalDistribution[r, m],
{c, c, ..., c}] = q. *)

MultinormalDistribution/: Quantile[MultinormalDistribution[mu_, sigma_, l___],
   q_] :=
   {Quantile[NormalDistribution[mu[[1]], Sqrt[sigma[[1, 1]]]], q]} /;
	(Length[{l}] <= 1  || nonsingularMultinormalQ[mu, sigma]   ) &&
	Length[mu] == 1 && NumberQ[q] && (0 <= q <= 1) 

(* - - - - -  EllipsoidQuantile[MultinormalDistribution[], ellipsoid] - - - - *)
MultinormalDistribution/: EllipsoidQuantile[MultinormalDistribution[
	mu_?VectorQ, sigma_?MatrixQ, l___], q_] :=
  Module[{ellipsoid, quantile},
    (
    ellipsoid
    ) /; (quantile = Quantile[ChiSquareDistribution[Length[mu]], q];
	  ellipsoid = ellipsoidalLocus[mu, quantile sigma];
	  If[ellipsoid === $Failed,
		Message[EllipsoidQuantile::mnormeig, quantile sigma] ];
	  ellipsoid =!= $Failed)
  ] /; QuantileQ[q] &&
	 (Length[{l}] <= 1 || nonsingularMultinormalQ[mu, sigma]  )

EllipsoidQuantile::mnormeig =
"Unable to find eigensystem of ``."

(* - - - - -  RegionProbability[MultinormalDistribution[], ellipsoid] - - - - *)
MultinormalDistribution/: RegionProbability[MultinormalDistribution[
	mu_?VectorQ, sigma_?MatrixQ, l___], EllipsoidQuantile[mu1_, radii_]] :=
		0 /; radii === Table[0, {Length[mu]}]

MultinormalDistribution/: RegionProbability[MultinormalDistribution[
	mu_?VectorQ, sigma_?MatrixQ, l___], EllipsoidQuantile[mu1_, radii_]] :=
		1 /; radii === Table[Infinity, {Length[mu]}]

MultinormalDistribution/: RegionProbability[MultinormalDistribution[
	mu_, sigma_, l___],
	Ellipsoid[mu_, radii_?VectorQ, dir___?MatrixQ]] :=
   Module[{p, internaldir, normalized, diagonalizedMatrix, acc, prec, delta,
		 scaled, quantile, diagonal},
     (
	CDF[ ChiSquareDistribution[p], quantile ]
     ) /; (
	   ( p = Length[mu];
	     internaldir = If[{dir}==={}, IdentityMatrix[p], dir];
	     normalized = sigma.Transpose[internaldir];	
	     diagonalizedMatrix = internaldir.normalized;
	     acc = Max[Accuracy[diagonalizedMatrix], $MachinePrecision]-1;
             prec = Max[Precision[diagonalizedMatrix], $MachinePrecision]-1;
             delta = Max[10^(-acc), Abs[Det[diagonalizedMatrix]] 10^(-prec)];
	
	     diagonalizedMatrix = Simplify[Chop[diagonalizedMatrix, delta]];
	     If[MatchQ[diagonalizedMatrix, DiagonalMatrix[Table[_, {p}]] ],
		True,
		Message[RegionProbability::mnormell,
			 mu, radii, internaldir, sigma];
		False] &&	
	     (diagonal = DiagonalElements[diagonalizedMatrix]/radii^2;
	      (* Consider
		  (dist = MultinormalDistribution[{1, 2, 3},
			{{1/3, 1/4, 1/5}, {1/4, 1/5, 1/6}, {1/5, 1/6, 1/7}}];
		  ellipsoid = EllipsoidQuantile[dist, .5];
		  p = RegionProbability[dist, ellipsoid];
		  Chop[p-.5])
		in determining acc and prec below. *)
              acc = Max[Accuracy[diagonal], $MachinePrecision]-4;
              prec = Max[Precision[diagonal], $MachinePrecision]-4;
              delta = Max[10^(-acc), Apply[Times, Abs[diagonal]] 10^(-prec)];	

	      Apply[And, Map[(Chop[#, delta]==0)&,
		 (diagonal - RotateRight[diagonal, 1]) ]])    ) &&
	   ( scaled = Transpose[internaldir].DiagonalMatrix[radii^2];
	     quantile = Scan[If[ !TrueQ[#[[1]]==0], Return[#[[2]]/#[[1]]] ]&,
	  	       Transpose[{Flatten[normalized], Flatten[scaled]}] ];
	     quantile =!= Null  )
	  )	
   ] /; (Length[{l}] <= 1  || nonsingularMultinormalQ[mu, sigma])

RegionProbability::mnormell = 
"Ellipsoid[`1`, `2`, `3`] does not correspond to a constant-probability
contour of MultinormalDistribution[`1`, `4`]."


(* - - - - - - -  ExpectedValue[f, MultinormalDistribution[], x] - - - - - - *)
MultinormalDistribution/: ExpectedValue[f_Function,
	 MultinormalDistribution[mu_, sigma_, l_], opts___?OptionQ] :=
  Module[{n, (* number of arguments of function f *)
	  m = Length[mu], (* dimensionality of MultinormalDistribution *) 
	  xvec, x, assmp = Assumptions /. {opts} /. Options[ExpectedValue],
	  arglist, integral},
   (
    xvec = Array[x, m]; 
    assmp = Flatten[Join[	Map[(Im[#]==0)&, mu],
         			Map[(# > 0)&, DiagonalElements[sigma]],
	 			{Det[sigma]>0},
	 			assmp			] /. True -> {}];
    arglist = Prepend[ Map[{#, -Infinity, Infinity}&, xvec],
		       Apply[f, xvec] *
			PDF[MultinormalDistribution[mu, sigma, l], xvec] ];	
    If[assmp =!= {}, AppendTo[arglist, Assumptions -> assmp]];
    If[FreeQ[integral = Apply[Integrate, arglist], Integrate],
       integral,
       unique = Table[Unique[], {m}];	
       integral /. Thread[Rule[xvec, unique]]		] (* end If *)
   ) /; (If[Length[f]==1,
	    (* Function with only a body *)
	    n = Max[Cases[{f}, Slot[z_]->z, Infinity]],
	    (* Function with a list of formal parameters *)
	    n = Length[f[[1]]]	];
	 n <= m)
  ]
	
MultinormalDistribution/: ExpectedValue[f_,
	MultinormalDistribution[mu_, sigma_, l_], xvec_?VectorQ,
	opts___?OptionQ] :=			
  Module[{assmp = Assumptions /. {opts} /. Options[ExpectedValue], arglist},
    assmp = Flatten[Join[	Map[(Im[#]==0)&, AboveDiagonalElements[sigma]],
		 	        Map[(Im[#]==0)&, mu],
                                Map[(# > 0)&, DiagonalElements[sigma]],
                                {Det[sigma]>0},
                                assmp                   ] /. True -> {}];
    arglist = Prepend[ Map[{#, -Infinity, Infinity}&, xvec],
		       f PDF[MultinormalDistribution[mu, sigma, l], xvec] ];	
    If[assmp =!= {}, AppendTo[arglist, Assumptions -> assmp]];
    Apply[Integrate, arglist]
  ] /; Length[xvec] <= Length[mu]
	
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not check parameters on Random so that Random will always
        evaluate and be efficient. *)

(* single random vector from a nonsingular multinormal *)
MultinormalDistribution/: Random[MultinormalDistribution[mu_, sigma_, l_]] :=
  Module[{z = RandomArray[NormalDistribution[0, 1], Length[mu]]},
	mu + l.z  
  ]

(* list of random vectors from a nonsingular multinormal *)
MultinormalDistribution/: RandomArray[
	MultinormalDistribution[mu_, sigma_, l_], dim_] :=
  Module[{m, array},
	m = If[VectorQ[dim], Apply[Times, dim], dim];
	array = RandomArray[NormalDistribution[0, 1], {m, Length[mu]}];
	array = Map[mu + l . # &, array];
	If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]

(* single random vector from a singular multinormal *)
MultinormalDistribution/: Random[MultinormalDistribution[mu_, sigma_,
	 w_?VectorQ, v_?MatrixQ]] :=
  Module[{k = Length[w], y},
	y = Random[MultinormalDistribution[Table[0, {k}], DiagonalMatrix[w]]]; 
	Transpose[v].y + mu
  ] (* Thm. 2.5.6, p. 42, Mardia, Kent, Bibby *)



(* list of random vectors from a singular multinormal *)
MultinormalDistribution/:RandomArray[MultinormalDistribution[mu_,sigma_,
	w_?VectorQ,v_?MatrixQ],dim_]:=
  Module[{k=Length[w],ydist,array,m},
   	m=If[VectorQ[dim],Apply[Times,dim],dim];
   	If[k==1
   	   ,
    	   ydist=NormalDistribution[0,Sqrt[w[[1]]]];
    	   array=RandomArray[ydist,m];
    	   array=Map[(v[[1]]*#+mu)&,array]
    	   ,
    	   ydist=MultinormalDistribution[Table[0,{k}],DiagonalMatrix[w]];
    	   array=RandomArray[ydist,m];
    	   array=Map[(Transpose[v].#+mu)&,array]];
   	If[VectorQ[dim]&&Length[dim]>1,
    	   Fold[Partition[#1,#2]&,array,Reverse[Drop[dim,1]]],array]
  ]/;(IntegerQ[dim]&&dim>0)||VectorQ[dim,(IntegerQ[#]&&#>0)&]
	(* Thm. 2.5.6, p. 42, Mardia, Kent, Bibby *)

  
   

(* ========================== Wishart Distribution ======================== *)

WishartDistribution/: ParameterQ[WishartDistribution[sigma_, m_, l_]] :=
   MatrixQ[sigma] &&
   If[NumberQ[m], IntegerQ[m] && TrueQ[m >= Length[sigma]+1], True] &&
   sigma === Transpose[sigma] &&
   If[Apply[And, Map[NumberQ, Flatten[N[sigma]] ]],
      TrueQ[Det[N[sigma]] > 0] &&
         Apply[And, Map[TrueQ[# > 0]&, N[DiagonalElements[sigma]] ]],
      True]

validWishartQ[sigma_, m_, warnflag_:False] :=
  (
   MatrixQ[sigma] &&
   If[NumberQ[m], IntegerQ[m] && TrueQ[m >= Length[sigma]+1], True] &&
   If[sigma === Transpose[sigma], True,
       If[TrueQ[warnflag], Message[WishartDistribution::cmsym]];False] &&
   If[Apply[And, Map[NumberQ, Flatten[N[sigma]] ]],
      TrueQ[Det[N[sigma]] > 0] &&
         Apply[And, Map[TrueQ[# > 0]&, N[DiagonalElements[sigma]] ]],
      True]
  )

WishartDistribution::cmsym =
"The covariance matrix must be symmetric.";

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not condition DomainQ on ParameterQ so that DomainQ will always
        evaluate and be efficient. *)
WishartDistribution/: DomainQ[WishartDistribution[sigma_, m_, l___],
	 list_List] :=
  Module[{p},
    (
	Scan[Module[{matrix = #},
	       If[!TrueQ[matrix === Transpose[matrix] &&
	                 Det[N[matrix]] > 0 &&
	                 Apply[And, Map[(# > 0)&,
				 N[DiagonalElements[matrix]] ]]		],
		  Return[False]
	       ]
	     ]&, list] =!= False	
    ) /; (p = Length[sigma];  MatchQ[Dimensions[list], {_, p, p}])
  ] 
WishartDistribution/: DomainQ[WishartDistribution[sigma_, m_, l___],
	 x_?MatrixQ] :=
  Module[{p = Length[sigma]},
       TrueQ[Dimensions[x] == {p, p} && x === Transpose[x] &&
	     Det[N[x]] > 0 &&
	     Apply[And, Map[(# > 0)&, N[DiagonalElements[x]] ]]
       ]
  ]

(* The following iDomainQ allows for symbolic arguments *)
iDomainQ[WishartDistribution[sigma_, m_, l___], x_] :=
  Module[{p = Length[sigma]},
   TrueQ[Dimensions[x] == {p, p} && x === Transpose[x] &&
	 (Apply[And, Map[!NumberQ[#]&, Flatten[N[x]] ]] ||
          (Det[N[x]] > 0 && Apply[And, Map[(# > 0)&, N[DiagonalElements[x]] ]])
	 )	]
  ]

(* - - - - - - - - - - - - - -- - - - - - - - - - - - - - - - - - - - - - *)
(* compute lower triangular matrix for use in calculations involving
	WishartDistribution *)
WishartDistribution[sigma_:{{1}}, m_:1] :=
  Module[{u},
    (
    WishartDistribution[sigma, m, Transpose[u]
    ]	
    ) /; FreeQ[u = CholDecomp[sigma], CholDecomp]
  ] /; validWishartQ[sigma, m, True]

(* hide lower triangular matrix using Format *)
WishartDistribution /: Format[WishartDistribution[sigma_, m_, l_]] :=
	WishartDistribution[Short[sigma], m]

(* special rule for sigma = IdentityMatrix[p] *)
WishartDistribution[sigma_?MatrixQ, m_?IntegerQ] :=
  Module[{p},
    (
  	WishartDistribution[sigma, m, 
		Table[Prepend[Table[0, {i}], 1], {i, 0, p-1}]]
    ) /; (p = Length[sigma];
	  (m >= p + 1) && sigma === IdentityMatrix[p])
  ] /; Apply[Equal, Dimensions[sigma]]

(* - - - - - - - - - - - - - -- - - - - - - - - - - - - - - - - - - - - - *)

(* The following indicates that the domain of WishartDistribution is the
	set of all appropriately-sized symmetric positive definite matrices. *)
WishartDistribution/: Domain[WishartDistribution[sigma_, m_, l___]] :=
 (
 (Dimensions[#] == Dimensions[sigma] && # === Transpose[#] && Det[#1] > 0)&
 ) /; ({l} =!= {} || validWishartQ[sigma, m]) 

WishartDistribution/: PDF[WishartDistribution[sigma_, m_, l___], x_?MatrixQ] :=
  0 /; ({l} =!= {} || validWishartQ[sigma, m]) &&
       !iDomainQ[WishartDistribution[sigma, m], x]

WishartDistribution/: PDF[WishartDistribution[sigma_, m_, l___], x_?MatrixQ] :=
  With[{invsigma = Inverse[sigma], p = Length[sigma]},
    iK[m, invsigma] Det[x]^((m-p-1)/2) Exp[trace[-1/2 invsigma.x]]
  ] /; ({l} =!= {} || validWishartQ[sigma, m]) &&
       iDomainQ[WishartDistribution[sigma, m], x]

(*  - - - - - - - - - -  CDF[WishartDistribution[], ]  - - - - - - - - - - - *)

WishartDistribution/: CDF[WishartDistribution[sigma_, m_, l___], x_?MatrixQ] :=
  (
  0
  ) /; ({l} =!= {} || validWishartQ[sigma, m]) &&
       !iDomainQ[WishartDistribution[sigma, m], x]

WishartDistribution/: CDF[WishartDistribution[sigma_, m_, l___], x_?MatrixQ] :=
   (
   GammaRegularized[ m/2, 0, x[[1, 1]]/2/sigma[[1, 1]] ]
   ) /; Length[x] == 1 &&
       ({l} =!= {} || validWishartQ[sigma, m]) &&
       iDomainQ[WishartDistribution[sigma, m], x]


(* Johnson & Kotz, p. 162, eq. (11) *)
iK[nu_, C_] :=
  With[{p = Length[C]},
    Det[C]^(nu/2) ( 2^(nu p/2) MultivariateGamma[p, nu/2] )^(-1)
  ]	

(* Johnson & Kotz, p. 162, eq. (12) *)
MultivariateGamma[p_, z_] := 
	 Pi^(p(p-1)/4) Product[Gamma[(2z - j + 1)/2], {j, 1, p}]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
 
WishartDistribution/: Mean[WishartDistribution[sigma_, m_, l___]] := m sigma /;
	({l} =!= {} || validWishartQ[sigma, m])

WishartDistribution/: Variance[WishartDistribution[sigma_, m_, l___]] :=
   With[{diag = DiagonalElements[sigma]},
	m (sigma^2 + Map[List, diag].{diag})
   ] /; ({l} =!= {} || validWishartQ[sigma, m])  

WishartDistributionDistribution/: StandardDeviation[
   WishartDistributionDistribution[sigma_, m_, l___]] :=
	Module[{var},
	  (
          Sqrt[var]	
	  ) /; FreeQ[var = Variance[WishartDistribution[sigma, m, l]], Variance]
	] /; ({l} =!= {} || validWishartQ[sigma, m])

(* NOTE: CovarianceMatrix and CorrelationMatrix are not implemented for the
	WishartDistribution.  They could be if the p(p+1)/2 unique entries of
	a Wishart matrix were ordered.  Currently, the distribution is
	considered to be matrix-valued, rather than vector-valued with the 
	entries taken from a matrix. *)

WishartDistribution/: Skewness[WishartDistribution[sigma_, m_, l___]] :=
   Module[{i, j, p = Length[sigma], rho},
	(2/Sqrt[m]) Table[If[i==j, Sqrt[2],
			     rho = sigma[[i, j]];
			     rho (3 + rho^2) (1 + rho^2)^(-3/2)
			  ], {i, p}, {j, p}]
   ] /; ({l} =!= {} || validWishartQ[sigma, m])  

(* NOTE: MultivariateSkewness[WishartDistribution[sigma, m]] not implemented. *)

WishartDistribution/: Kurtosis[WishartDistribution[sigma_, m_, l___]] :=
   Module[{i, j, p = Length[sigma], rho},
	3 + (2/m) Table[If[i==j, 6,
			   rho = sigma[[i, j]];
			   (1 + 6 rho^2 + rho^4)/(1 + rho^2)^2
		        ], {i, p}, {j, p}]
   ] /; ({l} =!= {} || validWishartQ[sigma, m]) 

(* NOTE: MultivariateKurtosis[WishartDistribution[sigma, m]] not implemented. *)

WishartDistribution/: KurtosisExcess[WishartDistribution[sigma_, m_, l___]] :=
   Module[{i, j, p = Length[sigma], rho},
	(2/m) Table[If[i==j, 6,
		       rho = sigma[[i, j]];
		       (1 + 6 rho^2 + rho^4)/(1 + rho^2)^2
		    ], {i, p}, {j, p}]
   ] /; ({l} =!= {} || validWishartQ[sigma, m]) 

(* NOTE: MultivariateKurtosisExcess[WishartDistribution[sigma, m]] not
	 implemented. *)

(* NOTE:
  CharacteristicFunction[dist, t] for a random matrix x is defined as the
	expected value of
  Exp[trace[-I(t + DiagonalMatrix[DiagonalElements[t]]).x]],
	where Dimensions[t] == Dimensions[x].
  Johnson & Kotz, Chap. 38, eq. 18.2
*)
WishartDistribution/: CharacteristicFunction[
                WishartDistribution[sigma_, m_, l___], t_?MatrixQ] :=
  With[{invsigma = Inverse[sigma]},
    ( Det[invsigma] )^(m/2) /
    ( Det[invsigma - I (t + DiagonalMatrix[DiagonalElements[t]])] )^(m/2)
  ] /; ({l} =!= {} || validWishartQ[sigma, m]) &&
       TrueQ[Dimensions[t] == Dimensions[sigma]] && t === Transpose[t]

(* NOTE:
Quantile is not implemented for WishartDistribution[sigma, m].
Actually since there is no unique matrix x such that
q = CDF[WishartDistribution[sigma, m], x], it is difficult to even
define Quantile[WishartDistribution[sigma, m], q]. *)

(* NOTE: ExpectedValue of WishartDistribution not implemented. *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not check parameters for Random so that Random will always
        evaluate and be efficient. *)

WishartDistribution/: Random[WishartDistribution[sigma_, m_, l_]] :=
   Module[{dist = MultinormalDistribution[Table[0, {Length[sigma]}], sigma, l],
	   dataMatrix},
     dataMatrix = RandomArray[dist, m];
     Transpose[dataMatrix].dataMatrix
   ] /; (IntegerQ[m] && m >= Length[sigma]+1 && MatrixQ[sigma] && 
   	Dimensions[sigma]===Dimensions[l])
   
   
WishartDistribution/: RandomArray[WishartDistribution[sigma_, m_, l_], dim_] :=
  Module[{dist = MultinormalDistribution[Table[0, {Length[sigma]}], sigma, l],
	  n, array},
    n = If[VectorQ[dim], Apply[Times, dim], dim];
    array = RandomArray[dist, {n, m}];
    array = Map[(Transpose[#].#)&, array];
    If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
 ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&] &&
	  (IntegerQ[m] && m >= Length[sigma]+1 && MatrixQ[sigma] && 
   	Dimensions[sigma]===Dimensions[l]) 


(* ======================== QuadraticForm Distribution ====================== *)


QuadraticFormDistribution/: ParameterQ[QuadraticFormDistribution[{a_, b_, c_},
	 {mu_, sigma_}, {lambda_, beta_, alpha_}, rest___]] := 
   ParameterQ[MultinormalDistribution[mu, sigma]] && 
   MatrixQ[a] && VectorQ[b] && (Head[c] =!= List) &&
   (Dimensions[a] === {Length[b], Length[b]} === Dimensions[sigma]) &&
   a == Transpose[a]

validQuadraticFormQ[{a_, b_, c_}, {mu_, sigma_}, warnflag_:False] :=
   If[ParameterQ[MultinormalDistribution[mu, sigma]],
      True,
      If[TrueQ[warnflag], Message[QuadraticFormDistribution::multinorm]];False] && 
   If[MatrixQ[a] && VectorQ[b] && (Head[c] =!= List),
      True,
      If[TrueQ[warnflag], Message[QuadraticFormDistribution::badcoefs]];False] &&
   If[(Dimensions[a] === {Length[b], Length[b]} === Dimensions[sigma]),
      True,
      If[TrueQ[warnflag], Message[QuadraticFormDistribution::coefdims]];False] &&
   If[a == Transpose[a],
      True,
      If[TrueQ[warnflag], Message[QuadraticFormDistribution::coefsym]];False]

QuadraticFormDistribution::multinorm =
"Second argument must be a list of valid MultinormalDistribution arguments.";

QuadraticFormDistribution::badcoefs =
"The quadratic coefficients should be a matrix, a vector, and a scalar,
respectively.";

QuadraticFormDistribution::coefdims =
"The quadratric coefficients must have dimensions commensurate with
the covariance matrix of the multinormal distribution.";

QuadraticFormDistribution::coefsym =
"The first quadratic coefficient must be a symmetric matrix.";

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not condition DomainQ on ParameterQ so that DomainQ will always
        evaluate and be efficient. *)
QuadraticFormDistribution/: DomainQ[QuadraticFormDistribution[{a_, b_, c_},
        {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], list_?VectorQ] :=
        FreeQ[N[list], Complex] &&
 	If[Det[a] > 0,
	   With[{low = alpha - Apply[Plus, beta^2/lambda]/4}, 
	     Scan[If[!TrueQ[# >= low], Return[False]]&, list] =!= False  ],
	   True,
	   False]
QuadraticFormDistribution/: DomainQ[QuadraticFormDistribution[{a_, b_, c_},
        {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], x_] :=
        FreeQ[N[x], Complex] &&
 	If[Det[a] > 0, x >= alpha - Apply[Plus, beta^2/lambda]/4, True, False]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* assume a = {{1}}, b = {0}, c = 0, mu = {0}, sigma = {{1}} *)
QuadraticFormDistribution[] := ChiSquareDistribution[1]

(* assume mu = Table[0, {Length[b]}] and sigma = IdentityMatrix[Length[b]] *)
QuadraticFormDistribution[{a_?MatrixQ, b_?VectorQ, c_}] :=
  Module[{p = Length[b]},
    QuadraticFormDistribution[{a, b, c}, {Table[0, {p}], IdentityMatrix[p]}]
  ]  /; Dimensions[a] === {Length[b], Length[b]} && Det[a] > 0

(* assume a = IdentityMatrix[Length[mu]] and b = Table[0, {Length[mu]}] and
	 c = 0 *)
QuadraticFormDistribution[{mu_, sigma_}] :=
  Module[{p = Length[mu]},
    QuadraticFormDistribution[{IdentityMatrix[p],
	 Table[0, {p}], 0}, {mu, sigma}]
  ] /; ParameterQ[MultinormalDistribution[mu, sigma]] 
	

(* QuadraticFormDistribution -> NormalDistribution for linear forms *)
QuadraticFormDistribution[{a_?MatrixQ, b_?VectorQ, c_}, {mu_, sigma_}] :=
	NormalDistribution[b.mu + c, Sqrt[b.sigma.b]] /; ZeroMatrixQ[a] &&
		ParameterQ[MultinormalDistribution[mu, sigma]] &&
		(Head[c] =!= List) && Length[b]==Length[mu]

(* compute reparametrization for use in calculations involving
	QuadraticFormDistribution *)
QuadraticFormDistribution[{a_?MatrixQ, b_?VectorQ, c_}, {mu_, sigma_}] :=
  Module[{u, esys, lambda, beta, alpha, evecs},
    (
    {lambda, evecs} = esys;
    evecs = (#/Sqrt[#.#])& /@ evecs; (* P P' = I *)
    beta = evecs.(u.b + 2u.a.mu);
    alpha = mu.a.mu + b.mu + c;
    QuadraticFormDistribution[
	{a, b, c}, {mu, sigma}, 
	{lambda, beta, alpha},	(* reparameterization *)
	Transpose[u] (* for random num gen *)
    ]
    ) /; FreeQ[u = CholDecomp[sigma], CholDecomp] &&
	 FreeQ[esys = Eigensystem[u.a.Transpose[u]], Eigensystem]
  ] /; validQuadraticFormQ[{a, b, c}, {mu, sigma}, True]

(* hide additional parameters using Format *)
QuadraticFormDistribution /: Format[QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_]] :=
	QuadraticFormDistribution[{Short[a], b, c}, {mu, Short[sigma]}]	

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* Mathai & Provost, p. 97 *)
(* if {lambda, beta, alpha} exist, assume valid distribution *)
QuadraticFormDistribution/: Domain[QuadraticFormDistribution[{a_, b_, c_},
	{mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_]] :=
	If[Det[a] > 0, 
	   Interval[{alpha - Apply[Plus, beta^2/lambda]/4, Infinity}],
	   Interval[{-Infinity, Infinity}]
	]

(*  - - - - - - - - - - - PDF of QuadraticFormDistribution  - - - - - - - - *)

QuadraticFormDistribution/: PDF[QuadraticFormDistribution[{a_, b_, c_},
  {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], x_] :=
     Module[{noncentralparm, gamma, delta},
	gamma = a[[1, 1]] sigma[[1, 1]];
 	delta = c - b[[1]]^2/(4 a[[1, 1]]);	
	noncentralparm = (b[[1]]/(2 a[[1, 1]]) + mu[[1]])^2 / sigma[[1, 1]];
	If[Evaluate[noncentralparm == 0],
	   Evaluate[PDF[ChiSquareDistribution[1], (x-delta)/gamma]],
	   Evaluate[PDF[NoncentralChiSquareDistribution[1, noncentralparm],
		(x-delta)/gamma]]	
        ] / Abs[gamma]
     ] /; Length[b] == 1

QuadraticFormDistribution/: PDF[QuadraticFormDistribution[{a_?MatrixQ, b_, c_},
    {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], x_] :=
	  (
	  0
	  ) /; TrueQ[x < alpha - Apply[Plus, beta^2/lambda]/4] &&
  		TrueQ[Det[a] > 0] 

(* Mathai & Provost, p. 93 *)
(* A LaguerreL polynomial expansion is preferable to the power series
	expansion provided by Series and should be implemented someday! *)
PDF/: Series[PDF[QuadraticFormDistribution[{a_?MatrixQ, b_, c_}, {mu_, sigma_},
	{lambda_, beta_, alpha_}, lowertri_], x_],
	{x_Symbol, x0_, n_Integer}] :=
  Module[{kmax, nmin, nmax, cvec, cc, dvec, dd, r, p = Length[mu], cutoff},
    (
    kmax = Max[0, Floor[n-p/2+1]];
			 (* kmax+1 is the number of nonzero non-order terms *)
    nmin = p/2 - 1;	   (* smallest exponent of (x-x0) *)
    nmax = p/2 + kmax - 1; (* largest exponent of (x-x0) *)
    cvec = Array[cc, n+1, 0]; dvec = Array[dd, n];
    dvec = Table[(-1)^r/2 Apply[Plus, (1 - r (beta/lambda)^2/4)/(2 lambda)^r],
		 {r, kmax}];
    cc[0] = Exp[-Apply[Plus, (beta/lambda)^2]/8] / Sqrt[Apply[Times, 2 lambda]];
    Scan[(cc[#] = (Reverse[Take[dvec, #]].Take[cvec, #])/#)&, Range[n]];
    cvec /= Table[Gamma[p/2 + r], {r, 0, kmax}];
    If[OddQ[p],	
       SeriesData[x, x0, Flatten[Map[{#, 0}&, cvec]],
		 Numerator[nmin], Numerator[nmax+1], 2],
       SeriesData[x, x0, cvec, nmin, nmax+1, 1] 
    ]
    ) /; (cutoff = alpha - Apply[Plus, beta^2/lambda]/4;
	  If[!TrueQ[N[x0-cutoff]==0||Simplify[x0-cutoff,TimeConstraint->1]===0],
	    Message[Series::qforig,
		    x, cutoff, 
		    PDF[QuadraticFormDistribution[{a, b, c}, {mu, sigma},
			{lambda, beta, alpha}, lowertri], x],
		    {x, cutoff, n}]; False, 
	    True])
  ]  /; n >= 0 && Apply[Equal, Dimensions[a]] &&
	If[TrueQ[Det[a] <= 0], Message[Series::qfpos]; False, True]

(*  - - - - - - - - - - - CDF of QuadraticFormDistribution  - - - - - - - - *)

QuadraticFormDistribution/: CDF[QuadraticFormDistribution[{a_, b_, c_},
  {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], x_] :=
     Module[{noncentralparm, gamma, delta},
	gamma = a[[1, 1]] sigma[[1, 1]];
 	delta = c - b[[1]]^2/(4 a[[1, 1]]);	
	noncentralparm = (b[[1]]/(2 a[[1, 1]]) + mu[[1]])^2 / sigma[[1, 1]];
	If[Evaluate[noncentralparm == 0],
	   Evaluate[If[Evaluate[a[[1, 1]] > 0],
	      Evaluate[CDF[ChiSquareDistribution[1], (x-delta)/gamma]],
	      Evaluate[1 - CDF[ChiSquareDistribution[1], (x-delta)/gamma]]	
	   ]],
	   Evaluate[If[Evaluate[a[[1, 1]] > 0],
	      Evaluate[CDF[NoncentralChiSquareDistribution[1, noncentralparm],
		(x-delta)/gamma]],
	      Evaluate[1 - CDF[NoncentralChiSquareDistribution[1,
		 noncentralparm], (x-delta)/gamma]]
	   ]]
	]
     ] /; Length[b] == 1

QuadraticFormDistribution/: CDF[QuadraticFormDistribution[{a_?MatrixQ, b_, c_},
    {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], x_] :=
	  (
	  0
	  ) /; TrueQ[x < alpha - Apply[Plus, beta^2/lambda]/4] &&
  		TrueQ[Det[a] > 0] 

CDF/: Series[CDF[QuadraticFormDistribution[{a_?MatrixQ, b_, c_}, {mu_, sigma_},
	{lambda_, beta_, alpha_}, lowertri_], x_],
	{x_Symbol, x0_, n_Integer}] :=
  Module[{pdfseries, cutoff},
    (
        pdfseries = Series[PDF[QuadraticFormDistribution[{a, b, c},
		{mu, sigma}, {lambda, beta, alpha}, lowertri], x],
		{x, x0, n-1}];
	Integrate[pdfseries, x]
    ) /; (cutoff = alpha - Apply[Plus, beta^2/lambda]/4;
	  If[!TrueQ[N[x0-cutoff]==0||Simplify[x0-cutoff,TimeConstraint->1]===0],
	    Message[Series::qforig,
		    x, cutoff, 
		    CDF[QuadraticFormDistribution[{a, b, c}, {mu, sigma},
			{lambda, beta, alpha}, lowertri], x],
		    {x, cutoff, n}]; False, 
	    True])
  ] /; n >= 1 && Apply[Equal, Dimensions[a]] &&
	If[TrueQ[Det[a] <= 0], Message[Series::qfpos]; False, True] 
	
CDF/: Series[CDF[QuadraticFormDistribution[{a_?MatrixQ, b_, c_}, {mu_, sigma_},
	{lambda_, beta_, alpha_}, lowertri_], x_], {x_Symbol, x0_, 0}] :=
  Module[{cutoff},
    (
    SeriesData[x, 0, List[], 1, 1, 1]
    ) /; (cutoff = alpha - Apply[Plus, beta^2/lambda]/4;
	  If[!TrueQ[N[x0-cutoff]==0],
             Message[Series::qforig,
		     x, cutoff,
		     CDF[QuadraticFormDistribution[{a, b, c}, {mu, sigma},	
                       {lambda, beta, alpha}, lowertri], x],
		     {x, cutoff, 0}]; False,
             True])
  ] /; Apply[Equal, Dimensions[a]] &&
	If[TrueQ[Det[a] <= 0], Message[Series::qfpos]; False, True]

Series::qfpos =
"The Series expansions of PDF[QuadraticFormDistribution[{A, B, C}, {mu, sigma}],
x] or CDF[QuadraticFormDistribution[{A, B, C}, {mu, sigma}], x] are implemented
only for the case of positive definite A matrix."

Series::qforig =
"The Series expansions of the PDF and CDF of the QuadraticFormDistribution
must be about the lower point of the domain `1` = `2`.
Try Series[`3`, `4`]."

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

(* Mathai & Provost, eq. 3.2b.8, used for Mean, StandardDeviation, Variance,
	Skewness, Kurtosis, and KurtosisExcess *)
QuadraticFormDistribution/: Mean[QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, ___]] :=
	(
       Apply[Plus, lambda] + alpha
	) (* if {lambda, beta, alpha} exist, assume valid distribution *)

QuadraticFormDistribution/: StandardDeviation[
	QuadraticFormDistribution[{a_, b_, c_}, {mu_, sigma_},
		 {lambda_, beta_, alpha_}, rest___]] :=
   Module[{var},
     (
     Sqrt[var]
     ) /; FreeQ[var = Variance[QuadraticFormDistribution[{a, b, c},
		{mu, sigma}, {lambda, beta, alpha}, rest]], Variance]
   ]

QuadraticFormDistribution/: Variance[QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, ___]] :=
	(
	2 Apply[Plus, lambda^2] + Apply[Plus, beta^2]
	) (* if {lambda, beta, alpha} exist, assume valid distribution *) 

QuadraticFormDistribution/: Skewness[QuadraticFormDistribution[
	{a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, rest___]] := 
  Module[{var},
    (
      (8 Apply[Plus, lambda^3] + 6 Apply[Plus, beta^2 lambda]) var^(-3/2)
    ) /; FreeQ[var = Variance[QuadraticFormDistribution[{a, b, c},
                {mu, sigma}, {lambda, beta, alpha}, rest]], Variance] 
  ]

QuadraticFormDistribution/: Kurtosis[qfd:QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, rest___]] := 
  (
	KurtosisExcess[qfd] + 3
  ) (* if {lambda, beta, alpha} exist, assume valid distribution *)

QuadraticFormDistribution/: KurtosisExcess[qfd:QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, rest___]] := 
  Module[{var},
    (
	(48 Apply[Plus, lambda^4 + beta^2 lambda^2])/var^2
    )/; FreeQ[var = Variance[qfd], Variance]
  ] (* if {lambda, beta, alpha} exist, assume valid distribution *)

QuadraticFormDistribution/: CharacteristicFunction[QuadraticFormDistribution[
   {a_, b_, c_}, {mu_, sigma_}, {lambda_, beta_, alpha_}, rest___], t_] :=
   Module[{terms = Map[(1 - 2 t I #)&, lambda]},
     Exp[alpha t I - t^2/2 (beta^2) . (terms^(-1))] *
       Apply[Times, terms^(-1/2)]
   ] (* if {lambda, beta, alpha} exist, assume valid distribution *)


(* ================== Quantile for QuadraticFormDistribution =============== *)

(* NOTE:
  Quantile is not implemented for QuadraticFormDistribution[{a, b, c},
  {mu, sigma}, {lambda, beta, alpha}] for p > 1.  Since
  QuadraticFormDistribution is univariate,
  Quantile[QuadraticFormDistribution[], q] does have a unique definition.
  Unfortunately, only various approximations are available for p > 1. *)

QuadraticFormDistribution/: Quantile[QuadraticFormDistribution[{a_, b_, c_},
  {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], q_] :=
     Module[{noncentralparm, gamma, delta},
	gamma = a[[1, 1]] sigma[[1, 1]];
	delta = c - b[[1]]^2/(4 a[[1, 1]]);
	noncentralparm = (b[[1]]/(2 a[[1, 1]]) + mu[[1]])^2 / sigma[[1, 1]];
	If[Evaluate[noncentralparm == 0],
	   Evaluate[gamma Quantile[ChiSquareDistribution[1], q] + delta],
	   Evaluate[gamma Quantile[NoncentralChiSquareDistribution[1,
		    noncentralparm], q] + delta]
	]		
     ] /; Length[b] == 1 && NumberQ[q] && 0 <= q <= 1


(* NOTE: ExpectedValue of QuadraticFormDistribution is not implemented. *)

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not check parameters on Random so that Random will always
        evaluate and be efficient. *)

QuadraticFormDistribution/: Random[QuadraticFormDistribution[{a_, b_, c_},
	{mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_]] :=
  Module[{z = Random[MultinormalDistribution[mu, sigma]]},
    z.a.z + b.z + c
  ] (* if {lambda, beta, alpha} exist, assume valid distribution *)

QuadraticFormDistribution/: RandomArray[QuadraticFormDistribution[{a_, b_, c_},
        {mu_, sigma_}, {lambda_, beta_, alpha_}, lowertri_], dim_] :=
  Module[{m, array},
        m = If[VectorQ[dim], Apply[Times, dim], dim];
	array = RandomArray[MultinormalDistribution[mu, sigma], m];
	array = Map[(#.a.# + b.# + c)&, array];
	If[VectorQ[dim] && Length[dim] > 1,
           Fold[Partition[#1, #2]&, array, Reverse[Drop[dim, 1]] ],
           array  ]
  ] /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]





(* ==================== Hotelling T-squared Distribution =================== *)

HotellingTSquareDistribution/: ParameterQ[
   HotellingTSquareDistribution[p_, m_]] :=
	If[NumberQ[p], IntegerQ[p] && p > 0, True] &&
	If[NumberQ[m], IntegerQ[m] && m > 0, True] &&
	!TrueQ[m-p+1 <= 0]
	
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not condition DomainQ on ParameterQ so that DomainQ will always
        evaluate and be efficient. *)
HotellingTSquareDistribution/: DomainQ[HotellingTSquareDistribution[p_, m_],
	 list_?VectorQ] :=
	FreeQ[N[list], Complex] &&
	Scan[If[!TrueQ[# >= 0], Return[False]]&, list] =!= False
HotellingTSquareDistribution/: DomainQ[HotellingTSquareDistribution[p_, m_],
	 x_] :=
	FreeQ[N[x], Complex] && TrueQ[x >= 0]

HotellingTSquareDistribution/: Domain[HotellingTSquareDistribution[p_, m_]] :=
    Interval[{0, Infinity}] /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: PDF[HotellingTSquareDistribution[p_, m_], 
	x_?Negative] := 0 /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: PDF[HotellingTSquareDistribution[p_, m_], x_] :=
	(
	x^(p/2-1) / ( m^(p/2) (1 + x/m)^((m+1)/2) Beta[p/2, (m-p+1)/2] )	
	) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: CDF[HotellingTSquareDistribution[p_, m_], 
	x_?Negative] := 0 /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: CDF[HotellingTSquareDistribution[p_, m_], x_] :=
  	BetaRegularized[1/(1 + x/m), 1, (m-p+1)/2, p/2] /;
	  ParameterQ[HotellingTSquareDistribution[p, m]] && 1/(1 + x/m)<1/2

HotellingTSquareDistribution/: CDF[HotellingTSquareDistribution[p_, m_], x_] :=
  	BetaRegularized[x/(m+x), p/2, (m-p+1)/2] /;
	  ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: Mean[HotellingTSquareDistribution[p_, m_]] :=
	(m*p / (m-p-1)) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: Variance[HotellingTSquareDistribution[p_, m_]] :=
	(
	(2*(-1 + m)*m^2*p)/((-3 + m - p)*(-1 + m - p)^2)
	) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: StandardDeviation[
   HotellingTSquareDistribution[p_, m_]] :=
	Module[{var},
	  (
	  Sqrt[var]
	  ) /; FreeQ[var = Variance[HotellingTSquareDistribution[p, m]],
			Variance]
	] /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: Skewness[HotellingTSquareDistribution[p_, m_]] :=
   (
	(2*2^(1/2)*(-3 + m - p)^(1/2)*(-1 + m + p))/
		((-1 + m)^(1/2)*(-5 + m - p)*p^(1/2))
   ) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: Kurtosis[HotellingTSquareDistribution[p_, m_]] :=
    (
	3 +
	(12*((-3 + m - p)*(-1 + m - p)^2 + (-1 + m)*(-22 + 5*(1 + m - p))*p))/
	((-1 + m)*(-7 + m - p)*(-5 + m - p)*p)
    ) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: KurtosisExcess[
	HotellingTSquareDistribution[p_, m_]] :=
   (
	(12*((-3 + m - p)*(-1 + m - p)^2 + (-1 + m)*(-22 + 5*(1 + m - p))*p))/
	((-1 + m)*(-7 + m - p)*(-5 + m - p)*p)
   ) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: CharacteristicFunction[
	HotellingTSquareDistribution[p_, m_]] :=
   (
	Hypergeometric1F1[p/2, 1 + (-1 - m + p)/2, -I*m*t]
   ) /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: Quantile[HotellingTSquareDistribution[p_, m_],
	q] :=
   (
	m (1/InverseBetaRegularized[1, -q, (m-p+1)/2, p/2] - 1)
   ) /; ParameterQ[HotellingTSquareDistribution[p, m]] &&
	 NumberQ[q] && 0 <= q <= 1

HotellingTSquareDistribution/: ExpectedValue[f_Function,
		 HotellingTSquareDistribution[p_, m_], opts___?OptionQ] :=
   Module[{x, integral,
           assmp = Assumptions /. {opts} /. Options[ExpectedValue]},
     If[FreeQ[integral = Integrate[ f[x] *
				 PDF[HotellingTSquareDistribution[p, m], x],
                                        {x, 0, Infinity},
                        Assumptions -> Join[{p > 0, m > 0, m-p+1 > 0}, assmp]],
              Integrate],
        integral,
        integral /. x -> Unique[]       ]
   ] /; ParameterQ[HotellingTSquareDistribution[p, m]]

HotellingTSquareDistribution/: ExpectedValue[f_,
	 HotellingTSquareDistribution[p_, m_], x_, opts___?OptionQ] :=
  Module[{assmp = Assumptions /. {opts} /. Options[ExpectedValue]},
   Integrate[ f PDF[HotellingTSquareDistribution[p, m], x],
                {x, 0, Infinity},
                Assumptions -> Join[{p > 0, m > 0, m-p+1 > 0}, assmp]]
  ] /; ParameterQ[HotellingTSquareDistribution[p, m]]


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not check parameters on Random so that Random will always
        evaluate and be efficient. *)
 
HotellingTSquareDistribution/: Random[HotellingTSquareDistribution[p_, m_]] :=
	(m p/(m-p+1)) Random[FRatioDistribution[p, m-p+1]]

HotellingTSquareDistribution/: RandomArray[HotellingTSquareDistribution[p_, m_],
	dim_] :=
   (
	(m p/(m-p+1)) RandomArray[FRatioDistribution[p, m-p+1], dim]
   ) /; (IntegerQ[dim] && dim > 0) || VectorQ[dim, (IntegerQ[#] && # > 0)&]
	

(* =================== Multivariate StudentT Distribution =================  *)

MultivariateTDistribution /: ParameterQ[
	MultivariateTDistribution[r_, m_, l_]] :=
  Module[{i, j},
     MatrixQ[r] &&  If[NumberQ[m], IntegerQ[m] && m > 0, True] &&
     r === Transpose[r] &&
     If[Apply[And, Map[NumberQ, Flatten[N[r]] ]], TrueQ[Det[N[r]] > 0], True] &&
     Apply[And,
	 Flatten[Table[
	    With[{z = r[[i, j]]},
	      If[NumberQ[z], If[i==j, z==1, Abs[z] < 1], True]
	    ], {i, Length[r]}, {j, i, Length[r]}]]
     ]
  ]

validMultivariateTQ[r_, m_, warnflag_:False] :=
  Module[{i, j},
     MatrixQ[r] &&  If[NumberQ[m], IntegerQ[m] && m > 0, True] &&
     If[r === Transpose[r], True,
         If[TrueQ[warnflag], Message[MultivariateTDistribution::cmsym]]; False] &&
     If[Apply[And, Map[NumberQ, Flatten[N[r]] ]], TrueQ[Det[N[r]] > 0], True] &&
     Apply[And,
         Flatten[Table[
            With[{z = r[[i, j]]},
              If[NumberQ[z], If[i==j, z==1, Abs[z] < 1], True]
            ], {i, Length[r]}, {j, i, Length[r]}]]
     ]
  ]

MultivariateTDistribution::cmsym =
"The correlation matrix must be symmetric.";

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not condition DomainQ on ParameterQ so that DomainQ will always
        evaluate and be efficient. *)
MultivariateTDistribution/: DomainQ[MultivariateTDistribution[r_, m_, l___],
        list_?MatrixQ] :=
   MatchQ[Dimensions[list], {_, Length[r]}] && FreeQ[N[list], Complex]
MultivariateTDistribution/: DomainQ[MultivariateTDistribution[r_, m_, l___],
        x_?VectorQ] :=
   TrueQ[Length[x] == Length[r] && FreeQ[N[x], Complex]]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* compute lower triangular matrix for use in calculations involving
	MultinormalDistribution *)
MultivariateTDistribution[r_:{{1}}, m_:1] :=
  Module[{u},
    (
    MultivariateTDistribution[r, m, Transpose[u] ]
    ) /; FreeQ[u = CholDecomp[r], CholDecomp]
  ] /; validMultivariateTQ[r, m, True]

(* hide lower triangular matrix using Format *)
MultivariateTDistribution /:
	Format[MultivariateTDistribution[r_, m_, l_]] :=
	MultivariateTDistribution[Short[r], m]

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

MultivariateTDistribution/: Domain[
   MultivariateTDistribution[r_?MatrixQ, m_, l___]] :=
	Table[Interval[{-Infinity, Infinity}], {Length[r]}] /; 
	Apply[Equal, Dimensions[r]]

MultivariateTDistribution/: PDF[MultivariateTDistribution[r_, m_, l___], x_] :=
	Module[{rinv = Inverse[r], p = Length[r]},	
	  (1 + x.rinv.x/m)^(-(m+p)/2) Gamma[(m+p)/2] /
		( (m Pi)^(p/2) Gamma[m/2] PowerExpand[Sqrt[Det[r]]] )
	] /; ({l} =!= {} || validMultivariateTQ[r, m]) &&
	     VectorQ[x] && TrueQ[Length[x] == Length[r]]

(*  - - - - - - - - - CDF[MultivariateTDistribution[], ]  - - - - - - - - - *)



(* special case code for diagonal r.
   if r is diagonal, the CDF can be found as a single integration involving
   univariate normals 
*)


diagMVT[m_, x_ ,opts___]:=Module[{s, v, integrand, res},
	If[m<=400
	  	,
	  	(* this definition is generally faster, but has numerical 
      		   stability problems for large values of m *)
	  	integrand=(2^(1 - m/2))/Gamma[m/2]*s^(m - 1)*Exp[-(s^2)/2]*Apply[Times,
	  		Table[1/2(1 + Erf[s*x[[i]]/Sqrt[2*m]]), {i, Length[x]}]];
	  	NIntegrate[integrand, {s, 0, Infinity},opts]
	  	,
	  	(* the change of variable from s to v maps the integration into 
		   Interval[{0,1}] and makes the integration more numerically stable.
		   there is a speed trade-off, however, since the integrand involves more nested 
 		   special functions *)
	  	integrand=Apply[Times,Table[1/2(1 + Erf[s*x[[i]]/Sqrt[2*m]]), 
	  		{i, Length[x]}]] /. s->Sqrt[2]*Sqrt[InverseGammaRegularized[m/2, 0, v]];
	  	NIntegrate[integrand, {v, 0, 1},opts]
     		]
     	   ]
     	   

(* p-dimensional case, p >= 2, special structure r:
	r = {{1,     l1 l2,     l1 l3,     l1 l4, ...},
	     {l2 l1, 1,		l2 l3,     l2 l4, ...},
	     {l3 l1, l3 l2,     1,         l3 l4, ...}, ...}.
   where Abs[li] <= 1, i = 1, 2, 3, ... .

The integrand using this method becomes very unstable for degrees of freedom over about 70.
Some problems also exist for negative correlations, at least in the 2D case.  
For these reasons, tryTongMVT is currently not called by CDF[MultivariateTDistribution[]].
The code is left here because it could possibly be useful for high dimensional MultivariateTs
with a small number of degrees of freedom and the special structure mentioned above.
-darreng 
*)


tryTongMVT[r_, m_, x_, lowertri_List, opts___] := 
  Module[{p = Length[x], lambda, arg, product, res}, 
   lambda = If[p === 2, 
	Sqrt[r[[1,2]]], 
	(* p > 2 *)
	Module[{lam, l, temp, soln, zeros, acc, prec, delta, i, j}, 
	   lam = Array[l, p]; 
	   temp = lam; 
           Internal`DeactivateMessages[
		soln = Solve[Take[Flatten[Table[r[[i,j]] == Times @@ temp[[{i, j}]], 
             		{i, p - 1}, {j, i + 1, p}], 1], p], lam]]; 
           If[FreeQ[soln, Solve] && Length[soln] === 2, 
		lam = (lam /. soln)[[1]]; 
          	temp = lam; 
		zeros = Flatten[Table[r[[i,j]] - Times @@ temp[[{i, j}]], 
             		{i, p - 1}, {j, i + 1, p}]]; 
		acc = Max[Accuracy[r], $MachinePrecision] - 1; 
		prec = Max[Precision[r], $MachinePrecision] - 1; 
		delta = Max[10^(-acc), Abs[Det[r]]/10^prec]; 
          	zeros = Chop[zeros, delta]; 
		If[And @@ (N[Abs[#1]] <= 1 & ) /@ lam && 
			And @@ (#1 == 0 & ) /@ zeros, 
			lam, 
			$Failed], 
			$Failed](* end If[FreeQ[soln, Solve] && ...] *)
			]  (* end Module *)
			]; (* end If[p == 2, ] *)
    		If[lambda =!= $Failed, 
    		(* s is distributed according to ChiDistribution[m];
	       	s^2 is distributed according to ChiSquareDistribution[m] *)
		arg = (x*(s/Sqrt[m]) - lambda*z)/Sqrt[1 - lambda^2]; 
      		product = Times @@ ((Erf[#1/Sqrt[2]] + 1)/2 & ) /@ arg*
   			PDF[ChiDistribution[m], s]; 
      	If[FreeQ[res = NIntegrate[Evaluate[product/(E^(z^2/2)*(2*Pi)^(1/2))], 
          	{z, -Infinity, Infinity}, {s, 0, Infinity}, opts], NIntegrate], 
		res, 
       		$Failed], 
	$Failed]]


(* the following method is for higher dimensional multivariate t CDFs.
   the idea is similar to the method for multivariate normals
   
   Reference:
   Alan Genz and Frank Bretz, "Comparison of Methods for the Computation of Multivariate t-Probabilities,"
   revised version published in J. Comp. Graph. Stat. 11 (2002), pp. 950-971. 
   
   also available through Genz's Research link on
   http://www.math.wsu.edu/math/faculty/genz/homepage
*)

tryGenzMVT[r_, m_, x_, lowertri_List, opts___] := 
	Module[{wvars, p, integrand, intRange, evals, xscale, s, v, phiInv,
		wp = WorkingPrecision /. {opts}}, 
      		p = Length[x];
      		If[wp=!=MachinePrecision && Precision[{mu, sigma, x, lowertri}]=!=MachinePrecision
		   ,(* for efficiency, only use exact form if bignums are used *)
		   phiInv=(Sqrt[2]*InverseErf[0, -1 + 2#])&
		   , (* for MachinePrecision, use interpolation of InverseErf *)
		   phiInv=phiInverse];
      		wvars = Table[Unique[w], {p}];
      		xscale = s/Sqrt[m]*x;
      		evals = Table[0, {p}];
      		evals[[1]] = (1/2)*(1 + Erf[(xscale[[1]]/lowertri[[1, 1]])/
      			Sqrt[2]]);
      		Do[evals[[i]] = (1/2)*(1 + Erf[((xscale[[i]] - 
			Apply[Plus, Table[lowertri[[i, j]]*phiInv[wvars[[j]]*evals[[j]]], 
				{j, i - 1}]])/lowertri[[i, i]])/Sqrt[2]]), {i, 2, p}];
      		
      		If[m<=550
 		  ,
 		  (* this definition is generally faster, but has numerical 
      		   stability problems for large values of m *)
 		  integrand=s^(m-1)*Exp[-(s^2)/2]*Apply[Times,evals];
 		  intRange=Join[{{s,0,Infinity}},Map[{wvars[[#]],0,1}&,Range[p-1]]];
 		  2^(1-m/2)/Gamma[m/2]*Apply[NIntegrate[integrand,##,opts]&,intRange]
 		  ,
 		  (* the change of variable from s to v maps the inner-most integration into 
 		  Interval[{0,1}] and makes the integration more numerically stable.
 		  there is a speed trade-off, however, since the integrand involves more nested 
 		  special functions *)
 		  integrand=Apply[Times,evals]/.s->Sqrt[2]*Sqrt[InverseGammaRegularized[m/2,0,v]];
 		  intRange=Join[{{v,0,1}},Map[{wvars[[#]],0,1}&,Range[p-1]]];
 		  Apply[NIntegrate[integrand,##,opts]&,intRange]
 		  ]
 		 ] 


(* CDF definition for dimension 2 or higher MultivariateTs such that r is 
   neither diagonal nor singular.  Because of numerical instabilities that 
   arise in the extension of Tong's method to MultivariateTs, only the method 
   due to Genz is used.
*)

MultivariateTDistribution /: CDF[MultivariateTDistribution[r_, m_, lowertri___], 
   x_?VectorQ, opts___] := 
	Module[{p, lambda, z, product, s, accgoal, compiled, gpoints, mxpoints, 
	   mxrec, meth, mnrec, precgoal, sdepth, wprec, nint, ok, delta, agWarningQ, pgWarningQ},
      	   p = Length[r];
      	   {accgoal, compiled, gpoints, mxpoints, mxrec, meth, mnrec, precgoal, sdepth, 
      	     wprec} = {AccuracyGoal, Compiled, GaussPoints, MaxPoints, MaxRecursion, Method, 
      	     MinRecursion, PrecisionGoal, SingularityDepth, WorkingPrecision} /. {opts} /. Options[CDF];
      	   If[! (accgoal === Automatic || validGoalQ[accgoal]), Message[CDF::accg, accgoal]; accgoal = Automatic];
      	   If[! (precgoal === Automatic || validGoalQ[precgoal]), Message[CDF::precg, precgoal]; 
      	   	precgoal =  Automatic]; 
      	  (If[! FreeQ[nint, Complex],(*Note : agWarningQ and pgWarningQ aren't used here because 
      	  	they are checked for prior to NIntegrate.The purpose of these flags is to signal 
      	  	whether appropriate warning Messages (i.e., CDF::accg and CDF::precg) should be issued.*)
	   {delta, agWarningQ, pgWarningQ} = Delta[nint, accgoal, precgoal];
           nint = Chop[nint, delta]];
    	   N[Re[nint]]) /; (ok = FreeQ[
    	   		      nint = If[diagonalmatrixQ[r]
    	   			,
    	   			diagMVT[m, x, AccuracyGoal -> accgoal, Compiled -> compiled,
    	   			GaussPoints -> gpoints, MaxPoints -> mxpoints, MaxRecursion -> mxrec,
    	   			Method -> meth, MinRecursion -> mnrec, PrecisionGoal -> precgoal,
    	   			SingularityDepth -> sdepth, WorkingPrecision -> wprec]
    	   			,
    	   			tryGenzMVT[r, m, x, lowertri, AccuracyGoal -> accgoal, 
    	   			Compiled -> compiled, GaussPoints -> gpoints, MaxPoints -> mxpoints, 
    	   			MaxRecursion -> mxrec, Method -> meth, MinRecursion -> mnrec, 
                        	PrecisionGoal -> precgoal, SingularityDepth -> sdepth, 
                        	WorkingPrecision -> wprec]]
                              , 
               		      NIntegrate];
          If[ok, True, Message[CDF::nint]; False])
          	] /; ({lowertri} =!= {} || validMultivariateTQ[r,m]) && Length[r] >= 2 && 
          	TrueQ[Length[x] == Length[r]] && Apply[And, Map[((NumberQ[#] || 
          	Head[#] === DirectedInfinity) && FreeQ[#, Complex]) &, N[Flatten[{x, r, m}]]]]


(* 1-dimensional case *)
MultivariateTDistribution/: CDF[MultivariateTDistribution[r_, m_, lowertri___],
         x_?VectorQ] :=
	BetaRegularized[m/(m+x[[1]]^2),m/2,1/2]/2 /;
	  ({lowertri} =!= {} || validMultivariateTQ[r, m]) &&
       (Length[r] == 1 == Length[x]) && FreeQ[x[[1]], Complex] && x<0
MultivariateTDistribution/: CDF[MultivariateTDistribution[r_, m_, lowertri___],
         x_?VectorQ] :=
	(1 + Sign[x[[1]]] BetaRegularized[m/(m+x[[1]]^2), 1, m/2, 1/2])/2 /;
	  ({lowertri} =!= {} || validMultivariateTQ[r, m]) &&
       (Length[r] == 1 == Length[x]) && FreeQ[x[[1]], Complex]	


(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

MultivariateTDistribution/: Mean[MultivariateTDistribution[r_?MatrixQ,
   m_, l___]] :=
	Table[0, {Length[r]}] /; Apply[Equal, Dimensions[r]]

(* StandardDeviation yields a vector in multivar case *)
MultivariateTDistribution/: StandardDeviation[MultivariateTDistribution[
   r_?MatrixQ, m_, l___]] :=
	Sqrt[m/(m-2)] Table[1, {Length[r]}] /; Apply[Equal, Dimensions[r]] &&
					       !TrueQ[IntegerQ[m] && m <= 2]	

(* Variance yields a vector in multivar case *)
MultivariateTDistribution/: Variance[MultivariateTDistribution[r_?MatrixQ,
   m_, l___]] :=
	m/(m-2) Table[1, {Length[r]}] /; Apply[Equal, Dimensions[r]] &&
					 !TrueQ[IntegerQ[m] && m <= 2]

MultivariateTDistribution/: CovarianceMatrix[MultivariateTDistribution[
	r_?MatrixQ, m_, l___]] :=
	m/(m-2) r /; Apply[Equal, Dimensions[r]] &&
		     !TrueQ[IntegerQ[m] && m <= 2]

MultivariateTDistribution/: CorrelationMatrix[MultivariateTDistribution[
	r_?MatrixQ, m_, l___]] :=
	r /; Apply[Equal, Dimensions[r]]

MultivariateTDistribution/: Skewness[MultivariateTDistribution[r_?MatrixQ,
   m_, l___]] := 
	Table[0, {Length[r]}] /; Apply[Equal, Dimensions[r]]

MultivariateTDistribution/: MultivariateSkewness[MultivariateTDistribution[
	r_, m_, l___]] := 0

MultivariateTDistribution/: Kurtosis[MultivariateTDistribution[r_?MatrixQ,
   m_, l___]] :=
	(6/(m-4) + 3) Table[1, {Length[r]}] /; Apply[Equal, Dimensions[r]] &&
					       !TrueQ[IntegerQ[m] && m <= 4]

MultivariateTDistribution/: MultivariateKurtosis[MultivariateTDistribution[
	r_?MatrixQ, m_, l___]] := 
	With[{p = Length[r]},
		p(p + 2)(m - 2)/(m - 4)
	] /; !TrueQ[IntegerQ[m] && m <= 4]

MultivariateTDistribution/: KurtosisExcess[MultivariateTDistribution[
	r_?MatrixQ, m_, l___]] :=
		6/(m-4) Table[1, {Length[r]}] /; Apply[Equal, Dimensions[r]] &&
					         !TrueQ[IntegerQ[m] && m <= 4]

MultivariateTDistribution/: MultivariateKurtosisExcess[
	MultivariateTDistribution[r_?MatrixQ, m_, l___]] :=
	  With[{p = Length[r]},
		2 p (p + 2)/(m - 4)
	  ] /; !TrueQ[IntegerQ[m] && m <= 4]

(* NOTE:
  CharacteristicFunction[dist, t] for a random vector x is defined as the
        expected value of Exp[I t.x], where Dimensions[t] == Dimensions[x].
*)
(* Johnson & Kotz, p. 136 *)
MultivariateTDistribution/: CharacteristicFunction[
                MultivariateTDistribution[r_, m_, l___], t_] :=
	Module[{x = Unique[], head, int},
	  (
		If[head === If,
                        int[[2]] / Gamma[m/2],
                        int / Gamma[m/2]
	 	]
	  ) /; (head = Head[int = Integrate[x^(m/2-1) Exp[-x - m t.r.t / (4 x)],
			{x, 0, Infinity}]];
	        If[head =!= If, FreeQ[int, Integrate], True])
 	] /; ({l} =!= {} || validMultivariateTQ[r, m]) &&
	     VectorQ[t] && TrueQ[Length[t] == Length[r]]

(* NOTE:
Quantile is not implemented for MultivariateTDistribution[r, m] for p > 1.
Actually since there is no unique vector x such that
q = CDF[MultivariateTDistribution[r, m], x], it is difficult to even
define Quantile[MultivariateTDistribution[r, m], q].  One possibility is
the so-called "equicoordinate one-sided percentage points", i.e., the
quantile is {c, c, ..., c} such that CDF[MultivariateTDistribution[r, m],
{c, c, ..., c}] = q. *)

MultivariateTDistribution/: Quantile[MultivariateTDistribution[r_, m_, l_],
   q_] :=
   {Quantile[StudentTDistribution[m], q]} /;
	ParameterQ[MultivariateTDistribution[r, m]] &&
	NumberQ[q] && (0 <= q <= 1)

MultivariateTDistribution/: EllipsoidQuantile[MultivariateTDistribution[
	r_, m_, l___], q_] :=
  Module[{ellipsoid, quantile, p},
    (
    ellipsoid
    ) /; (p = Length[r];
	  quantile = Quantile[FRatioDistribution[p, m], q];
	  ellipsoid = ellipsoidalLocus[Table[0, {p}], quantile p r];
	  If[ellipsoid === $Failed,
		Message[EllipsoidQuantile::mvarteig, quantile p r] ];
	  ellipsoid =!= $Failed)
  ] /; QuantileQ[q] && ({l} =!= {} || validMultivariateTQ[r, m])

EllipsoidQuantile::mvarteig =
"Unable to find eigensystem of ``."

MultivariateTDistribution/: RegionProbability[MultivariateTDistribution[
	r_?MatrixQ, m_, l___], EllipsoidQuantile[mu1_, radii_]] :=
		0 /; radii === Table[0, {Length[r]}]

MultivariateTDistribution/: RegionProbability[MultivariateTDistribution[
	r_?MatrixQ, m_, l___], EllipsoidQuantile[mu1_, radii_]] :=
		1 /; radii === Table[Infinity, {Length[r]}]
	
MultivariateTDistribution/: RegionProbability[MultivariateTDistribution[
        r_, m_, l___],
        Ellipsoid[mu_, radii_?VectorQ, dir___?MatrixQ]] :=
   Module[{p, internaldir, normalized, diagonalizedMatrix, acc, prec, delta,
		 scaled, quantile, diagonal},
     (
        CDF[ FRatioDistribution[p, m], quantile ]
     ) /; ({l} =!= {} || validMultivariateTQ[r, m]) &&
          (p = Length[r];  TrueQ[mu == Table[0, {p}]]) &&
          (  internaldir = If[{dir}==={}, IdentityMatrix[p], dir];
             normalized = r.Transpose[internaldir];
	     diagonalizedMatrix = internaldir.normalized;
	     acc = Max[Accuracy[diagonalizedMatrix], $MachinePrecision]-1;
             prec = Max[Precision[diagonalizedMatrix], $MachinePrecision]-1;
             delta = Max[10^(-acc), Abs[Det[diagonalizedMatrix]] 10^(-prec)];	
             diagonalizedMatrix = Simplify[Chop[diagonalizedMatrix, delta]];
             If[ MatchQ[ diagonalizedMatrix, DiagonalMatrix[Table[_, {p}]] ],
		True,
		Message[RegionProbability::mvartell,
			 mu, radii, internaldir, r, m];
		False] &&
          (diagonal = DiagonalElements[diagonalizedMatrix]/radii^2;
	   acc = Max[Accuracy[diagonal], $MachinePrecision]-3;
           prec = Max[Precision[diagonal], $MachinePrecision]-3;
           delta = Max[10^(-acc), Apply[Times, Abs[diagonal]] 10^(-prec)];
	   Apply[Equal, Chop[diagonal, delta]]) &&
          (  scaled = Transpose[internaldir].DiagonalMatrix[radii^2];
             quantile = Scan[If[ !TrueQ[#[[1]]==0], Return[#[[2]]/#[[1]]/p] ]&,
                       Transpose[{Flatten[normalized], Flatten[scaled]}] ];
             quantile =!= Null)
          )
   ]

RegionProbability::mvartell = 
"Ellipsoid[`1`, `2`, `3`] does not correspond to a constant-probability
contour of MultivariateTDistribution[`4`, `5`]."

(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)

MultivariateTDistribution/: ExpectedValue[f_Function,
	 MultivariateTDistribution[r_, m_, lowertri_], opts___?OptionQ] :=
  Module[{n, (* number of arguments of function f *)
	  p = Length[r], (* dimensionality of MultinormalDistribution *)
	  xvec, x, assmp = Assumptions /. {opts} /. Options[ExpectedValue],
	  correlations = AboveDiagonalElements[r],
	  arglist, integral},
   (
    xvec = Array[x, p]; 
    assmp = Flatten[Join[	Map[(#^2 < 1)&, correlations],
	 			{Det[r] > 0, m > 0},
	 			assmp			] /. True -> {}];
    arglist = Prepend[ Map[{#, -Infinity, Infinity}&, xvec],
		       Apply[f, xvec] *
		        PDF[MultivariateTDistribution[r, m, lowertri], xvec] ];
    If[assmp =!= {}, AppendTo[arglist, Assumptions -> assmp]];
    If[FreeQ[integral = Apply[Integrate, arglist], Integrate],
       integral,
       unique = Table[Unique[], {p}];	
       integral /. Thread[Rule[xvec, unique]]		] (* end If *)
   ) /; (
	 n = If[Length[f]==1,
	        (* Function with only a body *)
	        Max[Cases[{f}, Slot[z_]->z, Infinity]],
	        (* Function with a list of formal parameters *)
	        Length[f[[1]]]	];
	 n <= p)
  ]
	
MultivariateTDistribution/: ExpectedValue[f_,
	MultivariateTDistribution[r_, m_, lowertri_], xvec_?VectorQ,
	opts___?OptionQ] :=			
  Module[{assmp = Assumptions /. {opts} /. Options[ExpectedValue],
	  correlations = AboveDiagonalElements[r], arglist},
    assmp = Flatten[Join[	Map[(#^2 < 1)&, correlations],      	
                                {Det[r] > 0, m > 0},
                                assmp			] /. True -> {}];
    arglist = Prepend[ Map[{#, -Infinity, Infinity}&, xvec],
	 	       f PDF[MultivariateTDistribution[r, m, lowertri], xvec] ];
    If[assmp =!= {}, AppendTo[arglist, Assumptions -> assmp]];
    Apply[Integrate, arglist]
  ] /; Length[xvec] <= Length[r] 
	
(* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *)
(* NOTE: do not check parameters on Random so that Random will always
        evaluate and be efficient. *)

MultivariateTDistribution/: Random[MultivariateTDistribution[r_, m_, l_]] :=
   Module[{x = Random[MultinormalDistribution[Table[0, {Length[r]}], r, l]],
	   s = Random[ChiSquareDistribution[m]]},
	x Sqrt[m/s]
   ] (* if l exists, assume valid distribution *)

MultivariateTDistribution/: RandomArray[
	MultivariateTDistribution[r_, m_, l_], dim_] :=
   Module[{x, s},
     x = RandomArray[MultinormalDistribution[Table[0, {Length[r]}], r, l], dim];
     s = RandomArray[ChiSquareDistribution[m], dim];
     x Sqrt[m/s]
   ] (* if l exists, assume valid distribution *)
     

(* ====================================================================== *)
End[]

SetAttributes[ MultinormalDistribution, ReadProtected];
SetAttributes[ WishartDistribution, ReadProtected];    
SetAttributes[ HotellingTSquareDistribution, ReadProtected];    
SetAttributes[ QuadraticFormDistribution, ReadProtected];    
SetAttributes[ MultivariateTDistribution, ReadProtected];    

(* Protect DescriptiveStatistics symbols *)
Protect[Mean, Variance, StandardDeviation, Skewness, Kurtosis,
         KurtosisExcess]

(* Protect MultinormalDistribution symbols. *)
Protect[MultinormalDistribution, WishartDistribution,
	HotellingTSquareDistribution, QuadraticFormDistribution,
	MultivariateTDistribution]


EndPackage[]

